# WMLDeck.pm - The Class that provides a WMLDeck Object
# Created by James Pattie, 12/21/2000.

# Copyright (c) 2000 PC & Web Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package HTMLObject::WMLDeck;
use strict;
use HTMLObject::WMLCard;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();
	
$VERSION = '2.27';

# new 
# instantiates an instance of a WMLDeck Object
# requires: id 
# optional: lang (language code), displayLang (bool), digestURL
# if displayLang = true, then the xml:lang="" attribute will be displayed else it is left out.
sub new
{
  my $that = shift;
  my $class = ref($that) || $that;
  my $self = bless {}, $class;
  my %args = ( id => "deck0", lang => "en-us", displayLang => "false", digestURL => "", @_ );
  my $errStr = "HTMLObject::WMLDeck->new() - Error!<br/>\n";

  $self->{id} = $args{id};
  $self->{lang} = $args{lang};
  $self->{displayLang} = $args{displayLang};
  $self->{digestURL} = $args{digestURL};
  $self->{cards} = {};
  $self->{cardsOrder} = [];
  $self->{currentCard} = "";
  $self->{head} = "";  # the contents of the head section.
  $self->{template} = "";  # the contents of the template section.
  
  # do validation
  $self->{error} = !$self->isValid;  
  if ($self->{error})
  {
    $self->{errorString} = $errStr . $self->{errorString};
  }

  return $self;
}

# isValid - Returns 0 or 1 to indicate if the object is valid.
sub isValid
{
  my $self = shift;
  my $error = 0;
  my $errorString = "";
  my $errStr = "HTMLObject::WMLDeck->isValid() - Error!<br/>\n";

  # do validation code here.
  if (length $self->{id} == 0)
  {
    $error = 1;
    $errorString .= "id = '$self->{id}' is invalid!<br/>\n";
  }
  if ($self->{displayLang} !~ /^(true|false)$/)
  {
    $error = 1;
    $errorString .= "displayLang = '$self->{displayLang}' is invalid!<br/>\n";
  }
  if (length $self->{lang} == 0)
  {
    $error = 1;
    $errorString .= "lang = '$self->{lang}' is invalid!<br/>\n";
  }

  $self->{error} = $error;
  $self->{errorString} = $errStr if $error;
  $self->{errorString} .= $errorString;

  return !$error;
}

sub DESTROY
{
  my $self = shift;
}

sub AUTOLOAD
{
  my $self = shift;
  my $type = ref($self) || die "$self is not an object";
  my $name = $AUTOLOAD;
  $name =~ s/.*://;	# strip fully-qualified portion
  unless (exists $self->{$name})
  {
    die "Can't access `$name' field in object of class $type";
  }
  if (@_)
  {
    return $self->{$name} = shift;
  }
  else
  {
    return $self->{$name};
  }
}

# setError
# requires: errorString
# returns: nothing
sub setError
{
  my $self = shift;
  my %args = ( errorString => "", @_ );
  my $errorString = $args{errorString};
  
  $self->{errorString} = $errorString;
  $self->{error} = 1;
}

# didErrorOccur
# returns the value of error.
sub didErrorOccur
{
  my $self = shift;

  return $self->{error};
}

# errorMessage
# returns the value of errorString
sub errorMessage
{
  my $self = shift;
  
  return $self->{errorString};
}

#createCard
# takes: id, title, newcontext, ordered, onenterforward, onenterbackward, ontimer
# returns: 1=OK, 0=Error, -1=Already exists
sub createCard
{
  my $self = shift;
  my %args = ( id => "card0", title => "", newcontext => "false", ordered => "true", onenterforward => "", onenterbackward => "", ontimer => "", @_ );
  my $errStr = "HTMLObject::WMLDeck->createCard()  - Error!<br/>\n";
  my $id = $args{id};
  my $title = $args{title};
  my $newcontext = $args{newcontext};
  my $ordered = $args{ordered};
  my $onenterforward = $args{onenterforward};
  my $onenterbackward = $args{onenterbackward};
  my $ontimer = $args{ontimer};
  
  if (length $id == 0)
  {
    $self->{error} = 1;
    $self->{errorString} = $errStr . "id = '$id' is invalid!<br/>\n";
    return 0;
  }
  
  if (exists $self->{cards}->{$id})
  {
    return -1;  # already exists!
  }
  
  # create the WMLCard object
  my $wmlCardObj = HTMLObject::WMLCard->new(id => $id, title => $title, newcontext => $newcontext, ordered => $ordered, onenterforward => $onenterforward,
                                            onenterbackward => $onenterbackward, ontimer => $ontimer);
  if ($wmlCardObj->didErrorOccur)
  {
    $self->{error} = 1;
    $self->{errorString} = $errStr. $wmlCardObj->errorMessage;
    return 0;
  } 
  
  # insert into the cards hash
  $self->{cards}->{$id} = $wmlCardObj;
  
  # append this card to the end of the list so I know the order to display them in.
  push @{$self->{cardsOrder}}, $id;
  
  # set the focus to this card
  $self->{currentCard} = $id;
  
  return 1;
}

# setFocus
# takes: card id to set focus to or nothing to just return current card
# returns: card that previously had focus or "" if no previous card
sub setFocus
{
  my $self = shift;
  my $card = shift;
  my $errStr = "HTMLObject::WMLDeck->setFocus()  - Error!<br/>\n";
  my $oldCard = $self->{currentCard};
  
  if (length $card == 0)
  {
    return $oldCard;
  }
  else
  {
    if (!exists $self->{cards}->{$card})
    {
      $self->{error} = 1;
      $self->{errorString} = $errStr . "card = '$card' does not exist!<br/>\n";
      return $oldCard;
    }
    $self->{currentCard} = $card;
  }
  
  return $oldCard;
}

# print
# takes: text to place in content of current card
# returns: nothing
sub print
{
  my $self = shift;
  my $content = shift;
  my $errStr = "HTMLObject::WMLDeck->print()  - Error!<br/>\n";
  
  if (length $self->{currentCard} == 0 || !exists $self->{cards}->{$self->{currentCard}})
  {
    $self->{error} = 1;
    $self->{errorString} = $errStr . "'$self->{currentCard}' is invalid!<br/>\n";
    return;
  }
  
  # print the content into the card.
  $self->{cards}->{$self->{currentCard}}->print($content);
}

# printHead
# takes: text to place in head of this deck
# returns: nothing
sub printHead
{
  my $self = shift;
  my $content = shift;
  my $errStr = "HTMLObject::WMLDeck->printHead()  - Error!<br/>\n";
  
  $self->{head} .= $content;
}

# printTemplate
# takes: text to place in the template of this deck
# returns: nothing
sub printTemplate
{
  my $self = shift;
  my $content = shift;
  my $errStr = "HTMLObject::WMLDeck->printTemplate()  - Error!<br/>\n";
  
  $self->{template} .= $content;
}

# display
# takes: nothing
# returns: string containing <wml> .. </wml> output to display
sub display
{
  my $self = shift;
  my $errStr = "HTMLObject::WMLDeck->display()  - Error!<br/>\n";
  my $output = "";
  
  $output .= "<wml";
  $output .= " xml:lang=\"$self->{lang}\"" if ($self->{displayLang} eq "true");
  $output .= ">\n";
  
  if (length $self->{head} > 0)
  {
    $output .= "  <head>\n";
    (my $head = $self->{head}) =~ s/^(.*)$/    $1/mg;  # indent 4 spaces.
    $output .= $head;
    $output .= "  </head>\n";
  }
  
  if (length $self->{template} > 0)
  {
    $output .= "  <template>\n";
    (my $template = $self->{template}) =~ s/^(.*)$/    $1/mg;  # indent 4 spaces.
    $output .= $template;
    $output .= "  </template>\n";
  }
  
  foreach my $card (@{$self->{cardsOrder}})
  {
    # print this card into the output stream
    $output .= $self->{cards}->{$card}->display();
    if ($self->{cards}->{$card}->didErrorOccur)
    {
      $self->{error} = 1;
      $self->{errorString} = $errStr . $self->{cards}->{$card}->errorMessage;
      return "";
    }
  }
  
  $output .= "</wml>\n";
  
  return $output;
}

1;
__END__

=head1 NAME

WMLDeck - Object used to build a WMLDeck Object Class.

=head1 SYNOPSIS

  use HTMLObject::WMLDeck;
  my $obj = HTMLObject::WMLDeck->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

WMLDeck is a WMLDeck class.

=head1 Exported FUNCTIONS

  scalar new(lang, displayLang)
    Creates a new instance of the HTMLObject::WMLDeck
    object.

  bool isValid(void)
    Returns 0 or 1 to indicate if the object is valid.  The error will be
    available via errorMessage().

  void setError(errorString)
    sets error and errorString to indicate an error occurred.

  scalar didErrorOccur(void)
    Returns the value of error.

  scalar errorMessage(void)
    Returns the value of errorString.
    
  int createCard(id, title, newcontext, ordered, onenterforward, 
                  onenterbackward, ontimer)
    Creates a new WMLCard Object and stores it in the cards hash under
    the id value.  The focus is automatically set to this new card
    until changed by the user.  Returns -1 if the card already exists.
    
  scalar setFocus(scalar)
    This sets the focus to the new card id and returns the old card id
    that previously had the focus.  If no card was previously created it
    will return an empty string.  If no value is given, then it just
    returns the card that has the focus.
    
  void print(scalar)
    Prints the contents of the string passed in into the card that 
    currently has the focus.  If no card is defined, then we blow an
    error.
    
  void printHead(scalar)
    Prints the contents of the string passed in into the head section
    of the deck.
    
  void printTemplate(scalar)
    Prints the contents of the string passed in into the template 
    section of the deck.
    
  scalar display(void)
    Returns the <wml>...</wml> string that represents the contents of
    this WMLDeck Object.


  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:htmlobject@pcxperience.com)

=head1 SEE ALSO

perl(1), HTMLObject::WAP(3)

=cut
