#!/usr/bin/perl
# form.cgi - Test the HTMLObject::Form module

use strict;
use HTMLObject::Form;
use HTMLObject::Normal;
use HTMLObject::CGILib;
use HTMLObject::Widgets;

my $widgets = HTMLObject::Widgets->new();

my @options = (
  { label => "First", data => ["1", "2", "2005-01-01", "#000000"] },
  { label => "Second", data => ["2", "3", "2005-02-02", "#1111FF"] },
  { label => "Third", data => ["3", "This isn't your code.", "2005-03-03", "#00FF00"] },
);

my %testOptions = (
  names => [ "One", "Two", "Three" ],
  values => [ 1, 2, 3 ],
);

my %result = $widgets->generatePopulator(name => "test", rows => 5,
  required => 3, options => \@options, selectLocation => "left",
  #htmlTemplate => "<td>#LFRI=field_x_0#</td>\n<td>#LFRI=field_x_1#</td>\n<td>#LFRI=field_x_2#</td>\n<td>#LFRI=field_x_3#</td>",
  manual => 0, customLabel => "text",
  optionsTypes => [
    { -Type => "text", -Label => "first", -Value => "" },
    { -Type => "select", -Label => "select", -Value => "", -Options => \%testOptions },
    { -Type => "datePicker", -Label => "date", -Value => "", -WidgetOptions => {year => "2005"} },
    { -Type => "colorPicker", -Label => "color", -Value => "", -WidgetOptions => {} },
    ],
  debugLevel => 0,
  );
if ($widgets->error)
{
  die $widgets->errorMessage;
}

my %commands = ( display => "Display the form", update => "Update the database", );

# build up the data structure to pass to the generate() command
# to build up and display the form from.
my %data = (  "fname" => { -Label => "First Name:",
                -Type => "text" },
              "lname" => { -Label => "Last Name:",
                -Type => "text" },
              "mname" => { -Label => "Middle Initial:",
                -Type => "text", size => 1, maxlength => 1 },
              "state" => { -Label => "State:",
                -Type => "searchBox", -Value => "MO", -WidgetOptions => { data => { "Arizona" => "AZ", "Arkansas" => "AR", "Hawaii" => "HI", "Illinois" => "IL", "Missouri" => "MO", "Mississippi" => "MS" }, isSorted => 1, displayEmpty => 1, displayHelp => 1 }},
              "color" => { -Label => "Your Favorite Color:",
                -Type => "select", -Options => getColors(), -Value => "blue", -onload => "this.selectedIndex = -1;", -onloadOnce => 1, -onbeforeunload => "if (this.selectedIndex != -1) { return; } else { return 'You need to select a color!'; }" },
              "color2" => { -Label => "All Your Favorite Colors:",
                -Type => "multi-select", -Options => getColors(), -Value => [ "yellow", "green" ],
                -Buttons => [ "SelectAll", "ToggleSelect" ] },
              "color3" => { -Label => "Pick a color:", -Type => "colorPicker", -Value => "#000000" },
              "startDate" => { -Label => "Start Date:", -Type => "datePicker", -Value => "2004-01-01", -WidgetOptions => { year => "2004" },
                -ReadOnly => 1, -ReadOnlyArgs => "style=\"color: blue;\"", -ReadOnlyMode => "text", -CreateTemplate => { -Type => "header" } },
              "endDate" => { -Label => "End Date:", -Type => "date-picker", -Value => "2004-12-31", -WidgetOptions => { year => "2004" }, },
              "comment" => { -Label => "Comment:",
                -Type => "textarea" },
              "num1" => { -Label => "Pick a number:", -Type => "select", -Options => { names => ["1", "2", "3", "4", "5"], values => [1, 2, 3, 4, 5] }, -Value => "3" },
              "num2" => { -Label => "Pick a number:", -Type => "multi-select", -Options => { names => ["1", "2", "3", "4", "5"], values => [1, 2, 3, 4, 5] }, -Value => [ "2", "4" ], -Buttons => ["ToggleSelect", "SelectAll"] },
              "num3" => { -Label => "Pick a number:", -Type => "radio", -Options => { names => ["1", "2", "3", "4", "5"], values => [1, 2, 3, 4, 5] }, -Value => "5" },
              "calc1" => { -Label => "Calculator:", -Type => "calculator", -Value => "F{num1} + F{num2} - F{num3}", -WidgetOptions => { calcButton => 1, undoButton => 1, helpLink => 1 } },
              "command" => { -Type => "hidden",
                -Value => "display" },
              "formSubmitted" => { -Type => "hidden",
                -Value => "1" },
              "check1" => { -Label => "Checkbox 1:", -Type => "checkbox", -Value => "one", checked => 1 },
              "check2" => { -Label => "Checkbox 2:", -Type => "checkbox", -Value => "two", checked => 0 },
              "check3" => { -Label => "Checkbox 3:", -Type => "checkbox", -Value => "three", -ReadOnly => 1, -ReadOnlyMode => "text" },
              "check4" => { -Label => "Checkbox 4:", -Type => "checkbox", -Value => "four", -ReadOnly => 1, -ReadOnlyMode => "text", checked => 1 },
              "useTemplate" => { -Label => "Use the prebuilt Template:", -Type => "checkbox", -Value => "template", checked => 0 },
              "resetForm" => { -Type => "reset", -Value => "Reset" },
              "submitForm" => { -Type => "submit", -Value => "Submit" },
              "pop2" => { -Type => "populator", -WidgetOptions => {rows => 2,
                  required => 0, options => \@options, selectLocation => "right",
                  #htmlTemplate => "<td>#LFRI=field_x_0#</td>\n<td>#LFRI=field_x_1#</td>\n<td>#LFRI=field_x_2#</td>\n<td>#LFRI=field_x_3#</td>",
                  manual => 0, customLabel => "text",
                  optionsTypes => [
                    { -Type => "searchBox", -Label => "state", -Value => "AZ", -WidgetOptions => { data => [ "AZ", "AR", "HI", "IL", "MO", "MS" ], isSorted => 1, displayEmpty => 1, displayHelp => 1 } },
                    { -Type => "select", -Label => "select", -Value => "Two", -Options => \%testOptions },
                    { -Type => "datePicker", -Label => "date", -Value => "2005-05-21", -WidgetOptions => {year => "2005"} },
                    { -Type => "colorPicker", -Label => "color", -Value => "transparent", -WidgetOptions => {} },
                    ],
                  debugLevel => 0, displayLabels => 0, displayColumnHeaders => 1, columnHeaders => ["First Name", "Select", "Date", "Color"]}, -ReadOnly => 0, -ReadOnlyMode => "DOM", },
              %{$result{data}},
           );

# create the order array that specifies the order form items should be processed by createTemplate().
my @order = qw( useTemplate fname mname lname state color color2 color3 startDate endDate num1 num2 num3 calc1 comment check1 check2 check3 check4 );
foreach my $entry (@{$result{order}})
{
  push @order, $entry;
}

# build up the profile to pass to the Data::FormValidator for validation
# of the input when the form is submitted back to us.
my %profile = ( required => [ qw( fname lname state color color2 color3 startDate endDate command formSubmitted calc1 ) ],
                optional => [ qw( mname comment check1 check2 check3 check4 useTemplate ) ],
                constraints => { fname => qr/^.+$/,
                                  mname => qr/^.$/,
                                  lname => qr/^.+$/,
                                  color3 => qr/^((#([A-Fa-f0-9]){6})|transparent|inherit)( !important)?$/,
                                  command => sub { my $value = shift; return (exists $commands{$value} ? $value : 0); },
                                },
              );
foreach my $entry (@{$result{profile}->{required}})
{
  push @{$profile{required}}, $entry;
}

my $template = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=command#
  #FIELD=formSubmitted#
  <center>
  #FORMREQUIREDSTRING#<br />
  #FORMERRORSTRING#
  </center>
  <table border="0" width="100%" cellpadding="0" cellspacing="2">
    <tr>
      <td align="right">#LR=useTemplate#</td>
      <td align="left">#FI=useTemplate#</td>
    </tr>
    <tr>
      <td align="right">#LR=fname#</td>
      <td align="left">#FI=fname#</td>
    </tr>
    <tr>
      <td align="right">#LABEL=mname# #REQUIRED=mname#</td>
      <td align="left">#FIELD=mname# #INVALID=mname#</td>
    </tr>
    <tr>
      <td align="right">#LABEL=lname# #REQUIRED=lname#</td>
      <td align="left">#FIELD=lname# #INVALID=lname#</td>
    </tr>
    <tr>
      <td align="right">#LABEL=state# #REQUIRED=state#</td>
      <td align="left">#FIELD=state# #INVALID=state#</td>
    </tr>
    <tr>
      <td align="right">#LRI=color#</td>
      <td align="left">#FIELD=color#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=color2# #REQUIRED=color2# #INVALID=color2#</td>
      <td align="left">#FIELD=color2#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=color3# #REQUIRED=color3# #INVALID=color3#</td>
      <td align="left">#FIELD=color3#</td>
    </tr>
    <tr>
      <td colspan="2">
        <table border="0" width="100%" cellpadding="0" cellspacing="2">
          <tr>
            <td align="right">#LABEL=startDate# #REQUIRED=startDate# #INVALID=startDate#</td>
            <td align="left">#FIELD=startDate#</td>
            <td align="right">#LABEL=endDate# #REQUIRED=endDate# #INVALID=endDate#</td>
            <td align="left">#FIELD=endDate#</td>
          </tr>
        </table>
      </td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=num1# #REQUIRED=num1# #INVALID=num1#</td>
      <td align="left">#FIELD=num1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=num2# #REQUIRED=num2# #INVALID=num2#</td>
      <td align="left">#FIELD=num2#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=num3# #REQUIRED=num3# #INVALID=num3#</td>
      <td align="left">#FIELD=num3#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=calc1# #REQUIRED=calc1# #INVALID=calc1#</td>
      <td align="left">#FIELD=calc1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment# #REQUIRED=comment# #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    <tr>
      <td colspan="2" align="center">Testing the checkbox code</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LRI=check1#</td>
      <td align="left">#FIELD=check1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LRI=check2#</td>
      <td align="left">#FIELD=check2#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LRI=check3#</td>
      <td align="left">#FIELD=check3#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LRI=check4#</td>
      <td align="left">#FIELD=check4#</td>
    </tr>
    <tr>
      <td colspan="2">$result{html}->{body}</td>
    </tr>
    <tr>
      <td colspan="2">#FIELD=pop2#</td>
    </tr>
    <tr>
      <td colspan="2" align="center"><input type="submit" name="submitForm" value="Submit"></td>
    </tr>
  </table>
</form>
END_OF_FORM

my $doc = HTMLObject::Normal->new();
my $formObj = HTMLObject::Form->new();

$template .= $doc->br() . $doc->hr(width => "50%") . "\n";

$doc->setTitle("HTMLObject::Form test script");

# setup the JavaScript error handler support.
my $email = "changeme\@your.domain";
$doc->enableJavascriptErrorHandler(email => $email, prgName => "form.cgi", prgVersion => "1.0");

use vars qw ( %input %clientFileNames %fileContentTypes %serverFileNames );

HTMLObject::CGILib::ReadParse(*input, \%clientFileNames,
                              \%fileContentTypes, \%serverFileNames);

my %form = ();  # the variable that holds the generated form data.
my $displayForm = 0; # signal when we need to display the form.
my $updateData = 0;  # signal when the data is ok to work with
if (exists $input{formSubmitted})
{
  # validate the submitted form
  my $result = $formObj->validate(input => \%input, profile => \%profile, data => \%data);
  if ($formObj->error)
  {
    # display the error message.
    die $formObj->errorMessage;
  }
  if (!$result)
  {
    $displayForm = 1;
  }
  else
  {
    # we have a valid form filled out.
    # update the database and continue, etc.
    # use the valid entries from the $formObj instance.
    $displayForm = 1;
    $updateData = 1;
  }
}
else
{
  $displayForm = 1;
}

if ($displayForm)
{
  my %extraArgs = ();
  $extraArgs{template} = $template if (exists $input{useTemplate});
  %form = $formObj->generate(
            data => \%data, %extraArgs,
            name => "mainForm", action => "",
            method => "post", profile => \%profile,
            order => \@order, onsubmit => $result{html}{onsubmit});
  if ($formObj->error)
  {
    # display the error message
    die $formObj->errorMessage;
  }
  $form{javascript} .= $result{html}->{javascript};
  push @{$form{link}}, $result{html}->{link};
}

if ($updateData)
{
  # output the found entries
  $form{body} .= $doc->br() . $doc->br() . "Valid Entries:" . $doc->br() . "\n";
  foreach my $f ($formObj->getValid)
  {
    my $value = $formObj->getValidEntry($f);
    $form{body} .= "$f = '" . (ref($value) eq "ARRAY" ? $doc->formEncode(join(", ", @{$value})) : $doc->formEncode($value)) . "'" . $doc->br() . "\n";
  }
  $form{body} .= $doc->br() . "Unknown Entries:" . $doc->br() . "\n";
  foreach my $f ($formObj->getUnknown)
  {
    my $value = $formObj->getUnknownEntry($f);
    $form{body} .= "$f = '" . (ref($value) eq "ARRAY" ? $doc->formEncode(join(", ", @{$value})) : $doc->formEncode($value)) . "'" . $doc->br() . "\n";
  }
}

# at this time, print the form into your document.
$doc->print(%form);

# test the new optionsBuilder() method.
$doc->print($doc->br . "optionsBuilder test: " . $doc->select(name => "color2", -content => $doc->optionsBuilder(options => getColors(), selected => $input{color2})));

$doc->print(bodyStyle => "color: purple;");
$doc->setBodyID("myBody");

$doc->display();

exit 0;

# returns a hashref with names and values entries that
# are anonymous arrays containg the pairs of data
# to put in the options tags for a select box.
sub getColors
{
  # do a database lookup, etc.
  my %result = ();
  $result{names} = [ qw( Red Blue Green Yellow Black White ) ];
  $result{values} = [ qw( red blue green yellow black white ) ];

  return \%result;
}

