# Config.pm - The Object Class that provides a Config Data Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::Data::Config;

# updated on 05/24/2002 to work with XML config files.

use strict;
use Portal::Base;
use Portal::XML::ConfigParser;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

my $configVersion = "0.1";

=head1 NAME

Config - Object used to build a Config Data Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::Data::Config;
  my $obj = undef;
  eval { $obj = Portal::UserProperties::Data::Config->new; };
  if ($@)
  {
    die "Error instantiating Config Object!  $@\n";
  }
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Config is a Config class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::Data::Config object.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = (  @_ );
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  # source in the config file from /etc/pcx_portal/UserProperties.xml and suck the settings hash into ourselves.
  my $cfgFile = "/etc/pcx_portal/UserProperties.xml";

  eval { $self->{cfgObj} = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    die $errStr . "Instantiation of ConfigParser failed!<br>\n$@";
  }
  if ($self->{cfgObj}->didErrorOccur)
  {          
    die "$errStr" . $self->{cfgObj}->errorMessage;
  }
  eval { $self->{settingsObj} = $self->{cfgObj}->parse(file => $cfgFile, module => "UserProperties", version => $configVersion); };
  if ($@)
  {
    die $errStr . "Parse of '$cfgFile' failed!<br>\n$@";
  }
  if ($self->{settingsObj}->didErrorOccur)
  {
    die "$errStr" . $self->{settingsObj}->errorMessage;
  }

  my %settings = ();
  %settings = %{$self->{settingsObj}->{settings}};

  foreach my $name (keys %settings)
  {
    $self->{$name} = $settings{$name};
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  if ($self->{settingsObj}->{version} ne $configVersion)
  {
    die $errStr . "Config File '$cfgFile' is at version '$self->{settingsObj}->{version}' not at the current version of '$configVersion'!<br>\n";
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  dbName                - set to appName_db if this app needs its own
                          database.
  dbPerCompany          - set to true if this app needs its own database
                          for each company.
  appType               - administration, help, bugs, regular, tools,
                          news
  appAdminType          - normal or system
  appDescription        - describe (briefly) what the app does.  This
                          will be used in the App purchasing and
                          assigning Apps.
  appIconName           - name the icon this app will display on the
                          desktop.
  height                - default height of the app.
  width                 - default width of the app.
  autorun               - do you want this app to automatically startup
                          when the user logs in?
  dbType                - Database backend to use. Pg or mysql
  dbPort                - The Port to connect to for this database
                          server. 3306 for mysql, 5432 for Pg. Username
                          and password will come from the portals
                          settings.
  appName               - This is the name of the Applications perl
                          module.  Case sensitive.
  isAppFree             - if false, then app is a for fee app.
  assignToAllCompanies  - if false, don't assign to any companies.
  assignToAllUsers      - if true, then assign to all users, else check
                          the admin and sysadmin flags.
  assignToAdminUsers    - if true, then assign to all admin and sysadmin
                          users, else check sysadmin flag.
  assignToSysAdminUsers - if true, then assign to only sysadmin users.
  appCost               - The default cost of the app.  0.00 indicates
                          free.
  unit                  - 0 = per month, 1 = per user.
  numLicenses           - The number of licenses to assign to a company
                          if assignToAllCompanies is true.  0 means
                          unlimited.
  wapAware              - Is this app WAP aware?

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::Base(3)

=cut
