# Template.pm - The Object Class that provides a Template Language Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::Language::Template;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

# The Phrases used by the App.  - Add these to the Template and convert in each language as created.

my %phrases = {};
$phrases{close} = "Close App";
$phrases{help} = "Help";
$phrases{appVersion} = "App Version = <font color=\"%s\">%s</font>";
$phrases{refresh} = "Refresh";
$phrases{title} = "UserProperties %s - %s";
$phrases{mainMenu} = "Main Menu";
$phrases{closeMessage} = "Press 'OK' if you really want to Close this Application,\\nelse press 'Cancel'.";
$phrases{error} = "Error";
$phrases{status} = "Status";
$phrases{changePassword} = "Change Password";
$phrases{userPreferences} = "User Preferences";
$phrases{userProperties} = "User Properties";
$phrases{edit} = "Edit";
$phrases{update} = "Update";
$phrases{password} = "Password";
$phrases{repeat} = "Repeat";
$phrases{original} = "Original";
$phrases{new} = "New";
$phrases{passwordLength} = "Your %s password must be >= 6 characters in length!";
$phrases{passwordNotEqual} = "Your %s password does not match!";
$phrases{passwordError} = "An error occured that should not have!  Error Code = '%s'.";
$phrases{successPassword} = "Successfully updated password!";
$phrases{colorScheme} = "Color Scheme";
$phrases{dynamicContent} = "Dynamic Content";
$phrases{dynamicContentConfig} = "Dynamic Content Configuration Options";
$phrases{frameBorder} = "Frame Border";
$phrases{frameScrollable} = "Frame Scrollable";
$phrases{frameWidth} = "Frame Width";
$phrases{frameHeight} = "Frame Height";
$phrases{successPreferences} = "Successfully updated Portal preferences!";
$phrases{user} = "User";
$phrases{mainMenuMessage} = "Welcome to the User Properties Application.<br><br>Use the menu across the top to quit, get help or refresh the screen you are on.<br><br>Use the menu down the left to actually navigate around and get work done.";

# new
# instantiates an instance of the Template Language Phrases.
sub new
{
  my $that = shift;
  my $class = ref($that) || $that;
  my $self = bless {}, $class;
  my %args = (  @_ );

  $self->{phrases} = { %phrases };
  $self->{encoding} = "iso-8859-1";

  # do validation
  $self->{error} = !$self->isValid;
  if ($self->{error})
  {
    $self->setError($self->{errorString});
  }

  return $self;
}

# isValid - Returns 0 or 1 to indicate if the object is valid.
sub isValid
{
  my $self = shift;
  my $error = 0;
  my $errorString = "";

  # do validation code here.

  if (length $errorString > 0)
  {
    $error = 1;
    $self->setError($errorString);
  }

  return !$error;
}

sub DESTROY
{
  my $self = shift;
}

sub AUTOLOAD  # act as though all calls are just queries, setting of values will not work.
{
  my $self = shift;
  my $type = ref($self) || die "$self is not an object";
  my $name = $AUTOLOAD;
  $name =~ s/.*://;	# strip fully-qualified portion
  unless (exists $self->{$name})
  {
    die "Can't access `$name' field in object of class $type";
  }
  return $self->{$name};
}

# setError
# requires: errorString
# returns: nothing
sub setError
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $callerErrStr = "$subName() - Error!<br>\n";

  if (scalar @_ == 1)
  {
    $self->{errorString} = $callerErrStr . @_[0];
  }
  else
  {
    my %args = ( @_ );
    if (!exists $args{errorString})  # make sure we get the errorString argument!
    {
      $self->setError("<b>errorString</b> is missing!<br>\n");
    }
    else
    {
      $self->{errorString} = $callerErrStr . $args{errorString};
    }
  }
  $self->{error} = 1;
}

# didErrorOccur
# returns the value of error.
sub didErrorOccur
{
  my $self = shift;

  return $self->{error};
}

# errorMessage
# returns the value of errorString
sub errorMessage
{
  my $self = shift;

  return $self->{errorString};
}

1;
__END__

=head1 NAME

Template - Object used to build a Template Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::Language::Template;
  my $obj = Portal::UserProperties::Language::Template->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Template is a Template Language class.

=head1 Exported FUNCTIONS

  scalar new(void)
    Creates a new instance of the Portal::UserProperties::Language::Template
    object.

  bool isValid(void)
    Returns 0 or 1 to indicate if the object is valid.  The error will be
    available via errorMessage().

  void setError(errorString)
    Sets error = 1 and errorString = errorString.

  scalar didErrorOccur(void)
    Returns the value of error.

  scalar errorMessage(void)
    Returns the value of errorString.

=head1 Exported VARIABLES

  hash   phrases  - converted phrases
  scalar encoding - content type encoding for document.

  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3)

=cut
