# Main.pm - The Object Class that provides a Main Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::Main;
use strict;
use Portal::AppState;
use Portal::UserProperties::Objects::Menu;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Main Menu", close => "Close App", );

=head1 NAME

Main - Object used to build a Main Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::Main;
  my $obj = Portal::UserProperties::Main->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

Main is a Main class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::Main module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc;

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  $doc = $self->setupCSS();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }

  my $mainMenuMessagePhrase = $self->{langObj}->map("mainMenuMessage");
  # create the help JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  # create the validateClose JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  my $menuObj = Portal::UserProperties::Objects::Menu->new(methods => $self->{methods},
                sessionObj => $self->{sessionObj}, langObj => $self->{langObj}, userObj => $self->{userObj},
                portalSession => $self->{portalSession});
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  my $menuString = $menuObj->print(location => "menubar", state => "Main", command => "display", doc => $doc);
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  my $sidemenuString = $menuObj->print(location => "sidemenu", state => "Main", command => "display", doc => $doc);
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top">$sidemenuString</td>
    <td width="75%" align="center">$mainMenuMessagePhrase</td>
  </tr>
</table>
END_OF_CODE

  return $doc;
}

=item HTMLObject c_close()

=cut
sub c_close
{
  my $self = shift;
  my $doc;

  $doc = $self->processCloseAppEvent();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }

  return $doc;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  HTMLObject run(command)
    Runs the specified command and returns the resulting HTMLObject.


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::AppState(3)

=cut
