# StateTemplate.pm - The Object Class that provides a StateTemplate Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::StateTemplate;
use strict;
use Portal::AppState;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Main Menu" );

=head1 NAME

StateTemplate - Object used to build a StateTemplate Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::StateTemplate;
  my $obj = Portal::UserProperties::StateTemplate->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

StateTemplate is a StateTemplate class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::StateTemplate module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{portalVariables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{portalVariables}->didErrorOccur)
  {
    $self->setError($self->{portalVariables}->errorMessage);
    return $self;
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# run
# takes: command
# returns: The HTMLObject document to display or undef if an error.
sub run
{
  my $self = shift;
  my %args = ( command => "", @_ );
  my $command = $args{command};
  my $doc;

  if (!exists $commands{$command})
  {
    $self->setError(errorString => sprintf($self->langObj->map("commandNotKnown"), $command));
    return undef;
  }

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  $doc = HTMLObject::Normal->new;

  # get the users colorScheme to work with.
  my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => $self->{userObj}->{id});
  if ($self->{applicationObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $colorScheme)
  {
    $self->setError(errorString => "No valid colorScheme returned!<br>\n");
    return undef;
  }

  # create the needed css entries
  $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  if ($command eq "display")
  {
    $doc->setFocus("body");
    $doc->print("Hello World!");
  }
  else
  {
    $doc->setFocus("body");
    $doc->print("I should never get here!");
  }

  if (defined $doc)
  {
    # figure out what JavaScript Error Handler needs to be generated.
    if ($browserCap{'document.getElementById'} && $browserCap{'document.createElement'})
    {
      $doc->setJavascriptErrorHandlerVersion(version => "v5");
    }
    elsif ($browserCap{'document.layers'} || ($browserCap{'document.all'} && !$browserCap{'document.createElement'}))
    {
      $doc->setJavascriptErrorHandlerVersion(version => "v4");
    }
  }

  return $doc;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  HTMLObject run(command)
    Runs the specified command and returns the resulting HTMLObject.


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::AppState(3)

=cut
