# User.pm - The Object Class that provides a User Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::User;
use strict;
use Portal::AppState;
use Portal::UserProperties::Objects::Menu;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use Portal::Objects::UserPreferenceObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "User Properties", passwd => "Change User Password", preferences => "User Preferences", );

=head1 NAME

User - Object used to build a User Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::User;
  my $obj = Portal::UserProperties::User->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

User is a User class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::User module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }

  if (! exists $self->{input}->{update})
  {
    $doc = $self->displayUserProperties(doc => $doc);
    if ($self->didErrorOccur)
    {
      $self->prefixError();
      return undef;
    }
  }
  else  # validate and update the users properties.
  {
  }

  return $doc;
}

=item HTMLObject c_passwd

=cut
sub c_passwd
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }
  my $passwordLengthPhrase = $self->langObj->map("passwordLength");
  my $passwordNotEqualPhrase = $self->langObj->map("passwordNotEqual");
  my $passwordErrorPhrase = $self->langObj->map("passwordError");
  my $successPasswordPhrase = $self->langObj->map("successPassword");

  if (! exists $self->{input}->{update})
  {
    $doc = $self->displayPasswordScreen(doc => $doc);
    if ($self->didErrorOccur)
    {
      $self->prefixError();
      return undef;
    }
  }
  else  # validate and update the password.
  {
    # make sure they specified original, password1 and password2
    my $original = $self->{input}->{original};
    my $password1 = $self->{input}->{password1};
    my $password2 = $self->{input}->{password2};

    my $errorString = "";
    if (length $original < 6)
    {
      $errorString .= sprintf($passwordLengthPhrase, "original") . "<br>\n";
    }
    else
    {
      # actually validate the original password.
      my $result = $self->{authObj}->isUserValid(uname => $self->{userObj}->{uname}, password => $original);
      if ($result == 0)
      {
        $errorString .= sprintf($passwordNotEqualPhrase, "original") . "<br>\n";
      }
      elsif ($result < 0)
      {
        $errorString .= sprintf($passwordErrorPhrase, "$result") . "<br>\n";
      }
    }
    if (length $password1 < 6)
    {
      $errorString .= sprintf($passwordLengthPhrase, "") . "<br>\n";
    }
    if ($password1 ne $password2)
    {
      $errorString .= sprintf($passwordNotEqualPhrase, "") . "<br>\n";
    }

    if (length $errorString > 0)
    {
      $doc = $self->displayPasswordScreen(doc => $doc, error => $errorString);
      if ($self->didErrorOccur)
      {
        $self->prefixError();
        return undef;
      }
    }
    else
    {
      # now we update the userObj with the new password and signal that it needs to be encrypted
      # and then update the users data.

      $self->{userObj}->{password} = $password1;
      $self->{userObj}->{passCrypted} = "false";

      my $result = $self->{authObj}->updateUser(userObj => $self->{userObj});
      if ($self->{authObj}->didErrorOccur)
      {
        $doc = $self->displayPasswordScreen(doc => $doc, error => $self->{authObj}->errorMessage);
        if ($self->didErrorOccur)
        {
          $self->prefixError();
          return undef;
        }
      }
      $doc = $self->displayPasswordScreen(doc => $doc, status => $successPasswordPhrase);
      if ($self->didErrorOccur)
      {
        $self->prefixError();
        return undef;
      }
    }
  }

  return $doc;
}

=item HTMLObject c_preferences()

=cut
sub c_preferences
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }
  my $successPreferencesPhrase = $self->langObj->map("successPreferences");

  if (! exists $self->{input}->{update})
  {
    $doc = $self->displayUserPreferences(doc => $doc);
    if ($self->didErrorOccur)
    {
      $self->prefixError();
      return undef;
    }
  }
  else  # validate and update the users preferences.
  {
    my $colorScheme = $self->{input}->{colorScheme};
    my $dynamicContent = $self->{input}->{dynamicContent};
    my $frameBorder = $self->{input}->{frameBorder};
    my $frameScrolling = $self->{input}->{frameScrolling};
    my $frameWidth = $self->{input}->{frameWidth};
    my $frameHeight = $self->{input}->{frameHeight};

    # do validation
    my $errorString = "";
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{applicationObj}->errorMessage);
      return undef;
    }
    my $systemColorScheme = grep($colorScheme, @systemColorSchemes);
    my $userColorScheme = grep($colorScheme, @userColorSchemes);
    my $invalidArgument = $self->{langObj}->map("invalidArgument");
    if ($systemColorScheme == 0 && $userColorScheme == 0)
    {
      $errorString .= sprintf($invalidArgument, "colorScheme", $colorScheme);
    }
    if ($dynamicContent !~ /^(true|false)$/)
    {
      $errorString .= sprintf($invalidArgument, "dynamicContent", $dynamicContent);
    }
    if ($frameBorder !~ /^(0|1)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameBorder", $frameBorder);
    }
    if ($frameScrolling !~ /^(yes|no|auto)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameScrolling", $frameScrolling);
    }
    if ($frameWidth !~ /^(\d+%?)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameWidth", $frameWidth);
    }
    if ($frameHeight !~ /^(\d+)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameHeight", $frameHeight);
    }

    if (length $errorString > 0)
    {
      $doc = $self->displayUserPreferences(doc => $doc, error => $errorString);
      if ($self->didErrorOccur)
      {
        $self->prefixError();
        return undef;
      }
    }
    else
    {
      # update the user preferences.

      # handle colorScheme, dynamicContent
      foreach my $name ("colorScheme", "dynamicContent", "dynamicContentConfig")
      {
        my $configObjStr;
        if ($name eq "dynamicContentConfig")
        {
          # handle dynamicContentConfig
          my $configObj;
          eval { $configObj = Portal::XML::ConfigObject->new(langObj => $self->{langObj}); };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          $configObj->{version} = "1.0";
          $configObj->{module} = "DynamicContentConfig";
          $configObj->{settings}->{frameBorder} = $frameBorder;
          $configObj->{settings}->{frameScrolling} = $frameScrolling;
          $configObj->{settings}->{frameHeight} = $frameHeight;
          $configObj->{settings}->{frameWidth} = $frameWidth;

          eval { $configObjStr = $configObj->generateXML; };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if (length $configObjStr == 0)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{langObj}->map("noConfigXMLGenerated"));
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
        }

        my $module = ($name eq "colorScheme" ? "Global" : "Desktop");
        my $prefObj = Portal::Objects::UserPreferenceObject->new(id => $self->{userObj}->{id},
                      app => "Portal", module => $module, name => "$name",
                      value => ($name ne "dynamicContentConfig" ? $self->{input}->{$name} : $configObjStr), langObj => $self->{langObj});
        if ($prefObj->didErrorOccur)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $prefObj->errorMessage);
          if ($self->didErrorOccur)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        my $result = $self->{authObj}->updateUserPreference(userPreferenceObj => $prefObj);
        if ($self->{authObj}->didErrorOccur)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
          if ($self->didErrorOccur)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        if ($result == -1)
        { # have to create it
          $result = $self->{authObj}->createUserPreference(userPreferenceObj => $prefObj);
          if ($self->{authObj}->didErrorOccur)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if ($result == -1)
          { # already existed!  don't know how!
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceNowExists"), $prefObj->print));
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == -2)
          { # app doesn't exist
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceAppNotExist"), $prefObj->print));
            if ($self->didErrorOccur)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == 1)
          { # we are ok.
          }
        }
        elsif ($result == -2)
        { # app doesn't exist
          $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->langObj->map("updateUserPreferenceAppNotExist"), $prefObj->print));
          if ($self->didErrorOccur)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        elsif ($result == 1)
        { # we are ok.
        }
      }

      $doc = $self->displayUserPreferences(doc => $doc, status => $successPreferencesPhrase);
      if ($self->didErrorOccur)
      {
        $self->prefixError();
        return undef;
      }
    }
  }

  return $doc;
}

sub displayUserProperties
{
  my $self = shift;
  my %args = ( doc => undef, error => "", status => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $status = $args{status};

  my $changePasswordPhrase = $self->langObj->map("changePassword");
  my $passwordPhrase = $self->langObj->map("password");
  my $repeatPhrase = $self->langObj->map("repeat");

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  $doc = $self->displayMenu(doc => $doc, command => "passwd");
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $doc;
  }

  if ($error)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "error", message => $error,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }
  if ($status)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "status", message => $status,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }

  return $doc;
}

sub displayPasswordScreen
{
  my $self = shift;
  my %args = ( doc => undef, error => "", status => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $status = $args{status};

  my $changePasswordPhrase = $self->langObj->map("changePassword");
  my $passwordPhrase = $self->langObj->map("password");
  my $repeatPhrase = $self->langObj->map("repeat");
  my $originalPhrase = $self->langObj->map("original");
  my $newPhrase = $self->langObj->map("new");

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  $doc = $self->displayMenu(doc => $doc, command => "passwd");
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $doc;
  }

  if ($error)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "error", message => $error,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }
  if ($status)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "status", message => $status,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }

  # generate the form to let them specify their new password in.
  $doc->print(<<"END_OF_CODE");
$changePasswordPhrase
<br>
<form name="passwds" method="post" action="$url">
  <input type="hidden" name="app" value="UserProperties">
  <input type="hidden" name="state" value="User">
  <input type="hidden" name="command" value="passwd">
  <input type="hidden" name="update" value="1">
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right">$originalPhrase $passwordPhrase</td>
      <td align="left"><input type="password" name="original" value=""></td>
    </tr>
    <tr>
      <td align="right">$newPhrase $passwordPhrase</td>
      <td align="left"><input type="password" name="password1" value=""></td>
    </tr>
    <tr>
      <td align="right">$repeatPhrase</td>
      <td align="left"><input type="password" name="password2" value=""></td>
    </tr>
  </table>
  <input type="submit" name="update" value="$changePasswordPhrase">
</form>
    </td>
  </tr>
</table>
END_OF_CODE

  $doc->setOnload(code => "document.passwds.original.focus();");

  return $doc;
}

sub displayUserPreferences
{
  my $self = shift;
  my %args = ( doc => undef, error => "", status => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $status = $args{status};

  my $colorSchemePhrase = $self->langObj->map("colorScheme");
  my $dynamicContentPhrase = $self->langObj->map("dynamicContent");
  my $dynamicContentConfigPhrase = $self->langObj->map("dynamicContentConfig");
  my $frameBorderPhrase = $self->langObj->map("frameBorder");
  my $frameScrollablePhrase = $self->langObj->map("frameScrollable");
  my $frameWidthPhrase = $self->langObj->map("frameWidth");
  my $frameHeightPhrase = $self->langObj->map("frameHeight");
  my $editPhrase = $self->langObj->map("edit");
  my $userPreferencesPhrase = $self->langObj->map("userPreferences");

  # setup the XML parser needed.
  my $parserObj;
  eval { $parserObj = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->setError(errorString => $@);
    return $doc;
  }

  # get the Defaults for the preferences.
  my %defaults = ();  # store the defaults the system gives us.
  foreach my $prefModule ("Global", "Desktop")
  {
    my @names = ($prefModule eq "Global" ? ("colorScheme") : ("dynamicContent", "dynamicContent_Config"));
    foreach my $prefName (@names)
    {
      my $prefNameDefault = "Portal_" . $prefModule . "_" . $prefName . "_Default";
      my $prefValueDefault = $self->{methods}->getConfigValue(name => $prefNameDefault, portalDB => $self->{applicationObj}->{portalDB});
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return $doc;
      }
      if (!defined $prefValueDefault)
      {
        $self->setError(errorString => sprintf($self->{langObj}->map("configEntryDoesNotExistUpgradeDatabase"), $prefNameDefault));
        return $doc;
      }
      if ($prefName eq "dynamicContent_Config" && $prefModule eq "Desktop")
      {
        # have to do the XML parsing
        my $configObj;
        eval { $configObj = $parserObj->parse(string => $prefValueDefault, module => "DynamicContentConfig", version => "1.0"); };
        if ($@)
        {
          $self->setError(errorString => sprintf($self->{langObj}->map("parseFailed"), $@));
          return $doc;
        }
        foreach my $name (keys %{$configObj->{settings}})
        {
          $defaults{$prefModule}{$name} = $configObj->{settings}->{$name};
        }
      }
      else
      {
        $defaults{$prefModule}{$prefName} = $prefValueDefault;
      }
    }
  }

  # pull in the Portal preferences for the user.
  my @userPreferences = $self->{authObj}->getUserPreferences(userId => $self->{userObj}->{id},
                        app => "Portal");
  if ($self->{authObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{authObj}->errorMessage);
    return $doc;
  }
  my %preferences = ();  # store the preferences info that we get from the system.
  foreach my $prefObj (@userPreferences)
  {
    if ($prefObj->{name} eq "dynamicContentConfig" && $prefObj->{module} eq "Desktop")
    {
      # have to do the XML parsing
      my $configObj;
      eval { $configObj = $parserObj->parse(string => $prefObj->{value}, module => "DynamicContentConfig", version => "1.0"); };
      if ($@)
      {
        $self->setError(errorString => sprintf($self->{langObj}->map("parseFailed"), $@));
        return $doc;
      }
      foreach my $name (keys %{$configObj->{settings}})
      {
        $preferences{$prefObj->{module}}{$name} = $configObj->{settings}->{$name};
      }
    }
    else
    {
      $preferences{$prefObj->{module}}{$prefObj->{name}} = $prefObj->{value};
    }
  }

  # verify we have preferences to edit, if not we specify the default.
  foreach my $module ("Global", "Desktop")
  {
    my @names = ($module eq "Global" ? ("colorScheme") : ("frameScrolling", "frameBorder", "frameWidth", "frameHeight", "dynamicContent"));
    foreach my $name (@names)
    {
      $preferences{$module}{$name} = (exists $self->{input}->{$name} ? $self->{input}->{$name} : (exists $preferences{$module}{$name} ? $preferences{$module}{$name} : $defaults{$module}{$name}));
    }
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  $doc = $self->displayMenu(doc => $doc, command => "preferences");
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $doc;
  }

  if ($error)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "error", message => $error,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }
  if ($status)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "status", message => $status,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->didErrorOccur)
    {
      $self->setError(errorString => $self->{methods}->errorMessage);
      return undef;
    }
  }

  # generate the form to let them edit their Portal Preferences.
  $doc->print(<<"END_OF_CODE");
$editPhrase $userPreferencesPhrase
<br>
<form name="preferences" method="post" action="$url">
  <input type="hidden" name="app" value="UserProperties">
  <input type="hidden" name="state" value="User">
  <input type="hidden" name="command" value="preferences">
  <input type="hidden" name="update" value="1">
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right">$colorSchemePhrase</td>
      <td align="left">
        <select name="colorScheme">
END_OF_CODE
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{applicationObj}->errorMessage);
      return undef;
    }
    foreach my $option (@systemColorSchemes, @userColorSchemes)
    {
      my $selected = ($preferences{Global}{colorScheme} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$dynamicContentPhrase</td>
      <td align="left">
        <select name="dynamicContent">
END_OF_CODE
    foreach my $option ("true", "false")
    {
      my $selected = ($preferences{Desktop}{dynamicContent} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td colspan="2" align="center"><b>$dynamicContentConfigPhrase</b></td>
    </tr>
    <tr>
      <td align="right">$frameBorderPhrase</td>
      <td align="left">
        <select name="frameBorder">
END_OF_CODE
    foreach my $option ("0", "1")
    {
      my $selected = ($preferences{Desktop}{frameBorder} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameScrollablePhrase</td>
      <td align="left">
        <select name="frameScrolling">
END_OF_CODE
    foreach my $option ("auto", "yes", "no")
    {
      my $selected = ($preferences{Desktop}{frameScrolling} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameWidthPhrase</td>
      <td align="left"><input type="text" name="frameWidth" value="$preferences{Desktop}{frameWidth}" size="4" maxlength="4"></td>
    </tr>
    <tr>
      <td align="right">$frameHeightPhrase</td>
      <td align="left"><input type="text" name="frameHeight" value="$preferences{Desktop}{frameHeight}" size="3" maxlength="3"></td>
    </tr>
  </table>
  <input type="submit" name="update" value="$editPhrase $userPreferencesPhrase">
</form>
    </td>
  </tr>
</table>
END_OF_CODE

  return $doc;
}

sub displayMenu
{
  my $self = shift;
  my %args = ( doc => undef, command => "", @_ );
  my $doc = $args{doc};
  my $command = $args{command};

  # create the help JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  # create the validateClose JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  my $menuObj = Portal::UserProperties::Objects::Menu->new(methods => $self->{methods},
                sessionObj => $self->{sessionObj}, langObj => $self->{langObj}, userObj => $self->{userObj},
                portalSession => $self->{portalSession});
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  my $menuString = $menuObj->print(location => "menubar", state => "User", command => $command, doc => $doc);
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  my $sidemenuString = $menuObj->print(location => "sidemenu", state => "User", command => $command, doc => $doc);
  if ($menuObj->didErrorOccur)
  {
    $self->setError(errorString => $menuObj->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top">$sidemenuString</td>
    <td width="75%" align="center">
END_OF_CODE

  return $doc;
}

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::AppState(3)

=cut
