# CompanyAdmin.pm - The Object class that provides the CompanyAdmin Application Object.
# Created by James A. Pattie, 07/11/2000.

package Portal::CompanyAdmin;
# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

use strict;
use Portal::App;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::App Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

# global variables
my %states = ( Main => "Main Menu", User => "User Administration", Apps => "Apps Administration" );

=head1 NAME

CompanyAdmin - Object used to build a CompanyAdmin Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin;
  my $obj = Portal::CompanyAdmin->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

CompanyAdmin is the main class for the CompanyAdmin Application.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::CompanyAdmin object.
 See Portal::App(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{app} = "CompanyAdmin";
  $self->{appVersion} = $VERSION;
  $self->{states} = \%states;
  $self->{logOk} = 1;  # we can log safely.  Change to 0 if you don't have a log_tb defined in your database.
  $self->{useAppStateCommandSecurity} = 0;

  # do validation
  if (!$self->Portal::CompanyAdmin::isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  # change the dbHost => "App" when you use this as a template and are working with a different database than the Portal's.
  $self->instantiatePortalApp(dbHost => "Portal");
  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # store the users companyObj in the App Session so the code knows what company to work with.
  $self->{appSessions}->{apps}->{$self->{app}}->{store}->{companyObj} = $self->{companyObj};
  
  return $self;
}

=item % getStates(void)

  This method returns the states hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getStates
{
  return %states;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

PC&Web Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::CompanyAdmin(3), Portal::App(3)

=cut

