# Apps.pm - The Object Class that provides a Apps Object
# Created by James A. Pattie, 2004-08-19.

# Copyright (c) 2000-2004, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::CompanyAdmin::Apps;
use strict;
use Portal::AppState;
use Portal::Forms::Apps;
use Portal::CompanyAdmin::Objects::Menu;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Select Apps", displayApp => "Display App Info", updateServerInfo => "Update Server Info", editUserAppSettings => "Edit User App Settings", sellApp => "Sell App", purchaseApp => "Purchase App", updateAppAssignments => "Assign/UnAssign App To/From Users", purchaseLicenses => "Purchase Licenses" );

=head1 NAME

Apps - Object used to build a Apps Object Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin::Apps;
  my $obj = Portal::CompanyAdmin::Apps->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

Apps is a Apps class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::CompanyAdmin::Apps module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;
  $self->{state} = "Apps";
  
  my $arguments = $self->arguments();
  eval "\$self->{appsForms} = Portal::Forms::Apps->new($arguments);";
  if ($@)
  {
    $self->error($@);
    return $self;
  }
  if ($self->{appsForms}->error)
  {
    $self->error($self->{appsForms}->errorMessage);
    return $self;
  }
  

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "display");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # call the Portal::Forms::Apps module to do the form display work.
  my %output = $self->{appsForms}->selectAppsAdminForm(companyObj => $self->{sessionObj}->{store}->{companyObj},
      url => $url, app => $self->{app}, state => $self->{state} );
  if ($self->{appsForms}->error)
  {
    $self->error($self->{appsForms}->errorMessage);
    return undef;
  }
  
  # now display the output to the user.
  $doc->print(%output);
  
  # finish off the screen, closing out what the displayMenu() method started.
  $doc = $self->closeMenu(doc => $doc);

  return $doc;
}

=item HTMLObject c_displayApp()

=cut
sub c_displayApp
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "displayApp", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have a appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{appId} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  # lookup the specified application.
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_display();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_display();
    return $doc;
  }
  
  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my %output = $self->{appsForms}->displayAppForAdmin(companyObj => $self->{sessionObj}->{store}->{companyObj}, companyAppObj => $tmpCompanyApp,
      url => $url, app => $self->{app}, state => $self->{state});
  if ($self->{appsForms}->error)
  {
    $self->error($self->{appsForms}->errorMessage);
    return undef;
  }
  
  # now display the output to the user.
  $doc->print(%output);
  
  # finish off the screen, closing out what the displayMenu() method started.
  $doc = $self->closeMenu(doc => $doc);

  return $doc;
}

=item HTMLObject c_updateServerInfo

=cut
sub c_updateServerInfo
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successEditPhrase = $self->langObj->map("SuccessfullyUpdatedServerInfo");

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "updateServerInfo", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have the appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_displayApp();
    return $doc;
  }
  elsif (length $self->{input}->{appId} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", "");
    $doc = $self->c_displayApp();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_displayApp();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my @result = $self->{appsForms}->updateServerInfo(companyObj => $self->{sessionObj}->{store}->{companyObj},
      companyAppObj => $tmpCompanyApp,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{appsForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{appsForms}->errorMessage;
    $self->{appsForms}->clearErrors("all");
    $self->{appsForms}->resetError;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the edit suceeded.
    $self->{input}->{statusMessage} = $successEditPhrase;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_updateAppAssignments()

=cut
sub c_updateAppAssignments
{
  my $self = shift;
  my $doc = undef;
  my $errStr = "";
  
  # make sure we got the assignedUsers and unassignedUsers variables.
  foreach my $var (qw( assignedUsers unassignedUsers appId ))
  {
    if (!exists $self->{input}->{$var})
    {
      $errStr .= sprintf($self->{langObj}->map("missingArgument"), $var);
    }
    elsif ($var eq "appId" && $self->{input}->{$var} !~ /^(\d+)$/)
    {
      $errStr .= sprintf($self->{langObj}->map("invalidArgument"), $var, $self->{input}->{$var});
    }
  }
  
  if (length $errStr > 0)
  {
    $self->{input}->{errorMessage} = $errStr;
    $doc = $self->c_displayApp();
    return $doc;
  }
 
  # now verify that the specified app still exists.
  my $appId = $self->{input}->{appId};
  
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_displayApp();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  my @assignedUsers = split(/,/, $self->{input}->{assignedUsers});
  my @unassignedUsers = split(/,/, $self->{input}->{unassignedUsers});

  # call the Portal::Forms::Apps->updateAppAssignments to do the work.
  my $result = $self->{appsForms}->updateAppAssignments(companyObj => $self->{sessionObj}->{store}->{companyObj}, companyAppObj => $tmpCompanyApp,
      assignedUsers => \@assignedUsers, unassignedUsers => \@unassignedUsers);
  if ($self->{appsForms}->error)
  {
    $self->error($self->{appsForms}->errorMessage);
    return undef;
  }
  
  # call the c_displayApp() method to inform the caller of results.
  if (!$result->{result})
  {
    $self->{input}->{errorMessage} = $result->{message};
  }
  else
  {
    $self->{input}->{statusMessage} = $result->{message};
  }
  $doc = $self->c_displayApp();
    
  return $doc;
}

=item HTMLObject c_purchaseApp

=cut
sub c_purchaseApp
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "purchaseApp", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have an appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_display();
    return $doc;
  }
  elsif ($self->{input}->{appId} !~ /^(\d+)$/)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", $self->{input}->{appId});
    $doc = $self->c_display();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  # lookup the specified app.
  my $tmpApp = $self->{applicationObj}->getApplicationInfo(id => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_display();
    return $doc;
  }
  my $appPurchasedPhrase = sprintf($self->langObj->map("AppPurchased"), $tmpApp->{name});
  
  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my @result = $self->{appsForms}->purchaseApp(appObj => $tmpApp, 
      companyObj => $self->{sessionObj}->{store}->{companyObj},
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{appsForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{appsForms}->errorMessage;
    $self->{appsForms}->clearErrors("all");
    $self->{appsForms}->resetError;
    $doc = $self->c_display();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the purchase suceeded.
    $self->{input}->{statusMessage} = $appPurchasedPhrase . $result[1]->{body};
    $doc = $self->c_display();
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_sellApp

=cut
sub c_sellApp
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "sellApp", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have the appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_displayApp();
    return $doc;
  }
  elsif (length $self->{input}->{appId} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", "");
    $doc = $self->c_displayApp();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_displayApp();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my @result = $self->{appsForms}->sellAppLicenses(companyObj => $self->{sessionObj}->{store}->{companyObj},
      companyAppObj => $tmpCompanyApp,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{appsForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{appsForms}->errorMessage;
    $self->{appsForms}->clearErrors("all");
    $self->{appsForms}->resetError;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    #$doc->print($self->{methods}->getCurrentDate(format => "%F %T"));
    #$doc->print($doc->br . "Debug: $result[6]");
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the edit suceeded.
    my $successSellPhrase = sprintf($self->langObj->map("SuccessfullySoldLicensesForApp"), $result[4], $result[5]);
    $self->{input}->{statusMessage} = $successSellPhrase;
    if ($result[4] eq "all")
    { # we sold the app, so it is no longer assigned to us.
      $doc = $self->c_display();
    }
    else
    { # we just sold some licenses, so we can re-display the App info screen.
      $doc = $self->c_displayApp();
    }
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_editUserAppSettings

=cut
sub c_editUserAppSettings
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "editUserAppSettings", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have the appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_displayApp();
    return $doc;
  }
  elsif (length $self->{input}->{appId} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", "");
    $doc = $self->c_displayApp();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  # we need the userId
  
  $self->{input}->{errorMessage} = "editUserAppSettings - This feature not yet implemented.  Sorry.";
  $doc = $self->c_displayApp();
  return $doc;
  
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_displayApp();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my @result = $self->{appsForms}->sellAppLicenses(companyObj => $self->{sessionObj}->{store}->{companyObj},
      companyAppObj => $tmpCompanyApp,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{appsForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{appsForms}->errorMessage;
    $self->{appsForms}->clearErrors("all");
    $self->{appsForms}->resetError;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    #$doc->print($self->{methods}->getCurrentDate(format => "%F %T"));
    #$doc->print($doc->br . "Debug: $result[6]");
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the edit suceeded.
    my $successSellPhrase = sprintf($self->langObj->map("SuccessfullySoldLicensesForApp"), $result[4], $result[5]);
    $self->{input}->{statusMessage} = $successSellPhrase;
    if ($result[4] eq "all")
    { # we sold the app, so it is no longer assigned to us.
      $doc = $self->c_display();
    }
    else
    { # we just sold some licenses, so we can re-display the App info screen.
      $doc = $self->c_displayApp();
    }
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_purchaseLicenses

=cut
sub c_purchaseLicenses
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "purchaseLicenses", appId => $self->{input}->{appId});
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have the appId specified.
  if (!exists $self->{input}->{appId})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "appId");
    $doc = $self->c_displayApp();
    return $doc;
  }
  elsif (length $self->{input}->{appId} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "appId", "");
    $doc = $self->c_displayApp();
    return $doc;
  }
  my $appId = $self->{input}->{appId};
  
  # we need the userId
  
  $self->{input}->{errorMessage} = "purchaseLicenses - This feature not yet implemented.  Sorry.";
  $doc = $self->c_displayApp();
  return $doc;
  
  my $tmpCompanyApp = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{sessionObj}->{store}->{companyObj}->{id}, appId => $appId);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpCompanyApp)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidApp"), $appId);
    $doc = $self->c_displayApp();
    return $doc;
  }
  if (!$tmpCompanyApp->isValid)
  {
    $self->{input}->{errorMessage} = $tmpCompanyApp->errorMessage;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # call the Portal::Forms::Apps module to do the core display, validation and update work
  my @result = $self->{appsForms}->sellAppLicenses(companyObj => $self->{sessionObj}->{store}->{companyObj},
      companyAppObj => $tmpCompanyApp,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{appsForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{appsForms}->errorMessage;
    $self->{appsForms}->clearErrors("all");
    $self->{appsForms}->resetError;
    $doc = $self->c_displayApp();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    #$doc->print($self->{methods}->getCurrentDate(format => "%F %T"));
    #$doc->print($doc->br . "Debug: $result[6]");
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the edit suceeded.
    my $successSellPhrase = sprintf($self->langObj->map("SuccessfullySoldLicensesForApp"), $result[4], $result[5]);
    $self->{input}->{statusMessage} = $successSellPhrase;
    if ($result[4] eq "all")
    { # we sold the app, so it is no longer assigned to us.
      $doc = $self->c_display();
    }
    else
    { # we just sold some licenses, so we can re-display the App info screen.
      $doc = $self->c_displayApp();
    }
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

sub displayMenu
{
  my $self = shift;
  my %args = ( doc => undef, command => "", @_ );
  my $doc = $args{doc};
  my $command = $args{command};
  my $appId = (exists $args{appId} ? $args{appId} : "");

  # create the help JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # create the validateClose JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  my $menuObj = Portal::CompanyAdmin::Objects::Menu->new(methods => $self->{methods},
                sessionObj => $self->{sessionObj}, langObj => $self->{langObj}, userObj => $self->{userObj},
                portalSession => $self->{portalSession});
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $menuString = $menuObj->print(location => "menubar", state => $self->{state}, command => $command, doc => $doc, appId => $appId);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $sidemenuString = $menuObj->print(location => "sidemenu", state => $self->{state}, command => $command, doc => $doc, appId => $appId);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top">$sidemenuString</td>
    <td width="75%" align="center">
END_OF_CODE

  return $doc;
}

sub closeMenu
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};
  
  $doc->print(<<"END_OF_FORM");
    </td>
  </tr>
</table>
END_OF_FORM

  return $doc;  
}

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::CompanyAdmin(3), Portal::AppState(3)

=cut
