# Help.pm - The Object Class that provides a Help Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::CompanyAdmin::Data::Help;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

# Help Data structure
my %Topics = (
             en => {
                     Normal => {
                               "_order_" => "Close App,_seperator_,Refresh,Help,_seperator_,User,_line_,Apps",
                               "_overview_" => "This is the Help for the CompanyAdmin Portal app. &nbsp;Links with a &gt; to their left will take you to more topics about that module.",
                               "Close App" => "This closes the Company Admin App, and brings you back to the Portal Desktop.",
                               "Refresh" => "This redisplays the page you are currently on. &nbsp;Use this to reset back to defaults if things get changed too drastically.",
                               "Help" => "This is what you are currently seeing.",
                               "User" => {
                                         "_order_" => "Create User,Change Password,Edit User Info,View User Info,Edit User Preferences",
                                         "_summary_" => "User related Help Topics",
                                         "_overview_" => "This section covers the screens available while working with Users.",
                                         "Create User" => "This module will allow you to create a new user. &nbsp;At most, it can be an admin user.",
                                         "Change Password" => "This module will allow you to specify a new password for the selected user. &nbsp;It will not go into affect until they log out and log back in.",
                                         "Edit User Info" => "This module will allow you to edit the selected user. &nbsp;You will not be able to change their uname, id, companyId or lastEdited values.",
                                         "View User Info" => "Allows you to see the info about the selected user account.&nbsp; Go <help app=\"\" content=\"Normal,User,Edit User Info\">here</help> to edit a selected users info.",
                                         "Edit User Preferences" => "This module allows you to edit the selected users preferences. &nbsp;These include their Color Schemes and Desktop Dynamic Content settings.",
                                         },
                               "Apps" => {
                                         "_order_" => "Manage Apps,Select App,Edit User Info,Manage User Assignments,Edit Server Info,_line_,Purchase App,Sell App/Licenses",
                                         "_summary_" => "Application Management related Help Topics",
                                         "_overview_" => "This section covers the screens available while Managing the Applications assigned to your company.",
                                         "Manage Apps" => "This screen shows you all the apps that are assigned to your company and gives you a brief overview of them. &nbsp;It also lists the apps that are available for purchasing and a brief overview of them.<br /><br />Click on the App's icon or name to work with it.",
                                         "Select App" => "This screen shows you the detailed info about the application you selected (one that is assigned to your company). &nbsp;The screen displays a list of the users that are assigned to the application and allows you to see their settings and be able to select one to modify. &nbsp;There are also forms that allow you to assign/un-assign the app to users (limited to those users that are of the correct type for this application). &nbsp;You can also edit the <b>Server Info</b> which specifies which server the Portal knows about that users will be pointed to for the application and where the database is located.",
                                         "Edit User Info" => "This screen allows you to edit the preferences the selected user has for this app. &nbsp;You can change whether the app should start automatically when the user logs into the Portal and what the window width and height are for the main window of the app.",
                                         "Manage User Assignments" => "This form allows you to specify what users (that are valid for this app type) have the app assigned to them or not. &nbsp;Use the &gt;, &gt;&gt;, &lt;, &lt;&lt; buttons to move individual users or all users between the Assigned/Un-Assigned select boxes.",
                                         "Edit Server Info" => "This form allows you to specify what server (from the known configurations for this application) that the Portal will direct your companies users to when they access this application. &nbsp;This will also affect the database settings, but there is no migration support for the database at this time.",
                                         "Purchase App" => "This screen will allow you to purchase the selected Application for your company and specify how many licenses you need and what server the application will be hosted on.",
                                         "Sell App/Licenses" => "This screen will allow you to sell off some unused licenses or opt to sell the application and un-assign it from your company.",
                                         },
                               },
                     System => {
                               "_order_" => "This should never be reached",
                               "This should never be reached" => "The Company Admin app always runs in Normal mode!<br><br>Go <help app=\"\" content=\"Normal,Help\">here</help> to return to the Normal Help System.",
                               },
                   },
             );

=head1 NAME

Help - Object used to build a Help Object Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin::Data::Help;
  my $obj = Portal::CompanyAdmin::Data::Help->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Help is a Help class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(lang)

 Creates a new instance of the Portal::CompanyAdmin::Data::Help object.
 See Portal::Base(3) for a listing of required arguments.

 requires:
   lang - language code to work with.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( lang => "en", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{lang} = $args{lang};  # we will assume this language is valid as the Language module will have already validated it for us.
  $self->{Topics} = \%Topics;   # store the help data in the class.
  if (!exists $self->{Topics}->{$self->{lang}})
  {
    $self->{lang} = "en";  # This must always exist!
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS

  Topics - Help topics hash.  The language code is the first index,
           followed by Normal and System.  They provide the main indexes
           for the Normal and System modes.  Under them are the topics
           of interest for the 2 modes.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Help(3), Portal::Base(3)

=cut
