# Menu.pm - The Object Class that provides the Menus for CompanyAdmin App
# Created by James A. Pattie, 2004-06-25.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::CompanyAdmin::Objects::Menu;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

=head1 NAME

Menu - Object used to build a UserOpbject Menu Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin::Objects::Menu;
  my $obj = Portal::CompanyAdmin::Objects::Menu->new(methods,
                sessionObj, langObj, userObj);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Menu is a menu class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(methods, sessionObj, langObj, userObj, portalSession)

 Creates a new instance of the Portal::CompanyAdmin::Objects::Menu
 object.
 See Portal::Base(3) for a listing of required arguments.

 requires: methods - methods object,
           sessionObj - sessionObj for app,
           userObj - userObj for user

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( methods => undef, sessionObj => undef, userObj => undef, portalSession => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{methods} = $args{methods};
  $self->{sessionObj} = $args{sessionObj};
  $self->{userObj} = $args{userObj};
  $self->{portalSession} = $args{portalSession};
  $self->{app} = (split /::/, ref $self)[1];

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  $self->SUPER::isValid();

  # validate our parameters.
  if (! defined $self->{methods})
  {
    $self->missing("methods");
  }
  if (! defined $self->{sessionObj})
  {
    $self->missing("sessionObj");
  }
  if (! defined $self->{userObj})
  {
    $self->missing("userObj");
  }
  if (! defined $self->{portalSession})
  {
    $self->missing("portalSession");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# string print(location, state, command, doc, appName, appId)
# requires: location - (menubar, sidemenu), state, command, doc
# optional: appName, appId
# returns:  generated Menubar string
# generates the specified Menu.
sub print
{
  my $self = shift;
  my %args = ( location => "menubar", state => "Main", command => "display", doc => undef,
               appId => "", @_ );
  my $location = $args{location};
  my $state = $args{state};
  my $command = $args{command};
  my $doc = $args{doc};
  my $appId = $args{appId};
  my $result = "";

  if ($location !~ /^(menubar|sidemenu)$/)
  {
    $self->invalid("location", $location);
  }
  if ($state !~ /^(Main|User|Apps)$/)
  {
    $self->invalid("state", $state);
  }
  if ($command !~ /^(.+)$/)
  {
    $self->invalid("command", $command);
  }
  if (! defined $doc)
  {
    $self->missing("doc");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $result;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $result;
  }

  my $closeUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $self->{app}, state => "Main", command=> "close" });
  my $mainMenuUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $self->{app}, state => "Main", command => "display" });
  my %refreshArgs = (app => $self->{app}, state => $state, command => $command);
  $refreshArgs{appId} = $appId if ($appId ne "");
  my $refreshUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => \%refreshArgs);
  my $helpUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "display", helpApp => $self->{app}, content => "Normal" . ($state ne "Main" ? ",$state" : "") });
  my $createUserUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "createUser" });
  my $selectUserUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "display" });
  my $manageAppsUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $self->{app}, state => "Apps", command => "display" });

  my $close = $self->langObj->map("close");
  my $refresh = $self->langObj->map("refresh");
  my $help = $self->langObj->map("help");
  my $closeMessage = $self->langObj->map("closeMessage");
  my $changePasswordPhrase = $self->langObj->map("changePassword");
  my $userPreferencesPhrase = $self->langObj->map("userPreferences");
  my $userInfoPhrase = $self->langObj->map("userInfo");
  my $editUserInfoPhrase = $self->langObj->map("EditUserInfo");
  my $createUserPhrase = $self->langObj->map("CreateUser");
  my $editPhrase = $self->langObj->map("edit");
  my $userPhrase = $self->langObj->map("user");
  my $selectUserPhrase = $self->langObj->map("SelectUser");
  my $manageAppsPhrase = $self->langObj->map("ManageApps");
  my $editAppPhrase = $self->langObj->map("EditApp");
  my $purchaseAppPhrase = $self->langObj->map("PurchaseApplication");
  my $sellAppPhrase = $self->langObj->map("SellAppLicenses");

  my $mainMenu = $self->langObj->map("mainMenu");
  my $titleCommand = "";
  if ($state eq "Main")
  {
    $titleCommand = $mainMenu;
  }
  elsif ($state eq "User")
  {
    $titleCommand = ($command eq "changePassword" ? $changePasswordPhrase : ($command eq "display" ? $selectUserPhrase : ($command eq "createUser" ? $createUserPhrase : ($command eq "editUser" ? $editUserInfoPhrase : $userPreferencesPhrase))));
  }
  elsif ($state eq "Apps")
  {
    $titleCommand = ($command eq "display" ? $manageAppsPhrase : ($command eq "displayApp" ? $editAppPhrase : ($command eq "purchaseApp" ? $purchaseAppPhrase : ($command eq "sellApp" ? $sellAppPhrase : "unknown command"))));
  }
  my $title = sprintf($self->langObj->map("title"), $Portal::CompanyAdmin::VERSION, $titleCommand);

  $doc->setTitle($title);

  # build up the menuItems arrays
  my @itemsLeft = ();
  my @itemsRight = ();
  my @itemsCenter = ();

  if ($location eq "menubar")
  {
    my $helpWindowName = $self->{portalSession}->generateHelpWindowName($self->{app});
    if ($self->{portalSession}->error)
    {
      $self->error($self->{portalSession}->errorMessage);
      return undef;
    }
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$closeUrl", langObj => $self->{langObj},
                    text => $close, title => $close, onMouseOver => "window.status='$close'; return true;", onClick => "return validateClose();");
    $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "link", url => "$helpUrl", langObj => $self->{langObj},
                    text => $help, onMouseOver => "window.status='$help'; return true;",
                    onClick => "showHelp('$helpWindowName', '$helpUrl', 480, 640); return false;", title => $help);
    $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "link", url => "$refreshUrl", langObj => $self->{langObj},
                    text => $refresh, onMouseOver => "window.status='$refresh'; return true;", title => "$refresh");
    $itemsRight[0] = Portal::Objects::MenuItem->new(type => "text", text => "$userPhrase = (", langObj => $self->{langObj});
    $itemsRight[1] = Portal::Objects::MenuItem->new(type => "hilightedText", text => "$self->{userObj}->{uname}", langObj => $self->{langObj});
    $itemsRight[2] = Portal::Objects::MenuItem->new(type => "text", text => ")", langObj => $self->{langObj});

    # now validate they created ok.
    for (my $i=0; $i < scalar @itemsLeft; $i++)
    {
      if ($itemsLeft[$i]->error)
      {
        $self->error("itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
        return $result;
      }
    }
    for (my $i=0; $i < scalar @itemsRight; $i++)
    {
      if ($itemsRight[$i]->error)
      {
        $self->error("itemsRight[$i]\n<br>" . $itemsRight[$i]->errorMessage);
        return $result;
      }
    }
    $result = $self->{methods}->displayMenu(doc => $doc, orientation => "horizontal",
                    itemsLeft => \@itemsLeft, itemsRight => \@itemsRight, indent => 6);
  }
  elsif ($location eq "sidemenu")
  {
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$selectUserUrl", langObj => $self->{langObj},
                    text => $selectUserPhrase, title => $selectUserPhrase, onMouseOver => "window.status='$selectUserPhrase'; return true;");
    $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "link", url => "$createUserUrl", langObj => $self->{langObj},
                    text => "$createUserPhrase", title => $createUserPhrase, onMouseOver => "window.status='$createUserPhrase'; return true;");
    $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "link", url => "$manageAppsUrl", langObj => $self->{langObj},
                    text => $manageAppsPhrase, title => $manageAppsPhrase, onMouseOver => "window.status='$manageAppsPhrase'; return true;");
    if ($state ne "Main")
    {
      $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
      $itemsLeft[5] = Portal::Objects::MenuItem->new(type => "link", url => "$mainMenuUrl", langObj => $self->{langObj},
                      text => $mainMenu, onMouseOver => "window.status='$mainMenu'; return true;", title => "$mainMenu");
    }

    # now validate they created ok.
    for (my $i=0; $i < scalar @itemsLeft; $i++)
    {
      if ($itemsLeft[$i]->error)
      {
        $self->error("itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
        return $result;
      }
    }
    $result = $self->{methods}->displayMenu(orientation => "vertical",
                    itemsLeft => \@itemsLeft, indent => 6);
  }
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $result;
  }

  return $result;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  string print(location, state, command, doc, appId)
    requires: location - (menubar, sidemenu), state, command, doc
    optional: appId
    returns:  generated Menubar string
    generates the specified Menu.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::CompanyAdmin(3), Portal::Base(3)

=cut
