# StateTemplate.pm - The Object Class that provides a StateTemplate Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::CompanyAdmin::StateTemplate;
use strict;
use Portal::AppState;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Main Menu" );

=head1 NAME

StateTemplate - Object used to build a StateTemplate Object Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin::StateTemplate;
  my $obj = Portal::CompanyAdmin::StateTemplate->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

StateTemplate is a StateTemplate class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::CompanyAdmin::StateTemplate module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;
  $self->{state} = "StateTemplate";
  
  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "display");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # call the Portal::Forms::User module to do the form display work.
  my %output = $self->{userForms}->selectUserAdminForm(companyObj => $self->{companyObj},
      url => $url, app => $self->{app}, state => $self->{state} );
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }
  
  # now display the output to the user.
  $doc->print(%output);
  
  # finish off the screen, closing out what the displayMenu() method started.
  $doc = $self->closeMenu(doc => $doc);

  return $doc;
}

sub displayMenu
{
  my $self = shift;
  my %args = ( doc => undef, command => "", @_ );
  my $doc = $args{doc};
  my $command = $args{command};

  # create the help JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # create the validateClose JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  my $menuObj = Portal::CompanyAdmin::Objects::Menu->new(methods => $self->{methods},
                sessionObj => $self->{sessionObj}, langObj => $self->{langObj}, userObj => $self->{userObj},
                portalSession => $self->{portalSession});
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $menuString = $menuObj->print(location => "menubar", state => $self->{state}, command => $command, doc => $doc);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $sidemenuString = $menuObj->print(location => "sidemenu", state => $self->{state}, command => $command, doc => $doc);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top">$sidemenuString</td>
    <td width="75%" align="center">
END_OF_CODE

  return $doc;
}

sub closeMenu
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};
  
  $doc->print(<<"END_OF_FORM");
    </td>
  </tr>
</table>
END_OF_FORM

  return $doc;  
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  HTMLObject run(command)
    Runs the specified command and returns the resulting HTMLObject.


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::CompanyAdmin(3), Portal::AppState(3)

=cut
