# User.pm - The Object Class that provides a User Object
# Created by James A. Pattie, 2004-06-25.

# Copyright (c) 2000-2004, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::CompanyAdmin::User;
use strict;
use Portal::AppState;
use Portal::Forms::User;
use Portal::CompanyAdmin::Objects::Menu;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use Portal::Objects::UserPreferenceObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Select User", viewUser => "View User Info", editUser => "Edit User Info", changePassword => "Change User Password", editPreferences => "User Preferences", deleteUser => "Delete User", createUser => "Create User" );

=head1 NAME

User - Object used to build a User Object Class.

=head1 SYNOPSIS

  use Portal::CompanyAdmin::User;
  my $obj = Portal::CompanyAdmin::User->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

User is a User class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::CompanyAdmin::User module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;
  $self->{state} = "User";
  
  my $arguments = $self->arguments();
  eval "\$self->{userForms} = Portal::Forms::User->new($arguments);";
  if ($@)
  {
    $self->error($@);
    return $self;
  }
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return $self;
  }
  

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "display");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # call the Portal::Forms::User module to do the form display work.
  my %output = $self->{userForms}->selectUserAdminForm(companyObj => $self->{sessionObj}->{store}->{companyObj},
      url => $url, app => $self->{app}, state => $self->{state} );
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }
  
  # now display the output to the user.
  $doc->print(%output);
  
  # finish off the screen, closing out what the displayMenu() method started.
  $doc = $self->closeMenu(doc => $doc);

  return $doc;
}

=item HTMLObject c_viewUser()

=cut
sub c_viewUser
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "changePassword");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have a uname specified.
  if (!exists $self->{input}->{uname})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "uname");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{uname} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "uname", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $uname = $self->{input}->{uname};
  
  # lookup the specified user.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $uname);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpUser)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("userNotFound"), $uname);
    $doc = $self->c_display();
    return $doc;
  }
  
  # call the Portal::Forms::User module to do the core display, validation and update work
  my %output = $self->{userForms}->viewUser(user => $tmpUser, companyObj => $self->{sessionObj}->{store}->{companyObj});
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }
  
  # now display the output to the user.
  $doc->print(%output);
  
  # finish off the screen, closing out what the displayMenu() method started.
  $doc = $self->closeMenu(doc => $doc);

  return $doc;
}

=item HTMLObject c_editUser

=cut
sub c_editUser
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successEditPhrase = $self->langObj->map("successUserEdit");

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "editUser");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have a uname specified.
  if (!exists $self->{input}->{uname})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "uname");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{uname} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "uname", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $uname = $self->{input}->{uname};
  
  # lookup the specified user.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $uname);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpUser)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("userNotFound"), $uname);
    $doc = $self->c_display();
    return $doc;
  }
  
  # call the Portal::Forms::User module to do the core display, validation and update work
  my @result = $self->{userForms}->editUser(user => $tmpUser,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{userForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{userForms}->errorMessage;
    $self->{userForms}->clearErrors("all");
    $self->{userForms}->resetError;
    $doc = $self->c_display();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # update the session for our userObj.
    if ($uname eq $self->{userObj}->{uname})
    {
      $self->{portalSession}->{store}->{userObj} = $result[2];
      $self->{portalSession}->{store}->{changed} = 1;
    }
    
    # let the user know the edit suceeded.
    $self->{input}->{statusMessage} = $successEditPhrase;
    $doc = $self->c_display();
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_changePassword

=cut
sub c_changePassword
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successPasswordPhrase = $self->langObj->map("successPassword");

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "changePassword");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have a uname specified.
  if (!exists $self->{input}->{uname})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "uname");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{uname} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "uname", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $uname = $self->{input}->{uname};
  
  # lookup the specified user.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $uname);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpUser)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("userNotFound"), $uname);
    $doc = $self->c_display();
    return $doc;
  }
  
  # call the Portal::Forms::User module to do the core display, validation and update work
  my @result = $self->{userForms}->changePassword(user => $tmpUser,
      url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{userForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{userForms}->errorMessage;
    $self->{userForms}->clearErrors("all");
    $self->{userForms}->resetError;
    $doc = $self->c_display();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[2]);
  }
  elsif ($result[0] == 1)
  {
    $self->{input}->{statusMessage} = $successPasswordPhrase;
    $doc = $self->c_display();
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_deleteUser()

=cut
sub c_deleteUser
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successDeletePhrase = sprintf($self->langObj->map("successUserDelete"), $self->{input}->{uname});

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "deleteUser");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure we have a uname specified.
  if (!exists $self->{input}->{uname})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "uname");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{uname} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "uname", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $uname = $self->{input}->{uname};
  
  # lookup the specified user.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $uname);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpUser)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("userNotFound"), $uname);
    $doc = $self->c_display();
    return $doc;
  }
  
  if ($tmpUser->{uname} eq $self->{userObj}->{uname})
  {
    $self->{input}->{errorMessage} = $self->{langObj}->map("YouCanNotDeleteYourself");
    $doc = $self->c_display();
    return $doc;
  }
  
  # call the Portal::Forms::User module to do the update work
  my $result = $self->{userForms}->deleteUser(userObj => $tmpUser, companyObj => $self->{sessionObj}->{store}->{companyObj});
      
  if ($self->{userForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{userForms}->errorMessage;
    $self->{userForms}->clearErrors("all");
    $self->{userForms}->resetError;
    $doc = $self->c_display();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result->{result} == 1)
  {
    $self->{input}->{statusMessage} = $successDeletePhrase . $result->{message};
    $doc = $self->c_display();
    return $doc;
  }
  
  return $doc;
}

=item HTMLObject c_createUser

=cut
sub c_createUser
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successCreatePhrase = sprintf($self->langObj->map("successUserCreate"), $self->{input}->{uname});
  my $companyObj = $self->{sessionObj}->{store}->{companyObj};

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # start the page display overall with the menus.
  $doc = $self->displayMenu(doc => $doc, command => "createUser");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # call the Portal::Forms::User module to do the core display, validation and create work
  my @result = $self->{userForms}->createUser(companyObj => $companyObj, url => $url, app => $self->{app}, state => $self->{state});
      
  if ($self->{userForms}->error)
  {
    $self->{input}->{errorMessage} = $self->{userForms}->errorMessage;
    $self->{userForms}->clearErrors("all");
    $self->{userForms}->resetError;
    $doc = $self->c_display();
    return $doc;
  }
  
  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    # let the user know the create suceeded.
    $self->{input}->{statusMessage} = $successCreatePhrase . "<br />" . $result[4];
    $doc = $self->c_display();
    return $doc;
  }
  
  # finish off the screen, closing out what the displayMenu() method started.
  if ($result[0] != 0)
  {
    $doc = $self->closeMenu(doc => $doc);
  }

  return $doc;
}

=item HTMLObject c_editPreferences()

=cut
sub c_editPreferences
{
  my $self = shift;
  my $doc = $self->setupCSS();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successPreferencesPhrase = $self->langObj->map("successPreferences");

  # make sure we have a uname specified.
  if (!exists $self->{input}->{uname})
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("missingArgument"), "uname");
    $doc = $self->c_display();
    return $doc;
  }
  elsif (length $self->{input}->{uname} == 0)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("invalidArgument"), "uname", "");
    $doc = $self->c_display();
    return $doc;
  }
  my $uname = $self->{input}->{uname};
  
  # lookup the specified user.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $uname);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (!defined $tmpUser)
  {
    $self->{input}->{errorMessage} = sprintf($self->{langObj}->map("userNotFound"), $uname);
    $doc = $self->c_display();
    return $doc;
  }
  
  if (! exists $self->{input}->{update})
  {
    $doc = $self->displayUserPreferences(doc => $doc);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
  }
  else  # validate and update the users preferences.
  {
    my $colorScheme = $self->{input}->{colorScheme};
    my $dynamicContent = $self->{input}->{dynamicContent};
    my $frameBorder = $self->{input}->{frameBorder};
    my $frameScrolling = $self->{input}->{frameScrolling};
    my $frameWidth = $self->{input}->{frameWidth};
    my $frameHeight = $self->{input}->{frameHeight};

    # do validation
    my $errorString = "";
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my $systemColorScheme = grep($colorScheme, @systemColorSchemes);
    my $userColorScheme = grep($colorScheme, @userColorSchemes);
    my $invalidArgument = $self->{langObj}->map("invalidArgument");
    if ($systemColorScheme == 0 && $userColorScheme == 0)
    {
      $errorString .= sprintf($invalidArgument, "colorScheme", $colorScheme);
    }
    if ($dynamicContent !~ /^(1|0)$/)
    {
      $errorString .= sprintf($invalidArgument, "dynamicContent", $dynamicContent);
    }
    if ($frameBorder !~ /^(0|1)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameBorder", $frameBorder);
    }
    if ($frameScrolling !~ /^(yes|no|auto)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameScrolling", $frameScrolling);
    }
    if ($frameWidth !~ /^(\d+%?)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameWidth", $frameWidth);
    }
    if ($frameHeight !~ /^(\d+)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameHeight", $frameHeight);
    }

    if (length $errorString > 0)
    {
      $doc = $self->displayUserPreferences(doc => $doc, error => $errorString);
      if ($self->error)
      {
        $self->prefixError();
        return undef;
      }
    }
    else
    {
      # update the user preferences.

      # handle colorScheme, dynamicContent
      foreach my $name ("colorScheme", "dynamicContent", "dynamicContentConfig")
      {
        my $configObjStr;
        if ($name eq "dynamicContentConfig")
        {
          # handle dynamicContentConfig
          my $configObj;
          eval { $configObj = Portal::XML::ConfigObject->new(langObj => $self->{langObj}); };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          $configObj->{version} = "1.0";
          $configObj->{module} = "DynamicContentConfig";
          $configObj->{settings}->{frameBorder} = $frameBorder;
          $configObj->{settings}->{frameScrolling} = $frameScrolling;
          $configObj->{settings}->{frameHeight} = $frameHeight;
          $configObj->{settings}->{frameWidth} = $frameWidth;

          eval { $configObjStr = $configObj->generateXML; };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if (length $configObjStr == 0)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{langObj}->map("noConfigXMLGenerated"));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
        }

        my $module = ($name eq "colorScheme" ? "Global" : "Desktop");
        my $prefObj = Portal::Objects::UserPreferenceObject->new(id => $self->{userObj}->{id},
                      app => "Portal", module => $module, name => "$name",
                      value => ($name ne "dynamicContentConfig" ? $self->{input}->{$name} : $configObjStr), langObj => $self->{langObj});
        if ($prefObj->error)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $prefObj->errorMessage);
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        my $result = $self->{authObj}->updateUserPreference(userPreferenceObj => $prefObj);
        if ($self->{authObj}->error)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        if ($result == -1)
        { # have to create it
          $result = $self->{authObj}->createUserPreference(userPreferenceObj => $prefObj);
          if ($self->{authObj}->error)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if ($result == -1)
          { # already existed!  don't know how!
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceNowExists"), $prefObj->print));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == -2)
          { # app doesn't exist
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceAppNotExist"), $prefObj->print));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == 1)
          { # we are ok.
          }
        }
        elsif ($result == -2)
        { # app doesn't exist
          $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->langObj->map("updateUserPreferenceAppNotExist"), $prefObj->print));
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        elsif ($result == 1)
        { # we are ok.
        }
      }

      $doc = $self->displayUserPreferences(doc => $doc, status => $successPreferencesPhrase);
      if ($self->error)
      {
        $self->prefixError();
        return undef;
      }
    }
  }

  return $doc;
}

sub displayUserPreferences
{
  my $self = shift;
  my %args = ( doc => undef, error => "", status => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $status = $args{status};

  my $colorSchemePhrase = $self->langObj->map("colorScheme");
  my $dynamicContentPhrase = $self->langObj->map("dynamicContent");
  my $dynamicContentConfigPhrase = $self->langObj->map("dynamicContentConfig");
  my $frameBorderPhrase = $self->langObj->map("frameBorder");
  my $frameScrollablePhrase = $self->langObj->map("frameScrollable");
  my $frameWidthPhrase = $self->langObj->map("frameWidth");
  my $frameHeightPhrase = $self->langObj->map("frameHeight");
  my $editPhrase = $self->langObj->map("edit");
  my $userPreferencesPhrase = $self->langObj->map("userPreferences");

  # setup the XML parser needed.
  my $parserObj;
  eval { $parserObj = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->error($@);
    return $doc;
  }

  # get the Defaults for the preferences.
  my %defaults = ();  # store the defaults the system gives us.
  foreach my $prefModule ("Global", "Desktop")
  {
    my @names = ($prefModule eq "Global" ? ("colorScheme") : ("dynamicContent", "dynamicContent_Config"));
    foreach my $prefName (@names)
    {
      my $prefNameDefault = "Portal_" . $prefModule . "_" . $prefName . "_Default";
      my $prefValueDefault = $self->{methods}->getConfigValue(name => $prefNameDefault, portalDB => $self->{applicationObj}->{portalDB});
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return $doc;
      }
      if (!defined $prefValueDefault)
      {
        $self->error(sprintf($self->{langObj}->map("configEntryDoesNotExistUpgradeDatabase"), $prefNameDefault));
        return $doc;
      }
      if ($prefName eq "dynamicContent_Config" && $prefModule eq "Desktop")
      {
        # have to do the XML parsing
        my $configObj;
        eval { $configObj = $parserObj->parse(string => $prefValueDefault, module => "DynamicContentConfig", version => "1.0"); };
        if ($@)
        {
          $self->error(sprintf($self->{langObj}->map("parseFailed"), $@));
          return $doc;
        }
        foreach my $name (keys %{$configObj->{settings}})
        {
          $defaults{$prefModule}{$name} = $configObj->{settings}->{$name};
        }
      }
      else
      {
        $defaults{$prefModule}{$prefName} = $prefValueDefault;
      }
    }
  }

  # pull in the Portal preferences for the user.
  my @userPreferences = $self->{authObj}->getUserPreferences(userId => $self->{userObj}->{id},
                        app => "Portal");
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return $doc;
  }
  my %preferences = ();  # store the preferences info that we get from the system.
  foreach my $prefObj (@userPreferences)
  {
    if ($prefObj->{name} eq "dynamicContentConfig" && $prefObj->{module} eq "Desktop")
    {
      # have to do the XML parsing
      my $configObj;
      eval { $configObj = $parserObj->parse(string => $prefObj->{value}, module => "DynamicContentConfig", version => "1.0"); };
      if ($@)
      {
        $self->error(sprintf($self->{langObj}->map("parseFailed"), $@));
        return $doc;
      }
      foreach my $name (keys %{$configObj->{settings}})
      {
        $preferences{$prefObj->{module}}{$name} = $configObj->{settings}->{$name};
      }
    }
    else
    {
      $preferences{$prefObj->{module}}{$prefObj->{name}} = $prefObj->{value};
    }
  }

  # verify we have preferences to edit, if not we specify the default.
  foreach my $module ("Global", "Desktop")
  {
    my @names = ($module eq "Global" ? ("colorScheme") : ("frameScrolling", "frameBorder", "frameWidth", "frameHeight", "dynamicContent"));
    foreach my $name (@names)
    {
      $preferences{$module}{$name} = (exists $self->{input}->{$name} ? $self->{input}->{$name} : (exists $preferences{$module}{$name} ? $preferences{$module}{$name} : $defaults{$module}{$name}));
    }
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  $doc = $self->displayMenu(doc => $doc, command => "preferences");
  if ($self->error)
  {
    $self->prefixError();
    return $doc;
  }

  if ($error)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "error", message => $error,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }
  if ($status)
  {
    $self->{methods}->displayMessage(doc => $doc, type => "status", message => $status,
                                     langObj => $self->{langObj}, break => "below");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }

  # generate the form to let them edit their Portal Preferences.
  $doc->print(<<"END_OF_CODE");
$editPhrase $userPreferencesPhrase
<br>
<form name="preferences" method="post" action="$url">
  <input type="hidden" name="app" value="$self->{app}">
  <input type="hidden" name="state" value="$self->{state}">
  <input type="hidden" name="command" value="preferences">
  <input type="hidden" name="update" value="1">
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right">$colorSchemePhrase</td>
      <td align="left">
        <select name="colorScheme">
END_OF_CODE
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    foreach my $option (@systemColorSchemes, @userColorSchemes)
    {
      my $selected = ($preferences{Global}{colorScheme} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$dynamicContentPhrase</td>
      <td align="left">
        <select name="dynamicContent">
END_OF_CODE
    foreach my $option ("1", "0")
    {
      my $selected = ($preferences{Desktop}{dynamicContent} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>" . ($option ? "true" : "false") . "\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td colspan="2" align="center"><b>$dynamicContentConfigPhrase</b></td>
    </tr>
    <tr>
      <td align="right">$frameBorderPhrase</td>
      <td align="left">
        <select name="frameBorder">
END_OF_CODE
    foreach my $option ("0", "1")
    {
      my $selected = ($preferences{Desktop}{frameBorder} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameScrollablePhrase</td>
      <td align="left">
        <select name="frameScrolling">
END_OF_CODE
    foreach my $option ("auto", "yes", "no")
    {
      my $selected = ($preferences{Desktop}{frameScrolling} eq "$option" ? " selected" : "");
      $doc->print("          <option value=\"$option\"$selected>$option\n");
    }
  $doc->print(<<"END_OF_CODE");
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameWidthPhrase</td>
      <td align="left"><input type="text" name="frameWidth" value="$preferences{Desktop}{frameWidth}" size="4" maxlength="4"></td>
    </tr>
    <tr>
      <td align="right">$frameHeightPhrase</td>
      <td align="left"><input type="text" name="frameHeight" value="$preferences{Desktop}{frameHeight}" size="3" maxlength="3"></td>
    </tr>
  </table>
  <input type="submit" name="update" value="$editPhrase $userPreferencesPhrase">
</form>
    </td>
  </tr>
</table>
END_OF_CODE

  return $doc;
}

sub displayMenu
{
  my $self = shift;
  my %args = ( doc => undef, command => "", @_ );
  my $doc = $args{doc};
  my $command = $args{command};

  # create the help JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # create the validateClose JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  my $menuObj = Portal::CompanyAdmin::Objects::Menu->new(methods => $self->{methods},
                sessionObj => $self->{sessionObj}, langObj => $self->{langObj}, userObj => $self->{userObj},
                portalSession => $self->{portalSession});
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $menuString = $menuObj->print(location => "menubar", state => $self->{state}, command => $command, doc => $doc);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  my $sidemenuString = $menuObj->print(location => "sidemenu", state => $self->{state}, command => $command, doc => $doc);
  if ($menuObj->error)
  {
    $self->error($menuObj->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top">$sidemenuString</td>
    <td width="75%" align="center">
END_OF_CODE

  return $doc;
}

sub closeMenu
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};
  
  $doc->print(<<"END_OF_FORM");
    </td>
  </tr>
</table>
END_OF_FORM

  return $doc;  
}

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::CompanyAdmin(3), Portal::AppState(3)

=cut
