# Language.pm - The Object Class that provides a Language Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Example::Language;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.10';

=head1 NAME

Language - Object used to build a Language Object Class.

=head1 SYNOPSIS

  use Portal::Example::Language;
  my $obj = Portal::Example::Language->new(lang => $lang);
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Language is a Language class used to hold phrases of a specific language.  This object will be referenced for internationalization of the application.

=head1 Exported FUNCTIONS

=head2 scalar new(lang)

    Creates a new instance of the Portal::Example::Language
    object.  Internally sources the specified language if able and
    creates an internal data object that holds the language hash to
    work with for the Portal.

=cut
sub new
{
  my $that = shift;
  my $class = ref($that) || $that;
  my $self = bless {}, $class;
  my %args = ( lang => 'en', @_ );

  $self->{lang} = $args{lang};

  my $moduleName = $self->{lang};
  if (length $moduleName == 0) # it wasn't found, default to English.
  {
    $moduleName = $self->{lang} = "en";
  }
  if (!-e "Language/$moduleName". ".pm")
  {
    $moduleName = $self->{lang} = "en";
  }
  $moduleName = "Portal::Example::Language::" . $moduleName;

  # bring it into existance.
  eval "use $moduleName;";
  if ($@)
  {
    $self->setError(errorString => "Eval of $moduleName failed!<br>\nError = '$@'.<br>\n");

    return $self;
  }
  eval { $self->{phraseObj} = $moduleName->new; };  # Instantiate the Language Phrase object.  It will contain the charEncoding to use.
  if ($@)
  {
    $self->setError(errorString => "Instantiation of $moduleName failed!<br>\nError = '$@'.<br>\n");

    return $self;
  }
  if ($self->{phraseObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{phraseObj}->errorMessage);

    return $self;
  }

  # do validation
  $self->{error} = !$self->isValid;
  if ($self->{error})
  {
    $self->setError($self->{errorString});
  }

  return $self;
}

=head2 bool isValid(void)

    Returns 0 or 1 to indicate if the object is valid.  The error will be
    available via errorMessage().

=cut
sub isValid
{
  my $self = shift;
  my $error = 0;
  my $errorString = "";

  # do validation code here.

  if (length $errorString > 0)
  {
    $error = 1;
    $self->setError($errorString);
  }

  return !$error;
}

sub DESTROY
{
  my $self = shift;
}

sub AUTOLOAD   # Read only, setting does not work.
{
  my $self = shift;
  my $type = ref($self) || die "$self is not an object";
  my $name = $AUTOLOAD;
  $name =~ s/.*://;	# strip fully-qualified portion
  unless (exists $self->{$name})
  {
    die "Can't access `$name' field in object of class $type";
  }
  return $self->{$name};
}

=head2  void setError(errorString)

    Sets error = 1 and $self->{errorString} = errorString.

=cut
sub setError
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $callerErrStr = "$subName() - Error!<br>\n";

  if (scalar @_ == 1)
  {
    $self->{errorString} = $callerErrStr . @_[0];
  }
  else
  {
    my %args = ( @_ );
    if (!exists $args{errorString})  # make sure we get the errorString argument!
    {
      $self->setError("<b>errorString</b> is missing!<br>\n");
    }
    else
    {
      $self->{errorString} = $callerErrStr . $args{errorString};
    }
  }
  $self->{error} = 1;
}

=head2 scalar didErrorOccur(void)

    Returns the value of error.

=cut
sub didErrorOccur
{
  my $self = shift;

  return $self->{error};
}

=head2 scalar errorMessage(void)

    Returns the value of errorString.

=cut
sub errorMessage
{
  my $self = shift;

  return $self->{errorString};
}

=head2 string map(phrase_variable)

  maps language phrase variable to actual text.
  returned string will be "Error: ..." if phrase is missing.

=cut
sub map
{
  my $self = shift;
  my $phrasevar = shift;

  if ( $phrasevar eq "" )  
  { 
    return "Error, no phrase variable given!"; 
  }
  if ( exists $self->{phraseObj}->{phrases}->{$phrasevar} )
  {
    return $self->{phraseObj}->{phrases}->{$phrasevar}; 
  }  
  return "Error: $phrasevar";
}

1;
__END__

=head1 Exported VARIABLES

  LanguageObject phraseObj - The Language that has phrases and encoding.

  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Example(3)

=cut
