# Template.pm - The Object Class that provides a Template Language Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Example::Language::Template;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

# The Phrases used by the App.  - Add these to the Template and convert in each language as created.

my %phrases = {};
$phrases{stateNeeded} = "The State of the Application must be specified!";
$phrases{commandNeeded} = "The Command to execute must be specified!";
$phrases{sessionCookieNeeded} = "The Session Cookie must be defined!";
$phrases{sessionNotCreated} = "The Session does not exist but Session Cookie is defined!";
$phrases{commandNotKnown} = "Command = '%s' is not known!";
$phrases{stateNotKnown} = "State = '%s' is not known!";
$phrases{stateEvalFailed} = "Eval of State = '%s' failed!<br>\nError = '%s'.";
$phrases{appStateCommand} = "App = '%s', State = '%s', Command = '%s'.";
$phrases{close} = "Close App";
$phrases{help} = "Help";
$phrases{appVersion} = "Example Version = <font color=\"%s\">%s</font>";
$phrases{refresh} = "Refresh";
$phrases{title} = "Portal Example %s - %s";
$phrases{mainMenu} = "Main Menu";
$phrases{closeMessage} = "Press 'OK' if you really want to Close this Application,\\nelse press 'Cancel'.";
$phrases{helloWorld} = "Hello World!";
$phrases{error} = "Error";
$phrases{status} = "Status";
$phrases{lock} = "Lock";
$phrases{unlock} = "Un-Lock";
$phrases{allUsers} = "All Users";
$phrases{selectedUsers} = "Selected Users";
$phrases{specificCompany} = "Specific Company";
$phrases{breakLock} = "Break Lock";
$phrases{appNotAssigned} = "You do not have this application assigned to you";
$phrases{notAdminLockMessage} = "You are not an administrator so you can not lock/unlock this app";
$phrases{errorExclam} = "Error!";
$phrases{missingArgument} = "<b>%s</b> is missing";
$phrases{invalidArgument} = "<b>%s</b> = '%s' is invalid";

# new
# instantiates an instance of the Template Language Phrases.
sub new
{
  my $that = shift;
  my $class = ref($that) || $that;
  my $self = bless {}, $class;
  my %args = (  @_ );

  $self->{phrases} = { %phrases };
  $self->{encoding} = "iso-8859-1";

  # do validation
  $self->{error} = !$self->isValid;
  if ($self->{error})
  {
    $self->setError($self->{errorString});
  }

  return $self;
}

# isValid - Returns 0 or 1 to indicate if the object is valid.
sub isValid
{
  my $self = shift;
  my $error = 0;
  my $errorString = "";

  # do validation code here.

  if (length $errorString > 0)
  {
    $error = 1;
    $self->setError($errorString);
  }

  return !$error;
}

sub DESTROY
{
  my $self = shift;
}

sub AUTOLOAD  # act as though all calls are just queries, setting of values will not work.
{
  my $self = shift;
  my $type = ref($self) || die "$self is not an object";
  my $name = $AUTOLOAD;
  $name =~ s/.*://;	# strip fully-qualified portion
  unless (exists $self->{$name})
  {
    die "Can't access `$name' field in object of class $type";
  }
  return $self->{$name};
}

# setError
# requires: errorString
# returns: nothing
sub setError
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $callerErrStr = "$subName() - Error!<br>\n";

  if (scalar @_ == 1)
  {
    $self->{errorString} = $callerErrStr . @_[0];
  }
  else
  {
    my %args = ( @_ );
    if (!exists $args{errorString})  # make sure we get the errorString argument!
    {
      $self->setError("<b>errorString</b> is missing!<br>\n");
    }
    else
    {
      $self->{errorString} = $callerErrStr . $args{errorString};
    }
  }
  $self->{error} = 1;
}

# didErrorOccur
# returns the value of error.
sub didErrorOccur
{
  my $self = shift;

  return $self->{error};
}

# errorMessage
# returns the value of errorString
sub errorMessage
{
  my $self = shift;

  return $self->{errorString};
}

1;
__END__

=head1 NAME

Template - Object used to build a Template Object Class.

=head1 SYNOPSIS

  use Portal::Example::Language::Template;
  my $obj = Portal::Example::Language::Template->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Template is a Template Language class.

=head1 Exported FUNCTIONS

  scalar new(void)
    Creates a new instance of the Portal::Example::Language::Template
    object.

  bool isValid(void)
    Returns 0 or 1 to indicate if the object is valid.  The error will be
    available via errorMessage().
    
  void setError(errorString)
    Sets error = 1 and errorString = errorString.

  scalar didErrorOccur(void)
    Returns the value of error.

  scalar errorMessage(void)
    Returns the value of errorString.

=head1 Exported VARIABLES

  hash   phrases  - converted phrases
  scalar encoding - content type encoding for document.

  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Example(3)

=cut
