# Main.pm - The Object Class that provides a Main Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2001, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Example::Main;
use strict;
use Portal::AppState;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Main Menu", close => "Close App", lock => "Lock App", unlock => "Unlock App",
                 breakLock => "Break App Lock" );

=head1 NAME

Main - Object used to build a Main Object Class.

=head1 SYNOPSIS

  use Portal::Example::Main;
  my $obj = Portal::Example::Main->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

Main is a Main class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Example::Main module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;
  $self->{portalVariables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{portalVariables}->didErrorOccur)
  {
    $self->setError($self->{portalVariables}->errorMessage);
    return $self;
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  my $doc = $self->setupCSS();

  my $close = $self->langObj->map("close");
  my $refresh = $self->langObj->map("refresh");
  my $help = $self->langObj->map("help");
  my $mainMenu = $self->langObj->map("mainMenu");
  my $title = sprintf($self->langObj->map("title"), $VERSION, $mainMenu);
  my $closeMessage = $self->langObj->map("closeMessage");
  my $helloWorld = $self->langObj->map("helloWorld");
  my $lockPhrase = $self->langObj->map("lock");
  my $unlockPhrase = $self->langObj->map("unlock");
  my $allUsersPhrase = $self->langObj->map("allUsers");
  my $selectedUsersPhrase = $self->langObj->map("selectedUsers");
  my $specificCompanyPhrase = $self->langObj->map("specificCompany");
  my $breakLockPhrase = $self->langObj->map("breakLock");

  $doc->setTitle($title);

  # create the help and validateClose JS functions.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help, closeApp", langObj => $self->{langObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }
  my $closeUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Example", state => "Main", command => "close" });
  my $refreshUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Example", state => "Main", command => "display" });
  my $helpUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "display", helpApp => "Example", content => "Normal" });

  # build up the menuItems arrays
  my @itemsLeft = ();
  my @itemsRight = ();
  my @itemsCenter = ();

  my $helpWindow = "help_Example_" . $self->{portalSession}->{store}->{windowTitleTimestamp};

  $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$closeUrl", langObj => $self->{langObj},
                  text => $close, title => $close, onMouseOver => "window.status='$close'; return true;", onClick => "return validateClose();");
  $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
  $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "link", url => "$helpUrl", langObj => $self->{langObj},
                  text => $help, onMouseOver => "window.status='$help'; return true;",
                  onClick => "showHelp('$helpWindow', '$helpUrl', 480, 640); return false;", title => $help);
  $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
  $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "link", url => "$refreshUrl", langObj => $self->{langObj},
                  text => $refresh, onMouseOver => "window.status='$refresh'; return true;", title => "$refresh");

  # now validate they created ok.
  for (my $i=0; $i < scalar @itemsLeft; $i++)
  {
    if ($itemsLeft[$i]->didErrorOccur)
    {
      $self->setError(errorString => "itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
      return undef;
    }
  }
  my $menuString = $self->{methods}->displayMenu(orientation => "horizontal",
                  itemsLeft => \@itemsLeft, indent => 6);
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
$menuString
<br>
<h2>$helloWorld</h2>
END_OF_CODE
  if ($self->{userObj}->{admin})
  {
    my @lockedApps = $self->{applicationObj}->getLockedAppInfo(app => $self->{appObj}->{id});
    if ($self->{applicationObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{applicationObj}->errorMessage);
      return undef;
    }
    if (scalar @lockedApps > 0)
    {
      # give the form to select which lock to unlock.
      $doc->print(<<"END_OF_CODE");
<form name="unlockAppForm" method="post" action="$url">
  <input type="hidden" name="app" value="Example">
  <input type="hidden" name="state" value="Main">
  <input type="hidden" name="command" value="unlock">
  <select name="locks" size="5">
END_OF_CODE
    foreach my $lockedApp (@lockedApps)
    {
      next if ($lockedApp->{admin} != $self->{userObj}->{id});
      my $value = "admin=$lockedApp->{admin}|type=$lockedApp->{type}|users=$lockedApp->{users}|company=$lockedApp->{company}";
      $doc->print("    <option value=\"$value\">$value\n");
    }
    $doc->print(<<"END_OF_CODE");
  </select>
  <br>
  <input type="submit" name="unlock" value="$unlockPhrase">
</form>
END_OF_CODE

    # now give the lock break form if they are a sysadmin
    if ($self->{userObj}->{sysadmin})
    {
      $doc->print(<<"END_OF_CODE");
<form name="breakAppLockForm" method="post" action="$url">
  <input type="hidden" name="app" value="Example">
  <input type="hidden" name="state" value="Main">
  <input type="hidden" name="command" value="breakLock">
  <select name="locks" size="5">
END_OF_CODE
      foreach my $lockedApp (@lockedApps)
      {
        next if ($lockedApp->{admin} == $self->{userObj}->{id});
        my $value = "admin=$lockedApp->{admin}|type=$lockedApp->{type}|users=$lockedApp->{users}|company=$lockedApp->{company}";
        $doc->print("    <option value=\"$value\">$value\n");
      }
      $doc->print(<<"END_OF_CODE");
  </select>
  <br>
  <input type="submit" name="breakLock" value="$breakLockPhrase">
</form>
END_OF_CODE
    }
  }
  # give the form to lock it.

  # find the available users.
  my @users;
  if ($self->{userObj}->{sysadmin})
  {
    @users = $self->{authObj}->getAllUsersInPortal();
  }
  else
  {
    @users = $self->{authObj}->getListOfUsersForCompany(companyObj => $self->{companyObj});
  }
  if ($self->{authObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{authObj}->errorMessage);
    return undef;
  }
  my $availableUsers = "";
  foreach my $userObj (@users)
  {
    $availableUsers .= "  <option value=\"$userObj->{id}\"> $userObj->{fname} " . ($userObj->{mname} ? $userObj->{mname} . " " : "") . "$userObj->{lname} ($userObj->{uname})\n";
  }

  # find the available companies.
  my $availableCompanies = "";
  if ($self->{userObj}->{sysadmin})
  {
    my @companies = $self->{authObj}->getListOfCompanies();
    if ($self->{authObj}->didErrorOccur)
    {
      $self->setError(errorString => $self->{authObj}->errorMessage);
      return undef;
    }
    foreach my $companyObj (@companies)
    {
      $availableCompanies .= "  <option value=\"$companyObj->{id}\"> $companyObj->{name} ($companyObj->{code})\n";
    }
  }
  else
  {
    $availableCompanies .= "  <option value=\"$self->{companyObj}->{id}\"> $self->{companyObj}->{name} ($self->{companyObj}->{code})\n";
  }

  $doc->print(<<"END_OF_CODE");
<form name="lockAppForm" method="post" action="$url">
  <input type="hidden" name="app" value="Example">
  <input type="hidden" name="state" value="Main">
  <input type="hidden" name="command" value="lock">
END_OF_CODE
  if ($self->{userObj}->{sysadmin})
  {
    $doc->print(<<"END_OF_CODE");
  <input type="radio" name="lockType" value="a" checked> $allUsersPhrase<br>
END_OF_CODE
  }
  $doc->print(<<"END_OF_CODE");
  <input type="radio" name="lockType" value="u"> $selectedUsersPhrase<br>
  <select name="users" size="5" multiple>
  $availableUsers
  </select><br>
  <input type="radio" name="lockType" value="c"> $specificCompanyPhrase <select name="company">
  $availableCompanies
  </select><br>
  <input type="submit" name="lock" value="$lockPhrase">
</form>
END_OF_CODE
  }
  else
  {
    $doc->print("<br>" . $self->langObj->map("notAdminLockMessage") . "!<br>\n");
  }

  return $doc;
}

=item HTMLObject c_close()

=cut
sub c_close
{
  my $self = shift;
  my $doc;

  $doc = $self->processCloseAppEvent();
  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return undef;
  }

  return $doc;
}

=item HTMLObject c_lock()

=cut
sub c_lock
{
  my $self = shift;

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  my $doc = $self->setupCSS();

  my $lockType = $self->{input}->{lockType};
  my $users = $self->{input}->{users};
  my $company = $self->{input}->{company};

  if ($lockType !~ /^(a|u|c)$/)
  {
    $self->setError(errorString => "lockType = '$lockType' is invalid!<br>\n");
    return undef;
  }
  if ($lockType eq "u")
  {
    if (length $users == 0)
    {
      $self->setError(errorString => "lockType = 'u' and no users specified!<br>\n");
      return undef;
    }
    my @users = split /\0/, $users; # split on the null's.
    $users = join (",", @users);  # put it back together w/ commas
    if ($users !~ /^(\d+(,\d+)*)$/)
    {
      $self->setError(errorString => "users = '$users' is invalid!<br>\n");
      return undef;
    }
    $company = 0;
  }
  if ($lockType eq "c" && $company !~ /^(\d+)$/)
  {
    $self->setError(errorString => "company = '$company' is invalid!<br>\n");
    return undef;
  }
  # attempt to lock the app.

  $self->{applicationObj}->lockApp(app => $self->{appObj}->{id}, type => $lockType, users => $users, company => $company);
  if ($self->{applicationObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{applicationObj}->errorMessage);
    return undef;
  }

  # redirect back to the display command screen.
  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }
  my $displayUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Example", state => "Main", command => "display" });

  $doc->setMetaTag('http-equiv' => "Refresh", content => "5; URL=$displayUrl");
  $doc->setTitle("Lock OK");
  $doc->setFocus("body");
  $doc->print("<center><h1>Lock Succeeded!</h1></center>");

  return $doc;
}

=item HTMLObject c_unlock()

=cut
sub c_unlock
{
  my $self = shift;

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  my $doc = $self->setupCSS();

  my $lock = $self->{input}->{locks};
  my @values = split /\|/, $lock;
  if (scalar @values != 4)
  {
    $self->setError(errorString => "locks='$lock' is invalid!<br>\n");
    return undef;
  }
  (my $lockType = $values[1]) =~ s/^(type=)(a|c|u)$/$2/;
  (my $users = $values[2]) =~ s/^(users=)(.*)$/$2/;
  (my $company = $values[3]) =~ s/^(company=)(\d+)$/$2/;

  # attempt to unlock the app.

  $self->{applicationObj}->unlockApp(app => $self->{appObj}->{id}, type => $lockType, users => $users,
                                    company => $company);
  if ($self->{applicationObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{applicationObj}->errorMessage);
    return undef;
  }

  # redirect back to the display command screen.
  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }
  my $displayUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Example", state => "Main", command => "display" });

  $doc->setMetaTag('http-equiv' => "Refresh", content => "5; URL=$displayUrl");
  $doc->setTitle("Unlock OK");
  $doc->setFocus("body");
  $doc->print("<center><h1>Un-Lock Succeeded!</h1></center>");

  return $doc;
}

=item HTMLObject c_breakLock()

=cut
sub c_breakLock
{
  my $self = shift;

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  my $doc = $self->setupCSS();

  my $lock = $self->{input}->{locks};
  my @values = split /\|/, $lock;
  if (scalar @values != 4)
  {
    $self->setError(errorString => "locks='$lock' is invalid!<br>\n");
    return undef;
  }
  (my $admin = $values[0]) =~ s/^(admin=)(\d+)$/$2/;
  (my $lockType = $values[1]) =~ s/^(type=)(a|c|u)$/$2/;
  (my $users = $values[2]) =~ s/^(users=)(.*)$/$2/;
  (my $company = $values[3]) =~ s/^(company=)(\d+)$/$2/;

  # attempt to unlock the app by breaking the lock.

  $self->{applicationObj}->breakAppLock(admin => $admin, app => $self->{appObj}->{id}, type => $lockType, users => $users,
                                    company => $company);
  if ($self->{applicationObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{applicationObj}->errorMessage);
    return undef;
  }

  # redirect back to the display command screen.
  my $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }
  my $displayUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Example", state => "Main", command => "display" });

  $doc->setMetaTag('http-equiv' => "Refresh", content => "5; URL=$displayUrl");
  $doc->setTitle("Lock Broken OK");
  $doc->setFocus("body");
  $doc->print("<center><h1>Un-Lock Succeeded!</h1></center>");

  return $doc;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  HTMLObject run(command)
    Runs the specified command and returns the resulting HTMLObject.


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Example(3), Portal::AppState(3)

=cut
