# PortalSetup.pm - The Object Class that provides a PortalSetup Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Example::PortalSetup;
use strict;
use Portal::AppState;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( cleanupDatabase => "Cleans up the Database", setupCompany => "Sets up a Company in the app", setupUser => "Sets up a User in the app", updateUser => "Updates a User in the app",
                 cleanupUser => "Removes a User from the App", );

=head1 NAME

PortalSetup - Object used to build a PortalSetup Object Class.

=head1 SYNOPSIS

  use Portal::Example::PortalSetup;
  my $obj = Portal::Example::PortalSetup->new;
  if ($obj->didErrorOccur())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

PortalSetup is a PortalSetup class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Example::PortalSetup module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );


  if ($self->didErrorOccur)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;


  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# run
# takes: command
# returns: The HTMLObject document to display or undef if an error.
sub run
{
  my $self = shift;
  my %args = ( command => "", @_ );
  my $command = $args{command};
  my $doc;


  if (!exists $commands{$command})
  {
    $self->setError(errorString => sprintf($self->langObj->map("commandNotKnown"), $command));
    return undef;
  }

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{portalSession}->{store}->{browserCap}};

  $doc = HTMLObject::Normal->new;

  # get the users colorScheme to work with.
  my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => $self->{userObj}->{id});
  if ($self->{applicationObj}->didErrorOccur)
  {
    $self->setError(errorString => $self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $colorScheme)
  {
    $self->setError(errorString => "No valid colorScheme returned!<br>\n");
    return undef;
  }

  # create the needed css entries
  $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
  if ($self->{methods}->didErrorOccur)
  {
    $self->setError(errorString => $self->{methods}->errorMessage);
    return undef;
  }

  if ($command eq "cleanupDatabase")
  {
    $self->cleanupDatabase(dataDir => $self->{input}->{dataDir});
    if ($self->didErrorOccur)
    {
      $doc->setFocus("body");
      $doc->print("Cleanup of Database failed!...<br>\nError = " . $self->errorMessage);

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 0;");
    }

    $doc->setFocus("body");
    $doc->print("Cleanup Of Database Succeeded...");

    $doc->setOnload(code => "window.opener.configForm.configResult.value = 1;\nwindow.close();");
  }
  elsif ($command eq "setupCompany")
  {
    $doc = $self->setupCompany();
    if ($self->didErrorOccur)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }
      $doc->setFocus("body");
      $doc->print("Setup of Company failed!...<br>\nError = " . $self->errorMessage);

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 0;");
    }

    if (!defined $doc)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }

      $doc->setFocus("body");
      $doc->print("Setup Of Company Succeeded...");

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 1;\nwindow.close();");
    }
  }
  elsif ($command eq "setupUser")
  {
    $doc = $self->setupUser();
    if ($self->didErrorOccur)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }
      $doc->setFocus("body");
      $doc->print("Setup of User failed!...<br>\nError = " . $self->errorMessage);

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 0;");
    }

    if (!defined $doc)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }

      $doc->setFocus("body");
      $doc->print("Setup Of User Succeeded...");

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 1;\nwindow.close();");
    }
  }
  elsif ($command eq "updateUser")
  {
    $doc = $self->updateUser();
    if ($self->didErrorOccur)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }
      $doc->setFocus("body");
      $doc->print("Update of User failed!...<br>\nError = " . $self->errorMessage);

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 0;");
    }

    if (!defined $doc)
    {
      $doc = HTMLObject::Normal->new;
      # create the needed css entries
      $doc = $self->{methods}->generateColorSchemeCSS(doc => $doc, colorScheme => $colorScheme);
      if ($self->{methods}->didErrorOccur)
      {
        $self->setError(errorString => $self->{methods}->errorMessage);
        return undef;
      }

      $doc->setFocus("body");
      $doc->print("Update Of User Succeeded...");

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 1;\nwindow.close();");
    }
  }
  elsif ($command eq "cleanupUser")
  {
    $self->cleanupUser();
    if ($self->didErrorOccur)
    {
      $doc->setFocus("body");
      $doc->print("Cleanup of User failed!...<br>\nError = " . $self->errorMessage);

      $doc->setOnload(code => "window.opener.configForm.configResult.value = 0;");
    }

    $doc->setFocus("body");
    $doc->print("Cleanup Of User Succeeded...");

    $doc->setOnload(code => "window.opener.configForm.configResult.value = 1;\nwindow.close();");
  }

  return $doc;
}

# cleanupDatabase
# takes: dataDir
# returns: nothing
# This routine dumps any data necessary and if a database is defined, destroys the database.
sub cleanupDatabase
{
  my $self = shift;
  my $companyObj = $self->{companyObj};
  my $userObj = $self->{userObj};
  my $appObj = $self->{appObj};
  my %args = ( @_ );
  my $dataDir = $args{dataDir};


  if (length $dataDir == 0 || ! -d $dataDir)
  {
    $self->setError(errorString => "dataDir = '$dataDir' is invalid or does not exist!<br>\n");
    return;
  }

  # do any cleanup necessary
}

# setupCompany
# takes: nothing
# returns: doc
# This routine proceeds to create a database for the company if needed and does any initialization necessary.  The application needs to provide an HTMLObject document that will start the user going through a configuration
# program for that application.  If the doc is undef on return and no error is set, then the calling app will assume that no extra configuration needs to be done.
# The calling program should have this output being done in another window and provide the user with a link to close that window when all configuration is done.
sub setupCompany
{
  my $self = shift;
  my $companyObj = $self->{companyObj};
  my $userObj = $self->{userObj};
  my $appObj = $self->{appObj};
  my $doc = undef;


  # do any setup necessary

  return $doc;
}

# setupUser
# takes: nothing
# returns: doc
# This routine configures the user in the application.  The application needs to provide an HTMLObject document that will start the user going through a configuration
# program for that application.  If the doc is undef on return and no error is set, then the calling app will assume that no extra configuration needs to be done.
# The calling program should have this output being done in another window and provide the user with a link to close that window when all configuration is done.
sub setupUser
{
  my $self = shift;
  my $companyObj = $self->{companyObj};
  my $userObj = $self->{userObj};
  my $appObj = $self->{appObj};
  my $doc = undef;


  # do any setup necessary

  return $doc;
}

# updateUser
# takes: nothing
# returns: doc
# This routine updates the users settings in the application.  This is mainly to synchronize any changes that were made to the users info in the portal_db
# that the application duplicates or relies on for user settings in its own database.  The application needs to provide an HTMLObject document that will start the user going through a configuration
# program for that application.  If the doc is undef on return and no error is set, then the calling app will assume that no extra configuration needs to be done.
# The calling program should have this output being done in another window and provide the user with a link to close that window when all configuration is done.
sub updateUser
{
  my $self = shift;
  my $companyObj = $self->{companyObj};
  my $userObj = $self->{userObj};
  my $appObj = $self->{appObj};
  my $doc = undef;


  # do any setup necessary

  return $doc;
}

# cleanupUser
# takes: nothing
# returns: nothing
# This routine archives any info on the user and then removes their data from active use.
sub cleanupUser
{
  my $self = shift;
  my $companyObj = $self->{companyObj};
  my $userObj = $self->{userObj};
  my $appObj = $self->{appObj};


  # do any cleanup necessary
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  HTMLObject run(command)
    Runs the specified command and returns the resulting HTMLObject.


  void cleanupDatabase(dataDir)
    This routine dumps any data necessary and if a database is defined,
    destroys the database.

  HTMLObject setupCompany()
    This routine proceeds to create a database for the company if needed
    and does any initialization necessary.  The application needs to
    provide an HTMLObject document that will start the user going through
    a configuration program for that application.  If the doc is undef on
    return and no error is set, then the calling app will assume that no
    extra configuration needs to be done.  The calling program should
    have this output being done in another window and provide the user
    with a link to close that window when all configuration is done.

  HTMLObject setupUser()
    This routine configures the user in the application.  The application
    needs to provide an HTMLObject document that will start the user going
    through a configuration program for that application.  If the doc is
    undef on return and no error is set, then the calling app will assume
    that no extra configuration needs to be done.  The calling program
    should have this output being done in another window and provide the
    user with a link to close that window when all configuration is done.

  HTMLObject updateUser()
    This routine updates the users settings in the application.  This is
    mainly to synchronize any changes that were made to the users info in
    the portal_db that the application duplicates or relies on for user
    settings in its own database.  The application needs to provide an
    HTMLObject document that will start the user going through a
    configuration program for that application.  If the doc is undef on
    return and no error is set, then the calling app will assume that no
    extra configuration needs to be done.  The calling program should have
    this output being done in another window and provide the user with a
    link to close that window when all configuration is done.

  void cleanupUser()
    This routine archives any info on the user and then removes their data
    from active use.


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Example(3), Portal::AppState(3)

=cut

