# Auth.pm - The Object Class that provides a Auth Object
# Created by James Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Auth;
use strict;
use Portal::Base;
use DBIWrapper;
use Portal::Objects::User;
use Portal::Objects::CompanyObject;
use Portal::Objects::CompanyBillingObject;
use Portal::Objects::UserRightObject;
use Portal::Objects::UserPreferenceObject;
use Portal::Data::Variables;
use Portal::Session;
use Portal::Replication;
use Portal::Objects::ReplicationObject;
use Digest::MD5;
use Crypt::CBC;
use Crypt::IDEA;
use Portal::Methods;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.15';

=head1 NAME

Auth - Object used to build a Auth Object Class.

=head1 SYNOPSIS

  use Portal::Auth;
  my $obj = Portal::Auth->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Auth is a Auth class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

  Any method that takes log as an optional parameter has log defaulting
  to 1 so that the successfull action will be logged.

=over 4

=item scalar new(portalDB, billingDB, sessionObj)

 Creates a new instance of the Application module.
 See Portal::Base(3) for a listing of required arguments.

 requires: portalDB - DBIWrapper pointing to the Portal Database
 optional: sessionObj - Portal Session
           billingDB - DBIWrapper pointing to the Billing Database.
                       (This is checked for when working with billing methods)

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( portalDB => undef, billingDB => undef, sessionObj => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{portalDB} = $args{portalDB};
  $self->{billingDB} = $args{billingDB};
  $self->{sessionObj} = $args{sessionObj};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  $self->{replicationObj} = Portal::Replication->new(portalDB => $self->{portalDB}, sessionObj => $self->{sessionObj}, langObj => $self->{langObj});
  if ($self->{replicationObj}->error)
  {
    $self->error($self->{replicationObj}->errorMessage);
    return $self;
  }

  $self->{methods} = Portal::Methods->new(langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{portalDB})
  {
    $self->missing("portalDB");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# Authentication helper functions.

# isUserValid
# takes: uname, password (unencrypted)
# returns: 1=OK, 0=Password does not match, -1=User not active, -2=Company not active, -3=Company does not exist that user belongs to, -4=User not found, -5=User locked out.
sub isUserValid
{
  my $self = shift;
  my %args = (uname => "", password => "", @_);
  my $uname = $args{uname};
  my $password = $args{password};

  if (length $uname == 0)
  {
    $self->error("uname must be specified!<br>\n");
    return 0;
  }
  if (length $password < 6)
  {
    $self->error("password must be at least 6 characters long!<br>\n");
    return 0;
  }
  if (!$self->isValid)
  {
    $self->error($self->{errorString});
    return 0;
  }

  # first see if this user is locked out, even if they don't exist!
  if ($self->isUserLockedOut(uname => $uname))
  {
    return -5;
  }
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }

  # now start by seeing if there is a user with this name.
  my $userObj = $self->getUserInfo(uname => $uname);
  if (not defined $userObj)  # verify we were returned an object.  This is how database errors are returned.
  {
    if ($self->error)
    {
      $self->error($self->{errorString});
      return 0;
    }
    else
    {  # user not found!
      $self->incrementInvalidLoginAttempt(uname => $uname);
      if ($self->error)
      {
        $self->prefixError();
      }
      return -4;
    }
  }
  if ($userObj->error)
  {
    $self->error($userObj->errorMessage);
    return 0;
  }

  # then we need to crypt the password.
  my $cryptedPassword = $self->md5EncryptPassword($password);

  # now we need to check and make sure the company this user is part of exists and is active.
  my $companyObj = $self->getCompanyInfo(id => $userObj->companyId);
  if (not defined $companyObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else
    {
      $self->incrementInvalidLoginAttempt(uname => $uname);
      if ($self->error)
      {
        $self->prefixError();
      }
      return -3; # company does not exist!
    }
  }
  else
  {
    if ($companyObj->error)
    {
      $self->error($companyObj->errorMessage);
      return 0;
    }
    else
    {
      # now make sure the company is active.
      if ($companyObj->active == 0)
      {
        $self->incrementInvalidLoginAttempt(uname => $uname);
        if ($self->error)
        {
          $self->prefixError();
        }
        return -2;  # company not active!
      }

      # now make sure the user is active.
      if ($userObj->active == 0)
      {
        $self->incrementInvalidLoginAttempt(uname => $uname);
        if ($self->error)
        {
          $self->prefixError();
        }
        return -1;  # user not active!
      }

      # now we check to make sure the password is valid.
      if ($cryptedPassword ne $userObj->password)
      {
        $self->incrementInvalidLoginAttempt(uname => $uname);
        if ($self->error)
        {
          $self->prefixError();
        }
        return 0;
      }
    }
  }

  # make sure there is no lockout entry still around.
  $self->clearInvalidLoginLock(uname => $uname);
  if ($self->error)
  {
    $self->prefixError();
  }

  return 1;  # all's ok.
}

=item bool isUserLockedOut(uname)

 Requires: uname - user name to look up in the user_lockouts_tb.

 Returns: 1 if the user is currently locked out, or 0 if not.

=cut
sub isUserLockedOut
{
  my $self = shift;
  my %args = ( uname => "", @_ );
  my $uname = $args{uname};
  my $lockedOut = 0;

  # get the config options
  my $maxFailedLogins = $self->{methods}->getConfigValue(name => "maxFailedLogins", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return 0;
  }
  my $failedLoginsTimeoutPeriod = $self->{methods}->getConfigValue(name => "failedLoginsTimeoutPeriod", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return 0;
  }

  if (length $uname)
  {
    # do a query and see if we get anything from the database.
    my $sth = $self->{portalDB}->read(sql => "SELECT uname, numtries, first_time, last_time FROM user_lockouts_tb WHERE uname = ?",
                                      plug => [ $uname ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    my @info = $sth->fetchrow_array;
    if (defined $info[0])
    {
      my $tries = $info[1];
      my $firstTime = $info[2];
      my $lastTime = $info[3];

      if ($tries >= $maxFailedLogins)
      {
        # first see if the timeout period has expired.

        # convert the lastTime value into the number of seconds past the epoc.
        my $currTimeSecs = $self->{methods}->formatDateString(date => "now", format => "%s");
        my $elapsedTimeSecs = $self->{methods}->getDateOffset(date => $lastTime, offset => "$failedLoginsTimeoutPeriod", format => "%s");
        if ($elapsedTimeSecs <= $currTimeSecs)
        {
          # they have spent their time locked out.  Clear the lock
          $self->clearInvalidLoginLock(uname => $uname);
          if ($self->error)
          {
            $self->prefixError();
            return 0;
          }
        }
        else
        {
          # they are still locked out.
          $lockedOut = 1;
        }
      }
    }
  }

  return $lockedOut;
}

=item void clearInvalidLoginLock(uname)

  Requires: uname - user name to remove the lock from.

  Returns: nothing

=cut
sub clearInvalidLoginLock
{
  my $self = shift;
  my %args = ( uname => "", @_ );
  my $uname = $args{uname};

  if (length $uname)
  {
    $self->{portalDB}->write(sql => "DELETE FROM user_lockouts_tb WHERE uname = ?", plug => [ $uname ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return;
    }
    $self->{portalDB}->commit;  # make the changes permanent.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
  }
}

=item void incrementInvalidLoginAttempt(uname)

 Requires: uname - user name to mark as invalidly logging in.

 Returns: nothing

=cut
sub incrementInvalidLoginAttempt
{
  my $self = shift;
  my %args = ( uname => "", @_ );
  my $uname = $args{uname};

  # get the config options
  my $maxFailedLogins = $self->{methods}->getConfigValue(name => "maxFailedLogins", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return 0;
  }
  my $failedLoginsTimeoutPeriod = $self->{methods}->getConfigValue(name => "failedLoginsTimeoutPeriod", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return 0;
  }

  if (length $uname)
  {
    # do a query and see if we get anything from the database.
    my $sth = $self->{portalDB}->read(sql => "SELECT uname, numtries, first_time, last_time FROM user_lockouts_tb WHERE uname = ?",
                                      plug => [ $uname ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    my @info = $sth->fetchrow_array;
    if (defined $info[0])
    {
      # we have to update the entry.
      my $tries = $info[1];
      $self->{portalDB}->write(sql => "UPDATE user_lockouts_tb set numtries = ?, last_time = now() WHERE uname = ?", plug => [ $tries + 1, $uname ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;  # make the changes permanent.
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
    }
    else
    {
      # we have to create the entry.
      $self->{portalDB}->write(sql => "INSERT INTO user_lockouts_tb (uname, numtries, first_time, last_time) VALUES (?, 1, now(), now())", plug => [ $uname ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;  # make the changes permanent.
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
    }
  }
}

# getUserInfo
# takes: id or uname - User name to lookup
# returns: User Object or undef if user not found.
sub getUserInfo
{
  my $self = shift;
  my %args = ( @_ );
  my $type = "";

  if (exists $args{uname})
  {
    $type = "uname";
    if (length $args{$type} == 0)
    {
      $self->error("uname must be specified!<br>\n");
      return undef;
    }
  }
  elsif (exists $args{id})
  {
    $type = "id";
    if (length $args{$type} == 0 || $args{$type} < 0)
    {
      $self->error("id = '$args{$type}' is invalid!<br>\n");
      return undef;
    }
  }
  else
  {
    $self->error("You must specify either id or uname!<br>\n");
    return undef;
  }
  my $userObj;

  # now we select this user from the system and return the object.
  my $sth = $self->{portalDB}->read(sql => "SELECT id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, company_id, time_format, tz, last_edited FROM user_tb WHERE $type = ?",
                                    plug => [ $args{$type} ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return undef;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $userObj = Portal::Objects::User->new(langObj => $self->{langObj});
    $userObj->populate(id => $info[0], uname => $info[1], fname => $info[2], mname => $info[3], lname => $info[4], email => $info[5],
             password => $info[6], empId => $info[7], ssn => $info[8], language => $info[9], comment => $info[10], active => $info[11],
             admin => $info[12], sysadmin => $info[13], companyId => $info[14], timeFormat => $info[15], tz => $info[16], lastEdited => $info[17]) if (!$userObj->error);
  }
  else
  {
    $userObj = undef;
  }

  return $userObj;
}

# getUserId
# takes: uname
# returns: id (-1 on error, -2 = user not found)
sub getUserId
{
  my $self = shift;
  my %args = ( uname => "", @_ );
  my $uname = $args{uname};
  my $id = -1;

  if (length $uname == 0)
  {
    $self->error("uname must be specified!<br>\n");
    return $id;
  }

  # now we select this user from the system and return the id.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM user_tb WHERE uname = ?",
                                    plug => [ $uname ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return undef;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $id = $info[0];
  }
  else
  {
    $id = -2;  # user not found.
  }

  return $id;
}

=item scalar md5EncryptPassword(password)

 md5 hashes the specified password.

 Usage: $pass = $authObj->md5EncryptPassword($password);

=cut
sub md5EncryptPassword
{
  my $self = shift;
  my $md5 = Digest::MD5->new;
  $md5->add(shift);
  return $md5->hexdigest;
}

# createUser
# takes a User userObj (password unencrypted) and creates the entry in the database after crypting the password.
# optional: log - indicate if we should log this action.
# returns 1 if ok, 0 if error.
sub createUser
{
  my $self = shift;
  my %args = ( userObj => undef, log => 1, @_ );
  my $userObj = $args{userObj};
  my $log = $args{log};

  if (not defined $userObj)
  {
    $self->error("userObj must be specified!<br>\n");
    return 0;
  }
  if (!$userObj->isValid)
  {
    $self->error($userObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  if ($userObj->passCrypted == 0)
  {
    $userObj->{password} = $self->md5EncryptPassword($userObj->{password});
    $userObj->{passCrypted} = 1;  # indicate we have now crypted the password.
  }

  # verify that the companyId specified does exist and is currently active.
  my $companyObj = $self->getCompanyInfo(id => $userObj->companyId);
  if (not defined $companyObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company does not exist!
    {
      $self->error("Company with ID = '$userObj->{companyId}' does not exist!<br>\n");
      return 0;
    }
  }
  else
  {
    if ($companyObj->error)
    {
      $self->error($companyObj->errorMessage);
      return 0;
    }
    if ($companyObj->active == 0)
    {
      $self->error("Company ($companyObj->{code}), ID = '$companyObj->{id}' is inactive!<br>\n");
      return 0;
    }
  }

  # verify that the user doesn't already exist.
  my $tmpUser = $self->getUserInfo(uname => $userObj->uname);
  if (not defined $tmpUser)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # The user doesn't already exist!
    {
      my $comment = $userObj->comment;
      # make sure that an ' quotes are escaped in the comment string.
      $comment =~ s/([^\\][^\\])'/$1\\'/g;  # only do it to those ' that are not already escaped.
      my $active = $userObj->{active};
      my $admin = $userObj->{admin};
      my $sysadmin = $userObj->{sysadmin};

      $self->{portalDB}->write(sql => "INSERT INTO user_tb (company_id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, tz, time_format, last_edited) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $userObj->companyId, $userObj->uname, $userObj->fname, $userObj->mname, $userObj->lname, $userObj->email, $userObj->password,
                                     $userObj->empId, $userObj->ssn, $userObj->language, $comment, $active, $admin, $sysadmin, $userObj->tz, $userObj->timeFormat, $userObj->lastEdited ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      else
      {
        $self->{portalDB}->commit;  # have to save the changes in the database.
        if ($self->{portalDB}->error)
        {
          $self->error($self->{portalDB}->errorMessage);
          return 0;
        }

        if ($log)
        { # log the User Creation event.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 7, extraInfo => "Created User = '$userObj->{uname}' in Company = '$companyObj->{code}'", userId => (defined $self->{sessionObj} ? "" : -1));
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }

        # now see if we need to replicate this change.
        my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_tb", createDBIObjects => 1);
        if ($self->{replicationObj}->error)
        {
          $self->error($self->{replicationObj}->errorMessage);
          return 0;
        }
        my $userId = -1;  # This has to be the correct thing in the replication databases.
        if (scalar @repDBs > 0)
        {
          my $tmpUser = $self->getUserInfo(uname => $userObj->uname);
          if (not defined $tmpUser)
          {
            if ($self->error)
            {
              $self->error("In Replication:<br>\n" . $self->errorMessage);
              return 0;
            }
            else  # The user doesn't exist!
            {
              $self->error("In Replication - User doesn't exist even though we just created him!<br>\nUser = '$userObj->{uname}'.<br>\n");
              return 0;
            }
          }
          else
          {
            if ($tmpUser->error)
            {
              $self->error("In Replication:<br>\n" . $tmpUser->errorMessage);
              return 0;
            }
            else
            {
              $userId = $tmpUser->{id};
            }
          }
        }
        foreach my $repObj (@repDBs)
        {
          # do the replication.
          $repObj->{dbObj}->write(sql => "INSERT INTO user_tb (id, company_id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, tz, time_format, last_edited) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $userId, $userObj->companyId, $userObj->uname, $userObj->fname, $userObj->mname, $userObj->lname, $userObj->email, $userObj->password,
                                     $userObj->empId, $userObj->ssn, $userObj->language, $comment, $active, $admin, $sysadmin, $userObj->tz, $userObj->timeFormat, $userObj->lastEdited ]);
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }
          else
          {
            $repObj->{dbObj}->commit;  # have to save the changes in the database.
            if ($repObj->{dbObj}->error)
            {
              $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
              return 0;
            }

            # now log the replication event
            $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User = '$userObj->{uname}' in Company = '$companyObj->{code}' to " . $repObj->print, userId => (defined $self->{sessionObj} ? "" : -1));
            if ($self->{methods}->error)
            {
              $self->error($self->{methods}->errorMessage);
              return 0;
            }

            # log the replication event into the replication database.
            $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User = '$userObj->{uname}' in Company = '$companyObj->{code}'", userId => (defined $self->{sessionObj} ? "" : -1));
            if ($self->{methods}->error)
            {
              $self->error($self->{methods}->errorMessage);
              return 0;
            }
          }
        } # end of replication code
      }
    }
  }
  else  # The user appears to exist.
  {
    if ($tmpUser->error)
    {
      $self->error($tmpUser->errorMessage);
      return 0;
    }
    else
    {
      $self->error("User = '" . $userObj->uname . "' already exists!<br>\n");
      return 0;
    }
  }

  return 1;
}

# updateUser
# takes a User userObj (password possibly unencrypted) and updates the entry in the database after crypting the password.
# optional: log - indicate if we should log this action.
# returns 1 if ok, 0 if error.
sub updateUser
{
  my $self = shift;
  my %args = ( userObj => undef, log => 1, @_ );
  my $userObj = $args{userObj};
  my $log = $args{log};

  if (not defined $userObj)
  {
    $self->error("userObj must be specified!<br>\n");
    return 0;
  }
  if (!$userObj->isValid)
  {
    $self->error($userObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  if ($userObj->passCrypted == 0)
  {
    $userObj->{password} = $self->md5EncryptPassword($userObj->{password});
    $userObj->{passCrypted} = 1;  # indicate we have now crypted the password.
  }

  # verify that the user already exists.
  my $tmpUser = $self->getUserInfo(uname => $userObj->uname);
  if (defined $tmpUser)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # The user exists!
    {
      my $oldValues = "";
      foreach my $item (qw(fname mname lname email password empId ssn language comment active admin sysadmin tz timeFormat))
      {
        if ($userObj->{$item} ne $tmpUser->{$item})
        {
          $oldValues .= ", " if (length $oldValues > 0);
          $oldValues .= "$item = " . ($item ne "password" ? $tmpUser->{$item} : "<changed>");
        }
      }
      my $comment = $userObj->comment;
      # make sure that an ' quotes are escaped in the comment string.
      $comment =~ s/([^\\][^\\])'/$1\\'/g;  # only do it to those ' that are not already escaped.
      my $active = $userObj->{active};
      my $admin = $userObj->{admin};
      my $sysadmin = $userObj->{sysadmin};

      $self->{portalDB}->write(sql => "UPDATE user_tb SET fname = ?, mname = ?, lname = ?, email = ?, password = ?, emp_id = ?, ssn = ?, language = ?, comment = ?, active = ?, admin = ?, sysadmin = ?, tz = ?, time_format = ?, last_edited = ? WHERE uname = ?",
                           plug => [ $userObj->fname, $userObj->mname, $userObj->lname, $userObj->email, $userObj->password,
                                     $userObj->empId, $userObj->ssn, $userObj->language, $comment, $active, $admin, $sysadmin,
                                     $userObj->tz, $userObj->timeFormat, $userObj->lastEdited, $userObj->uname ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      else
      {
        $self->{portalDB}->commit;  # have to save the changes in the database.
        if ($self->{portalDB}->error)
        {
          $self->error($self->{portalDB}->errorMessage);
          return 0;
        }

        if ($log)
        { # log the User Update event.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 8, extraInfo => "Updated User = '$userObj->{uname}'.  oldvalues = '$oldValues'");
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }

        # now see if we need to replicate this change.
        my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_tb", createDBIObjects => 1);
        if ($self->{replicationObj}->error)
        {
          $self->error($self->{replicationObj}->errorMessage);
          return 0;
        }
        foreach my $repObj (@repDBs)
        {
          # do the replication.
          $repObj->{dbObj}->write(sql => "UPDATE user_tb SET fname = ?, mname = ?, lname = ?, email = ?, password = ?, emp_id = ?, ssn = ?, language = ?, comment = ?, active = ?, admin = ?, sysadmin = ?, tz = ?, time_format = ?, last_edited = ? WHERE uname = ?",
                           plug => [ $userObj->fname, $userObj->mname, $userObj->lname, $userObj->email, $userObj->password,
                                     $userObj->empId, $userObj->ssn, $userObj->language, $comment, $active, $admin, $sysadmin,
                                     $userObj->tz, $userObj->timeFormat, $userObj->lastEdited, $userObj->uname ]);
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }
          else
          {
            $repObj->{dbObj}->commit;  # have to save the changes in the database.
            if ($repObj->{dbObj}->error)
            {
              $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
              return 0;
            }

            # now log the replication event
            $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User = '$userObj->{uname}' to " . $repObj->print);
            if ($self->{methods}->error)
            {
              $self->error($self->{methods}->errorMessage);
              return 0;
            }

            # log the replication event into the replication database.
            $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User = '$userObj->{uname}'.  oldvalues = '$oldValues'");
            if ($self->{methods}->error)
            {
              $self->error($self->{methods}->errorMessage);
              return 0;
            }
          }
        } # end of replication code
      }
    }
  }
  else  # The user does not appear to exist.
  {
    if ($tmpUser->error)
    {
      $self->error($tmpUser->errorMessage);
      return 0;
    }
    else
    {
      $self->error("User = '" . $userObj->uname . "' does not exist!<br>\n");
      return 0;
    }
  }

  return 1;
}

# getCompanyInfo
# takes: id or code
# returns: CompanyObject or undefined if error.
sub getCompanyInfo
{
  my $self = shift;
  my %args = ( @_ );
  my $type = "";
  my $companyObj = undef;

  if (exists $args{id})
  {
    $type = "id";
    if ($args{$type} < 0)
    {
      $self->error("id = '$args{$type}' is invalid!<br>\n");
      return undef;
    }
  }
  elsif (exists $args{code})
  {
    $type = "code";
    if (length $args{$type} == 0)
    {
      $self->error("code must be specified!<br>\n");
      return undef;
    }
  }
  else
  {
    $self->error("You must specify either id or code!<br>\n");
    return undef;
  }

  my $sth = $self->{portalDB}->read(sql => "SELECT id, parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, time_format, country, tz, last_edited, url, logo_url, billing_method FROM company_tb WHERE $type = ?",
                                plug => [ $args{$type} ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return undef;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $companyObj = Portal::Objects::CompanyObject->new(id => $info[0], parentId => $info[1], code => $info[2], name => $info[3], address1 => $info[4], address2 => $info[5], city => $info[6], state => $info[7],
                  zip => $info[8], phone => $info[9], fax => $info[10], email => $info[11], comment => $info[12], language => $info[13],
                  active => $info[14], timeFormat => $info[15], country => $info[16], tz => $info[17], lastEdited => $info[18],
                  url => $info[19], logoUrl => $info[20], billingMethod => $info[21], langObj => $self->{langObj});
  }

  return $companyObj;
}

# getCompanyBillingInfo
# takes: id
# returns: CompanyBillingObject
sub getCompanyBillingInfo
{
  my $self = shift;
  my %args = ( id => -1, @_ );
  my $companyBillObj = undef;

  my $id = $args{id};
  if ($id < 0)
  {
    $self->error("id = '$id' is invalid!<br>\n");
    return $companyBillObj;
  }
  if (not defined $self->{billingDB})
  {
    $self->error("billingDB is not defined!<br>\n");
    return $companyBillObj;
  }

  my $sth = $self->{billingDB}->read(sql => "SELECT id, cc_type, cc_num, cc_expire FROM company_tb WHERE id = $id");
  if ($self->{billingDB}->error)
  {
    $self->error($self->{billingDB}->errorMessage);
    return $companyBillObj;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    my $date = $info[3];
    if ($self->{billingDB}->{dbType} eq "mysql" || ($self->{billingDB}->{dbType} eq "Pg" && !$self->{billingDB}->{setDateStyle}))
    {
      $date = $self->swapDate(date => $date);
    }
    my $secret = $info[1];
    my $cipher = Crypt::CBC->new($secret, 'IDEA');
    my $ccNum = ($info[2] =~ /^(encrypted{)([^}]+)(})$/ ? $cipher->decrypt_hex($2) : $info[2]);
    $companyBillObj = Portal::Objects::CompanyBillingObject->new(id => $info[0], ccType => $info[1], ccNum => $ccNum, ccExpire => $date, langObj => $self->{langObj});
  }

  return $companyBillObj;
}

# createCompany
# takes: companyObj - CompanyObject describing the company to create
# optional: log - indicate if we should log this action.
# returns: 1=OK, 0=Error, -1 = already exists
sub createCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, log => 1, @_ );
  my $companyObj = $args{companyObj};
  my $log = $args{log};

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return 0;
  }

  if (!$companyObj->isValid)
  {
    $self->error($companyObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this company doesn't exist.
  my $tempCompany = $self->getCompanyInfo(code => $companyObj->code);
  if (not defined $tempCompany)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company doesn't exist yet!
    {
      my $comment = $companyObj->comment;
      # make sure that an ' quotes are escaped in the comment string.
      $comment =~ s/([^\\][^\\])'/$1\\'/g;  # only do it to those ' that are not already escaped.
      my $active = $companyObj->{active};

      $self->{portalDB}->write(sql => "INSERT INTO company_tb (parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, tz, time_format, country, last_edited, url, logo_url, billing_method) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                               plug => [ $companyObj->parentId, $companyObj->code, $companyObj->name, $companyObj->address1, $companyObj->address2, $companyObj->city, $companyObj->state, $companyObj->zip,
                                         $companyObj->phone, $companyObj->fax, $companyObj->email, $comment, $companyObj->language,
                                         $active, $companyObj->tz, $companyObj->timeFormat, $companyObj->country, "now()", $companyObj->url, $companyObj->logoUrl, $companyObj->billingMethod ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;  # make the changes permanent.
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Creation event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 4, extraInfo => "Created Company = '$companyObj->{code}'", userId => (defined $self->{sessionObj} ? "" : -1));
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->{replicationObj}->getEntries(tbName => "company_tb", createDBIObjects => 1);
      if ($self->{replicationObj}->error)
      {
        $self->error($self->{replicationObj}->errorMessage);
        return 0;
      }
      my $companyId = -1;
      if (scalar @repDBs > 0)
      {
        my $tempCompany = $self->getCompanyInfo(code => $companyObj->code);
        if (not defined $tempCompany)
        {
          if ($self->error)
          {
            $self->prefixError();
            return 0;
          }
          else  # company doesn't exist!
          {
            $self->error("In Replication - Company = '$companyObj->{code}' no longer exists and we just created it!<br>\n");
            return 0;
          }
        }
        else
        {
          if ($tempCompany->error)
          {
            $self->error($tempCompany->errorMessage);
            return 0;
          }
          else
          {
            $companyId = $tempCompany->{id};
          }
        }
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->{dbObj}->write(sql => "INSERT INTO company_tb (id, parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, tz, time_format, country, last_edited, url, logo_url, billing_method) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                               plug => [ $companyId, $companyObj->parentId, $companyObj->code, $companyObj->name, $companyObj->address1, $companyObj->address2, $companyObj->city, $companyObj->state, $companyObj->zip,
                                         $companyObj->phone, $companyObj->fax, $companyObj->email, $comment, $companyObj->language,
                                         $active, $companyObj->tz, $companyObj->timeFormat, $companyObj->country, "now()", $companyObj->url, $companyObj->logoUrl, $companyObj->billingMethod ]);
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }
        else
        {
          $repObj->{dbObj}->commit;  # have to save the changes in the database.
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated Company = '$companyObj->{code}' to " . $repObj->print, userId => (defined $self->{sessionObj} ? "" : -1));
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated Company = '$companyObj->{code}'", userId => (defined $self->{sessionObj} ? "" : -1));
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }
  else
  {
    if ($tempCompany->error)
    {
      $self->error($tempCompany->errorMessage);
      return 0;
    }
    else
    {
      return -1;  # already exists!
    }
  }

  return 1;
}

# updateCompany
# takes: companyObj - CompanyObject
# optional: log - indicate if we should log this action.
# returns: 1 = OK, 0 = Error, -1 = Company doesn't exist
sub updateCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, log => 1, @_ );
  my $companyObj = $args{companyObj};
  my $log = $args{log};

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return 0;
  }

  if (!$companyObj->isValid)
  {
    $self->error($companyObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this company does exist.
  my $tempCompany = $self->getCompanyInfo(id => $companyObj->id);
  if (not defined $tempCompany)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company doesn't exist yet!
    {
      return -1;
    }
  }
  else
  {
    if ($tempCompany->error)
    {
      $self->error($tempCompany->errorMessage);
      return 0;
    }
    else
    {
      my $comment = $companyObj->comment;
      # make sure that an ' quotes are escaped in the comment string.
      $comment =~ s/([^\\][^\\])'/$1\\'/g;  # only do it to those ' that are not already escaped.
      my $active = $companyObj->{active};

      $self->{portalDB}->write(sql => "UPDATE company_tb SET name = ?, address1 = ?, address2 = ?, city = ?, state = ?, zip = ?, phone = ?, fax = ?, email = ?, comment = ?, language = ?, active = ?, tz = ?, time_format = ?, country = ?, last_edited = now(), url = ?, logo_url = ?, billing_method = ? WHERE id = ?",
                               plug => [ $companyObj->name, $companyObj->address1, $companyObj->address2, $companyObj->city, $companyObj->state, $companyObj->zip,
                                         $companyObj->phone, $companyObj->fax, $companyObj->email, $comment, $companyObj->language,
                                         $active, $companyObj->tz, $companyObj->timeFormat, $companyObj->country, $companyObj->url, $companyObj->logoUrl, $companyObj->billingMethod, $companyObj->id ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;  # make the changes permanent.
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Update event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 5, extraInfo => "Updated Company = '$companyObj->{code}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->{replicationObj}->getEntries(tbName => "company_tb", createDBIObjects => 1);
      if ($self->{replicationObj}->error)
      {
        $self->error($self->{replicationObj}->errorMessage);
        return 0;
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->{dbObj}->write(sql => "UPDATE company_tb SET name = ?, address1 = ?, address2 = ?, city = ?, state = ?, zip = ?, phone = ?, fax = ?, email = ?, comment = ?, language = ?, active = ?, tz = ?, time_format = ?, country = ?, last_edited = now(), url = ?, logo_url = ?, billing_method = ? WHERE id = ?",
                               plug => [ $companyObj->name, $companyObj->address1, $companyObj->address2, $companyObj->city, $companyObj->state, $companyObj->zip,
                                         $companyObj->phone, $companyObj->fax, $companyObj->email, $comment, $companyObj->language,
                                         $active, $companyObj->tz, $companyObj->timeFormat, $companyObj->country, $companyObj->url, $companyObj->logoUrl, $companyObj->billingMethod, $companyObj->id ]);
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }
        else
        {
          $repObj->{dbObj}->commit;  # have to save the changes in the database.
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated Company = '$companyObj->{code}' to " . $repObj->print);
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated Company = '$companyObj->{code}'");
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }

  return 1;
}

# createCompanyBilling
# takes: companyBillObj - CompanyBillingObject
# optional: log - indicate if we should log this action.
# returns: 1=OK, 0=Error, -1 = already exists
sub createCompanyBilling
{
  my $self = shift;
  my %args = ( companyBillObj => undef, log => 1, @_ );
  my $companyBillObj = $args{companyBillObj};
  my $log = $args{log};

  if (not defined $companyBillObj)
  {
    $self->error("companyBillObj is not defined!<br>\n");
    return 0;
  }
  if (!$companyBillObj->isValid)
  {
    $self->error($companyBillObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }
  if (not defined $self->{billingDB})
  {
    $self->error("billingDB is not defined!<br>\n");
    return 0;
  }

  # first make sure that no Billing info exists for this company.
  my $tempCompanyBill = $self->getCompanyBillingInfo(id => $companyBillObj->id);
  if (not defined $tempCompanyBill)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company Billing doesn't exist yet!
    {
      my $date = $companyBillObj->ccExpire;
      if ($self->{billingDB}->{dbType} eq "mysql" || ($self->{billingDB}->{dbType} eq "Pg" && !$self->{billingDB}->{setDateStyle}))
      {
        $date = $self->swapDate(date => $date);
      }
      my $secret = $companyBillObj->ccType;
      my $cipher = Crypt::CBC->new($secret, 'IDEA');
      my $ccNum = "encrypted{" . $cipher->encrypt_hex($companyBillObj->ccNum) . "}";
      $self->{billingDB}->write(sql => "INSERT INTO company_tb (id, cc_type, cc_num, cc_expire) VALUES (?, ?, ?, ?)",
                               plug => [ $companyBillObj->id, $companyBillObj->ccType, $ccNum, $date ]);
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }
      $self->{billingDB}->commit;  # make the changes permanent.
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Update event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{billingDB}, action => 4, extraInfo => "Created Company with ID = '$companyBillObj->{id}'", userId => (defined $self->{sessionObj} ? "" : -1));
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    }
  }
  else
  {
    if ($tempCompanyBill->error)
    {
      $self->error($tempCompanyBill->errorMessage);
      return 0;
    }
    else
    {
      return -1;  # already exists
    }
  }

  return 1;
}

# updateCompanyBilling
# takes: companyBillObj - CompanyBillingObject
# optional: log - indicate if we should log this action.
# returns: 1 = OK, 0 = Error, -1 = doesn't exist
sub updateCompanyBilling
{
  my $self = shift;
  my %args = ( companyBillObj => undef, log => 1, @_ );
  my $companyBillObj = $args{companyBillObj};
  my $log = $args{log};

  if (not defined $companyBillObj)
  {
    $self->error("companyBillObj is not defined!<br>\n");
    return 0;
  }
  if (!$companyBillObj->isValid)
  {
    $self->error($companyBillObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }
  if (not defined $self->{billingDB})
  {
    $self->error("billingDB is not defined!<br>\n");
    return 0;
  }

  # first make sure the Billing info exists for this company.
  my $tempCompanyBill = $self->getCompanyBillingInfo(id => $companyBillObj->id);
  if (not defined $tempCompanyBill)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company Billing doesn't exist yet!
    {
      return -1;
    }
  }
  else
  {
    if ($tempCompanyBill->error)
    {
      $self->error($tempCompanyBill->errorMessage);
      return 0;
    }
    else
    {
      my $date = $companyBillObj->ccExpire;
      if ($self->{billingDB}->{dbType} eq "mysql" || ($self->{billingDB}->{dbType} eq "Pg" && !$self->{billingDB}->{setDateStyle}))
      {
        $date = $self->swapDate(date => $date);
      }
      my $secret = $companyBillObj->ccType;
      my $cipher = Crypt::CBC->new($secret, 'IDEA');
      my $ccNum = "encrypted{" . $cipher->encrypt_hex($companyBillObj->ccNum) . "}";
      $self->{billingDB}->write(sql => "UPDATE company_tb SET cc_type = ?, cc_num = ?, cc_expire = ? WHERE id = ?",
                               plug => [ $companyBillObj->ccType, $ccNum, $date, $companyBillObj->id ]);
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }
      $self->{billingDB}->commit;  # make the changes permanent.
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Update event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{billingDB}, action => 5, extraInfo => "Updated Company with ID = '$companyBillObj->{id}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    }
  }

  return 1;
}

# getListOfCompanies
# optional: like (string to search against the company names via the LIKE SQL operator),
#           field (name of the field to compare against using LIKE - must be a string field),
#           orderBy (the order by string - must be valid)
#           parentId (specify the parentId value to match against)
# returns: Array of CompanyObject that represent all companies in the Portal.
sub getListOfCompanies
{
  my $self = shift;
  my %args = ( like => "", field => "", orderBy => "name", parentId => "", @_ );
  my $like = $args{like};
  my $field = $args{field};
  my $orderBy = $args{orderBy};
  my $parentId = $args{parentId};
  my @Companies = ();

  if (length $orderBy == 0)
  {
    $self->error("orderBy = '$orderBy' is invalid!<br>\n");
    return undef;
  }

  my $likeString = ((length $like > 0 && length $field > 0) ? "WHERE $field LIKE '$like\%'" : "");
  my $parentIdString = ($parentId =~ /^(-1|\d+)$/ ? (length $likeString > 0 ? " AND parent_id = $parentId" : "WHERE parent_id = $parentId") : "");

  my $sth = $self->{portalDB}->read(sql => "SELECT id, parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, time_format, country, tz, last_edited, url, logo_url, billing_method FROM company_tb $likeString$parentIdString ORDER BY $orderBy");
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return undef;
  }
  while (my @info = $sth->fetchrow_array)
  {
    my $companyObj = Portal::Objects::CompanyObject->new(id => $info[0], parentId => $info[1], code => $info[2], name => $info[3], address1 => $info[4], address2 => $info[5], city => $info[6], state => $info[7],
                  zip => $info[8], phone => $info[9], fax => $info[10], email => $info[11], comment => $info[12], language => $info[13],
                  active => $info[14], timeFormat => $info[15], country => $info[16], tz => $info[17], lastEdited => $info[18],
                  url => $info[19], logoUrl => $info[20], billingMethod => $info[21], langObj => $self->{langObj});
    if ($companyObj->error)
    {
      $self->error($companyObj->errorMessage);
      return undef;
    }
    $Companies[++$#Companies] = $companyObj;
  }

  return @Companies;
}

# getListOfUsersForCompany
# takes: companyObj CompanyObject, like (string to search against the user names via the LIKE SQL operator), admin, sysadmin
# returns: Array of User Objects that represent all users of the specified company.
# if admin or sysadmin are set to 1, then it will only return users that are admins or sysadmins if so defined.
sub getListOfUsersForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, like => "", admin => 0, sysadmin => 0, @_ );
  my $companyObj = $args{companyObj};
  my $like = $args{like};
  my $admin = $args{admin};
  my $sysadmin = $args{sysadmin};
  my @Users = ();

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return @Users;
  }

  if (!$companyObj->isValid)
  {
    $self->error($companyObj->errorMessage);
    return @Users;
  }

  if ($admin !~ /^(1|0)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return @Users;
  }
  if ($sysadmin !~ /^(1|0)$/)
  {
    $self->error("sysadmin = '$sysadmin' is invalid!<br>\n");
    return @Users;
  }

  my $likeString = (length $like > 0 ? " AND uname LIKE '$like\%'" : "");
  my $adminString = ($admin ? " AND admin = '1'" : "");
  $adminString .= ($sysadmin ? " AND sysadmin = '1'" : "");

  # first make sure this company exists.
  my $tempCompany = $self->getCompanyInfo(code => $companyObj->code);
  if (not defined $tempCompany)
  {
    if ($self->error)
    {
      $self->prefixError();
      return @Users;
    }
    else  # company doesn't exist yet!
    {
      $self->error("Company ($companyObj->{code}) doesn't exist!<br>\n");
      return @Users;
    }
  }
  else
  {
    if ($tempCompany->error)
    {
      $self->error($tempCompany->errorMessage);
      return @Users;
    }
    else
    {  # get list of Users for this company.
      my $sth = $self->{portalDB}->read(sql => "SELECT id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, company_id, time_format, tz, last_edited FROM user_tb WHERE company_id = ?$likeString$adminString ORDER BY lname, fname, uname",
                                        plug => [ $companyObj->{id} ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return @Users;
      }
      while (my @info = $sth->fetchrow_array)
      {
        my $userObj = Portal::Objects::User->new(langObj => $self->{langObj});
        $userObj->populate(id => $info[0], uname => $info[1], fname => $info[2], mname => $info[3], lname => $info[4], email => $info[5],
                 password => $info[6], empId => $info[7], ssn => $info[8], language => $info[9], comment => $info[10], active => $info[11],
                 admin => $info[12], sysadmin => $info[13], companyId => $info[14], timeFormat => $info[15], tz => $info[16], lastEdited => $info[17]) if (!$userObj->error);
        if ($userObj->error)
        {
          $self->error($userObj->errorMessage);
          return @Users;
        }
        $Users[++$#Users] = $userObj;
      }
    }
  }

  return @Users;
}

# getAllUsersInPortal
# takes: like, admin, sysadmin
# returns: Array of User Objects that define all users in the portal that match the requirements sorted by company.
# This routine works similar to getListOfUsersForCompany in parameters it takes.
sub getAllUsersInPortal
{
  my $self = shift;
  my %args = ( like => "", admin => 0, sysadmin => 0, @_ );
  my $like = $args{like};
  my $admin = $args{admin};
  my $sysadmin = $args{sysadmin};
  my @Users = ();

  if ($admin !~ /^(1|0)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return @Users;
  }
  if ($sysadmin !~ /^(1|0)$/)
  {
    $self->error("sysadmin = '$sysadmin' is invalid!<br>\n");
    return @Users;
  }

  my $likeString = (length $like > 0 ? " uname LIKE '$like\%'" : "");
  my $adminString = ($admin ? (length $like > 0 ? " AND " : " ") . "admin = '1'" : "");
  $adminString .= ($sysadmin ? (length $like > 0 || $admin ? " AND " : " ") . "sysadmin = '1'" : "");
  my $whereString = "";
  if (length $likeString > 0 || length $adminString > 0)
  {
    $whereString = "WHERE$likeString$adminString";
  }

  my $sth = $self->{portalDB}->read(sql => "SELECT id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, company_id, time_format, tz, last_edited FROM user_tb $whereString ORDER BY lname, fname, uname, company_id");
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return @Users;
  }
  while (my @info = $sth->fetchrow_array)
  {
    my $userObj = Portal::Objects::User->new(langObj => $self->{langObj});
    $userObj->populate(id => $info[0], uname => $info[1], fname => $info[2], mname => $info[3], lname => $info[4], email => $info[5],
             password => $info[6], empId => $info[7], ssn => $info[8], language => $info[9], comment => $info[10], active => $info[11],
             admin => $info[12], sysadmin => $info[13], companyId => $info[14], timeFormat => $info[15], tz => $info[16], lastEdited => $info[17]) if (!$userObj->error);
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return @Users;
    }
    $Users[++$#Users] = $userObj;
  }

  return @Users;
}

# deleteCompany
# takes: companyObj - CompanyObject
# optional: log - indicate if we should log this action.
# returns: 1=OK, 0=Error, -1=doesn't exist
# deletes both entries in the portal and billing databases for this company
# deletion of users, apps, etc. is up to other routines to do.
sub deleteCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, log => 1, @_ );
  my $companyObj = $args{companyObj};
  my $log = $args{log};

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return 0;
  }

  if (!$companyObj->isValid)
  {
    $self->error($companyObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this company exists.
  my $tempCompany = $self->getCompanyInfo(code => $companyObj->code);
  if (not defined $tempCompany)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempCompany->error)
    {
      $self->error($tempCompany->errorMessage);
      return 0;
    }
    else
    {  # delete the entries for this company from both the portal and billing databases.
      my $rc = $self->{portalDB}->write(sql => "DELETE FROM company_tb WHERE id = ?", plug => [ $companyObj->id ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Deletion event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 6, extraInfo => "Deleted Company = '$companyObj->{code}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # billing cleanup
      $rc = $self->{billingDB}->write(sql => "DELETE FROM company_tb WHERE id = ?", plug => [ $companyObj->id ]);
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }
      $self->{billingDB}->commit;
      if ($self->{billingDB}->error)
      {
        $self->error($self->{billingDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Company Delete event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{billingDB}, action => 6, extraInfo => "Deleted Company = '$companyObj->{code}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->{replicationObj}->getEntries(tbName => "company_tb", createDBIObjects => 1);
      if ($self->{replicationObj}->error)
      {
        $self->error($self->{replicationObj}->errorMessage);
        return 0;
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->{dbObj}->write(sql => "DELETE FROM company_tb WHERE id = ?", plug => [ $companyObj->id ]);
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }
        else
        {
          $repObj->{dbObj}->commit;  # have to save the changes in the database.
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Deleted Company = '$companyObj->{code}' from " . $repObj->print);
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Deleted Company = '$companyObj->{code}'");
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }

  return 1;
}

# deleteUser
# takes: userObj - User Object
# optional: log - indicate if we should log this action.
# returns: 1=OK, 0=Error, -1=doesn't exist
# deletion of apps is up to other routines to do.
sub deleteUser
{
  my $self = shift;
  my %args = ( userObj => undef, log => 1, @_ );
  my $userObj = $args{userObj};
  my $log = $args{log};

  if (not defined $userObj)
  {
    $self->error("userObj is not defined!<br>\n");
    return 0;
  }

  if (!$userObj->isValid)
  {
    $self->error($userObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this user exists.
  my $tempUser = $self->getUserInfo(uname => $userObj->uname);
  if (not defined $tempUser)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # user doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUser->error)
    {
      $self->error($tempUser->errorMessage);
      return 0;
    }
    else
    {  # delete this user from the portal database.
      my $rc = $self->{portalDB}->write(sql => "DELETE FROM user_tb WHERE id = ?", plug => [ $userObj->id ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the User Delete event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 9, extraInfo => "Deleted User = '$userObj->{uname}' from Company ID = '$userObj->{companyId}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_tb", createDBIObjects => 1);
      if ($self->{replicationObj}->error)
      {
        $self->error($self->{replicationObj}->errorMessage);
        return 0;
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->{dbObj}->write(sql => "DELETE FROM user_tb WHERE id = ?", plug => [ $userObj->id ]);
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }
        else
        {
          $repObj->{dbObj}->commit;  # have to save the changes in the database.
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Deleted User = '$userObj->{uname}' from " . $repObj->print);
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Deleted User = '$userObj->{uname}'");
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }

  return 1;
}

# deleteUsersOfCompany
# takes: companyObj - CompanyObject
# optional: log - indicate if we should log this action.
# returns: 1=OK, 0=Error, -1=doesn't exist
# deletion of apps is up to other routines to do.
sub deleteUsersOfCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, log => 1, @_ );
  my $companyObj = $args{companyObj};
  my $log = $args{log};

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return 0;
  }

  if (!$companyObj->isValid)
  {
    $self->error($companyObj->errorMessage);
    return 0;
  }

  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this company exists.
  my $tempCompany = $self->getCompanyInfo(code => $companyObj->code);
  if (not defined $tempCompany)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # company doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempCompany->error)
    {
      $self->error($tempCompany->errorMessage);
      return 0;
    }
    else
    {  # delete the users for this company from the portal database.
      my $rc = $self->{portalDB}->write(sql => "DELETE FROM user_tb WHERE company_id = ?", plug => [ $companyObj->id ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
      $self->{portalDB}->commit;
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Deletion of all Users in this Company event.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 9, extraInfo => "Deleted all users of Company = '$companyObj->{code}'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_tb", createDBIObjects => 1);
      if ($self->{replicationObj}->error)
      {
        $self->error($self->{replicationObj}->errorMessage);
        return 0;
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->{dbObj}->write(sql => "DELETE FROM user_tb WHERE company_id = ?", plug => [ $companyObj->id ]);
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }
        else
        {
          $repObj->{dbObj}->commit;  # have to save the changes in the database.
          if ($repObj->{dbObj}->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Deleted all users of Company = '$companyObj->{code}' from " . $repObj->print);
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Deleted all users of Company = '$companyObj->{code}'");
          if ($self->{methods}->error)
          {
            $self->error($self->{methods}->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }

  return 1;
}

# ********* User Rights and Preferences Methods **************

# getUserRightInfo
# requires:
# optional: counter or app, section, permission
# returns: UserRightObject or undef if not found
# summary: Finds the entry in the rights_tb table and returns the
#          UserRightObject that represents it.  If not found it
#          returns undef.  It can either look it up by the
#          counter or by the app, section, permission pair.
sub getUserRightInfo
{
  my $self = shift;
  my %args = ( counter => "", app => "", section => "", permission => "", @_ );
  my $counter = $args{counter};
  my $app = $args{app};
  my $section = $args{section};
  my $permission = $args{permission};
  my $userRightObj = undef;
  my $sql = "SELECT counter, permission, description, admin, app, section, depends_on FROM rights_tb WHERE ";

  if ($counter =~ /^(\d+)$/)
  {
    $sql .= "counter = $counter";
  }
  elsif (length $app > 0 && length $section > 0 && length $permission > 0)
  {
    $sql .= "app = '$app' AND section = '$section' AND permission = '$permission'";
  }
  else
  {
    $self->error("You must specify counter or app, section, permission!<br>\ncounter = '$counter', app = '$app', section = '$section', permission = '$permission'<br>\n");
    return $userRightObj;
  }

  my $sth = $self->{portalDB}->read(sql => $sql);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return $userRightObj;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $userRightObj = Portal::Objects::UserRightObject->new(counter => $info[0], permission => $info[1],
                            description => $info[2], admin => $info[3],
                            app => $info[4], section => $info[5], dependsOn => $info[6], langObj => $self->{langObj});
  }

  return $userRightObj;
}

# getUserRights
# requires: userId, app
# optional: section, permission, admin, dependsOn
# returns: Array of UserRightObjects
# summary: Finds all entries in the user_rights_tb table
#          that match the given parameters and returns an Array
#          of UserRightObjects.
sub getUserRights
{
  my $self = shift;
  my %args = ( userId => "", app => "", section => "", permission => "", admin => "", dependsOn => "", @_ );
  my $userId = $args{userId};
  my $app = $args{app};
  my $section = $args{section};
  my $permission = $args{permission};
  my $admin = $args{admin};
  my $dependsOn = $args{dependsOn};
  my @results = ();
  my $sql = "SELECT r.counter, r.permission, r.description, r.admin, r.app, r.section, r.depends_on FROM rights_tb AS r, user_rights_tb AS u WHERE u.id = $userId AND (r.counter = u.counter) AND r.app = '$app' ";

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return @results;
  }
  if (length $app == 0)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return @results;
  }

  if (length $section > 0)
  {
    $sql .= "AND r.section = '$section' ";
  }
  if (length $permission > 0)
  {
    $sql .= "AND r.permission = '$permission' ";
  }
  if (length $admin > 0)
  {
    if ($admin !~ /^(1|0)$/)
    {
      $self->error("admin = '$admin' is invalid!<br>\n");
      return @results;
    }
    $sql .= "AND r.admin = '$admin' ";
  }
  if (length $dependsOn > 0)
  {
    if ($dependsOn !~ /^(\d+)$/)
    {
      $self->error("dependsOn = '$dependsOn' is invalid!<br>\n");
      return @results;
    }
    $sql .= "AND r.depends_on = $dependsOn ";
  }
  $sql .= "ORDER BY r.section, r.depends_on, r.admin, r.permission";

  my $sth = $self->{portalDB}->read(sql => $sql);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return @results;
  }
  while (my @info = $sth->fetchrow_array)
  {
    my $userRightObj = Portal::Objects::UserRightObject->new(counter => $info[0],
                       permission => $info[1], description => $info[2],
                       admin => $info[3],
                       app => $info[4], section => $info[5], dependsOn => $info[6], langObj => $self->{langObj});
    if ($userRightObj->error)
    {
      $self->error($userRightObj->errorMessage);
      return @results;
    }
    $results[++$#results] = $userRightObj;
  }

  return @results;
}

# isUserRightAssigned
# requires: userId, permission, app, section
# optional:
# returns: 1 = Assigned, 0 = Not Assigned
sub isUserRightAssigned
{
  my $self = shift;
  my %args = ( userId => "", app => "", section => "", permission => "", @_ );
  my $userId = $args{userId};
  my $app = $args{app};
  my $section = $args{section};
  my $permission = $args{permission};

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if (length $app == 0)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return 0;
  }
  if (length $section == 0)
  {
    $self->error("section = '$section' is invalid!<br>\n");
    return 0;
  }
  if (length $permission == 0)
  {
    $self->error("permission = '$permission' is invalid!<br>\n");
    return 0;
  }

  my @result = $self->getUserRights(userId => $userId, permission => $permission, section => $section, app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (scalar @result == 0)
  {
    return 0;
  }
  elsif (scalar @result == 1)
  {
    return 1;  # it was found.
  }
  else
  {
    $self->error("Too many matches found for this specific UserRight! " . int(scalar @result) . " were returned.<br>\nuserId = '$userId', app = '$app', section = '$section', permission = '$permission'.<br>\n");
    return 0;
  }
}

# createUserRight
# requires: userRightObj
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = already exists,
#          -2 = app doesn't exist, -3 = dependsOn doesn't exist
# summary: Creates the User Right in the rights_tb table.
sub createUserRight
{
  my $self = shift;
  my %args = ( userRightObj => undef, log => 1, @_ );
  my $userRightObj = $args{userRightObj};
  my $log = $args{log};

  if (not defined $userRightObj)
  {
    $self->error("userRightObj must be specified!<br>\n");
    return 0;
  }
  if (!$userRightObj->isValid)
  {
    $self->error($userRightObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # immediately bomb out if they are attempting to specify a counter value other than -1 since there is a good chance
  # that it already exists.
  if ($userRightObj->counter ne "-1")
  {
    return -1;
  }

  # do the simplest thing first, see if this Right already has been created.
  my $tempUserRightObj = $self->getUserRightInfo(app => $userRightObj->app, section => $userRightObj->section, permission => $userRightObj->permission);
  if (!defined $tempUserRightObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    # we just drop through to the next part of this method.
  }
  else
  {
    if ($tempUserRightObj->error)
    {
      $self->error($tempUserRightObj->errorMessage);
      return 0;
    }
    else  # it exists!
    {
      return -1;
    }
  }

  # verify that the app specified does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $userRightObj->app ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (! defined $info[0])
  {
    return -2;
  }

  # verify the dependsOn value
  if ($userRightObj->dependsOn ne "-1")
  {
    $sth = $self->{portalDB}->read(sql => "SELECT app, section, permission FROM rights_tb WHERE counter = ?", plug => [ $userRightObj->dependsOn ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    @info = $sth->fetchrow_array;
    if (!defined $info[0])
    {
      return -3;
    }
  }

  # now create the entry in the database.
  my $admin = $userRightObj->{admin};

  $self->{portalDB}->write(sql => "INSERT INTO rights_tb (permission, description, admin, app, section, depends_on) VALUES (?, ?, ?, ?, ?, ?)",
                      plug => [ $userRightObj->permission, $userRightObj->description, $admin, $userRightObj->app, $userRightObj->section, $userRightObj->dependsOn ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    # get the counter for the right just created.
    $tempUserRightObj = $self->getUserRightInfo(app => $userRightObj->app, section => $userRightObj->section, permission => $userRightObj->permission);
    if (!defined $tempUserRightObj)
    {
      if ($self->error)
      {
        $self->prefixError();
        return 0;
      }
      else
      {
        $self->error("User Right does not exist and we just created it!<br>\n" . $userRightObj->print);
        return 0;
      }
    }
    else
    {
      if ($tempUserRightObj->error)
      {
        $self->error($tempUserRightObj->errorMessage);
        return 0;
      }
    }

    $userRightObj->counter($tempUserRightObj->counter);

    if ($log)
    { # log the User Right Creation event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 7, extraInfo => "Created User Right:" . $userRightObj->print);
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "INSERT INTO rights_tb (counter, permission, description, admin, app, section, depends_on) VALUES (?, ?, ?, ?, ?, ?, ?)",
                      plug => [ $userRightObj->counter, $userRightObj->permission, $userRightObj->description, $admin, $userRightObj->app, $userRightObj->section, $userRightObj->dependsOn ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Right = '" . $userRightObj->print . "' to " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Right = '" . $userRightObj->print . "'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# updateUserRight
# requires: userRightObj
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = doesn't exist,
#          -2 = app doesn't exist, -3 = dependsOn doesn't exist
# summary: Updates the User Right in the rights_tb table.
#          The counter can not change, everything else can.
sub updateUserRight
{
  my $self = shift;
  my %args = ( userRightObj => undef, log => 1, @_ );
  my $userRightObj => $args{userRightObj};
  my $log = $args{log};

  if (not defined $userRightObj)
  {
    $self->error("userRightObj must be specified!<br>\n");
    return 0;
  }
  if (!$userRightObj->isValid)
  {
    $self->error($userRightObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # immediately bomb out if they are attempting to specify a counter value = -1 since it can't exist.
  if ($userRightObj->counter eq "-1")
  {
    return -1;
  }

  # do the simplest thing first, make sure this Right has been created.
  my $tempUserRightObj = $self->getUserRightInfo(app => $userRightObj->app, section => $userRightObj->section, permission => $userRightObj->permission);
  if (!defined $tempUserRightObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserRightObj->error)
    {
      $self->error($tempUserRightObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify that the app specified does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $userRightObj->app ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (! defined $info[0])
  {
    return -2;
  }

  # verify the dependsOn value
  if ($userRightObj->dependsOn ne "-1")
  {
    $sth = $self->{portalDB}->read(sql => "SELECT app, section, permission FROM rights_tb WHERE counter = ?", plug => [ $userRightObj->dependsOn ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    @info = $sth->fetchrow_array;
    if (!defined $info[0])
    {
      return -3;
    }
  }

  # now update the entry in the database.
  my $admin = $userRightObj->{admin};

  $self->{portalDB}->write(sql => "UPDATE rights_tb SET permission = ?, description = ?, admin = ?, app = ?, section = ?, depends_on = ? WHERE counter = ?",
                      plug => [ $userRightObj->permission, $userRightObj->description, $admin, $userRightObj->app, $userRightObj->section, $userRightObj->dependsOn, $userRightObj->counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Update event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 8, extraInfo => "Updated User Right:" . $userRightObj->print);
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "UPDATE rights_tb SET permission = ?, description = ?, admin = ?, app = ?, section = ?, depends_on = ? WHERE counter = ?",
                      plug => [ $userRightObj->permission, $userRightObj->description, $admin, $userRightObj->app, $userRightObj->section, $userRightObj->dependsOn, $userRightObj->counter ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Right = '" . $userRightObj->print . "' to " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Right = '" . $userRightObj->print . "'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# deleteUserRight
# requires: counter
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = doesn't exist
# summary: Deletes the User Right from the rights_tb table.
sub deleteUserRight
{
  my $self = shift;
  my %args = ( counter => "-1", log => 1, @_ );
  my $counter = $args{counter};
  my $log = $args{log};

  if ($counter !~ /^(\d+)$/)
  {
    $self->error("counter = '$counter' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # immediately bomb out if they are attempting to specify a counter value = -1 since it can't exist.
  if ($counter eq "-1")
  {
    return -1;
  }

  # do the simplest thing first, make sure this Right has been created.
  my $tempUserRightObj = $self->getUserRightInfo(counter => $counter);
  if (!defined $tempUserRightObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserRightObj->error)
    {
      $self->error($tempUserRightObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # now delete the entry in the database.
  $self->{portalDB}->write(sql => "DELETE FROM rights_tb WHERE counter = ?",
                      plug => [ $counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Delete event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 9, extraInfo => "Deleted User Right: counter = '$counter'.");
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "DELETE FROM rights_tb WHERE counter = ?",
                      plug => [ $counter ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated Deletion of User Right where counter = '$counter' to " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated Deletion of User Right where counter = '$counter'.");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# deleteAllUserRights
# requires: app
# optional: section, dependsOn, admin, log = 1
# returns: >0 = number of rights deleted, 0 = Error,
#          -1 = no rights found to delete
# summary: Deletes all User Rights that match the requirements
#          given from the rights_tb table.
sub deleteAllUserRights
{
  my $self = shift;
  my %args = ( app => "", section => "", dependsOn => "", admin => "", log => 1, @_ );
  my $app = $args{app};
  my $section = $args{section};
  my $dependsOn = $args{dependsOn};
  my $admin = $args{admin};
  my $log = $args{log};

  if (length $app == 0)
  {
    $self->error("app is required!<br>\n");
    return 0;
  }
  if (length $dependsOn > 0 && $dependsOn !~ /^(-1|\d+)$/)
  {
    $self->error("dependsOn = '$dependsOn' is invalid!<br>\n");
    return 0;
  }
  if (length $admin > 0 && $admin !~ /^(1|0)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # create the SQL statement.
  my $sqlString = "DELETE FROM rights_tb WHERE app = '$app'";
  $sqlString .= " AND section = '$section'" if (length $section > 0);
  $sqlString .= " AND depends_on = '$dependsOn'" if (length $dependsOn > 0);
  $sqlString .= " AND admin = '$admin'" if (length $admin > 0);

  # now delete the entry in the database.
  my $result = $self->{portalDB}->write(sql => $sqlString);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Delete event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 9, extraInfo => "Deleted All User Rights where app = '$app', section = '$section', admin = '$admin', dependsOn = '$dependsOn'.");
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => $sqlString);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated Deletion of All User Rights where app = '$app', section = '$section', admin = '$admin', dependsOn = '$dependsOn'.");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated Deletion of All User Rights where app = '$app', section = '$section', admin = '$admin', dependsOn = '$dependsOn'.");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }
  return -1 if ($result == 0);

  return $result;
}

# assignUserRight
# requires: counter, userId
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = right doesn't exist,
#          -2 = user doesn't exist, -3 = already assigned
# summary: Assigns the User Right to the Specified User in
#          the user_rights_tb table.
sub assignUserRight
{
  my $self = shift;
  my %args = ( counter => "-1", userId => "-1", log => 1, @_ );
  my $counter = $args{counter};
  my $userId = $args{userId};
  my $log = $args{log};

  if ($counter !~ /^(\d+)$/)
  {
    $self->error("counter = '$counter' is invalid!<br>\n");
    return 0;
  }
  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # make sure this Right has been created.
  my $tempUserRightObj = $self->getUserRightInfo(counter => $counter);
  if (!defined $tempUserRightObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserRightObj->error)
    {
      $self->error($tempUserRightObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify that the user specified does exist.
  my $userObj = $self->getUserInfo(id => $userId);
  if (!defined $userObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else
    {
      return -2;
    }
  }
  else
  {
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # now check and see if this right has already been assigned to this user.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM user_rights_tb WHERE id = ? AND counter = ?", plug => [ $userId, $counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    return -3;  # already assigned!
  }

  # now assign the right to the user in the database.
  $self->{portalDB}->write(sql => "INSERT INTO user_rights_tb (id, counter) VALUES (?, ?)",
                      plug => [ $userId, $counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Assignment event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 7, extraInfo => "Assigned User Right: counter = '$counter' to user = '$userId'.");
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "INSERT INTO user_rights_tb (id, counter) VALUES (?, ?)",
                      plug => [ $userId, $counter ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Right Assignment: counter = '$counter' to user = '$userId' for " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Right Assignment: counter = '$counter' to user = '$userId'.");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# assignAllUserRights
# requires: userId, app
# optional: section, admin, dependsOn, log = 1
# returns: >0 = number of rights assigned, 0 = Error,
#          -1 = app doesn't exist,
#          -2 = user doesn't exist,
#          -3 = dependsOn doesn't exist (if used)
#          -4 = no rights found to assign
# summary: Assigns all User Rights that meet the specified
#          requirements to the Specified User.
sub assignAllUserRights
{
  my $self = shift;
  my %args = ( userId => "-1", app => "", section => "", admin => "", dependsOn => "", log => 1, @_ );
  my $userId = $args{userId};
  my $app = $args{app};
  my $section = $args{section};
  my $admin = $args{admin};
  my $dependsOn = $args{dependsOn};
  my $log = $args{log};

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if (length $app == 0)
  {
    $self->error("app is required!<br>\n");
    return 0;
  }
  if (length $dependsOn > 0 && $dependsOn !~ /^(-1|\d+)$/)
  {
    $self->error("dependsOn = '$dependsOn' is invalid!<br>\n");
    return 0;
  }
  if (length $admin > 0 && $admin !~ /^(1|0)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # verify that the app specified does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $app ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (! defined $info[0])
  {
    return -1;
  }

  # verify that the user specified does exist.
  my $userObj = $self->getUserInfo(id => $userId);
  if (!defined $userObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else
    {
      return -2;
    }
  }
  else
  {
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify the dependsOn value
  if (length $dependsOn > 0 && $dependsOn ne "-1")
  {
    $sth = $self->{portalDB}->read(sql => "SELECT app, section, permission FROM rights_tb WHERE counter = ?", plug => [ $dependsOn ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    @info = $sth->fetchrow_array;
    if (!defined $info[0])
    {
      return -3;
    }
  }

  # get a list of all Rights that apply.
  my $sqlString = "SELECT counter FROM rights_tb WHERE app = '$app'";
  $sqlString .= " AND section = '$section'" if (length $section > 0);
  $sqlString .= " AND admin = '$admin'" if (length $admin > 0);
  $sqlString .= " AND depends_on = '$dependsOn'" if (length $dependsOn > 0);

  $sth = $self->{portalDB}->read(sql => $sqlString);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }

  while (my @info = $sth->fetchrow_array)
  {
    # now assign the right to the user in the database.
    my $result = $self->assignUserRight(userId => $userId, counter => $info[0], log => $log);
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    if ($result == -1)
    {
      $self->error("right = '$info[0]' does not exist!<br>\nuserId = '$userId'.<br>\n");
      return 0;
    }
    elsif ($result == -2)
    {
      $self->error("user with ID = '$userId' does not exist!<br>\nright = '$info[0]'.<br>\n");
      return 0;
    }
    elsif ($result == -3)
    {
      # This is not an error!
    }
  }
  if ($sth->rows == 0)
  {
    return -4;  # no Rights found to assign to the user.
  }

  return $sth->rows;
}

# unassignUserRight
# requires: counter, userId
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = right doesn't exist,
#          -2 = user doesn't exist, -3 = not assigned
# summary: Unassigns the User Right from the Specified User
#          in the user_rights_tb table.
sub unassignUserRight
{
  my $self = shift;
  my %args = ( counter => "-1", userId => "-1", log => 1, @_ );
  my $counter = $args{counter};
  my $userId = $args{userId};
  my $log = $args{log};

  if ($counter !~ /^(\d+)$/)
  {
    $self->error("counter = '$counter' is invalid!<br>\n");
    return 0;
  }
  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # make sure this Right is created.
  my $tempUserRightObj = $self->getUserRightInfo(counter => $counter);
  if (!defined $tempUserRightObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserRightObj->error)
    {
      $self->error($tempUserRightObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify that the user specified does exist.
  my $userObj = $self->getUserInfo(id => $userId);
  if (!defined $userObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else
    {
      return -2;
    }
  }
  else
  {
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # now check and see if this right is assigned to this user.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM user_rights_tb WHERE id = ? AND counter = ?", plug => [ $userId, $counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0])
  {
    return -3;  # not assigned!
  }

  # now unassign the right from the user in the database.
  $self->{portalDB}->write(sql => "DELETE FROM user_rights_tb WHERE id = ? AND counter = ?",
                      plug => [ $userId, $counter ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right UnAssignment event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 9, extraInfo => "UnAssigned User Right: counter = '$counter' from user = '$userId'.");
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_rights_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "DELETE FROM user_rights_tb WHERE id = ? AND counter = ?",
                      plug => [ $userId, $counter ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Right UnAssignment: counter = '$counter' from user = '$userId' for " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Right UnAssignment: counter = '$counter' from user = '$userId'.");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# unassignAllUserRights
# requires: userId, app
# optional: section, admin, dependsOn, log = 1
# returns: >0 = number of rights unassigned, 0 = Error,
#          -1 = app doesn't exist,
#          -2 = user doesn't exist,
#          -3 = dependsOn doesn't exist (if used)
#          -4 = no rights found to unassign
# summary: Unassigns all User Rights that meet the specified
#          requirements from the Specified User.
sub unassignAllUserRights
{
  my $self = shift;
  my %args = ( userId => "-1", app => "", section => "", admin => "", dependsOn => "", log => 1, @_ );
  my $userId = $args{userId};
  my $app = $args{app};
  my $section = $args{section};
  my $admin = $args{admin};
  my $dependsOn = $args{dependsOn};
  my $log = $args{log};

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if (length $app == 0)
  {
    $self->error("app is required!<br>\n");
    return 0;
  }
  if (length $dependsOn > 0 && $dependsOn !~ /^(-1|\d+)$/)
  {
    $self->error("dependsOn = '$dependsOn' is invalid!<br>\n");
    return 0;
  }
  if (length $admin > 0 && $admin !~ /^(1|0)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # verify that the app specified does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $app ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (! defined $info[0])
  {
    return -1;
  }

  # verify that the user specified does exist.
  my $userObj = $self->getUserInfo(id => $userId);
  if (!defined $userObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else
    {
      return -2;
    }
  }
  else
  {
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify the dependsOn value
  if (length $dependsOn > 0 && $dependsOn ne "-1")
  {
    $sth = $self->{portalDB}->read(sql => "SELECT app, section, permission FROM rights_tb WHERE counter = ?", plug => [ $dependsOn ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    @info = $sth->fetchrow_array;
    if (!defined $info[0])
    {
      return -3;
    }
  }

  # get a list of all Rights that apply.
  my $sqlString = "SELECT counter FROM rights_tb WHERE app = '$app'";
  $sqlString .= " AND section = '$section'" if (length $section > 0);
  $sqlString .= " AND admin = '$admin'" if (length $admin > 0);
  $sqlString .= " AND depends_on = '$dependsOn'" if (length $dependsOn > 0);

  $sth = $self->{portalDB}->read(sql => $sqlString);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }

  while (my @info = $sth->fetchrow_array)
  {
    # now unassign the right from the user in the database.
    my $result = $self->unassignUserRight(userId => $userId, counter => $info[0], log => $log);
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    if ($result == -1)
    {
      $self->error("right = '$info[0]' does not exist!<br>\nuserId = '$userId'.<br>\n");
      return 0;
    }
    elsif ($result == -2)
    {
      $self->error("user with ID = '$userId' does not exist!<br>\nright = '$info[0]'.<br>\n");
      return 0;
    }
    elsif ($result == -3)
    {
      $self->error("right = '$info[0]' is not assigned to user with ID = '$userId'!<br>\n");
      return 0;
    }
  }
  if ($sth->rows == 0)
  {
    return -4;  # no Rights found to unassign from the user.
  }

  return $sth->rows;
}

=item UserPreferenceObject getOrCreateUserPreference(userId, app, module, preference)

  requires: userId, app, module, preference

  returns: UserPreferenceObject or undef if error ocurred.
  summary: Calls getUserPreferenceInfo() to attempt to get the
    preference.  If undef is returned and no error happened,
    then we check for the default for this preference in the
    config system.  If the default is not found, we blow an
    error letting the user know the preference default is not
    available.  Else, if the default was found, we create the
    preference object using the default, call createUserPreference()
    to actually create the preference in the database, and then
    return the UserPreferenceObject we built up.

=cut
sub getOrCreateUserPreference
{
  my $self = shift;
  my %args = ( userId => "", app => "", module => "", preference => "", @_ );

  my $userId = $args{userId};
  my $app = $args{app};
  my $module = $args{module};
  my $preference = $args{preference};
  my $userPreferenceObj = undef;

  $userPreferenceObj = $self->getUserPreferenceInfo(userId => $userId, app => $app, module => $module, preference => $preference);
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  if (!defined $userPreferenceObj)
  {
    # we have to lookup the default value in the config system.
    my $default = $self->methods->getConfigDefault(app => $app, module => $module, prefName => $preference . "_Default", portalDB => $self->portalDB);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }
    if (!defined $default)
    {
      $self->error(sprintf($self->langObj->map("DefaultForPreferenceDoesNotExist"), $preference, $app, $module, $userId));
      return undef;
    }

    # now we need to instantiate the UserPreferenceObject and create the preference.
    $userPreferenceObj = Portal::Objects::UserPreferenceObject->new(id => $userId, name => $preference,
                            value => $default, app => $app, module => $module, langObj => $self->{langObj});
    if ($userPreferenceObj->error)
    {
      $self->error($userPreferenceObj->errorMessage);
      return undef;
    }
    else
    {
      # create the preference in the system.
      my $result = $self->createUserPreference(userPreferenceObj => $userPreferenceObj);
      if ($self->error)
      {
        $self->prefixError();
        return undef;
      }
      if ($result == -1)
      {
        # already exists
        $self->error($self->langObj->map("createUserPreferenceNowExists"), $preference);
        return undef;
      }
      elsif ($result == -2)
      {
        # app doesn't exist
        $self->error($self->langObj->map("createUserPreferenceAppNotExist"), $preference);
        return undef;
      }

      # at this point the preference is valid and has been created in the system.
    }
  }

  return $userPreferenceObj;
}

# UserPreferenceObject getUserPreferenceInfo(userId, app, module,
#                                             preference)
#   requires: userId, app, module, preference
#   optional:
#   returns: UserPreferenceObject or undef if not found
#   summary: Finds the User Preference in the user_preferences_tb
#            table and returns the UserPreferenceObject if found
#            otherwise we return undef to signal we didn't find it.
sub getUserPreferenceInfo
{
  my $self = shift;
  my %args = ( userId => "", app => "", module => "", preference => "", @_ );

  my $userId = $args{userId};
  my $app = $args{app};
  my $module = $args{module};
  my $preference = $args{preference};
  my $userPreferenceObj = undef;
  my $sql = "SELECT id, name, value, app, module FROM user_preferences_tb WHERE id = ? AND app = ? AND module = ? AND name = ?";

  if (!($userId =~ /^(\d+)$/ && length $app > 0 && length $preference > 0))
  {
    $self->error("You must specify userId and app, module, preference!<br>\nuserId = '$userId', app = '$app', module = '$module', preference = '$preference'<br>\n");
    return $userPreferenceObj;
  }

  my $sth = $self->{portalDB}->read(sql => $sql, plug => [ $userId, $app, $module, $preference ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return $userPreferenceObj;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $userPreferenceObj = Portal::Objects::UserPreferenceObject->new(id => $info[0], name => $info[1],
                            value => $info[2], app => $info[3], module => $info[4], langObj => $self->{langObj});
  }

  return $userPreferenceObj;
}

# @ getUserPreferences(userId, app, module)
#   requires: userId
#   optional: app, module
#   returns: Array of UserPreferenceObjects
#   summary: Finds all entries in the user_preferences_tb table
#            that match the given parameters and returns an Array
#            of UserPreferenceObjects.
sub getUserPreferences
{
  my $self = shift;
  my %args = ( userId => "", app => "", module => "", @_ );

  my $userId = $args{userId};
  my $app = $args{app};
  my $module = $args{module};
  my @userPreferences = ();

  my $sql = "SELECT id, name, value, app, module FROM user_preferences_tb WHERE id = '$userId'";
  if (length $app > 0)
  {
    $sql .= " AND app = '$app'";
    if (length $module > 0)
    {
      $sql .= " AND module = '$module'";
    }
  }

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("You must specify userId and optionally app, module!<br>\nuserId = '$userId', app = '$app', module = '$module'<br>\n");
    return @userPreferences;
  }

  my $sth = $self->{portalDB}->read(sql => $sql);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return @userPreferences;
  }
  while (my @info = $sth->fetchrow_array)
  {
    if (defined $info[0])
    {
      my $userPreferenceObj = Portal::Objects::UserPreferenceObject->new(id => $info[0], name => $info[1],
                              value => $info[2], app => $info[3], module => $info[4], langObj => $self->{langObj});
      push @userPreferences, $userPreferenceObj;
    }
  }

  return @userPreferences;
}

# createUserPreference
# requires: userPreferenceObj
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = already exists,
#          -2 = app doesn't exist
# summary: Creates the User Preference in the user_preferences_tb table.
sub createUserPreference
{
  my $self = shift;
  my %args = ( userPreferenceObj => undef, log => 1, @_ );
  my $userPreferenceObj = $args{userPreferenceObj};
  my $log = $args{log};

  if (not defined $userPreferenceObj)
  {
    $self->error("userPreferenceObj must be specified!<br>\n");
    return 0;
  }
  if (!$userPreferenceObj->isValid)
  {
    $self->error($userPreferenceObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # do the simplest thing first, see if this User Preference already has been created.
  my $tempUserPreferenceObj = $self->getUserPreferenceInfo(userId => $userPreferenceObj->{id},
      app => $userPreferenceObj->app, module => $userPreferenceObj->module,
      preference => $userPreferenceObj->name);
  if (!defined $tempUserPreferenceObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    # we just drop through to the next part of this method.
  }
  else
  {
    if ($tempUserPreferenceObj->error)
    {
      $self->error($tempUserPreferenceObj->errorMessage);
      return 0;
    }
    else  # it exists!
    {
      return -1;
    }
  }

  # verify that the app specified does exist.
  if ($userPreferenceObj->{app} ne "Portal")
  {
    my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $userPreferenceObj->app ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    my @info = $sth->fetchrow_array;
    if (! defined $info[0])
    {
      return -2;
    }
  }

  # now create the entry in the database.
  $self->{portalDB}->write(sql => "INSERT INTO user_preferences_tb (id, name, value, app, module) VALUES (?, ?, ?, ?, ?)",
                      plug => [ $userPreferenceObj->id, $userPreferenceObj->name, $userPreferenceObj->value,
                      $userPreferenceObj->app, $userPreferenceObj->module ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Preference Creation event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 7, extraInfo => "Created User Preference:" . $userPreferenceObj->print, userId => (defined $self->{sessionObj} ? "" : -1));
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_preferences_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "INSERT INTO user_preferences_tb (id, name, value, app, module) VALUES (?, ?, ?, ?, ?)",
                      plug => [ $userPreferenceObj->id, $userPreferenceObj->name, $userPreferenceObj->value,
                      $userPreferenceObj->app, $userPreferenceObj->module ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Preference = '" . $userPreferenceObj->print . "' to " . $repObj->print, userId => (defined $self->{sessionObj} ? "" : -1));
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Preference = '" . $userPreferenceObj->print . "'", userId => (defined $self->{sessionObj} ? "" : -1));
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# updateUserPreference
# requires: userPreferenceObj
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = doesn't exist,
#          -2 = app doesn't exist
# summary: Updates the User Preference in the user_preferences_tb table.
#          The value is the only thing that can change.
sub updateUserPreference
{
  my $self = shift;
  my %args = ( userPreferenceObj => undef, log => 1, @_ );
  my $userPreferenceObj = $args{userPreferenceObj};
  my $log = $args{log};

  if (not defined $userPreferenceObj)
  {
    $self->error("userPreferenceObj must be specified!<br>\n");
    return 0;
  }
  if (!$userPreferenceObj->isValid)
  {
    $self->error($userPreferenceObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # do the simplest thing first, make sure this Preference has been created.
  my $tempUserPreferenceObj = $self->getUserPreferenceInfo(userId => $userPreferenceObj->id,
     app => $userPreferenceObj->app, module => $userPreferenceObj->module,
     preference => $userPreferenceObj->name);
  if (!defined $tempUserPreferenceObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserPreferenceObj->error)
    {
      $self->error($tempUserPreferenceObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # verify that the app specified does exist.
  if ($userPreferenceObj->app ne "Portal")
  {
    my $sth = $self->{portalDB}->read(sql => "SELECT id FROM app_tb WHERE name = ?", plug => [ $userPreferenceObj->app ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    my @info = $sth->fetchrow_array;
    if (! defined $info[0])
    {
      return -2;
    }
  }

  # now update the entry in the database.
  $self->{portalDB}->write(sql => "UPDATE user_preferences_tb SET value = ? WHERE id = ? AND app = ? AND module = ? AND name = ?",
                      plug => [ $userPreferenceObj->value, $userPreferenceObj->id, $userPreferenceObj->app,
                      $userPreferenceObj->module, $userPreferenceObj->name ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Update event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 8, extraInfo => "Updated User Preference:" . $userPreferenceObj->print);
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_preferences_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "UPDATE user_preferences_tb SET value = ? WHERE id = ? AND app = ? AND module = ? AND name = ?",
                      plug => [ $userPreferenceObj->value, $userPreferenceObj->id, $userPreferenceObj->app,
                      $userPreferenceObj->module, $userPreferenceObj->name ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Preference = '" . $userPreferenceObj->print . "' to " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Preference = '" . $userPreferenceObj->print . "'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# deleteUserPreference
# requires: userPreferenceObj
# optional: log = 1
# returns: 1 = OK, 0 = Error, -1 = doesn't exist
# summary: Deletes the User Preference from the user_preferences_tb table.
sub deleteUserPreference
{
  my $self = shift;
  my %args = ( userPreferenceObj => undef, log => 1, @_ );
  my $userPreferenceObj = $args{userPreferenceObj};
  my $log = $args{log};

  if (not defined $userPreferenceObj)
  {
    $self->error("userPreferenceObj must be specified!<br>\n");
    return 0;
  }
  if (!$userPreferenceObj->isValid)
  {
    $self->error($userPreferenceObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # do the simplest thing first, make sure this Preference has been created.
  my $tempUserPreferenceObj = $self->getUserPreferenceInfo(userId => $userPreferenceObj->id,
     app => $userPreferenceObj->app, module => $userPreferenceObj->module,
     preference => $userPreferenceObj->name);
  if (!defined $tempUserPreferenceObj)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # it doesn't exist!
    {
      return -1;
    }
  }
  else
  {
    if ($tempUserPreferenceObj->error)
    {
      $self->error($tempUserPreferenceObj->errorMessage);
      return 0;
    }
    # we just drop through to the next part of this method.
  }

  # now delete the entry in the database.
  $self->{portalDB}->write(sql => "DELETE FROM user_preferences_tb WHERE id = ? AND app = ? AND module = ? AND name = ?",
                      plug => [ $userPreferenceObj->id, $userPreferenceObj->app,
                      $userPreferenceObj->module, $userPreferenceObj->name ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  else
  {
    $self->{portalDB}->commit;  # have to save the changes in the database.
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }

    if ($log)
    { # log the User Right Update event.
      $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 8, extraInfo => "Updated User Preference:" . $userPreferenceObj->print);
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->{replicationObj}->getEntries(tbName => "user_preferences_tb", createDBIObjects => 1);
    if ($self->{replicationObj}->error)
    {
      $self->error($self->{replicationObj}->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->{dbObj}->write(sql => "DELETE FROM user_preferences_tb WHERE id = ? AND app = ? AND module = ? AND name = ?",
                      plug => [ $userPreferenceObj->id, $userPreferenceObj->app,
                      $userPreferenceObj->module, $userPreferenceObj->name ]);
      if ($repObj->{dbObj}->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
        return 0;
      }
      else
      {
        $repObj->{dbObj}->commit;  # have to save the changes in the database.
        if ($repObj->{dbObj}->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->{dbObj}->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $self->{portalDB}, action => 23, extraInfo => "Replicated User Preference Deletion = '" . $userPreferenceObj->print . "' to " . $repObj->print);
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->{methods}->doLog(sessionObj => $self->{sessionObj}, db => $repObj->{dbObj}, action => 23, extraInfo => "Replicated User Preference Deletion = '" . $userPreferenceObj->print . "'");
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# deleteAllUserPreferences
# requires: userId
# optional: app, module, log = 1
# returns: >= 0 = number of preferences deleted,
#          -1 = no preferences found to delete
# summary: Deletes all User Preferences that match the requirements
#          given from the user_preferences_tb table.
sub deleteAllUserPreferences
{
  my $self = shift;
  my %args = ( userId => "", app => "", module => "", log => 1, @_ );
  my $userId = $args{userId};
  my $app = $args{app};
  my $module = $args{module};
  my $log = $args{log};
  my $count = 0;  # number of entries deleted.

  if ($userId !~ /^(\d+)$/)
  {
    $self->error("You must specify userId and optionally app, module!<br>\nuserId = '$userId', app = '$app', module = '$module', log = '$log'<br>\n");
    return $count;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return $count;
  }

  # get the array of preferences that meet the qualifications and delete them.
  my @userPreferences = $self->getUserPreferences(userId => $userId, app => $app, module => $module);
  if ($self->error)
  {
    $self->prefixError();
    return $count;
  }
  foreach my $userPreferenceObj (@userPreferences)
  {
    my $result = $self->deleteUserPreference(userPreferenceObj => $userPreferenceObj, log => $log);
    if ($self->error)
    {
      $self->prefixError();
      return $count;
    }
    if ($result == -1)
    {
      $self->error("userPreference = " . $userPreferenceObj->print . " does not exist!<br>\n");
      return $count;
    }
    $count++;  # keep track of the number deleted.
  }

  return $count;
}

# ******* End User Rights and Preferences Methods ************

# swapDate
# takes: date - a date in format MM/DD/YYYY and returns YYYY/MM/DD or vice versa
sub swapDate
{
  my $self = shift;
  my %args = ( date => "", @_ );
  my $date = $args{date};

  if (length $date == 0)
  {
    $date = "01/01/2000"; # default date.
  }
  if ($date =~ /^\d{2}.\d{2}.\d{4}$/)  # MM/DD/YYYY
  {
    $date =~ s/^(\d{2})(.)(\d{2})(.)(\d{4})$/$5$2$1$4$3/;
  }
  else # YYYY/MM/DD
  {
    $date =~ s/^(\d{4})(.)(\d{2})(.)(\d{2})$/$3$2$5$4$1/;
  }

  return $date;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  int isUserValid(uname, password)
      password is unencrypted.  This routine will crypt password itself.
    Returns: 1=OK, 0=Password does not match, -1=User not active,
             -2=Company not active, -3=Company does not exist that
             user belongs to, -4=User not found, -5=User locked out.

  User getUserInfo(id, uname)
    Returns the User Object that contains all the info on the
    specified user or undef if user not found.  You can specify
    either id or uname to look up the user by.

  int getUserId(uname)
    Returns id of user, -1 if error, -2 if user not found.

  int createUser(userObj, log)
    Takes a User Object and attempts to create the specified user
    account.  Returns 1 if successfull, 0 otherwise.
    If log = 1 then the successful action will be logged.

  int updateUser(userObj, log)
    Takes a User Object and updates the entry in the database.
    Returns 1 if ok, 0 if error.
    If log = 1 then the successful action will be logged.

  CompanyObject getCompanyInfo(id, code)
    Uses either the id or code value for the company and looks them up
    and returns the CompanyObject with the info for that company.
    If the company is not found, we return undef.  If an error occurred,
    we also return undef and set the error and errorString values.

  CompanyBillingObject getCompanyBillingInfo(id)
    Returns the billing info for the specified company or undef if the
    company can not be found or an error occurred.  The error and
    errorString values are set if an error occurred.

  int createCompany(companyObj, log)
    Takes the companyObj CompanyObject and creates the entry in the
    portal database after validating the company doesn't exist yet.
    Returns 1 if successfull, 0 otherwise.
    If log = 1 then the successful action will be logged.

  int createCompanyBilling(companyBillObj, log)
    Takes the companyBillObj CompanyBillingObject and creates the entry
    in the billing database after validating the company doesn't exist
    yet.  Returns 1 if successfull, 0 otherwise.
    If log = 1 then the successful action will be logged.

  int updateCompany(companyObj, log)
    takes: companyObj - CompanyObject
    returns: 1 = OK, 0 = Error, -1 = Company doesn't exist
    If log = 1 then the successful action will be logged.

  int updateCompanyBilling(companyBillObj, log)
    takes: companyBillObj - CompanyBillingObject
    returns: 1 = OK, 0 = Error, -1 = doesn't exist
    If log = 1 then the successful action will be logged.

  @ getListOfCompanies(like, field, parentId, orderBy)
    optional: like (string to search against the company names via the
                    LIKE SQL operator),
              field (name of the field to compare against using LIKE -
                     must be a string field),
              orderBy (the order by string - must be valid)
              parentId (specify the parentId value to match against)
    returns: Array of CompanyObject that represent all companies in the
             Portal that matched the given criteria.

  @ getListOfUsersForCompany(companyObj, like, admin, sysadmin)
    takes: companyObj CompanyObject, like (string to search against the
           user names via the LIKE SQL operator), admin, sysadmin
    returns: Array of User Objects that represent all users of the
             specified company.
             If admin or sysadmin are set to 1, then it will only
             return users that are admins or sysadmins if so defined.

  @ getAllUsersInPortal(like, admin, sysadmin)
    takes: like, admin, sysadmin
    returns: Array of User Objects that define all users in the portal
             that match the requirements sorted by company.
             This routine works similar to getListOfUsersForCompany in
             parameters it takes.

  int deleteCompany(companyObj, log)
    takes: companyObj - CompanyObject
    returns: 1=OK, 0=Error, -1=doesn't exist
             deletes both entries in the portal and billing databases for
             this company.  Deletion of users, apps, etc. is up to other
             routines to do.
    If log = 1 then the successful action will be logged.

  int deleteUser(userObj, log)
    takes: userObj - User Object
    returns: 1=OK, 0=Error, -1=doesn't exist
             deletion of apps is up to other routines to do.
    If log = 1 then the successful action will be logged.

  int deleteUsersOfCompany(companyObj, log)
    takes: companyObj - CompanyObject
    returns: 1=OK, 0=Error, -1=doesn't exist
             deletion of apps is up to other routines to do.
    If log = 1 then the successful action will be logged.

  UserRightObject getUserRightInfo(counter, app, section, permission)
    requires:
    optional: counter or app, section, permission
    returns: UserRightObject or undef if not found
    summary: Finds the entry in the rights_tb table and returns the
             UserRightObject that represents it.  If not found it
             returns undef.  It can either look it up by the
             counter or by the app, section, permission pair.

  @ getUserRights(userId, app, section, permission, admin, dependsOn)
    requires: userId, app
    optional: section, permission, admin, dependsOn
    returns: Array of UserRightObjects
    summary: Finds all entries in the user_rights_tb table
             that match the given parameters and returns an Array
             of UserRightObjects.

  bool isUserRightAssigned(userId, permission, app, section)
    requires: userId, permission, app, section
    optional:
    returns: 1 = Assigned, 0 = Not Assigned
    summary: Determines if the user right is assigned to the user.

  int createUserRight(userRightObj, log)
    requires: userRightObj
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = already exists,
             -2 = app doesn't exist, -3 = dependsOn doesn't exist
    summary: Creates the User Right in the rights_tb table.

  int updateUserRight(userRightObj, log)
    requires: userRightObj
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = doesn't exist,
             -2 = app doesn't exist, -3 = dependsOn doesn't exist
    summary: Updates the User Right in the rights_tb table.
             The counter can not change, everything else can.

  int deleteUserRight(counter, log)
    requires: counter
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = doesn't exist
    summary: Deletes the User Right from the rights_tb table.

  int deleteAllUserRights(app, section, dependsOn, admin, log)
    requires: app
    optional: section, dependsOn, admin, log = 1
    returns: >0 = number of rights deleted, 0 = Error,
             -1 = no rights found to delete
    summary: Deletes all User Rights that match the requirements
             given from the rights_tb table.

  int assignUserRight(counter, userId, log)
    requires: counter, userId
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = right doesn't exist,
             -2 = user doesn't exist, -3 = already assigned
    summary: Assigns the User Right to the Specified User in
             the user_rights_tb table.

  int assignAllUserRights(userId, app, section, admin, dependsOn, log)
    requires: userId, app
    optional: section, admin, dependsOn, log = 1
    returns: >0 = number of rights assigned, 0 = Error,
             -1 = app doesn't exist,
             -2 = user doesn't exist,
             -3 = dependsOn doesn't exist (if used)
             -4 = no rights found to assign
    summary: Assigns all User Rights that meet the specified
             requirements to the Specified User.

  int unassignUserRight(counter, userId, log)
    requires: counter, userId
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = right doesn't exist,
             -2 = user doesn't exist, -3 = not assigned
    summary: Unassigns the User Right from the Specified User
             in the user_rights_tb table.

  int unassignAllUserRights(userId, app, section, admin, dependsOn, log)
    requires: userId, app
    optional: section, admin, dependsOn, log = 1
    returns: >0 = number of rights unassigned, 0 = Error,
             -1 = app doesn't exist,
             -2 = user doesn't exist,
             -3 = dependsOn doesn't exist (if used)
             -4 = no rights found to unassign
    summary: Unassigns all User Rights that meet the specified
             requirements from the Specified User.

  UserPreferencesObject getUserPreferenceInfo(userId, app, module,
                                              preference)
    requires: userId, app, module, preference
    optional:
    returns: UserPreferenceObject or undef if not found
    summary: Finds the User Preference in the user_preferences_tb
             table and returns the UserPreferenceObject if found
             otherwise we return undef to signal we didn't find it.

  @ getUserPreferences(userId, app, module)
    requires: userId
    optional: app, module
    returns: Array of UserPreferenceObjects
    summary: Finds all entries in the user_preferences_tb table
             that match the given parameters and returns an Array
             of UserPreferenceObjects.

  int createUserPreference(userPreferenceObj, log)
    requires: userPreferenceObj
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = already exists,
             -2 = app doesn't exist
    summary: Creates the User Preference in the user_preferences_tb table

  int updateUserPreference(userPreferenceObj, log)
    requires: userPreferenceObj
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = doesn't exist,
             -2 = app doesn't exist
    summary: Updates the User Preference in the user_preferences_tb table
             The value is the only thing that can change.

  int deleteUserPreference(userPreferenceObj, log)
    requires: counter
    optional: log = 1
    returns: 1 = OK, 0 = Error, -1 = doesn't exist
    summary: Deletes the User Preference from the user_preferences_tb
             table.

  int deleteAllUserPreferences(userId, app, module, log)
    requires: userId
    optional: app, module, log = 1
    returns: >= 0 = number of preferences deleted,
             -1 = no preferences found to delete
    summary: Deletes all User Preferences that match the requirements
             given from the user_preferences_tb table.

  string swapDate(date)
    takes: date - a date in format MM/DD/YYYY and returns YYYY/MM/DD or
           vice versa

  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::Objects::User(3), Portal::Objects::CompanyObject(3), Portal::Objects::CompanyBillingObject(3), Portal::Base(3)

=cut
