# ColorScheme.pm - The config file for the Portal.
# Created by James A. Pattie (james@pcxperience.com)
# 11/13/2000 - Copyright (c) 2000-2003, Xperience, Inc.
# (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Data::ColorScheme;

use strict;
use Portal::Base;
use Portal::XML::ConfigParser;
use vars qw ($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter);
@EXPORT = qw();

$VERSION = '0.03';

my $configVersion = "0.1";

=head1 NAME

ColorScheme - Object used to store the Portal ColorScheme Data.

=head1 SYNOPSIS

  use Portal::Data::ColorScheme;
  my $obj = undef;
  eval { $obj = Portal::Data::ColorScheme->new(name => $name,
                string => $xmlString, userId => $userId,
                langObj => $langObj); };
  if ($@) # error occurred!
  {
    die "Error instantiating ColorScheme object! $@";
  }
  print "fgColor = $obj->{fgColor}\n";

=head1 DESCRIPTION

ColorScheme is the Portal Data ColorScheme class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(string, name, userId, derivedFrom)

 Creates a new instance of the ColorScheme module.
 See Portal::Base(3) for a listing of required arguments.

 requires: string - xml document representing the colorScheme to parse
           name - name of colorScheme
           userId - id associated w/ this colorScheme (-1 or real value)
           derivedFrom - name of the parent colorScheme or
                         empty if a system entry.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( string => "", name => "", userId => "-1", derivedFrom => "", @_ );
  
  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  my $string = $args{string};
  my $name = $args{name};
  my $userId = $args{userId};
  my $derivedFrom = $args{derivedFrom};

  eval { $self->{cfgObj} = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    die("Instantiation of ConfigParser failed!<br>\n$@");
  }
  if ($self->{cfgObj}->error)
  {
    die $self->{cfgObj}->errorMessage;
  }
  eval { $self->{settingsObj} = $self->{cfgObj}->parse(string => $string, module => "PortalColorScheme", version => $configVersion); };
  if ($@)
  {
    die("Parse of '$string' failed!<br>\n$@");
  }
  if ($self->{settingsObj}->error)
  {
    die $self->{settingsObj}->errorMessage;
  }

  $self->{name} = $name;  # store the name of this colorScheme.
  $self->{userId} = $userId;
  $self->{derivedFrom} = $derivedFrom;

  my %settings = ();
  %settings = %{$self->{settingsObj}->{settings}};

  foreach my $name (keys %settings)
  {
    $self->{$name} = $settings{$name};
  }

  my @names = keys %{$self->{settingsObj}->{settings}};
  $self->{"_names_"} = \@names;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  if ($self->{settingsObj}->{version} ne $configVersion)
  {
    $self->error("ColorScheme $name = '$string' is at version '$self->{settingsObj}->{version}' not at the current version of '$configVersion'!<br>\n");
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{userId} !~ /^(-1|\d+)$/)
  {
    $self->invalid("userId", "$self->{userId}");
  }
  if (length $self->{name} == 0)
  {
    $self->missing("name");
  }
  elsif ($self->{userId} != -1 && $self->{name} !~ /^(.+)( \(user\))$/)
  {
    $self->invalid("name", "'$self->{name}' is invalid for userId = '$self->{userId}'");
  }
  if ($self->{userId} != -1 && $self->{derivedFrom} !~ /^(.+)$/)
  {
    $self->invalid("derivedFrom", "'$self->{derivedFrom}' is invalid for userId = '$self->{userId}', name = '$self->{name}'");
  }
  foreach my $name (@{$self->{"_names_"}})
  {
    if (!exists $self->{$name})
    {
      $self->missing("$name");
    }
    if ($name !~ /(CSS|(fg|bg)Color)$/)
    {
      # make sure we do have color: and background-color: strings specified
      if ($self->{$name} !~ /(^|\s+)color:/)
      {
        $self->missing("$name", "color: attribute");
      }
      if ($self->{$name} !~ /background-color:/)
      {
        $self->missing("$name", "background-color: attribute");
      }
    }
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# xml export(void)
#    returns the xml document based upon the current
#    values of the colorScheme variables.
sub export
{
  my $self = shift;
  my $string = "";

  if (!$self->isValid())
  {
    $self->prefixError();
    return $string;
  }

  # make sure that the settings hash is empty first.
  %{$self->{settingsObj}->{settings}} = ();

  foreach my $name (@{$self->{"_names_"}})
  {
    $self->{settingsObj}->{settings}->{$name} = $self->{$name};
  }
  eval { $string = $self->{settingsObj}->generateXML(); };
  if ($@)
  {
    $self->error($@);
  }
  return $string;
}

# string print(void)
# returns a string with name, userId and derivedFrom printed in it.
sub print
{
  my $self = shift;
  my $string = "name='$self->{name}', userId='$self->{userId}', derivedFrom='$self->{derivedFrom}'";
  return $string;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  xml export(void)
    returns the xml document based upon the current
    values of the colorScheme variables.

  scalar print(void)
    returns a string with name, userId and derivedFrom printed in it.

=head1 Exported VARIABLES

  name             - The name of the colorScheme
  userId           - Id of the user this colorScheme is for.
  derivedFrom      - The colorScheme we are derived from or empty if a
                     system entry.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
