# Config.pm - The config file for the Portal.
# Created by James A. Pattie (james@pcxperience.com)
# 11/13/2000 - Copyright (c) 2000-2004, Xperience, Inc.
# (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Data::Config;

use strict;
use Portal::Base;
use Portal::XML::ConfigParser;
use vars qw ($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter);
@EXPORT = qw();

$VERSION = '0.11';

=head1 NAME

Config - Object used to store the Portal Configuration Data.

=head1 SYNOPSIS - use by the Portal itself

  use Portal::Data::Config;
  my $obj = undef;
  eval { $obj = Portal::Data::Config->new(langObj => $langObj); };
  if ($@) # error occurred!
  {
    die "Error instantiating Config object! $@";
  }
  print "myHostName = $obj->{myHostName}\n";

=head1 SYNOPSIS - use by a Portal app

  use Portal::Data::Config;
  my $obj = undef;
  eval { $obj = Portal::Data::Config->new(langObj => $langObj, app => $self->{app}); };
  if ($@) # error occurred!
  {
    die "Error instantiating $self->{app} Config object! $@";
  }
  print "dbName = $obj->{dbName}\n";

=head1 DESCRIPTION

Config is the Portal Data Configuration class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(prefix, app, version)

 Creates a new instance of the Portal::Data::Config module.
 See Portal::Base(3) for a listing of required arguments.

 optional:
   prefix - prefix directory to go before /etc/pcx_portal.
   app - name of app to source in config file for.  Defaults
     to 'Portal'.
   version - explicitly request this version of the config file
     and throw an error if the versions don't match.  If you
     don't specify a version, then the version that is on disk
     is considered to be the correct one and is used internally.
     This feature will probably not be used that much.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( prefix => "", app => "Portal", version => "", @_ );
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  my $prefix = $args{prefix};
  my $app = $args{app};
  my $version = $args{version};

  # source in the config file from $prefix/etc/pcx_portal/$app.xml and suck the settings hash into ourselves.
  my $cfgFile = "$prefix/etc/pcx_portal/$app.xml";

  if ( ! -e $cfgFile )
  {
    die $errStr . "config file '$cfgFile' does not exist!<br />\n$!";
  }

  if ($version eq "")
  {
    # get the version info from the config file.
    $version = `cat $cfgFile | grep '<config module'`;
    chomp $version;
    $version =~ s/^(<config\s+module="\w+"\s+version=")(\d+\.\d+)(".*>)$/$2/;
  }
  if ($version !~ /^(\d+\.\d+)$/)
  {
    die $errStr . "config file '$cfgFile' has an invalid version = '$version'!";
  }

  eval { $self->{cfgObj} = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    die $errStr . "Instantiation of ConfigParser failed!<br>\n$@";
  }
  if ($self->{cfgObj}->error)
  {
    die $errStr . $self->{cfgObj}->errorMessage;
  }
  eval { $self->{settingsObj} = $self->{cfgObj}->parse(file => $cfgFile, module => $app, version => $version); };
  if ($@)
  {
    die $errStr . "Parse of '$cfgFile' failed!<br />\n$@";
  }
  if ($self->{settingsObj}->error)
  {
    die $errStr . $self->{settingsObj}->errorMessage;
  }

  my %settings = ();
  %settings = %{$self->{settingsObj}->{settings}};

  foreach my $name (keys %settings)
  {
    $self->{$name} = $settings{$name};
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  if ($self->{settingsObj}->{version} ne $version)
  {
    die $errStr . "Config File '$cfgFile' is at version '$self->{settingsObj}->{version}' not at the current version of '$version'!<br />\n";
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported Portal VARIABLES

  version       - Version of Config File being used
  myHostName    - Portal host name or IP Address
  httpType      - Type of Web Server (http or https)
  sessionType   - Type of Sessions (Database or DB_File)
  sessionDirectory - Directory to use for Session Files
  sessionLockDir- Directory to use for File Locking
  dbType        - Database Type (Postgres or MySQL)
  dbHost        - host name or IP Address of Database Server
  dbName        - name of the Portal database
  dbPort        - Database Port to connect to
  dbAdminUser   - Admin User to connect to database as
  dbAdminPasswd - Admin Password to use when connecting to the database
  dbUser        - User to connect to database as
  dbPasswd      - Password to use when connecting to the database
  billingdbHost - host name or IP Address of Billing Database Server
  billingdbName - name of the Billing database
  webRoot       - The root directory of the web server
  siteDir       - The name of the directory index.cgi is in
  cookieDomain  - The domain that cookies are to be created in
  cookieLife    - The number of minutes that session cookies should stay
  emailAddress  - The email address for JavaScript Error Handler to use
  allowFileUploads - Do we allow file uploads (true/false)
  maxFileUploadSize - Max size of all data coming into the portal
  fileUploadLocation - Location files will be wrote to
  fileUploadPrefix - name prefixed to all incoming files
  hostingURL    - The URL of the hosting companies web site.
  hostingAltTag - The Alt Tag to use for the hostingURL.
  hostingLogoUrl- The URL of the hosting companies logo.
  cookieDebug   - Indicates if the cookie debug info should be shown at
                  the bottom of each portal generated page.

=head1 Exported Portal App VARIABLES that all apps share

  dbName                - set to appName_db if this app needs its own
                          database.
  dbPerCompany          - set to true if this app needs its own database
                          for each company.
  appType               - administration, help, bugs, regular, tools,
                          news
  appAdminType          - normal or system
  appDescription        - describe (briefly) what the app does.  This
                          will be used in the App purchasing and
                          assigning Apps.
  appIconName           - name the icon this app will display on the
                          desktop.
  height                - default height of the app.
  width                 - default width of the app.
  autorun               - do you want this app to automatically startup
                          when the user logs in?
  dbType                - Database backend to use. Pg or mysql
  dbPort                - The Port to connect to for this database
                          server. 3306 for mysql, 5432 for Pg. Username
                          and password will come from the portals
                          settings.
  appName               - This is the name of the Applications perl
                          module.  Case sensitive.
  isAppFree             - if false, then app is a for fee app.
  assignToAllCompanies  - if false, don't assign to any companies.
  assignToAllUsers      - if true, then assign to all users, else check
                          the admin and sysadmin flags.
  assignToAdminUsers    - if true, then assign to all admin and sysadmin
                          users, else check sysadmin flag.
  assignToSysAdminUsers - if true, then assign to only sysadmin users.
  appCost               - The default cost of the app.  0.00 indicates
                          free.
  unit                  - 0 = per month, 1 = per user.
  numLicenses           - The number of licenses to assign to a company
                          if assignToAllCompanies is true.  0 means
                          unlimited.
  wapAware              - Is this app WAP aware?

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
