# DetectDistro.pm - The Object Class that provides a DetectDistro Object
# Created by James A. Pattie, 2004-02-05.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Data::DetectDistro;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT $errorMessage);

require Exporter;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

$errorMessage = "";

=head1 NAME

DetectDistro - Object used to detect the Distro we are on.

=head1 SYNOPSIS

  use Portal::Data::DetectDistro;
  my @distroInfo = Portal::Data::DetectDistro::detect();
  if (!defined @distroInfo)
  {
    die $Portal::Data::DetectDistro::errorMessage;
  }
  my $distroName = $distroInfo[0];
  my $distroVersion = $distroInfo[1];

=head1 DESCRIPTION

DetectDistro attempts to detect the *nix Distro you are using.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item @(name, version) detect(void)

  returns the detected name and version in an array or undef
  if an error occured.

=cut
sub detect
{
  my $distroName = "";
  my $distroVersion = "";
  if (! -f "/etc/issue")
  {
    $errorMessage = "/etc/issue does not exist!  $!\n";
    return undef;
  }
  my $etcIssue = `cat /etc/issue`;
  if ($etcIssue =~ /^(Debian GNU)(\/\\s\s+)([^\s]+)(\s+.+)$/m)
  {
    $distroName = $1;
    $distroVersion = $3;
  }
  elsif ($etcIssue =~/^(Debian GNU)(\/Linux)\s+([^\s]+)(.+)$/m)
  {
    $distroName = $1;
    $distroVersion = $3;
  }
  elsif ($etcIssue =~ /^(Red Hat Linux)(\s+release\s+)([^\s]+)(\s+.+)$/m)
  {
    $distroName = $1;
    $distroVersion = $3;
  }
  elsif ($etcIssue =~ /^(Red Hat Enterprise Linux WS)(\s+release\s+)([^\s]+)(\s+\(.+\))$/m)
  {
    $distroName = $1;
    $distroVersion = $3;
  }
  else
  {
    $errorMessage = "Unsupported distro detected!\n/etc/issue = '$etcIssue'\n";
    return undef;
  }
  return ($distroName, $distroVersion);
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
