# Variables.pm - The Object Class that provides a Variables Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Data::Variables;
use strict;
use Portal::Base;
use HTMLObject::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.06';

# global variables
my %trueFalse = ( 1 => "true", 0 => "false" );
my %falseTrue = ( "true" => 1, "false" => 0 );
my @alphabet = qw( a b c d e f g h i j k l m n o p q r s t u v w x y z A B C D E F G H I J K L M N O P Q R S T U V W X Y Z );
my @numbers = qw( 1 2 3 4 5 6 7 8 9 0 );
my %states = ( AL => "Alabama", AK => "Alaska", AS => "American Samoa", AZ => "Arizona", AR => "Arkansas", CA => "California", CO => "Colorado",
               CT => "Connecticut", DE => "Deleware", DC => "District Of Columbia", FM => "Federated States Of Micronesia", FL => "Florida",
               GA => "Georgia", GU => "Guam", HI => "Hawaii", ID => "Idaho", IL => "Illinois", IN => "Indiana", IA => "Iowa", KS => "Kansas",
               KY => "Kentucky", LA => "Louisiana", ME => "Maine", MH => "Marshall Islands", MD => "Maryland", MA => "Massachusetts", MI => "Michigan",
               MN => "Minnesota", MS => "Mississippi", MO => "Missouri", MT => "Montana", "NE" => "Nebraska", NV => "Nevada", NH => "New Hampshire",
               NJ => "New Jersey", NM => "New Mexico", NY => "New York", NC => "North Carolina", ND => "North Dakota", MP => "Northern Mariana Islands",
               OH => "Ohio", OK => "Oklahoma", OR => "Oregon", PW => "Palau", PA => "Pennsylvania", PR => "Puerto Rico", RI => "Rhode Island",
               SC => "South Carolina", SD => "South Dakota", TN => "Tennessee", TX => "Texas", UT => "Utah", VT => "Vermont", VI => "Virgin Islands",
               VA => "Virginia", WA => "Washington", WV => "West Virginia", WI => "Wisconsin", WY => "Wyoming" );

my %ccTypes = ( Visa => "Visa", Mastercard => "Mastercard", "American Express" => "American Express" );
my %appTypes = ( administration => "Administration", help => "Help",
                 bugs => "Bugs", news => "News", regular => "Regular", tools => "User Tools" );
my %adminTypes = ( normal => "Normal", system => "System" );
my %timeFormats = ( 12 => "AM/PM", 24 => "Military Time" );

# Time Zone info taken from TimeDate Perl Module - Created by Graham Barr <gbarr@pobox.com>, David Muir Sharnoff <muir@idiom.com>, Paul Foley <paul@ascent.com>

# timeZones returns the number of hours that need to be added/subtracted to/from the UTC time to display in the timezone of the user.
my %timeZones = ("GMT" =>   0,  # Greenwich Mean
     "UTC"       =>   0,
     "WET"       =>   0,  # Western European
     "WAT"       =>  -1,  # West Africa
     "AT"        =>  -2,  # Azores
     "EWT"       =>  -4,  # U.S. Eastern War Time
     "AST"       =>  -4,  # Atlantic Standard
     "EST"       =>  -5,  # Eastern Standard
     "CST"       =>  -6,  # Central Standard
     "MST"       =>  -7,  # Mountain Standard
     "PST"       =>  -8,  # Pacific Standard
     "YST" =>  -9,  # Yukon Standard
     "HST" => -10,  # Hawaii Standard
     "CAT" => -10,  # Central Alaska
     "AHST" => -10,  # Alaska-Hawaii Standard
     "NT" => -11,  # Nome
     "IDLW" => -12,  # International Date Line West
     "CET" =>  +1,  # Central European
     "MEZ" =>  +1,  # Central European (German)
     "ECT" =>  +1,  # Central European (French)
     "MET" =>  +1,  # Middle European
     "MEWT" =>  +1,  # Middle European Winter
     "SWT" =>  +1,  # Swedish Winter
     "SET" =>  +1,  # Seychelles
     "FWT" =>  +1,  # French Winter
     "EET" =>  +2,  # Eastern Europe, USSR Zone 1
     "UKR" =>  +2,  # Ukraine
     "BT" =>  +3,  # Baghdad, USSR Zone 2
     "ZP4" =>  +4,  # USSR Zone 3
     "ZP5" =>  +5,  # USSR Zone 4
     "ZP6" =>  +6,  # USSR Zone 5
     "WST" =>  +8,  # West Australian Standard
     "HKT" =>  +8,  # Hong Kong
     "CCT" =>  +8,  # China Coast, USSR Zone 7
     "JST" =>  +9,  # Japan Standard, USSR Zone 8
     "KST" =>  +9,  # Korean Standard
     "EAST" => +10,  # Eastern Australian Standard
     "GST" => +10,  # Guam Standard, USSR Zone 9
     "NZT" => +12,  # New Zealand
     "NZST" => +12,  # New Zealand Standard
     "IDLE" => +12,  # International Date Line East
);

# timeZoneNames returns the Descriptive string for the zone code.
my %timeZoneNames = ("GMT" =>   "Greenwich Mean",
     "UTC"       => "Universal (Coordinated)",
     "WET"       => "Western European",
     "WAT"       => "West Africa",
     "AT"        => "Azores",
     "EWT"       => "U.S. Eastern War Time",
     "AST"       => "Atlantic Standard",
     "EST"       => "Eastern Standard",
     "CST"       => "Central Standard",
     "MST"       => "Mountain Standard",
     "PST"       => "Pacific Standard",
     "YST" => "Yukon Standard",
     "HST" => "Hawaii Standard",
     "CAT" => "Central Alaska",
     "AHST" => "Alaska-Hawaii Standard",
     "NT" => "Nome",
     "IDLW" => "International Date Line West",
     "CET" => "Central European",
     "MEZ" => "Central European (German)",
     "ECT" => "Central European (French)",
     "MET" => "Middle European",
     "MEWT" => "Middle European Winter",
     "SWT" => "Swedish Winter",
     "SET" => "Seychelles",
     "FWT" => "French Winter",
     "EET" => "Eastern Europe, USSR Zone 1",
     "UKR" => "Ukraine",
     "BT" => "Baghdad, USSR Zone 2",
     "ZP4" => "USSR Zone 3",
     "ZP5" => "USSR Zone 4",
     "ZP6" => "USSR Zone 5",
     "WST" => "West Australian Standard",
     "HKT" => "Hong Kong",
     "CCT" => "China Coast, USSR Zone 7",
     "JST" => "Japan Standard, USSR Zone 8",
     "KST" => "Korean Standard",
     "EAST" => "Eastern Australian Standard",
     "GST" => "Guam Standard, USSR Zone 9",
     "NZT" => "New Zealand",
     "NZST" => "New Zealand Standard",
     "IDLE" => "International Date Line East");

# this used to be the unit_tb in the portal database.
my %units = ( 0 => "per month", 1 => "per user" );

my %sideMenuLocations = ( left => "left", right => "right" );

my %countryCodes = ("AF" => "Afghanistan",
    "AL" => "Albania",
    "DZ" => "Algeria",
    "AS" => "American Samoa",
    "AD" => "Andorra",
    "AO" => "Angola",
    "AI" => "Anguilla",
    "AQ" => "Antarctica",
    "AG" => "Antigua and Barbuda",
    "AR" => "Argentina",
    "AM" => "Armenia",
    "AW" => "Aruba",
    "AU" => "Australia",
    "AT" => "Austria",
    "AZ" => "Azerbaijan",
    "BS" => "Bahamas",
    "BH" => "Bahrain",
    "BD" => "Bangladesh",
    "BB" => "Barbados",
    "BY" => "Belarus",
    "BE" => "Belgium",
    "BZ" => "Belize",
    "BJ" => "Benin",
    "BM" => "Bermuda",
    "BT" => "Bhutan",
    "BO" => "Bolivia",
    "BA" => "Bosnia and Herzegovina",
    "BW" => "Botswana",
    "BV" => "Bouvet Island",
    "BR" => "Brazil",
    "IO" => "British Indian Ocean Territory",
    "BN" => "Brunei Darussalam",
    "BG" => "Bulgaria",
    "BF" => "Burkina Faso",
    "BI" => "Burundi",
    "KH" => "Cambodia",
    "CM" => "Cameroon",
    "CA" => "Canada",
    "CV" => "Cape Verde",
    "KY" => "Cayman Islands",
    "CF" => "Central African Republic",
    "TD" => "Chad",
    "CL" => "Chile",
    "CN" => "China",
    "CX" => "Christmas Island",
    "CC" => "Cocos (Keeling Islands)",
    "CO" => "Colombia",
    "KM" => "Comoros",
    "CG" => "Congo",
    "CK" => "Cook Islands",
    "CR" => "Costa Rica",
    "CI" => "Cote D'Ivoire (Ivory Coast)",
    "HR" => "Croatia (Hrvatska)",
    "CU" => "Cuba",
    "CY" => "Cyprus",
    "CZ" => "Czech Republic",
    "DK" => "Denmark",
    "DJ" => "Djibouti",
    "DM" => "Dominica",
    "DO" => "Dominican Republic",
    "TP" => "East Timor",
    "EC" => "Ecuador",
    "EG" => "Egypt",
    "SV" => "El Salvador",
    "GQ" => "Equatorial Guinea",
    "ER" => "Eritrea",
    "EE" => "Estonia",
    "ET" => "Ethiopia",
    "FK" => "Falkland Islands (Malvinas)",
    "FO" => "Faroe Islands",
    "FJ" => "Fiji",
    "FI" => "Finland",
    "FR" => "France",
    "FX" => "France, Metropolitan",
    "GF" => "French Guiana",
    "PF" => "French Polynesia",
    "TF" => "French Southern Territories",
    "GA" => "Gabon",
    "GM" => "Gambia",
    "GE" => "Georgia",
    "DE" => "Germany",
    "GH" => "Ghana",
    "GI" => "Gibraltar",
    "GR" => "Greece",
    "GL" => "Greenland",
    "GD" => "Grenada",
    "GP" => "Guadeloupe",
    "GU" => "Guam",
    "GT" => "Guatemala",
    "GN" => "Guinea",
    "GW" => "Guinea-Bissau",
    "GY" => "Guyana",
    "HT" => "Haiti",
    "HM" => "Heard and McDonald Islands",
    "HN" => "Honduras",
    "HK" => "Hong Kong",
    "HU" => "Hungary",
    "IS" => "Iceland",
    "IN" => "India",
    "ID" => "Indonesia",
    "IR" => "Iran",
    "IQ" => "Iraq",
    "IE" => "Ireland",
    "IL" => "Israel",
    "IT" => "Italy",
    "JM" => "Jamaica",
    "JP" => "Japan",
    "JO" => "Jordan",
    "KZ" => "Kazakhstan",
    "KE" => "Kenya",
    "KI" => "Kiribati",
    "KP" => "Korea (North) (People's Republic)",
    "KR" => "Korea (South) (Republic)",
    "KW" => "Kuwait",
    "KG" => "Kyrgyzstan (Kyrgyz Republic)",
    "LA" => "Laos",
    "LV" => "Latvia",
    "LB" => "Lebanon",
    "LS" => "Lesotho",
    "LR" => "Liberia",
    "LY" => "Libya",
    "LI" => "Liechtenstein",
    "LT" => "Lithuania",
    "LU" => "Luxembourg",
    "MO" => "Macau",
    "MK" => "Macedonia",
    "MG" => "Madagascar",
    "MW" => "Malawi",
    "MY" => "Malaysia",
    "MV" => "Maldives",
    "ML" => "Mali",
    "MT" => "Malta",
    "MH" => "Marshall Islands",
    "MQ" => "Martinique",
    "MR" => "Mauritania",
    "MU" => "Mauritius",
    "YT" => "Mayotte",
    "MX" => "Mexico",
    "FM" => "Micronesia",
    "MD" => "Moldova",
    "MC" => "Monaco",
    "MN" => "Mongolia",
    "MS" => "Montserrat",
    "MA" => "Morocco",
    "MZ" => "Mozambique",
    "MM" => "Myanmar",
    "NA" => "Namibia",
    "NR" => "Nauru",
    "NP" => "Nepal",
    "NL" => "Netherlands",
    "AN" => "Netherlands Antilles",
    "NT" => "Neutral Zone (Saudia Arabia/Iraq)",
    "NC" => "New Caledonia",
    "NZ" => "New Zealand",
    "NI" => "Nicaragua",
    "NE" => "Niger",
    "NG" => "Nigeria",
    "NU" => "Niue",
    "NF" => "Norfolk Island",
    "MP" => "Northern Mariana Islands",
    "NO" => "Norway",
    "OM" => "Oman",
    "PK" => "Pakistan",
    "PW" => "Palau",
    "PA" => "Panama",
    "PG" => "Papua New Guinea",
    "PY" => "Paraguay",
    "PE" => "Peru",
    "PH" => "Philippines",
    "PN" => "Pitcairn",
    "PL" => "Poland",
    "PT" => "Portugal",
    "PR" => "Puerto Rico",
    "QA" => "Qatar",
    "RE" => "Reunion",
    "RO" => "Romania",
    "RU" => "Russian Federation",
    "RW" => "Rwanda",
    "GS" => "S. Georgia and S. Sandwich Isls.",
    "KN" => "Saint Kitts and Nevis",
    "LC" => "Saint Lucia",
    "VC" => "Saint Vincent and The Grenadines",
    "WS" => "Samoa",
    "SM" => "San Marino",
    "ST" => "Sao Tome and Principe",
    "SA" => "Saudi Arabia",
    "SN" => "Senegal",
    "SC" => "Seychelles",
    "SL" => "Sierra Leone",
    "SG" => "Singapore",
    "SK" => "Slovakia (Slovak Republic)",
    "SI" => "Slovenia",
    "SB" => "Solomon Islands",
    "SO" => "Somalia",
    "ZA" => "South Africa",
    "SU" => "Soviet Union (former)",
    "ES" => "Spain",
    "LK" => "Sri Lanka",
    "SH" => "St. Helena",
    "PM" => "St. Pierre and Miquelon",
    "SD" => "Sudan",
    "SR" => "Suriname",
    "SJ" => "Svalbard and Jan Mayen Islands",
    "SZ" => "Swaziland",
    "SE" => "Sweden",
    "CH" => "Switzerland",
    "SY" => "Syria",
    "TW" => "Taiwan",
    "TJ" => "Tajikistan",
    "TZ" => "Tanzania",
    "TH" => "Thailand",
    "TG" => "Togo",
    "TK" => "Tokelau",
    "TO" => "Tonga",
    "TT" => "Trinidad and Tobago",
    "TN" => "Tunisia",
    "TR" => "Turkey",
    "TM" => "Turkmenistan",
    "TC" => "Turks and Caicos Islands",
    "TV" => "Tuvalu",
    "UG" => "Uganda",
    "UA" => "Ukraine",
    "AE" => "United Arab Emirates",
    "UK" => "United Kingdom (Great Britain)",
    "US" => "United States",
    "UY" => "Uruguay",
    "UM" => "US Minor Outlying Islands",
    "UZ" => "Uzbekistan",
    "VU" => "Vanuatu",
    "VA" => "Vatican City State (Holy See)",
    "VE" => "Venezuela",
    "VN" => "Viet Nam",
    "VG" => "Virgin Islands (British)",
    "VI" => "Virgin Islands (US)",
    "WF" => "Wallis and Futuna Islands",
    "EH" => "Western Sahara",
    "YE" => "Yemen",
    "YU" => "Yugoslavia",
    "ZR" => "Zaire",
    "ZM" => "Zambia",
    "ZW" => "Zimbabwe",
);

=head1 NAME

Variables - Object used to build a Variables Object Class.

=head1 SYNOPSIS

  use Portal::Data::Variables;
  my $obj = Portal::Data::Variables->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Variables is a Variables class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Data::Variables module.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{trueFalse} = \%trueFalse;
  $self->{falseTrue} = \%falseTrue;
  $self->{alphabet} = \@alphabet;
  $self->{numbers} = \@numbers;
  $self->{states} = \%states;
  $self->{ccTypes} = \%ccTypes;
  $self->{appTypes} = \%appTypes;
  $self->{adminTypes} = \%adminTypes;
  $self->{timeZones} = \%timeZones;
  $self->{timeZoneNames} = \%timeZoneNames;
  $self->{units} = \%units;
  $self->{timeFormats} = \%timeFormats;
  $self->{countryCodes} = \%countryCodes;
  $self->{sideMenuLocations} = \%sideMenuLocations;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item % formBoolean()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines true = 1, false = 0.

=cut
sub formBoolean
{
  my $self = shift;

  my %result = ();
  $result{names} = [ qw( true false ) ];
  $result{values} = [ qw ( 1 0 ) ];

  return \%result;
}

=item % formTZ()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines the available TimeZones the user can select from.

=cut
sub formTZ
{
  my $self = shift;
  my %result = ();

  $result{names} = [];
  $result{values} = [];

  foreach my $name (sort keys %timeZoneNames)
  {
    push @{$result{names}}, $name . " (" . $timeZoneNames{$name} . ")";
    push @{$result{values}}, $name;
  }

  return \%result;
}

=item % formTimeFormat()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines the available TimeFormats the user can select from.

=cut
sub formTimeFormat
{
  my $self = shift;
  my %result = ();

  $result{names} = [];
  $result{values} = [];

  foreach my $name (sort keys %timeFormats)
  {
    push @{$result{names}}, $timeFormats{$name};
    push @{$result{values}}, $name;
  }

  return \%result;
}

=item % formLanguage()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines the available Languages the user can select from.

=cut
sub formLanguage
{
  my $self = shift;
  my %result = ();

  $result{names} = [];
  $result{values} = [];

  my $doc = HTMLObject::Base->new;
  foreach my $name (sort keys %{$doc->{codeToLanguageHash}})
  {
    push @{$result{names}}, $name . " (" . $doc->{codeToLanguageHash}->{$name} . ")";
    push @{$result{values}}, $name;
  }

  return \%result;
}

=item % formUnits()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines the available Billing Units the user can select from.

=cut
sub formUnits
{
  my $self = shift;
  my %result = ();

  $result{names} = [];
  $result{values} = [];

  foreach my $name (sort keys %units)
  {
    push @{$result{names}}, $units{$name};
    push @{$result{values}}, $name;
  }

  return \%result;
}

=item % formSideMenuLocations()

 Returns the hashref containing the names and values
 arrays defined for use by the HTMLObject::Form module.

 Defines left = left, right = right.

=cut
sub formSideMenuLocations
{
  my $self = shift;

  my %result = ();
  $result{names} = [ qw( left right ) ];
  $result{values} = [ qw ( left right ) ];

  return \%result;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  hash trueFalse (1 => "true", 0 => "false")
  hash falseTrue ( "true" => 1, "false" => 0)
  array alphabet [ lowercase then upper case letters ]
  array numbers [ 1 to 9 then 0 ]
  hash states ( state abbreviation => full name ) All States/Possessions
  hash ccTypes - hash of credit card types we recognize
  hash appTypes - hash of application types and their display names
  hash adminTypes - hash of application admin types and their display names
  hash timeZones - hash of TimeZone codes to hours off
  hash timeZoneNames - hash of TimeZone codes to descriptions
  hash units - hash of unit descriptions from unit_tb.
  hash timeFormats - AM/PM or Military Time definitions.
  hash countryCodes - 2 letter Country Codes and their full names.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
