# Apps.pm - The Object Class that provides a Apps Object
# Created by James A. Pattie, 2004-08-17.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Forms::Apps;
use strict;
use Portal::AppState;
use Portal::Objects::UserApplicationObject;
use Portal::Objects::CompanyApplicationObject;
use Portal::Objects::SOAPResult;
use HTMLObject::Form;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

=head1 NAME

Apps - Object used to build the Apps Forms Class.

=head1 SYNOPSIS

  use Portal::Forms::Apps;
  my $obj = Portal::Forms::Apps->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

  # ask the $obj to process Apps related forms.

=head1 DESCRIPTION

Apps is a Apps Forms class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Forms::Apps object.
 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{formObj} = HTMLObject::Form->new();
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->langObj);
  if ($self->variables->error)
  {
    $self->error($self->variables->errorMessage);

    return $self;
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item % selectAppsAdminForm(companyObj, url, app, state)

 companyObj - the company to show application assignments for.
 url - the url to specify in the forms action.
 app - the app that is working with the applications.
 state - the state in the app that is working with the applications.

 generates the form that provides an admin user with a list
 of all purchased apps for the selected company and allows them
 to manage the purchased apps or purchase new apps.

 returns %html, where:
   html: hash that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 If an error occured, we return undef.

=cut
sub selectAppsAdminForm
{
  my $self = shift;
  my %args = ( companyObj => undef, url => "", app => "", state => "", @_ );
  my $companyObj = $args{companyObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my $caller = $self->userObj;

  # make sure I got valid input.
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  if (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to change the user's password.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  elsif (!$caller->sysadmin && $caller->admin && ($caller->companyId != $companyObj->id))
  {
    $self->invalid("caller", $caller->uname, sprintf($self->langObj->map("YouAreCompanyAdminButWrongCompany", $companyObj->code)));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my %form = ();
  my $purchasedAppsForCompanyPhrase = sprintf($self->langObj->map("PurchasedAppsForCompany"), $companyObj->name);
  my $unpurchasedAppsForCompanyPhrase = sprintf($self->langObj->map("UnPurchasedAppsForCompany"), $companyObj->name);
  # display. :)  Let the appName be the link that then lets you drill in to manage it.

  # see if we need to display any success/failure messages.
  if (exists $self->input->{errorMessage})
  {
    my $string = $self->methods->displayMessageStr(type => "error", message => $self->input->{errorMessage},
       break => "both");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }
    $form{body} .= $string;
  }
  if (exists $self->input->{statusMessage})
  {
    my $string = $self->methods->displayMessageStr(type => "status", message => $self->input->{statusMessage},
       break => "both");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }
    $form{body} .= $string;
  }

  my $namePhrase = $self->langObj->map("Name");
  my $typePhrase = $self->langObj->map("Type");
  my $costPhrase = $self->langObj->map("Cost");
  my $numLicensesPhrase = $self->langObj->map("numLicenses");
  my $numAssignedPhrase = $self->langObj->map("numAssigned");
  my $descriptionPhrase = $self->langObj->map("Description");
  my $usersPhrase = $self->langObj->map("Users");
  my $doc = HTMLObject::Normal->new();

  $form{body} .= <<"END_OF_CODE";
<span style="font-size: smaller;">$purchasedAppsForCompanyPhrase</span><br />
<table border="1" width="100%" cellpadding="2" cellspacing="1">
  <tr>
    <th><span style="font-size: smaller;">$namePhrase</span></th>
    <th><span style="font-size: smaller;">$descriptionPhrase</span></th>
    <th><span style="font-size: smaller;">$typePhrase</span></th>
    <th><span style="font-size: smaller;">$costPhrase</span></th>
    <th><span style="font-size: smaller;">$numLicensesPhrase</span></th>
    <th><span style="font-size: smaller;">$numAssignedPhrase</span></th>
    <th><span style="font-size: smaller;">$usersPhrase</span></th>
  </tr>
END_OF_CODE

  my $url = $self->methods->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->sessionObj->store->{companyAppObj});
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage);
    return undef;
  }

  # get the list of purchased apps.
  my @purchasedApps = $self->applicationObj->getAppsForCompany(companyObj => $companyObj);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  # now walk the list of returned apps and output them.
  my $counter = 0;
  foreach my $appObj (@purchasedApps)
  {
    my $class = ($counter % 2 == 0 ? "primaryRow" : "secondaryRow");
    $counter++;

    my $cost = ($appObj->cost eq "0.00" ? "<b>Free</b>" : ("\$" . $appObj->cost . " " .  $self->portalVariables->units->{$appObj->unit}));

    # get the License Info.
    my $licenseObj = $self->applicationObj->getCompanyLicenseInfo(companyId => $companyObj->id, appId => $appObj->id);
    if ($self->applicationObj->error)
    {
      $self->error($self->applicationObj->errorMessage);
      return undef;
    }
    my $total = ($licenseObj->numTotal == 0 ? $self->langObj->map("Unlimited") : $licenseObj->numTotal);
    my $used = ($licenseObj->numTotal == 0 ? $licenseObj->numFree : $licenseObj->numTotal - $licenseObj->numFree);

    # get the list of users that are assigned this app.
    my @assignedUsers = $self->applicationObj->getUsersAssignedToAppForCompany(companyObj => $companyObj, appObj => $appObj);
    if ($self->applicationObj->error)
    {
      $self->error($self->applicationObj->errorMessage);
      return undef;
    }
    my $assignedUsers = "";
    foreach my $user (@assignedUsers)
    {
      $assignedUsers .= ", " if ($assignedUsers);
      $assignedUsers .= $user->uname;
    }

    my $appUrl = $self->methods->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $app, state => $state, command=> "displayApp", appId => $appObj->id });
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }

    my $displayPhrase = sprintf($self->langObj->map("Display"), $appObj->name);

    # add the apps' icon to the link.
    my $icon = $self->methods->createBaseURL(type => "App", linkType => "image", appConfigObj => $appObj, appName => $appObj->name) . $appObj->iconName;
    my $iconWidth = 40;
    my $iconHeight = 40;

    my %hvars = ( description => $appObj->description, name => $appObj->name, type => $appObj->type );
    $form{body} .= <<"END_OF_CODE";
  <tr class="$class">
    <td><span style="font-size: smaller;"><a href="$appUrl" title="$displayPhrase" class="$class"><img src="$icon" border="0" alt="$hvars{description}" width="$iconWidth" height="$iconHeight" /> $hvars{name}</a></span></td>
    <td><span style="font-size: smaller;">$hvars{description}</span></td>
    <td><span style="font-size: smaller;">$hvars{type}</span></td>
    <td align="right"><span style="font-size: smaller;">$cost</span></td>
    <td align="right"><span style="font-size: smaller;">$total</span></td>
    <td align="right"><span style="font-size: smaller;">$used</span></td>
    <td><span style="font-size: smaller;">$assignedUsers</span></td>
  </tr>
END_OF_CODE
  }

  $form{body} .= <<"END_OF_CODE";
</table>
<br />
<span style="font-size: smaller;">$unpurchasedAppsForCompanyPhrase</span><br />
<table border="1" width="100%" cellpadding="2" cellspacing="1">
  <tr>
    <th><span style="font-size: smaller;">$namePhrase</span></th>
    <th><span style="font-size: smaller;">$descriptionPhrase</span></th>
    <th><span style="font-size: smaller;">$typePhrase</span></th>
    <th><span style="font-size: smaller;">$costPhrase</span></th>
  </tr>
END_OF_CODE

  # now get the list of un-purchased apps.
  my @unpurchasedApps = $self->applicationObj->getUnAssignedAppsForCompany(companyObj => $companyObj);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  # now walk the list of returned apps and output them.
  $counter = 0;
  foreach my $appObj (@unpurchasedApps)
  {
    my $class = ($counter % 2 == 0 ? "primaryRow" : "secondaryRow");
    $counter++;

    my $cost = ($appObj->cost eq "0.00" ? "<b>Free</b>" : ("\$" . $appObj->cost . " " .  $self->portalVariables->units->{$appObj->unit}));

    my $appUrl = $self->methods->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $app, state => $state, command=> "purchaseApp", appId => $appObj->id });
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }

    my $purchaseAppPhrase = sprintf($self->langObj->map("PurchaseApp"), $appObj->name);

    # add the apps' icon to the link.
    my $icon = $self->methods->createBaseURL(type => "App", linkType => "image", appConfigObj => $appObj, appName => $appObj->name) . $appObj->iconName;
    my $iconWidth = 40;
    my $iconHeight = 40;

    my %hvars = ( description => $appObj->description, name => $appObj->name, type => $appObj->type );
    $form{body} .= <<"END_OF_CODE";
  <tr class="$class">
    <td><span style="font-size: smaller;"><a href="$appUrl" title="$purchaseAppPhrase" class="$class"><img src="$icon" border="0" alt="$hvars{description}" width="$iconWidth" height="$iconHeight" /> $hvars{name}</a></span></td>
    <td><span style="font-size: smaller;">$hvars{description}</span></td>
    <td><span style="font-size: smaller;">$hvars{type}</span></td>
    <td align="right"><span style="font-size: smaller;">$cost</span></td>
  </tr>
END_OF_CODE
  }

  $form{body} .= <<"END_OF_CODE";
</table>
END_OF_CODE

  return %form;
}

=item % displayAppForAdmin(companyObj, companyAppObj, url, app, state)

 companyObj - the Company Object to work with.
 companyAppObj - the companyAppObj to show details of.
 url - the url to specify in the forms action.
 app - the app that is working with the applications.
 state - the state in the app that is working with the applications.

 generates the form that provides an admin user with a list
 of the users assigned to this app and the app's settings.

 returns %html, where:
   html: hash that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 If an error occured, we return undef.

=cut
sub displayAppForAdmin
{
  my $self = shift;
  my %args = ( companyObj => undef, companyAppObj => undef, url => "", app => "", state => "", @_ );
  my $companyObj = $args{companyObj};
  my $companyAppObj = $args{companyAppObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my $caller = $self->userObj;

  # make sure I got valid input.
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to change the view the company/app info.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  elsif (!$caller->sysadmin && $caller->admin && ($caller->companyId != $companyObj->id))
  {
    $self->invalid("caller", $caller->uname, sprintf($self->langObj->map("YouAreCompanyAdminButWrongCompany", $companyObj->code)));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my %form = ();

  # see if we need to display any success/failure messages.
  if (exists $self->input->{errorMessage})
  {
    my $string = $self->methods->displayMessageStr(type => "error", message => $self->input->{errorMessage},
       break => "both");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }
    $form{body} .= $string;
  }
  if (exists $self->input->{statusMessage})
  {
    my $string = $self->methods->displayMessageStr(type => "status", message => $self->input->{statusMessage},
       break => "both");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }
    $form{body} .= $string;
  }

  # lookup the Application info for the CompanyAppObj->get("appId") value.
  my $appObj = $self->applicationObj->getApplicationInfo(id => $companyAppObj->get("appId"));
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  if (!defined $appObj)
  {
    $self->error(sprintf($self->langObj->map("invalidApp"), $companyAppObj->get("appId")));
    return undef;
  }
  if (!$appObj->isValid)
  {
    $self->error($appObj->errorMessage);
    return undef;
  }

  my $cost = ($appObj->cost eq "0.00" ? "<b>Free</b>" : ("\$" . $appObj->cost . " " .  $self->portalVariables->units->{$appObj->unit}));

  # get the License Info.
  my $licenseObj = $self->applicationObj->getCompanyLicenseInfo(companyId => $companyObj->id, appId => $appObj->id);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  my $total = ($licenseObj->numTotal == 0 ? $self->langObj->map("Unlimited") : $licenseObj->numTotal);
  my $used = ($licenseObj->numTotal == 0 ? $licenseObj->numFree : $licenseObj->numTotal - $licenseObj->numFree);

  # display the info about this App (Name, Description, type, cost, # licenses, # free, Server, Port)
  # The Server and Port, DB values are editable.

  # Provide a link to Sell the App and another link to Buy more licenses, but only if the app is not free.

  # display the users of this company that have the app assigned and their App preferences.
  # Clicking on the users name will let you edit their preferences.

  # Provide a mechanism to assign/unassign users to this app.
  # Have 2 multi-select edit boxes (the left = Assigned Users, the right = Un-Assigned Users).
  # Have buttons between the boxes (<- and ->) to move users between the boxes.  As you move
  # users between them, we update 2 associative arrays that will be converted to strings of comma delimited users
  # that get sent to the backend to let us deteremine what users are to have the app assigned and what
  # users need the app un-assigned.

  # NOTE: Make sure that the backend code uses the SOAP infrastructure as much as possible.

  my $namePhrase = $self->langObj->map("Name");
  my $typePhrase = $self->langObj->map("Type");
  my $costPhrase = $self->langObj->map("Cost");
  my $numLicensesPhrase = $self->langObj->map("numLicenses");
  my $numAssignedPhrase = $self->langObj->map("numAssigned");
  my $descriptionPhrase = $self->langObj->map("Description");
  my $usersPhrase = $self->langObj->map("Users");
  my $sellAppLicensesPhrase = $self->langObj->map("SellAppLicenses");
  my $purchaseMoreLicensesPhrase = $self->langObj->map("PurchaseMoreLicenses");
  my $userAppSettingsPhrase = $self->langObj->map("UserAppSettings");
  my $userNamePhrase = $self->langObj->map("userName");
  my $autoRunPhrase = $self->langObj->map("AutoRun");
  my $wapEnabledPhrase = $self->langObj->map("WAPEnabled");
  my $widthPhrase = $self->langObj->map("Width");
  my $heightPhrase = $self->langObj->map("Height");
  my $editPhrase = $self->langObj->map("Edit");
  #my $Phrase = $self->langObj->map("");
  my $doc = HTMLObject::Normal->new();

  my $url = $self->methods->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->sessionObj->store->{companyAppObj});
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage);
    return undef;
  }

  my $sellUrl = $self->methods->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $app, state => $state, command=> "sellApp", appId => $appObj->id });
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage);
    return undef;
  }

  my $purchaseUrl = $self->methods->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $app, state => $state, command=> "purchaseLicenses", appId => $appObj->id });
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage);
    return undef;
  }
  # don't display the Sell URL if we are managing the calling App.
  my $sellStr = ($appObj->name eq $app ? "&nbsp;" : "<a href=\"$sellUrl\" class=\"main\">$sellAppLicensesPhrase</a>");
  my $purchaseStr = ($licenseObj->numTotal == 0 ? "&nbsp;" : "<a href=\"$purchaseUrl\" class=\"main\">$purchaseMoreLicensesPhrase</a>");

  my %hvars = (name => $appObj->name, description => $appObj->description, type => $appObj->type );
  $form{body} .= <<"END_OF_CODE";
<table border="0" width="100%" cellpadding="2" cellspacing="1">
  <tr>
    <td><span style="font-size: smaller;"><b>$namePhrase</b>: <span style="font-size: smaller;">$hvars{name}</span></span></td>
    <td><span style="font-size: smaller;"><b>$descriptionPhrase</b>: <span style="font-size: smaller;">$hvars{description}</span></span></td>
  </tr>
  <tr>
    <td><span style="font-size: smaller;"><b>$typePhrase</b>: <span style="font-size: smaller;">$hvars{type}</span></span></td>
    <td><span style="font-size: smaller;"><b>$costPhrase</b>: <span style="font-size: smaller;">$cost</span></span></td>
  </tr>
  <tr>
    <td><span style="font-size: smaller;"><b>$numLicensesPhrase</b>: <span style="font-size: smaller;">$total</span></span></td>
    <td><span style="font-size: smaller;"><b>$numAssignedPhrase</b>: <span style="font-size: smaller;">$used</span></span></td>
  </tr>
  <tr>
    <td align="left"><span style="font-size: smaller;">$sellStr</span></td>
    <td align="left"><span style="font-size: smaller;">$purchaseStr</span></td>
  </t>
</table>
<hr />
<span style="font-size: smaller;">$userAppSettingsPhrase</span><br />
<table border="0" width="100%" cellpadding="2" cellspacing="0">
  <tr>
    <th align="left"><span style="font-size: smaller;">$userNamePhrase</span></th>
    <th><span style="font-size: smaller;">$autoRunPhrase</span></th>
    <th><span style="font-size: smaller;">$wapEnabledPhrase</span></th>
    <th><span style="font-size: smaller;">$widthPhrase</span></th>
    <th><span style="font-size: smaller;">$heightPhrase</span></th>
  </tr>
END_OF_CODE

  # get the list of users that are assigned this app.
  my @assignedUsers = $self->applicationObj->getUsersAssignedToAppForCompany(companyObj => $companyObj, appObj => $appObj);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  my $counter = 0;
  foreach my $userObj (@assignedUsers)
  {
    my $class = ($counter % 2 == 0 ? "primaryRow" : "secondaryRow");
    $counter++;

    my $editAppUrl = $self->methods->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => $app, state => $state, command=> "editUserAppSettings", appId => $appObj->id, userId => $userObj->id });
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return undef;
    }

    # now lookup the user_app_tb info for this user and app pair.
    my $userAppObj = $self->applicationObj->getUserAppInfo(userObj => $userObj, appObj => $appObj);
    if ($self->applicationObj->error)
    {
      $self->error($self->applicationObj->errorMessage);
      return undef;
    }
    if (!defined $userAppObj)
    {
      $self->error("getUserAppInfo failed!  userAppObj not defined");
      return undef;
    }
    if (!$userAppObj->isValid)
    {
      $self->error($userAppObj->errorMessage);
      return undef;
    }

    # make sure there is nothing left over in the formObj that can mess us up when we are coming
    # back from one of the update screens.
    $self->formObj->clearErrors("all");

    my %userAppForm = $self->formObj->generate(data => $userAppObj->data->{view},
      template => $userAppObj->template->{view},
      name => "userApp",
      action => "",
      method => "post",
      profile => $userAppObj->profile->{view},
      readOnly => 1);
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return undef;
    }

    # build up the #EDITLINK# replacement - Username
    # 2005-01-28 tested the ${\( func )} code, incase you wonder what is happening with $userObj->uname.
    my $editLink = "<a href=\"$editAppUrl\" class=\"$class\" title=\"$editPhrase ${\( $userObj->uname )}\">" . $userObj->uname . " (" . $userObj->fname . " " . $userObj->mname . " " . $userObj->lname . ")</a>";
    my $formBody = $userAppForm{body};
    $formBody =~ s/#TRCLASS#/$class/gm;
    $formBody =~ s/#EDITLINK#/$editLink/gm;

    # now display the UserApp info and make the entry editable
    $form{body} .= $formBody;
  }

  $form{body} .= "</table>\n";

  # build up the assignedUsers data structures for the assignments form.
  my $assignedUsers = "";
  my $assignedUsersOptions;
  foreach my $userObj (@assignedUsers)
  {
    $assignedUsers .= "," if ($assignedUsers);
    $assignedUsers .= $userObj->uname;
  }
  my $assignedUsersOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@assignedUsers,
    handler => $self->methods->getHandlerSub("User"));
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return undef;
  }

  # now get the list of users in this company so we can build the un-assigned users lists.
  my %availableUsersArgs = (companyObj => $companyObj);
  if ($appObj->type eq "administration")
  {
    if ($appObj->adminType eq "system")
    {
      $availableUsersArgs{sysadmin} = 1;
    }
    else
    {
      $availableUsersArgs{admin} = 1;
    }
  }
  my @availableUsers = $self->authObj->getListOfUsersForCompany(%availableUsersArgs);
  if ($self->authObj->error)
  {
    $self->error($self->authObj->errorMessage);
    return undef;
  }
  my $unassignedUsers = "";
  my $unassignedUsersOptions;
  my @unassignedUsers = ();
  foreach my $userObj (@availableUsers)
  {
    # only deal with the user as long as they are not in the assignedUsers array.
    my $found = 0;
    foreach my $tmpUser (@assignedUsers)
    {
      if ($userObj == $tmpUser)
      {
        $found = 1;
        last; # break out of the loop.
      }
    }
    if (!$found)
    {
      $unassignedUsers .= "," if ($unassignedUsers);
      $unassignedUsers .= $userObj->uname;

      push @unassignedUsers, $userObj;
    }
  }
  my $unassignedUsersOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@unassignedUsers,
    handler => $self->methods->getHandlerSub("User"));
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return undef;
  }

  # get the serverInfo entries.
  my @serverInfoObjs = $self->applicationObj->getAppServerEntries(id => $appObj->id);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  my $serverInfoOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@serverInfoObjs,
    handler => $self->methods->getHandlerSub("AppServerObject"));
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return undef;
  }

  # now we build the Assignments form and the ServerInfo form.
  $companyAppObj->data->{assignment}->{users}->{-assignedOptions} = $assignedUsersOptions;
  $companyAppObj->data->{assignment}->{users}->{-unassignedOptions} = $unassignedUsersOptions;
  $companyAppObj->data->{assignment}->{users}->{-assignedValues} = $assignedUsers;
  $companyAppObj->data->{assignment}->{users}->{-unassignedValues} = $unassignedUsers;
  $companyAppObj->data->{serverInfo}->{serverInfo}->{-Options} = $serverInfoOptions;
  foreach my $formName (qw (assignment serverInfo))
  {
    $companyAppObj->data->{$formName}->{appId}->{-Value} = $appObj->id;
  }

  # make sure there is nothing left over in the formObj that can mess us up when we are coming
  # back from one of the update screens.
  $self->formObj->clearErrors("all");

  my %assignForm = $companyAppObj->printForm(form => "assignment",
    name => "assignments", action => $url, app => $app, state => $state );
  if ($companyAppObj->error)
  {
    $self->error($companyAppObj->errorMessage);
    return undef;
  }

  my %serverInfoForm = $companyAppObj->printForm(form => "serverInfo",
    action => $url, app => $app, state => $state );
  if ($companyAppObj->error)
  {
    $self->error($companyAppObj->errorMessage);
    return undef;
  }

  # start the table that will wrap the forms.
  $form{body} .= <<"END_OF_CODE";
<hr />
<table border="0" width="100%" cellpadding="0" cellspacing="0">
  <tr>
    <td align="center" valign="top">
END_OF_CODE

  foreach my $part (keys %assignForm)
  {
    if (ref $assignForm{$part} ne "ARRAY")
    {
      $form{$part} .= $assignForm{$part};
    }
    else
    {
      foreach my $include (@{$assignForm{$part}})
      {
        push @{$form{$part}}, $include;
      }
    }
  }

  # close out the assignments form and start on the serverInfo form.
  $form{body} .= <<"END_OF_CODE";
    </td>
    <td>&nbsp;&nbsp;&nbsp;</td>
    <td align="center" valign="top">
END_OF_CODE

  foreach my $part (keys %serverInfoForm)
  {
    if (ref $serverInfoForm{$part} ne "ARRAY")
    {
      $form{$part} .= $serverInfoForm{$part};
    }
    else
    {
      foreach my $include (@{$serverInfoForm{$part}})
      {
        push @{$form{$part}}, $include;
      }
    }
  }

  # wrap up the table.
  $form{body} .= <<"END_OF_CODE";
    </td>
  </tr>
</table>
END_OF_CODE

  return %form;
}

=item SOAPResult updateAppAssignments(companyObj, companyAppObj, assignedUsers, unassignedUsers)

 companyObj - the Company Object to work with.
 companyAppObj - the CompanyApplicationObject to work with.
 assignedUsers - arrayref of users the app is to be assigned to.
 unassignedUsers - arrayref of users the app should not be assigned to.

 updates the app assignments.

 returns SOAPResult result: 0 = error occured,
           1 = app assignments successfully done.,
         message: Any text you want displayed to the user.

         undef if SOAPResult failed to instantiate.

=cut
sub updateAppAssignments
{
  my $self = shift;
  my %args = ( companyObj => undef, companyAppObj => undef, assignedUsers => undef, unassignedUsers => undef, @_ );
  my $companyObj = $args{companyObj};
  my $companyAppObj = $args{companyAppObj};
  my $assignedUsers = $args{assignedUsers};
  my $unassignedUsers = $args{unassignedUsers};
  my $caller = $self->userObj;

  # start our result object, default to a failure result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->langObj, result => -1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  # make sure I got valid input.
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  # verify the caller is allowed to assign apps to users.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  elsif (!$caller->sysadmin && $caller->admin && ($caller->companyId != $companyObj->id))
  {
    $self->invalid("caller", $caller->uname, sprintf($self->langObj->map("YouAreCompanyAdminButWrongCompany", $companyObj->code)));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  # first lookup the application we are dealing with.
  my $appObj = $self->applicationObj->getApplicationInfo(id => $companyAppObj->get("appId"));
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  if (!defined $appObj)
  {
    $self->error(sprintf($self->langObj->map("invalidApp"), $companyAppObj->get("appId")));
    return undef;
  }
  if (!$appObj->isValid)
  {
    $self->error($appObj->errorMessage);
    return undef;
  }

  # next get a list of all users assigned this app in this company.
  my @currentlyAssignedUsers = $self->applicationObj->getUsersAssignedToAppForCompany(companyObj => $companyObj, appObj => $appObj);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }

  # now instantiate the soapObj for this application.
  my $soapObj = undef;
  eval { $soapObj = $self->methods->getPortalSOAPObj(caller => $caller, configObj => $self->configObj,
    location => $appObj->name, companyAppObj => $companyAppObj); };
  if ($@)
  {
    $self->error("getPortalSOAPObj eval (app = '" . $appObj->name . "'): " . $@);
    return $result;
  }
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage);
    return $result;
  }
  if (!defined $soapObj)
  {
    $self->error("soapObj is not defined! (app = '" . $appObj->name . "')<br />\n" . $self->methods->errorMessage);
    return $result;
  }
  if ($soapObj->error)
  {
    $self->error($soapObj->errorMessage);
    return $result;
  }

  # now walk the assignedUsers variable and make sure all users on it are
  # already assigned the app or assign the app to them if not.
  foreach my $user (@{$assignedUsers})
  {
    # see if the user already has this app assigned.
    my $found = 0;
    foreach my $assigned (@currentlyAssignedUsers)
    {
      if ($assigned->uname eq $user)
      {
        $found = 1;
        last;
      }
    }
    if (!$found)
    {
      # lookup the userObj.
      my $userObj = $self->authObj->getUserInfo(uname => $user);
      if ($self->authObj->error)
      {
        $result->result(0);
        $result->message($result->message . $self->authObj->errorMessage);
      }
      else
      {
        # create a UserApplicationObject instance.
        my $userAppObj = Portal::Objects::UserApplicationObject->new(langObj => $self->langObj);
        if ($userAppObj->error)
        {
          $self->error($userAppObj->errorMessage);
          return $result;
        }
        $userAppObj->populate(companyId => $userObj->companyId,
          appId => $appObj->id, userId => $userObj->id, height => $companyAppObj->get("height"),
          width => $companyAppObj->get("width"), autorun => $companyAppObj->get("autorun"), wap => $companyAppObj->get("wap"));
        if ($userAppObj->error)
        {
          $self->error($userAppObj->errorMessage);
          return $result;
        }

        # now assign the app to them.
        my $code = $self->applicationObj->assignAppToUser(userAppObj => $userAppObj);
        if ($self->applicationObj->error)
        {
          $self->error($self->applicationObj->errorMessage);
          return $result;
        }
        if ($code != 1)
        {
          # codes -1 to -5 should not happen since we have pretty much checked for all their cases before getting here.
          if ($code == -6)
          {
            $result->result(0);
            $result->message($result->message . "License violation when assigning " . $appObj->name . " to " . $userObj->uname . "!");
          }
          if ($code == -7)
          {
            $result->result(0);
            $result->message($result->message . $appObj->name . " is an administration app and " . $userObj->uname . " is not an admin!");
          }
        }
        else
        {
          $result->message($result->message . $userObj->uname . " has app assigned...<br />\n");
          # let the app do any setup it needs to, now that the user has them assigned.
          my $tmpResult = undef;
          # removed the SOAP:: or SOAP-> attribute to force it to go to the server.
          eval { $tmpResult = $soapObj->setupUser(companyObj => $companyObj, userObj => $userObj,
                  appObj => $appObj, companyAppObj => $companyAppObj); };
          if ($@)
          {
            $self->error("setupUser eval (app = '" . $appObj->name . "'): " . $@);
            return $result;
          }
          if ($soapObj->error)
          {
            $self->error($soapObj->errorMessage);
            return $result;
          }
          # Update the result info.
          $result->message($result->message . $tmpResult->message);
          $result->result($tmpResult->result) if ($tmpResult->result == 0);  # only overwrite our result value if an error occured.
        }
      }
    }
  }

  # now walk the unassignedUsers variable and make sure all users on it
  # do not have the app assigned.  Unassign the app if they do.
  foreach my $user (@{$unassignedUsers})
  {
    my $userObj = undef;
    # see if the user has this app assigned.
    my $found = 0;
    foreach my $assigned (@currentlyAssignedUsers)
    {
      if ($assigned->uname eq $user)
      {
        # make sure the user is not trying to un-assign the current app from themselves.
        if ($appObj->name eq $self->app && $user eq $caller->uname)
        {
          $result->message($result->message . "You can not unAssign " . $self->app . " from yourself!<br />\n");
        }
        else
        {
          $found = 1;
          $userObj = $assigned;
        }
        last;
      }
    }
    if ($found)
    {
      # create a UserApplicationObject instance.
      my $userAppObj = Portal::Objects::UserApplicationObject->new(langObj => $self->langObj);
      if ($userAppObj->error)
      {
        $self->error($userAppObj->errorMessage);
        return $result;
      }
      $userAppObj->populate(companyId => $userObj->companyId,
        appId => $appObj->id, userId => $userObj->id, height => $companyAppObj->get("height"),
        width => $companyAppObj->get("width"), autorun => $companyAppObj->get("autorun"), wap => $companyAppObj->get("wap"));
      if ($userAppObj->error)
      {
        $self->error($userAppObj->errorMessage);
        return $result;
      }

      # now unassign the app from them.
      my $code = $self->applicationObj->unAssignAppFromUser(userAppObj => $userAppObj);
      if ($self->applicationObj->error)
      {
        $self->error($self->applicationObj->errorMessage);
        return $result;
      }
      if ($code != 1)
      {
        if ($code == -1)
        {
          $result->result(0);
          $result->message($result->message . $appObj->name . " wasn't assigned to " . $userObj->uname . " even though it just was!");
        }
      }
      else
      {
        $result->message($result->message . $userObj->uname . " has app unAssigned...<br />\n");
        # let the app do any cleanup it needs to, now that the user has them unassigned.
        my $tmpResult = undef;
        # removed the SOAP:: or SOAP-> attribute to force it to go to the server.
        eval { $tmpResult = $soapObj->cleanupUser(companyObj => $companyObj, userObj => $userObj,
                appObj => $appObj, companyAppObj => $companyAppObj); };
        if ($@)
        {
          $self->error("cleanupUser eval (app = '" . $appObj->name . "'): " . $@);
          return $result;
        }
        if ($soapObj->error)
        {
          $self->error($soapObj->errorMessage);
          return $result;
        }
        # Update the result info.
        $result->message($result->message . $tmpResult->message);
        $result->result($tmpResult->result) if ($tmpResult->result == 0);  # only overwrite our result value if an error occured.
      }
    }
  }

  $soapObj = undef;  # force the App SOAP object to go out of scope.

  $result->result(1) if ($result->result == -1);  # signal we successfully updated app assignments, as long as an error didn't occur.
  return $result;
}

=item @ purchaseApp(appObj, companyObj, url, app, state)

 appObj - specifies the Application object of the app we are purchasing.
 companyObj - specifies the Company object purchasing the app.
 url - the url to specify in the forms action.
 app - the app that is purchasing the app.
 state - the state in the app that is purchasing the app.

 returns an array with ($result, \%html, $companyAppObj, $onload) as
 values, where:
   result: 0 = error occured,
           1 = info successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   companyAppObj: the CompanyApplicationObject.
   onload: javascript code to pass to setOnload() method.

 The resulting form is named: purchaseAppForm

=cut
sub purchaseApp
{
  my $self = shift;
  my %args = ( appObj => undef, companyObj => undef, url => "", app => "", state => "", @_ );
  my $appObj = $args{appObj};
  my $companyObj = $args{companyObj};
  my $caller = $self->userObj;
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, undef, "document.purchaseAppForm.numLicenses.focus();" );
  my $formName = "purchaseApp";

  # make sure I got valid input.
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to purchase an app.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  # instantiate a CompanyApplicationObject
  my $companyAppObj = Portal::Objects::CompanyApplicationObject->new(langObj => $self->langObj);
  if ($companyAppObj->error)
  {
    $self->error($companyAppObj->errorMessage);
    return @result;
  }

  # get the serverInfo entries.
  my @serverInfoObjs = $self->applicationObj->getAppServerEntries(id => $appObj->id);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return @result;
  }
  my $serverInfoOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@serverInfoObjs,
    handler => $self->methods->getHandlerSub("AppServerObject"));
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return @result;
  }

  # look up the App's config we are purchasing.
  my $appConfigObj;
  eval "\$appConfigObj = Portal::Data::Config->new(langObj => \$self->langObj, app => \$appObj->name);";
  if ($@)
  {
    $self->error($@);
    return @result;
  }

  my $dbName = ($appConfigObj->dbPerCompany ? $companyObj->code . "_" . $appConfigObj->dbName : $appConfigObj->dbName);

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  if (!exists $self->input->{formSubmitted})
  {
    my %companyAppInfo = ( appId => $appObj->id, companyId => $companyObj->id,
      cost => $appObj->cost, unit => $appObj->unit, dbName => $dbName, height => $appObj->height,
      width => $appObj->width, autorun => $appObj->autorun, wap => $appObj->wap );

    $companyAppObj->populate(%companyAppInfo);
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->formObj->validate(input => $self->input, profile => $companyAppObj->profile->{$formName},
      data => $companyAppObj->data->{$formName});
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, purchase the App.

      my $dbHost;
      my $dbPort;
      my $server;
      my $port;
      foreach my $appServer (@serverInfoObjs)
      {
        if ($appServer->counter == $self->input->{server})
        {
          $dbHost = $appServer->dbHost;
          $dbPort = $appServer->dbPort;
          $server = $appServer->server;
          $port = $appServer->port;
          last;
        }
      }

      if (!defined $server)
      {
        $self->formObj->invalid("server", $self->input->{server});
        $self->formObj->deleteErrorsEntry("valid", "server");
        $displayForm = 1;
      }
      else
      {
        # build up the values from the $appObj object and from the user input
        my %companyAppInfo = ( appId => $appObj->id, companyId => $companyObj->id,
          number => $self->formObj->getValidEntry("numLicenses"),
          server => $server, port => $port, dbHost => $dbHost, dbType => $appObj->dbType, dbPort => $dbPort,
          cost => $appObj->cost, unit => $appObj->unit, dbName => $dbName,
          height => $appObj->height, width => $appObj->width, autorun => $appObj->autorun, wap => $appObj->wap );

        # make sure the new info is valid via the Portal::Objects::CompanyApplicationObject class.
        my $tmpCompanyAppObj = Portal::Objects::CompanyApplicationObject->new(langObj => $self->langObj);
        $tmpCompanyAppObj->populate(%companyAppInfo);
        if ($tmpCompanyAppObj->error)
        {
          $displayForm = 1;
        }
        $companyAppObj = $tmpCompanyAppObj;  # make sure we have the populated form for display purposes.
      }
      if (!$displayForm)
      {
        # at this point, we have a valid companyAppObj, so purchase the app.
        my $result = $self->applicationObj->assignAppToCompany(companyAppObj => $companyAppObj);
        if ($self->applicationObj->error)
        {
          $self->error($self->applicationObj->errorMessage);
          return @result;
        }
        if ($result == 1)
        {
          # now instantiate the soapObj for this application.
          my $soapObj = undef;
          eval { $soapObj = $self->methods->getPortalSOAPObj(caller => $caller, configObj => $self->configObj,
            location => $appObj->name, companyAppObj => $companyAppObj); };
          if ($@)
          {
            $self->error("getPortalSOAPObj eval (app = '" . $appObj->name . "'): " . $@);
            return @result;
          }
          if ($self->methods->error)
          {
            $self->error($self->methods->errorMessage);
            return @result;
          }
          if (!defined $soapObj)
          {
            $self->error("soapObj is not defined! (app = '" . $appObj->name . "')<br />\n" . $self->methods->errorMessage);
            return @result;
          }
          if ($soapObj->error)
          {
            $self->error($soapObj->errorMessage);
            return @result;
          }

          my $tmpResult = undef;
          # removed the SOAP:: or SOAP-> attribute to force it to go to the server.
          eval { $tmpResult = $soapObj->setupCompany(companyObj => $companyObj,
                  appObj => $appObj, companyAppObj => $companyAppObj); };
          if ($@)
          {
            $self->error("setupCompany eval (app = '" . $appObj->name . "', company = '" . $companyObj->name . "'): " . $@);
            return @result;
          }
          if ($soapObj->error)
          {
            $self->error($soapObj->errorMessage);
            return @result;
          }
          # Update the result info.
          $form{body} .= "<br />\n" . $tmpResult->message;

          $result[0] = 1;  # signal we successfully purchased the app.
          $result[1] = \%form;
          $result[2] = $companyAppObj;  # pass back the CompanyApplicationObject.
        }
        elsif ($result == -1)
        {
          $self->error(sprintf($self->langObj->map("invalidApp"), $appObj->name));
          return @result;
        }
        elsif ($result == -2)
        {
          $self->error(sprintf($self->langObj->map("companyDoesNotExist"), $companyObj->id));
          return @result;
        }
        elsif ($result == -3)
        {
          $self->error(sprintf($self->langObj->map("AppAlreadyPurchasedByCompany"), $appObj->name, $companyObj->name));
          return @result;
        }
      }
    }
  }

  if ($displayForm)
  {
    # display the form.
    my $template = $companyAppObj->template->{$formName};

    my @numLicenses = ( 1, 2, 3, 4, 5, 10, 15, 20, 25, 50, 75, 100, 250, 500 );
    my $numLicensesOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@numLicenses,
      handler => $self->methods->getHandlerSub("Name = Value"));
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return undef;
    }
    if ($appObj->cost eq "0.00")
    {
      # add the 0 - Unlimited options.
      unshift @{$numLicensesOptions->{names}}, $self->langObj->map("Unlimited");
      unshift @{$numLicensesOptions->{values}}, 0;
    }

    my $data = $companyAppObj->data->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;
    $data->{name}->{-Value} = $appObj->name;
    $data->{description}->{-Value} = $appObj->description;
    $data->{server}->{-Options} = $serverInfoOptions;
    $data->{numLicenses}->{-Options} = $numLicensesOptions;
    $data->{unit}->{-Options} = $self->variables->formUnits;

    my %form = $self->formObj->generate(data => $data,
      template => $template,
      name => "purchaseAppForm",
      action => $url,
      method => "post",
      profile => $companyAppObj->profile->{$formName});
    # update the html hash in the result array.
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=item @ updateServerInfo(companyAppObj, companyObj, url, app, state)

 companyAppObj - specifies the CompanyApplication object to work with.
 companyObj - specifies the Company object working with the app.
 url - the url to specify in the forms action.
 app - the app that working with the app.
 state - the state in the app that is working with the app.

 returns an array with ($result, \%html, $companyAppObj, $onload) as
 values, where:
   result: 0 = error occured,
           1 = info successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   companyAppObj: the updated CompanyApplicationObject.
   onload: javascript code to pass to setOnload() method.

 The resulting form is named: serverInfoForm

=cut
sub updateServerInfo
{
  my $self = shift;
  my %args = ( companyAppObj => undef, companyObj => undef, url => "", app => "", state => "", @_ );
  my $companyAppObj = $args{companyAppObj};
  my $companyObj = $args{companyObj};
  my $caller = $self->userObj;
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, undef, "document.serverInfo.serverInfo.focus();" );
  my $formName = "serverInfo";

  # make sure I got valid input.
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to purchase an app.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  # get the serverInfo entries.
  my @serverInfoObjs = $self->applicationObj->getAppServerEntries(id => $companyAppObj->get("appId"));
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return @result;
  }
  my $serverInfoOptions = $self->formObj->createSelectOptionsFromArrayref(array => \@serverInfoObjs,
    handler => $self->methods->getHandlerSub("AppServerObject"));
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return @result;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  if (!exists $self->input->{formSubmitted})
  {
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->formObj->validate(input => $self->input, profile => $companyAppObj->profile->{$formName},
      data => $companyAppObj->data->{$formName});
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, update the company_app_tb entries.

      my $dbHost;
      my $dbPort;
      my $server;
      my $port;
      foreach my $appServer (@serverInfoObjs)
      {
        if ($appServer->counter == $self->input->{serverInfo})
        {
          $dbHost = $appServer->dbHost;
          $dbPort = $appServer->dbPort;
          $server = $appServer->server;
          $port = $appServer->port;
          last;
        }
      }

      if (!defined $server)
      {
        $self->formObj->invalid("serverInfo", $self->input->{serverInfo});
        $self->formObj->deleteErrorsEntry("valid", "serverInfo");
        $displayForm = 1;
      }
      else
      {
        # build up the values from the $appObj object and from the user input
        $companyAppObj->set(server => $server);
        $companyAppObj->set(port => $port);
        $companyAppObj->set(dbHost => $dbHost);
        $companyAppObj->set(dbPort => $dbPort);
        $companyAppObj->set(height => $self->formObj->getValidEntry("height"));
        $companyAppObj->set(width => $self->formObj->getValidEntry("width"));
        $companyAppObj->set(autorun => $self->formObj->getValidEntry("autorun"));
        $companyAppObj->set(wap => $self->formObj->getValidEntry("wap"));

        # make sure the new info is valid.
        if (!$companyAppObj->isValid)
        {
   # migrate the invalid/missing into the formObj.
   $companyAppObj->updateFormObj();
          $displayForm = 1;
        }
      }
      if (!$displayForm)
      {
        # at this point, we have a valid companyAppObj, so update the info.

        my $result = $self->applicationObj->updateCompanyAppInfo(companyAppObj => $companyAppObj);
        if ($self->applicationObj->error)
        {
          $self->error($self->applicationObj->errorMessage);
          return @result;
        }
        if ($result == 1)
        {
          $result[0] = 1;  # signal we successfully updated the CompanyApp info.
          $result[1] = \%form;
          $result[2] = $companyAppObj;  # pass back the CompanyApplicationObject.
        }
        elsif ($result == -1)
        {
          $self->error(sprintf($self->langObj->map("invalidApp"), $companyAppObj->get("appId")));
          return @result;
        }
        elsif ($result == -2)
        {
          $self->error(sprintf($self->langObj->map("companyDoesNotExist"), $companyAppObj->get("companyId")));
          return @result;
        }
        elsif ($result == -3)
        {
          $self->error($self->langObj->map("applicationNotAssigned"));
          return @result;
        }
      }
    }
  }

  if ($displayForm)
  {
    # display the form.
    my $template = $companyAppObj->template->{$formName};

    my $data = $companyAppObj->data->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;
    $data->{appId}->{-Value} = $companyAppObj->get("appId");
    $data->{serverInfo}->{-Options} = $serverInfoOptions;

    my %form = $self->formObj->generate(data => $data,
      template => $template,
      name => "serverInfo",
      action => $url,
      method => "post",
      profile => $companyAppObj->profile->{$formName});
    # update the html hash in the result array.
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=item @ sellAppLicenses(companyAppObj, companyObj, url, app, state)

 companyAppObj - specifies the CompanyApplication object to work with.
 companyObj - specifies the Company object working with the app.
 url - the url to specify in the forms action.
 app - the app that working with the app.
 state - the state in the app that is working with the app.

 returns an array with ($result, \%html, $companyAppObj, $onload,
   $licensesSold, $appName) as values, where:
   result: 0 = error occured,
           1 = info successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   companyAppObj: the updated CompanyApplicationObject.
   onload: javascript code to pass to setOnload() method.
   licensesSold: the number of licenses sold or 'all' if we sold
     the application.  If 'all', the companyAppObj is undef.
   appName: the name of the app being worked with.

 The resulting form is named: sellAppLicensesForm

=cut
sub sellAppLicenses
{
  my $self = shift;
  my %args = ( companyAppObj => undef, companyObj => undef, url => "", app => "", state => "", @_ );
  my $companyAppObj = $args{companyAppObj};
  my $companyObj = $args{companyObj};
  my $caller = $self->userObj;
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, undef, "document.sellAppLicensesForm.numLicenses.focus();", 0 );
  my $formName = "sellAppLicenses";

  # make sure I got valid input.
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to sell the app.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->langObj->map("YouAreNotSysOrCompanyAdmin"));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  # first lookup the application we are dealing with.
  my $appObj = $self->applicationObj->getApplicationInfo(id => $companyAppObj->get("appId"));
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  if (!defined $appObj)
  {
    $self->error(sprintf($self->langObj->map("invalidApp"), $companyAppObj->get("appId")));
    return undef;
  }
  if (!$appObj->isValid)
  {
    $self->error($appObj->errorMessage);
    return undef;
  }

  # find out how many licenses are currently assigned.
  my @currentlyAssignedUsers = $self->applicationObj->getUsersAssignedToAppForCompany(companyObj => $companyObj, appObj => $appObj);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }
  my $numLicensesUsed = scalar @currentlyAssignedUsers;

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  if (!exists $self->input->{formSubmitted})
  {
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->formObj->validate(input => $self->input, profile => $companyAppObj->profile->{$formName},
      data => $companyAppObj->data->{$formName});
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
      $result[6] .= "We didn't validate!<br />Error: " . $self->formObj->errorMessage;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, sell the app or the numLicenses free licenses.

      my $type = $self->input->{type};
      if ($type !~ /^(app|licenses)$/)
      {
        $self->formObj->invalid("type", $self->input->{type});
        $self->formObj->deleteErrorsEntry("valid", "type");
        $displayForm = 1;
      }
      elsif ($type eq "licenses" && ($self->input->{numLicenses} > ($companyAppObj->get("number") - $numLicensesUsed)))
      {
        $self->formObj->invalid("numLicenses", $self->input->{numLicenses});
        $self->formObj->deleteErrorsEntry("valid", "numLicenses");
        $displayForm = 1;
      }
      $result[6] .= "type = '$type', displayForm = '$displayForm'...<br />";
      if (!$displayForm)
      {
        # at this point, we have valid info so sell the app or free licenses.

        if ($type eq "app")
        {
          # first use the SOAP code and notify the app so it can do any cleanup.
          # now instantiate the soapObj for this application.
          $result[6] .= "building soapObj...<br />";
          my $soapObj = undef;
          eval { $soapObj = $self->methods->getPortalSOAPObj(caller => $caller, configObj => $self->configObj,
            location => $appObj->name, companyAppObj => $companyAppObj); };
          if ($@)
          {
            $self->error("getPortalSOAPObj eval (app = '" . $appObj->name . "'): " . $@);
            return @result;
          }
          if ($self->methods->error)
          {
            $self->error($self->methods->errorMessage);
            return @result;
          }
          if (!defined $soapObj)
          {
            $self->error("soapObj is not defined! (app = '" . $appObj->name . "')<br />\n" . $self->methods->errorMessage);
            return @result;
          }
          if ($soapObj->error)
          {
            $self->error($soapObj->errorMessage);
            return @result;
          }

          # it is upto the SOAP module in each application to append their app name and any other directories they want.
          my $dataDir = $self->configObj->webRoot . $self->configObj->siteDir . "/db_dump";

          $result[6] .= "dataDir = '$dataDir'...<br />";

          my $tmpResult = undef;
          # removed the SOAP:: or SOAP-> attribute to force it to go to the server.
          eval { $tmpResult = $soapObj->cleanupCompany(companyObj => $companyObj, appObj => $appObj, companyAppObj => $companyAppObj, dataDir => $dataDir); };
          if ($@)
          {
            $self->error("cleanupCompany eval (app = '" . $appObj->name . "', company = '" . $companyObj->name . "'): " . $@);
            return @result;
          }
          if ($soapObj->error)
          {
            $self->error($soapObj->errorMessage);
            return @result;
          }

          $result[6] .= "called \$soapObj->cleanupCompany()...<br />";

          # Update the result info.
          #$result->message($result->message . $tmpResult->message);
          #$result->result($tmpResult->result) if ($tmpResult->result == 0);  # only overwrite our result value if an error occured.

          # do we loop over all assigned users and do the SOAP un-assign or is the company cleanup code going to be
          # enough to cover this and be able to restore the users data if the app is ever "restored" or "re-purchased"?

          # then unAssign the app from the company.
          my $result = $self->applicationObj->unAssignAppFromCompany(companyAppObj => $companyAppObj);
          if ($self->applicationObj->error)
          {
            $self->error($self->applicationObj->errorMessage);
            return @result;
          }
          if ($result == 1)
          {
            $result[0] = 1;  # signal we successfully updated the CompanyApp info.
            $result[1] = \%form;
            $result[2] = $companyAppObj;  # pass back the CompanyApplicationObject.
            $result[4] = "all";
            $result[5] = $appObj->name;
          }
          elsif ($result == -1)
          {
            $self->error($self->langObj->map("applicationNotAssigned"));
            return @result;
          }
          else
          {
            $self->error("result = '$result' was not handled!");
            return @result;
          }
          $result[6] .= "result = '$result' from unAssignAppFromCompany()...<br />";
        }
        elsif ($type eq "licenses")
        {
          # update the now available number of licenses for this CompanyApplicationObject and update
          # the companyAppInfo in the database.
          $companyAppObj->subtract(number => $self->input->{numLicenses});
          my $result = $self->applicationObj->updateCompanyAppInfo(companyAppObj => $companyAppObj);
          if ($self->applicationObj->error)
          {
            $self->error($self->applicationObj->errorMessage);
            return @result;
          }
          if ($result == 1)
          {
            $result[0] = 1;  # signal we successfully updated the CompanyApp info.
            $result[1] = \%form;
            $result[2] = $companyAppObj;  # pass back the CompanyApplicationObject.
            $result[4] = $self->input->{numLicenses};
            $result[5] = $appObj->name;
          }
          elsif ($result == -1)
          {
            $self->error(sprintf($self->langObj->map("invalidApp"), $companyAppObj->get("appId")));
            return @result;
          }
          elsif ($result == -2)
          {
            $self->error(sprintf($self->langObj->map("companyDoesNotExist"), $companyAppObj->get("companyId")));
            return @result;
          }
          elsif ($result == -3)
          {
            $self->error($self->langObj->map("applicationNotAssigned"));
            return @result;
          }
        }
      }
    }
  }

  if ($displayForm)
  {
    # build the numLicenses select entries.
    my $numLicensesOptions = { names => [], values => [] };
    for (my $i=1; $i <= ($companyAppObj->get("number") - ($numLicensesUsed + 1)); $i++)
    {
      push @{$numLicensesOptions->{names}}, $i;
      push @{$numLicensesOptions->{values}}, $i;
    }

    # display the form.
    my $template = $companyAppObj->template->{$formName};

    # replace the #SELLAPPLICENSESPHRASE# tag.
    my $sellAppPhrase = sprintf($self->langObj->map("SellAppLicensesForApp"), $appObj->name);
    $template =~ s/#SELLAPPLICENSESPHRASE#/$sellAppPhrase/g;

    my $data = $companyAppObj->data->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;
    $data->{appId}->{-Value} = $companyAppObj->get("appId");
    $data->{numLicenses}->{-Options} = $numLicensesOptions;
    if ($companyAppObj->get("number") - ($numLicensesUsed + 1) <= 0 || $companyAppObj->get("number") == 0)
    {
      # disable the selection of licenses to sell.
      $data->{type}->{-Value} = "app";
      $data->{numLicenses}->{-ReadOnly} = 1;
      $data->{numLicenses}->{-ReadOnlyMode} = "text";
      $result[3] = "document.sellAppLicensesForm.type[1].disabled = true;";
    }

    my %form = $self->formObj->generate(data => $data,
      template => $template,
      name => "sellAppLicensesForm",
      action => $url,
      method => "post",
      profile => $companyAppObj->profile->{$formName});
    # update the html hash in the result array.
    if ($self->formObj->error)
    {
      $self->error($self->formObj->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::AppState(3)

=cut
