# User.pm - The Object Class that provides the User Form Module
# Created by James A. Pattie, 2004-06-23.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Forms::User;
use strict;
use Portal::AppState;
use Portal::Objects::UserApplicationObject;
use Portal::Objects::SOAPResult;
use HTMLObject::Form;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

=head1 NAME

User - Object used to build the User Forms Class.

=head1 SYNOPSIS

  use Portal::Forms::User;
  my $obj = Portal::Forms::User->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

  # ask the $obj to process User related forms.

=head1 DESCRIPTION

User is a User Forms class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Forms::User object.
 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{formObj} = HTMLObject::Form->new();

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# create the methods to create, edit, view the specified userObj data via the
# forms defined in the Portal::Objects::User object.

=item @ changePassword(user, url, app, state)

 user - specifies the User object of the user whose password is
   to be changed.
 url - the url to specify in the forms action.
 app - the app that is changing the users password.
 state - the state in the app that is changing the users password.

 returns an array with ($result, \%html, $onload) as values, where:
   result: 0 = error occured,
           1 = password successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   onload: javascript code to pass to the setOnload() method.

 The resulting form is named: passwordForm

=cut
sub changePassword
{
  my $self = shift;
  my %args = ( user => undef, url => "", app => "", state => "", @_ );
  my $user = $args{user};
  my $caller = $self->{userObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, "document.passwordForm.original.focus();" );
  my $formName = "";  # keeps track of the userPassword or adminPassword form data to work with.

  # make sure I got valid input.
  if (!defined $user)
  {
    $self->missing("user");
  }
  if (!$user->isValid)
  {
    $self->invalid("user", $user->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  if ($user->id != $caller->id)
  {
    # verify the caller is allowed to change the user's password.
    if (!$caller->sysadmin && !$caller->admin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
    }
    elsif (!$caller->sysadmin && $caller->admin && ($user->companyId != $caller->companyId))
    {
      $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $user->{uname})));
    }
    else
    {
      $formName = "adminPassword";
      $result[2] = "document.passwordForm.password1.focus();";
    }
  }
  else
  {
    $formName = "userPassword";
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  if (!exists $self->{input}->{formSubmitted})
  {
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->{formObj}->validate(input => $self->{input}, profile => $user->{profile}->{$formName},
      $user->data->{$formName});
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, update the database.

      if ($formName eq "userPassword")
      {
        # make sure the "original" password matches what is the current password.
        if ($user->{password} ne $self->{authObj}->md5EncryptPassword($self->{input}->{original}))
        {
          # remove the original from the valid hash and move it to the invalid hash.
          $self->{formObj}->invalid("original", $self->{formObj}->deleteErrorsEntry("valid", "original"), "Original password does not match!");
          $displayForm = 1;
        }
      }
      # make sure the new passwords match.
      if (length $self->{input}->{password1} < 6)
      {
        # remove password1 and password2 from the valid hash and move them to the invalid hash.
        $self->{formObj}->invalid("password1", $self->{formObj}->deleteErrorsEntry("valid", "password1"), "Password must be 6 characters!");
        $self->{formObj}->invalid("password2", $self->{formObj}->deleteErrorsEntry("valid", "password2"), "Password must be 6 characters!");
        $displayForm = 1;
      }
      elsif ($self->{input}->{password1} ne $self->{input}->{password2})
      {
        # remove password2 from the valid hash and move it to the invalid hash.
        $self->{formObj}->invalid("password2", $self->{formObj}->deleteErrorsEntry("valid", "password2"), "Does not match New Password!");
        $displayForm = 1;
      }
      if (!$displayForm)
      {
        # at this point, we have a valid password change, so update the users info.
        $user->{password} = $self->{input}->{password1};
        $user->{passCrypted} = 0;
        $user->{lastEdited} = "now()";

        my $result = $self->{authObj}->updateUser(userObj => $user);
        if ($self->{authObj}->error)
        {
          $self->error($self->{authObj}->errorMessage);
          return @result;
        }
        $result[0] = 1;  # signal we successfully changed the password.
      }
    }
  }

  if ($displayForm)
  {
    # display the form.
    my $template = $user->{template}->{$formName};

    # remove the #CHILD_FORM_ARGS# and #CHILD_FORM_TABLE# tags, if they haven't been replaced yet.
    $template =~ s/(#CHILD_FORM_(ARGS|TABLE)#)//g;

    my $data = $user->{data}->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;

    my %form = $self->{formObj}->generate(data => $data,
      template => $template,
      name => "passwordForm",
      action => $url,
      method => "post",
      profile => $user->{profile}->{$formName});
    # update the html hash in the result array.
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=item % viewUser(user, companyObj)

 user - specifies the User object of the user to view.
 companyObj - specifies the CompanyObject the user belongs to.

 returns %html, where:
   html: hash that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 If an error occured, we return undef.

=cut
sub viewUser
{
  my $self = shift;
  my %args = ( user => undef, companyObj => undef, @_ );
  my $user = $args{user};
  my $companyObj = $args{companyObj};

  # make sure I got valid input.
  if (!defined $user)
  {
    $self->missing("user");
  }
  elsif (!$user->isValid)
  {
    $self->invalid("user", $user->errorMessage);
  }
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  # display the form.
  my $formName = "view";
  my $template = $user->{template}->{$formName};

  # remove the #CHILD_FORM_ARGS# and #CHILD_FORM_TABLE# tags, if they haven't been replaced yet.
  $template =~ s/(#CHILD_FORM_(ARGS|TABLE)#)//g;

  my $data = $user->{data}->{$formName};

  # update the companyId to also include the name.
  $data->{companyId}->{-Value} .= " (" . $companyObj->{name} . ")" if ($data->{companyId}->{-Value} =~ /^(\d+)$/);

  my %form = $self->{formObj}->generate(data => $data,
    template => $template,
    name => "viewUserForm",
    action => "",
    method => "post",
    profile => $user->{profile}->{$formName},
    readOnly => 1, readOnlyMode => "text", readOnlyHidden => 0);
  # update the html hash in the result array.
  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage);
    return undef;
  }

  return %form;
}

=item @ editUser(user, url, app, state)

 user - specifies the User object of the user who we are editing.
 url - the url to specify in the forms action.
 app - the app that is changing the users info.
 state - the state in the app that is changing the users info.

 returns an array with ($result, \%html, $userObj, $onload) as
 values, where:
   result: 0 = error occured,
           1 = info successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   userObj: the updated userObj when result = 1.
   onload: javascript code to pass to setOnload() method.

 The resulting form is named: editUserForm

=cut
sub editUser
{
  my $self = shift;
  my %args = ( user => undef, url => "", app => "", state => "", @_ );
  my $user = $args{user};
  my $caller = $self->{userObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, undef, "document.editUserForm.fname.focus();" );
  my $formName = "";  # keeps track of the userPassword or adminPassword form data to work with.

  # make sure I got valid input.
  if (!defined $user)
  {
    $self->missing("user");
  }
  elsif (!$user->isValid)
  {
    $self->invalid("user", $user->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  if (defined $user && $user->id != $caller->id)
  {
    # verify the caller is allowed to change the user's info.
    if (!$caller->sysadmin && !$caller->admin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
    }
    elsif (!$caller->sysadmin && $caller->admin && ($user->companyId != $caller->companyId))
    {
      $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $user->{uname})));
    }
    elsif ($caller->sysadmin)
    {
      $formName = "sysadminEdit";
    }
    else
    {
      $formName = "adminEdit";
    }
  }
  else
  {
    $formName = "userEdit";
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  if (!exists $self->{input}->{formSubmitted})
  {
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->{formObj}->validate(input => $self->{input}, profile => $user->{profile}->{$formName},
      data => $user->data->{$formName});
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, update the database.

      # build up the values from the $user object and from the user input
      # based upon the mode we are in.
      my %userInfo = ( id => $user->{id}, uname => $user->{uname}, lastEdited => $user->{lastEdited},
                       companyId => $user->{companyId}, password => $user->{password} );

      if ($formName eq "adminEdit")
      {
        # all other fields are fair game from the user input.
        foreach my $field ( qw( active fname mname lname ssn empId language timeFormat tz admin email comment ) )
        {
          $userInfo{$field} = $self->{formObj}->getValidEntry($field);
        }
        $userInfo{sysadmin} = $user->{sysadmin};
      }
      elsif ($formName eq "sysadminEdit")
      {
        # all other fields are fair game from the user input.
        foreach my $field ( qw( active fname mname lname ssn empId language timeFormat tz admin sysadmin email comment ) )
        {
          $userInfo{$field} = $self->{formObj}->getValidEntry($field);
        }
      }
      else
      {
        # only some of the user input is usable.
        foreach my $field ( qw( fname mname lname language timeFormat tz email comment ) )
        {
          $userInfo{$field} = $self->{formObj}->getValidEntry($field);
        }
        foreach my $field ( qw( active ssn empId admin sysadmin ) )
        {
          $userInfo{$field} = $user->{$field};
        }
      }

      # make sure the new info is valid via the Portal::Objects::User class.
      my $tmpUser = Portal::Objects::User->new(langObj => $self->{langObj});
      $tmpUser->populate(%userInfo);
      if ($tmpUser->error)
      {
        # now we have to walk the $tmpUser invalid and missing structures and
        # figure out what entries are causing problems and set them invalid
        # in the formObj structure so we can redisplay the form and have the
        # user try and fix things.
        foreach my $entry ($tmpUser->getInvalid())
        {
          # add to the formObj invalid structure.
          $self->{formObj}->invalid($entry, $tmpUser->getInvalidEntry($entry), $tmpUser->getExtraInfoEntry($entry));
          # clear the formObj valid structure entry.
          $self->{formObj}->deleteErrorsEntry("valid", $entry);
        }
        foreach my $entry ($tmpUser->getMissing())
        {
          # add to the formObj invalid structure.
          $self->{formObj}->missing($entry, $tmpUser->getExtraInfoEntry($entry));
          # clear the formObj valid structure entry.
          $self->{formObj}->deleteErrorsEntry("valid", $entry);
        }
        $displayForm = 1;
      }
      if (!$displayForm)
      {
        # at this point, we have a valid tmpUser, so update the users info.
        if ($user != $tmpUser)
        {
          $tmpUser->{lastEdited} = "now()";

          my $result = $self->{authObj}->updateUser(userObj => $tmpUser);
          if ($self->{authObj}->error)
          {
            $self->error($self->{authObj}->errorMessage);
            return @result;
          }

          # now lookup the user so we have the correct lastEdited value.
          $tmpUser = $self->{authObj}->getUserInfo(uname => $tmpUser->{uname});
          if ($self->{authObj}->error)
          {
            $self->error($self->{authObj}->errorMessage);
            return @result;
          }
        }
        $result[0] = 1;  # signal we successfully updated the user.
        $result[2] = $tmpUser;  # pass back the updated User Object.
      }
    }
  }

  if ($displayForm)
  {
    # display the form.
    my $template = $user->{template}->{$formName};

    # remove the #CHILD_FORM_ARGS# and #CHILD_FORM_TABLE# tags, if they haven't been replaced yet.
    $template =~ s/(#CHILD_FORM_(ARGS|TABLE)#)//g;

    my $data = $user->{data}->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;

    my %form = $self->{formObj}->generate(data => $data,
      template => $template,
      name => "editUserForm",
      action => $url,
      method => "post",
      profile => $user->{profile}->{$formName});
    # update the html hash in the result array.
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=item % selectUserAdminForm(companyObj, url, app, state)

 companyObj - the company to select users for.
 url - the url to specify in the forms action.
 app - the app that is changing the users info.
 state - the state in the app that is changing the users info.

 generates the form that provides an admin user with a list
 of all users for the specified company and buttons to
 let them edit, view, change passwords, preferences, etc.
 for the selected user.

 If the input hash contains uname then that is the default
 value for the user select box, which can allow the interface
 to re-select the user so you can choose a different action.

 returns %html, where:
   html: hash that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 If an error occured, we return undef.

=cut
sub selectUserAdminForm
{
  my $self = shift;
  my %args = ( companyObj => undef, url => "", app => "", state => "", @_ );
  my $companyObj = $args{companyObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my $caller = $self->{userObj};

  # make sure I got valid input.
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  if (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to change the user's password.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
  }
  elsif (!$caller->sysadmin && $caller->admin && ($caller->companyId != $companyObj->id))
  {
    $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $companyObj->{code})));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  # display the form.
  my $formName = "selectUserAdminForm";
  my $indicatorsPhrase = $self->{langObj}->map("DisabledAdminSysadminIndicators");
  my $selectUserPhrase = $self->{langObj}->map("SelectUser");
  my $template = <<"END_OF_TEMPLATE";
$selectUserPhrase<br />
#ERROR_MESSAGE#
#STATUS_MESSAGE#
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="center">#LABEL=uname#:<br />
        #FIELD=uname#      </td>
      <td align="left">#FIELD=editUser#<br />
        #FIELD=changePassword#<br />
        #FIELD=editPreferences#<br />
        #FIELD=viewUser#<br />
        #FIELD=deleteUser#
      </td>
    </tr>
    <tr>
      <td align="center" colspan="2"><span style="font-size: smaller;">$indicatorsPhrase</span></td>
    </tr>
  </table>
</form>
END_OF_TEMPLATE

  my @users = $self->{authObj}->getListOfUsersForCompany(companyObj => $companyObj);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  # now walk the list of returned users and create the %users options structure.
  my $userOptions = $self->{formObj}->createSelectOptionsFromArrayref(array => \@users,
    handler => $self->{methods}->getHandlerSub("User"));
  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage);
    return undef;
  }

  my %data = ( "editUser" => { -Type => "submit", -Value => $self->{langObj}->map("EditUser"), onclick => "if (window.document.$formName.uname.selectedIndex == -1) { alert('" . $self->{langObj}->map("YouMustSelectUser") . "'); return false; } else { window.document.$formName.command.value='editUser'; return true; }" },
               "changePassword" => { -Type => "submit", -Value => $self->{langObj}->map("ChangePassword"), onclick => "if (window.document.$formName.uname.selectedIndex == -1) { alert('" . $self->{langObj}->map("YouMustSelectUser") . "'); return false; } else { window.document.$formName.command.value='changePassword'; return true; }" },
               "editPreferences" => { -Type => "submit", -Value => $self->{langObj}->map("EditPreferences"), onclick => "if (window.document.$formName.uname.selectedIndex == -1) { alert('" . $self->{langObj}->map("YouMustSelectUser") . "'); return false; } else { window.document.$formName.command.value='editPreferences'; return true; }" },
               "viewUser" => { -Type => "submit", -Value => $self->{langObj}->map("ViewUser"), onclick => "if (window.document.$formName.uname.selectedIndex == -1) { alert('" . $self->{langObj}->map("YouMustSelectUser") . "'); return false; } else { window.document.$formName.command.value='viewUser'; return true; }" },
               "deleteUser" => { -Type => "submit", -Value => $self->{langObj}->map("DeleteUser"), onclick => "if (window.document.$formName.uname.selectedIndex != -1) { if (confirm('" . $self->{langObj}->map("DoYouWantToDeleteUser") . "')) { window.document.$formName.command.value='deleteUser'; return true; } else { return false; } } else { alert('" . $self->{langObj}->map("YouMustSelectUser") . "'); return false; }" },
               "uname" => { -Type => "select", -Value => (exists $self->{input}->{uname} ? $self->{input}->{uname} : ""), -Label => $self->{langObj}->map("uname"), -Options => $userOptions, size => 10 },
               "app" => { -Type => "hidden", -Value => $app },
               "state" => { -Type => "hidden", -Value => $state },
               "command" => { -Type => "hidden", -Value => "" },
             );

  my %profile = ( required => [], optional => [], constraints => {} );

  # see if we need to display any success/failure messages.
  if (exists $self->{input}->{errorMessage})
  {
    my $string = $self->{methods}->displayMessageStr(type => "error", message => $self->{input}->{errorMessage},
       break => "both");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $template =~ s/#ERROR_MESSAGE#/$string/g;
  }
  else
  {
    $template =~ s/#ERROR_MESSAGE#//g;
  }
  if (exists $self->{input}->{statusMessage})
  {
    my $string = $self->{methods}->displayMessageStr(type => "status", message => $self->{input}->{statusMessage},
       break => "both");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $template =~ s/#STATUS_MESSAGE#/$string/g;
  }
  else
  {
    $template =~ s/#STATUS_MESSAGE#//g;
  }

  my %form = $self->{formObj}->generate(data => \%data,
    template => $template,
    name => $formName,
    action => $url,
    method => "post",
    profile => \%profile);
  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage);
    return undef;
  }

  return %form;
}

=item SOAPResult deleteUser(userObj, companyObj)

 userObj - specifies the User object of the user we are deleting.
 companyObj - The company Object for the user being deleted.

 returns SOAPResult result: 0 = error occured,
           1 = user successfully deleted,
           2 = you can not delete yourself!
         message: Any text you want displayed to the user.

         undef if SOAPResult failed to instantiate.

=cut
sub deleteUser
{
  my $self = shift;
  my %args = ( userObj => undef, companyObj => undef, @_ );
  my $userObj = $args{userObj};
  my $companyObj = $args{companyObj};
  my $caller = $self->{userObj};

  # start our result object, default to a failure result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => -1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  # make sure I got valid input.
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }
  if ($userObj->id != $caller->id)
  {
    # verify the caller is allowed to delete the user.
    if (!$caller->sysadmin && !$caller->admin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
    }
    elsif (!$caller->sysadmin && $caller->admin && ($userObj->companyId != $caller->companyId))
    {
      $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $userObj->{uname})));
    }
    elsif (!$caller->sysadmin && $caller->admin && $userObj->sysadmin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouCanNotDeleteASysadmin"));
    }
  }
  else
  {
    # This should not get here, but...
    $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouCanNotDeleteYourself"));
    $result->{result} = 2;
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $result->{result} = 0 if ($result->{result} == -1);
    return $result;
  }

  # unassign the user from all apps, remove preferences, colorSchemes, permissions

  # Use SOAP to cleanupUser in the Portal itself.
  my $soapObj = undef;
  eval { $soapObj = $self->{methods}->getPortalSOAPObj(caller => $self->{userObj}, configObj => $self->{configObj},
    location => "Portal"); };
  if ($@)
  {
    $self->error("getPortalSOAPObj eval: " . $@);
    return $result;
  }
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $result;
  }
  if (!defined $soapObj)
  {
    $self->error("soapObj is not defined!<br />\n" . $self->{methods}->errorMessage);
    return $result;
  }
  if ($soapObj->error)
  {
    $self->error($soapObj->errorMessage);
    return $result;
  }

  my $tmpResult = undef;
  # removed the SOAP:: or SOAP-> to make the call go remote.
  eval { $tmpResult = $soapObj->cleanupUser(companyObj => $companyObj, userObj => $userObj); };
  if ($@)
  {
    $self->error("cleanupUser eval: " . $@);
    return $result;
  }
  if ($soapObj->error)
  {
    $self->error($soapObj->errorMessage);
    return $result;
  }
  # Update the result info.
  $result->{message} .= $tmpResult->{message};
  $result->{result} = $tmpResult->{result} if ($tmpResult->{result} == 0);  # only overwrite our result value if an error occured.

  $soapObj = undef;  # force the Portal SOAP object to go out of scope.

 # get a list of all apps the user is assigned.
  my %apps = $self->{applicationObj}->getAppsForUser(userId => $userObj->{id});
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return $result;
  }
  my @apps = $self->{applicationObj}->flattenAppHash(apps => \%apps);
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return $result;
  }

  #$self->error("cleanupUser call to Portal appears to have worked!<br />\n" . $result->{message} . (exists $soapObj->{authObj}->{portalDBSave} ? "<br />\nauthObj has the portalDBSave variable!" : "not here"));
  #return $result;

  foreach my $app (@apps)
  {
    # unassign the user rights for this app from the user.
    my $status = $self->{authObj}->unassignAllUserRights(userId => $userObj->{id}, app => $app->{name});
    if ($self->{authObj}->error)
    {
      $self->error($self->{authObj}->errorMessage);
      return $result;
    }
    $result->{message} .= sprintf($self->{langObj}->map("UnAssignedAllUserRightsForApp"), $app->{name}, $userObj->{uname}) . "<br />\n";

    # create a UserApplicationObject instance.
    my $userAppObj = Portal::Objects::UserApplicationObject->new(langObj => $self->{langObj});
    if ($userAppObj->error)
    {
      $self->error($userAppObj->errorMessage);
      return $result;
    }
    $userAppObj->populate(companyId => $userObj->{companyId},
      appId => $app->{id}, userId => $userObj->{id}, height => $app->{height}, width => $app->{width},
      autorun => $app->{autorun}, wap => $app->{wap});
    if ($userAppObj->error)
    {
      $self->error($userAppObj->errorMessage);
      return $result;
    }

    # use SOAP to inform the app this user is being un-assigned.

    # first lookup the companyApplicationObject for this app and user company pair.
    my $companyAppObj = $self->{applicationObj}->getCompanyAppInfo(companyId => $userObj->{companyId}, appId => $app->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return $result;
    }

    # now instantiate the soapObj for this application.
    my $soapObj = undef;
    eval { $soapObj = $self->{methods}->getPortalSOAPObj(caller => $self->{userObj}, configObj => $self->{configObj},
      location => $app->{name}, companyAppObj => $companyAppObj); };
    if ($@)
    {
      $self->error("getPortalSOAPObj eval (app = '$app->{name}'): " . $@);
      return $result;
    }
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return $result;
    }
    if (!defined $soapObj)
    {
      $self->error("soapObj is not defined! (app = '$app->{name}')<br />\n" . $self->{methods}->errorMessage);
      return $result;
    }
    if ($soapObj->error)
    {
      $self->error($soapObj->errorMessage);
      return $result;
    }

    my $tmpResult = undef;
    # removed the SOAP:: or SOAP-> attribute to force it to go to the server.
    eval { $tmpResult = $soapObj->cleanupUser(companyObj => $companyObj, userObj => $userObj,
            appObj => $app, companyAppObj => $companyAppObj); };
    if ($@)
    {
      $self->error("cleanupUser eval (app = '$app->{name}'): " . $@);
      return $result;
    }
    if ($soapObj->error)
    {
      $self->error($soapObj->errorMessage);
      return $result;
    }
    # Update the result info.
    $result->{message} .= $tmpResult->{message};
    $result->{result} = $tmpResult->{result} if ($tmpResult->{result} == 0);  # only overwrite our result value if an error occured.

    $soapObj = undef;  # force the App SOAP object to go out of scope.

    # unassign the app from the user.
    $status = $self->{applicationObj}->unAssignAppFromUser(userAppObj => $userAppObj);
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return $result;
    }
    $result->{message} .= sprintf($self->{langObj}->map("UnAssignedAppFromUser"), $app->{name}, $userObj->{uname}) . "<br />\n";
  }

  # then delete the user.

  my $tmpResult = $self->{authObj}->deleteUser(userObj => $userObj);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return $result;
  }
  $result->{result} = 1 if ($result->{result} == -1 && $tmpResult == 1);  # signal we successfully deleted the user and all configuration settings, as long as an error didn't occur.

  return $result;
}

=item @ createUser(companyObj, url, app, state)

 companyObj - the company the user will belong to.
 url - the url to specify in the forms action.
 app - the app that is creating the user.
 state - the state in the app that is creating the user.

 returns an array with ($result, \%html, $userObj, $onload, $message)
 as values, where:
   result: 0 = error occured,
           1 = info successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.
   userObj: the created userObj when result = 1.
   onload: javascript code to pass to setOnload() method.
   message: message from the SOAP code.

 The resulting form is named: createUserForm

=cut
sub createUser
{
  my $self = shift;
  my %args = ( companyObj => undef, url => "", app => "", state => "", @_ );
  my $companyObj = $args{companyObj};
  my $caller = $self->{userObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {}, undef, "document.createUserForm.uname.focus();", "" );
  my $formName = "";  # keeps track of the userPassword or adminPassword form data to work with.

  # make sure I got valid input.
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  # verify the caller is allowed to create a new user.
  if (!$caller->sysadmin && !$caller->admin)
  {
    $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
  }
  elsif (!$caller->sysadmin && $caller->admin && ($companyObj->{id} != $caller->companyId))
  {
    $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $companyObj->{name})));
  }
  elsif ($caller->sysadmin)
  {
    $formName = "sysadminCreate";
  }
  else
  {
    $formName = "adminCreate";
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  my $tmpUser = Portal::Objects::User->new(langObj => $self->{langObj});
  if (!exists $self->{input}->{formSubmitted})
  {
    $displayForm = 1;
  }
  else
  {
    # validate the input and update if everything ok.
    my $result = $self->{formObj}->validate(input => $self->{input}, profile => $tmpUser->{profile}->{$formName},
      data => $tmpUser->data->{$formName});
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    if (!$result)
    {
      $displayForm = 1;
    }
    else
    {
      # we have a valid form to at least the Data::FormValidator.
      # do any final validation and if everything looks good, update the database.

      # make sure the new passwords match.
      if (length $self->{input}->{password1} < 6)
      {
        # remove password1 and password2 from the valid hash and move them to the invalid hash.
        $self->{formObj}->invalid("password1", $self->{formObj}->deleteErrorsEntry("valid", "password1"));
        $self->{formObj}->invalid("password2", $self->{formObj}->deleteErrorsEntry("valid", "password2"));
        $displayForm = 1;
      }
      elsif ($self->{input}->{password1} ne $self->{input}->{password2})
      {
        # remove password2 from the valid hash and move it to the invalid hash.
        $self->{formObj}->invalid("password2", $self->{formObj}->deleteErrorsEntry("valid", "password2"));
        $displayForm = 1;
      }

      if (!$displayForm)
      {
        # at this point our passwords pass muster also.
        # build up the values from the from the user input
        # based upon the mode we are in.
        my %userInfo = ( companyId => $companyObj->{id}, password => $self->{input}->{password1}, passCrypted => 0 );

        if ($formName eq "adminCreate")
        {
          # all other fields are fair game from the user input.
          foreach my $field ( qw( uname active fname mname lname ssn empId language timeFormat tz admin email comment ) )
          {
            $userInfo{$field} = $self->{formObj}->getValidEntry($field);
          }
          $userInfo{sysadmin} = 0;  # they can not create a sysadmin.
        }
        elsif ($formName eq "sysadminCreate")
        {
          # all other fields are fair game from the user input.
          foreach my $field ( qw( uname active fname mname lname ssn empId language timeFormat tz admin sysadmin email comment ) )
          {
            $userInfo{$field} = $self->{formObj}->getValidEntry($field);
          }
        }

        # make sure the new info is valid via the Portal::Objects::User class.
        $tmpUser->populate(%userInfo);
        if ($tmpUser->error)
        {
          # now we have to walk the $tmpUser invalid and missing structures and
          # figure out what entries are causing problems and set them invalid
          # in the formObj structure so we can redisplay the form and have the
          # user try and fix things.
          foreach my $entry ($tmpUser->getInvalid())
          {
            # add to the formObj invalid structure.
            $self->{formObj}->invalid($entry, $tmpUser->getInvalidEntry($entry), $tmpUser->getExtraInfoEntry($entry));
            # clear the formObj valid structure entry.
            $self->{formObj}->deleteErrorsEntry("valid", $entry);
          }
          foreach my $entry ($tmpUser->getMissing())
          {
            # add to the formObj invalid structure.
            $self->{formObj}->missing($entry, $tmpUser->getExtraInfoEntry($entry));
            # clear the formObj valid structure entry.
            $self->{formObj}->deleteErrorsEntry("valid", $entry);
          }
          $displayForm = 1;
        }
      }
      if (!$displayForm)
      {
        # at this point, we have a valid tmpUser, so create the user.
        $tmpUser->{lastEdited} = "now()";

        my $result = $self->{authObj}->createUser(userObj => $tmpUser);
        if ($self->{authObj}->error)
        {
          $self->error($self->{authObj}->errorMessage);
          return @result;
        }

        # now lookup the user so we have the correct id value.
        $tmpUser = $self->{authObj}->getUserInfo(uname => $tmpUser->{uname});
        if ($self->{authObj}->error)
        {
          $self->error($self->{authObj}->errorMessage);
          return @result;
        }

        # now we need to let the Portal do anything it wants to via the SOAP interface.
        my $soapObj = undef;
        eval { $soapObj = $self->{methods}->getPortalSOAPObj(caller => $caller, configObj => $self->{configObj},
          location => "Portal"); };
        if ($@)
        {
          $self->error("getPortalSOAPObj eval: " . $@);
          return @result;
        }
        if ($self->{methods}->error)
        {
          $self->error($self->{methods}->errorMessage);
          return @result;
        }
        if (!defined $soapObj)
        {
          $self->error("soapObj is not defined!<br />\n" . $self->{methods}->errorMessage);
          return @result;
        }
        if ($soapObj->error)
        {
          $self->error($soapObj->errorMessage);
          return @result;
        }

        my $tmpResult = undef;
        # removed the SOAP:: or SOAP-> to make the call go remote.
        eval { $tmpResult = $soapObj->setupUser(companyObj => $companyObj, userObj => $tmpUser); };
        if ($@)
        {
          $self->error("setupUser eval: " . $@);
          return @result;
        }
        if ($soapObj->error)
        {
          $self->error($soapObj->errorMessage);
          return @result;
        }
        # Update the result info.
        $result[4] = $tmpResult->{message};

        $soapObj = undef;  # force the Portal SOAP object to go out of scope.

        $result[0] = 1;  # signal we successfully created the user.
        $result[2] = $tmpUser;  # pass back the created User Object.
      }
    }
  }

  if ($displayForm)
  {
    # display the form.
    my $template = $tmpUser->{template}->{$formName};

    # remove the #CHILD_FORM_ARGS# and #CHILD_FORM_TABLE# tags, if they haven't been replaced yet.
    $template =~ s/(#CHILD_FORM_(ARGS|TABLE)#)//g;

    my $data = $tmpUser->{data}->{$formName};
    $data->{app}->{-Value} = $app;
    $data->{state}->{-Value} = $state;

    my %form = $self->{formObj}->generate(data => $data,
      template => $template,
      name => "createUserForm",
      action => $url,
      method => "post",
      profile => $tmpUser->{profile}->{$formName});
    # update the html hash in the result array.
    if ($self->{formObj}->error)
    {
      $self->error($self->{formObj}->errorMessage);
      return @result;
    }
    $result[0] = 2;  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::AppState(3)

=cut
