# Help.pm - The Object Class that provides a Help Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Help;
use strict;
use Portal::Session;
use Portal::SessionHandler;
use Portal::PortalArgs;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::PortalArgs Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.12';

# global variables.
# modify to represent the commands this state provides.
my %commands = ( display => "Display Help Screen", closeHelp => "Close Help Window" );

=head1 NAME

Help - Object used to build a Help Object Class.

=head1 SYNOPSIS

  use Portal::Help;
  my $obj = Portal::Help->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Help is a Help class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(helpApp, content)

 Creates a new instance of the Portal::Help object.
 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{helpApp} = $self->{input}->{helpApp};
  $self->{content} = $self->{input}->{content};
  my @contentArray = split /,/, $self->{content};
  $self->{contentArray} = \@contentArray;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  # get the portalSession
  $self->{session} = $self->{methods}->portalSession(portalDB => $self->{portalDB},
                                                     cookieObj => $self->{cookieObj},
                                                     appName => "pcx_session_id",
                                                     configObj => $self->{configObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $self;
  }
  if ($self->{session}->error)
  {
    $self->error($self->{session}->errorMessage);
    return $self;
  }

  # get the AppSessions
  $self->{appSessions} = $self->{methods}->portalSessionHandler(portalDB => $self->{portalDB},
                                                                sessionObj => $self->{session},
                                                                configObj => $self->{configObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $self;
  }
  if ($self->{appSessions}->error)
  {
    $self->error($self->{appSessions}->errorMessage);
    return $self;
  }

  # instantiate the Help Data Structure for the Topics hash.
  my $helpName = ($self->{helpApp} ne "Portal" ? "Portal::" : "") . $self->{helpApp} . "::Data::Help";
  eval "use $helpName;";
  if ($@)
  {
    $self->error("Eval of $helpName failed!<br />\nError = '$@'.<br />\n");
    return $self;
  }
  # retrieve the Help Data object and store it in our object.
  $self->{helpData} = $helpName->new(lang => $self->{langObj}->{lang}, langObj => $self->{langObj});
  if ($self->{helpData}->error)
  {
    $self->error($self->{helpData}->errorMessage);
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{content} == 0 && $self->{input}->{command} ne "closeHelp")
  {
    $self->missing("content");
  }
  if (scalar $self->{contentArray} == 0 && $self->{input}->{command} ne "closeHelp")
  {
    $self->error("No keywords found to work with!<br />\n"); # LANG this.
  }
  if (length $self->{helpApp} == 0)
  {
    $self->missing("helpApp");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject run(state, command)

 Pulls the specified state file into existance and calls
 run(command).

 returns: The HTMLObject document to display or undef if an error.

=cut

sub run
{
  my $self = shift;
  my %args = ( command => "", @_ );
  my $command = $args{command};
  my $doc;

  if (!exists $commands{$command})
  {
    $self->error(sprintf($self->langObj->map("commandNotKnown"), $command));
    return undef;
  }

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$self->{session}->{store}->{browserCap}};

  $doc = HTMLObject::Normal->new;
  if ($command eq "display")
  {
    # create/update the entry in the windows hash
    if (!exists $self->{input}->{inlineDocument})
    {
      my $window = $self->{session}->generateHelpWindowName($self->{helpApp});
      if ($self->{session}->error)
      {
        $self->error($self->{session}->errorMessage);
        return undef;
      }
      $self->{session}->registerWindow($window);
      if ($self->{session}->error)
      {
        $self->error($self->{session}->errorMessage);
        return undef;
      }
    }

    $doc = $self->displayHelp(doc => $doc);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
  }
  elsif ($command eq "closeHelp")
  {
    if (!exists $self->{input}->{inlineDocument})
    {
      # remove the help window from the windows hash
      my $window = $self->{session}->generateHelpWindowName($self->{helpApp});
      if ($self->{session}->error)
      {
        $self->error($self->{session}->errorMessage);
        return undef;
      }
      $self->{session}->unregisterWindow($window);
      if ($self->{session}->error)
      {
        $self->error($self->{session}->errorMessage);
        return undef;
      }

      # generate the JavaScript code to close the window.
      $doc->setTitle("Close Help");
      my $onLoadCode = "window.close();\n";
      $doc->setOnload(code => $onLoadCode);
      $doc->setFocus("body");
      $doc->print("Closing Window");
    }
    else
    {
      $doc->setTitle("Close Help - Invalid");
      $doc->setFocus("body");
      $doc->print("You can not Close the Help Window!<br />\nYou are in an Inline Document!");
    }
  }

  # have it process the session object so that any changes are saved in the session.
  $self->{session}->writeCookie(doc => $doc);

  # set the JavaScript Error Handler Email Address
  $self->{methods}->setJavaScriptErrorInfo(doc => $doc, email => $self->{configObj}->{emailAddress}, appName => "Portal", appVersion => $VERSION);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  return $doc;
}

sub displayHelp
{
  my $self = shift;
  my %args = ( @_ );
  my $doc = $args{doc};

  my $url = $self->{methods}->createBaseURL(type => "Portal", linkType => "cgi");
  my $closeUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "closeHelp", helpApp => $self->{helpApp} });
  my $printUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { print => 1 });
  my $indexUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "display", helpApp => $self->{helpApp}, content => $self->{contentArray}[0] });
  my $helpUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "display" }) . "&helpApp=\%s&content=\%s";

  # get the users colorScheme to work with.
  my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => $self->{session}->{store}->{userObj}->{id});
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $colorScheme)
  {
    $self->error("No valid colorScheme returned!<br />\n");
    return undef;
  }

  my $closeWindow = $self->langObj->map("closeWindow");
  my $help = $self->langObj->map("help");
  my $print = $self->langObj->map("print");

  $doc->setTitle(sprintf("%s - Help Screen", $self->{helpApp}));
  # create the needed css entries
  my %cssEntries = $self->{methods}->generateColorSchemeCSS(colorScheme => $colorScheme);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  $doc->print(%cssEntries);

  # create the showHelp JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # build up the menuItems arrays
  my @itemsLeft = ();
  my @itemsRight = ();

  if (exists $self->{input}->{inlineDocument})
  {
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "text", text => "$self->{helpApp}", langObj => $self->{langObj});
  }
  else
  {
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$closeUrl",
                    text => $closeWindow, title => $closeWindow, onMouseOver => "window.status='$closeWindow'; return true;", langObj => $self->{langObj});
  }
  $itemsRight[0] = Portal::Objects::MenuItem->new(type => "link", url => "$printUrl", onClick => "window.print(); return false;",
                   text => $print, title => $print, onMouseOver => "window.status='$print'; return true;", langObj => $self->{langObj});

  # now validate they created ok.
  for (my $i=0; $i < scalar @itemsLeft; $i++)
  {
    if ($itemsLeft[$i]->error)
    {
      $self->error($itemsLeft[$i]->errorMessage);
      return undef;
    }
  }
  for (my $i=0; $i < scalar @itemsRight; $i++)
  {
    if ($itemsRight[$i]->error)
    {
      $self->error($itemsRight[$i]->errorMessage);
      return undef;
    }
  }
  my $menuString = $self->{methods}->displayMenu(doc => $doc, orientation => "horizontal",
                  itemsLeft => \@itemsLeft, itemsRight => \@itemsRight,
                  indent => 0);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  $doc->setFocus("body");
  $doc->print($menuString);
  $doc->print(<<"END_OF_CODE");
<hr width="100%" />
END_OF_CODE

  # see if we need to display an error message.
  if (exists $args{errorMessage})
  {
    my $string = $self->{methods}->displayMessageStr(type => "error", message => $args{errorMessage},
       langObj => $self->{langObj}, break => "none");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $doc->print($string);
    $doc->print("<hr width=\"100%\" />\n");
  }

  # now display the list of topics that got us to this point.
  my $counter;
  for ($counter=0; $counter < scalar @{$self->{contentArray}}; $counter++)
  {
    my $url = sprintf($helpUrl, $doc->encodeString(string => $self->{helpApp}), $doc->encodeString(string => join ',', @{$self->{contentArray}}[0..$counter]));
    $url .= "&inlineDocument=1" if (exists $self->{input}->{inlineDocument});
    #$doc->print("\n<br />\n") if ($counter > 0);
    #$doc->print("&nbsp;&nbsp;&nbsp;&nbsp;" x $counter);  # seperator token
    $doc->print(" > ") if ($counter > 0);
    $doc->print("<a href=\"$url\" class=\"help\" onMouseOver=\"window.status='$self->{contentArray}[$counter]'; return true;\" title=\"$self->{contentArray}[$counter]\">$self->{contentArray}[$counter]</a>");
  }
  $doc->print("\n<br />\n<hr width=\"100%\" />\n");  # drop display down 1 lines.

  # now display the topics or info for this item.
  my $topicRef = $self->{helpData}->{Topics}->{$self->{helpData}->{lang}};  # This points us to the language hash.
  for ($counter=0; $counter < scalar @{$self->{contentArray}}; $counter++)
  {
    if (! exists $topicRef->{$self->{contentArray}[$counter]})
    {
      $doc->reset();
      my $topic = $self->{contentArray}[$counter];
      splice (@{$self->{contentArray}}, $counter);  # remove the offending topic and any children it had.
      return $self->displayHelp(doc => $doc, errorMessage => "Topic = '$topic' does not exist!");
    }
    $topicRef = $topicRef->{$self->{contentArray}[$counter]};
    if ($counter < scalar @{$self->{contentArray}} - 1 && ref($topicRef) ne "HASH")
    { # Problem!!!
      $doc->reset();
      splice (@{$self->{contentArray}}, $counter);  # remove the offending topic and any children it had.
      return $self->displayHelp(doc => $doc, errorMessage => "Topic = '$topicRef' is not a HASH and we are not drilled all the way down yet! &nbsp;It is an " . ref($topicRef) . "!");
    }
  }

  # we are now drilled down to the point the user specified, now display either the list of available topics (hash) or the info for the specified topic.
  if (ref $topicRef eq "HASH")
  {
    # get the _order_ entry and build the array of items to display in the user specified order.
    my $order = $topicRef->{_order_};
    if (length $order == 0)
    {
      $doc->reset();
      splice (@{$self->{contentArray}}, scalar (@{$self->{contentArray}}) - 1);  # remove the offending topic and any children it had.
      return $self->displayHelp(doc => $doc, errorMessage => "_order_ not specified for Topic = '$self->{content}'!");
    }

    if (exists $topicRef->{_overview_})
    {
      $doc->print(<<"END_OF_CODE");
<div class="helpTopic">$topicRef->{_overview_}</div>
<br />
END_OF_CODE
    }

    my @order = split /,/, $order;
    # start a table.
    $doc->print(<<"END_OF_CODE");
<table border="0" cellpadding="0" cellspacing="0" width="100%">
END_OF_CODE
    foreach my $topic (@order)
    {
      if (! exists $topicRef->{$topic} && $topic !~ /^(_seperator_|_line_)$/)
      {
        $doc->print(<<"END_OF_CODE");
  <tr>
    <td align="right" width="10%">&nbsp;</td>
    <td align="left" colspan="2"><span style="color: red;">Error</span>:  Topic Item = '$topic' does not exist!</td>
  </tr>
END_OF_CODE
        next;
      }

      my @tmpArray = @{$self->{contentArray}};
      $tmpArray[++$#tmpArray] = $topic;
      my $url = sprintf($helpUrl, $doc->encodeString(string => $self->{helpApp}), $doc->encodeString(string => join ',', @tmpArray));
      $url .= "&inlineDocument=1" if (exists $self->{input}->{inlineDocument});

      # figure out what needs to be displayed.  A link, seperator, line, etc.
      my $content = "<a href=\"$url\" class=\"help\" onMouseOver=\"window.status='$topic'; return true;\" title=\"$topic\">$topic</a>";
      $content = "<br />" if ($topic eq "_seperator_");
      $content = "<hr width=\"25%\" align=\"left\" />" if ($topic eq "_line_");

      my $summary = (ref($topicRef->{$topic}) eq "HASH" && exists $topicRef->{$topic}->{'_summary_'} ? $topicRef->{$topic}->{'_summary_'} : "&nbsp;");
      my $indicator = (ref($topicRef->{$topic}) eq "HASH" ? "&gt;&nbsp;" : "&nbsp;");

      $doc->print(<<"END_OF_CODE");
  <tr>
    <td align="right" width="10%">$indicator</td>
    <td align="left" width="40%">$content</td>
    <td align="left" width="50%">$summary</td>
  </tr>
END_OF_CODE
    }
    $doc->print(<<"END_OF_CODE");
</table>
END_OF_CODE
  }
  else # info
  {
    # we need to do substitutions for <help></help> tags in the string.
    while ($topicRef =~ /<help app="[^"]*" content="[^"]+">[^<]+<\/help>/)
    {
      $topicRef =~ s/((<help app=")([^"]*)(" content=")([^"]+)(">)([^<]+)(<\/help>))/MYHELPTOKEN/;
      my $helpString = $1;
      my $app = $3;
      my $content = $5;
      my $name = $7;

      # fixup the app if it is empty to point to the current helpApp.
      $app = $self->{helpApp} if ($app eq "");

      $content = $doc->encodeString(string => $content);  # make sure it is url encoded.

      # generate the url string and replace for MYHELPTOKEN.
      my $url = sprintf($helpUrl, $app, $content);
      $url .= "&inlineDocument=1" if (exists $self->{input}->{inlineDocument} && $app eq $self->{helpApp});
      my $window = "help_" . $app . "_" . $self->{session}->{store}->{windowTitleTimestamp};
      my $urlString = "<a href=\"" . ($app eq $self->{helpApp} ? $url : "#") . "\" class=\"help\" onMouseOver=\"window.status='$name'; return true;\" title =\"$name\"" . ($app eq $self->{helpApp} ? "" : " onClick=\"showHelp('$window', '$url', 480, 640); return false;\"") . ">$name</a>";

      $topicRef =~ s/MYHELPTOKEN/$urlString/;
    }
    $doc->print(<<"END_OF_CODE");
<div class="helpTopic">$topicRef</div>
END_OF_CODE
  }

  return $doc;
}

=back

=cut

1;
__END__

=head1 DATA STRUCTURE

 The applications Help structure is defined as:

 %Topics = (
   en => {
     Normal => {
       "_order_" => "Close App,_seperator_,Refresh,Help",
       "Close App" => "Info about the Close App link.",
       "Refresh" => "Info about the Refresh link.",
       "Help" => "Info about the Help link.",
     },
     System => {
       "_order_" => "This should never be reached",
       "This should never be reached" => "explain why",
     },
   # other languages would go here
 };

 There are 2 sections per language (Normal and System).
 Normal is what most Portal Apps will use and denotes
 the type of user.  The System section is used in
 apps that have special code for System Admin accounts
 and want to provide different help for the System
 related code.

=head1 KEYWORDS

  _order_ - defines the topics available for display by
    the help system.  If you want a sub-topic, then the
    topic entry will point to another anonymous hash that
    must contain it's own _order_ entry.
  _seperator_ - adds a blank line between topic entries.
  _line_ - adds a <hr /> between topic entries.
  _overview_ - special topic entry that displays an
    overview of the section before displaying the contents
    of the current topic.  This should not be referenced
    in the _order_ entry.
  _summary_ - special topic entry that displays next to
    the topic entry in the parent listing that gives you
    this topic section as an option to drill downto.
    This should not be referenced in the _order_ entry.

  Topic entries can be made up of any characters other
  than a comma as the comma is what seperates them in the
  _order_ entry.

  You can link to another help topic by using the <help>
  tag.  Syntax is:
  <help app="" content="">text to display</help>
  The "text to display" part can not contain any html
  tags in it (at this time).

  app can be either "" to reference the current
  application or the name of another Portal App.
  Ex: app="UserProperties" or app="CompanyAdmin"

  content is the comma seperated list of topics to define
  the entry you want to link to.  It must start with
  Normal or System.  Ex: content="Normal,User" or
  content="System,User,Edit User Info"

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::PortalArgs(3)

=cut
