# Language.pm - The Object Class that provides a Language Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Language;
use strict;
use Portal::LanguageBase;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::LanguageBase Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.11';

=head1 NAME

Language - Object used to build a Language Object Class.

=head1 SYNOPSIS

  use Portal::Language;
  my $obj = Portal::Language->new(lang => $lang, app => "Portal");
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Language is a Language class used to hold phrases of a specific language.  This object will be referenced for internationalization of the application.

=head1 Exported FUNCTIONS

=head2 scalar new(lang, app)

    Creates a new instance of the Portal::Language
    object.  Internally sources the specified language if able and
    creates an internal data object that holds the language hash to
    work with for the Portal.
    
    If app = "Portal" we work with language files from the
    Portal::Language:: namespace.
    
    Else, we work with language files from the 
    Portal::app::Language:: namespace.
    
    lang defaults to 'en'.
    app defaults to 'Portal'.

=cut
sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( lang => 'en', app => "Portal", @_ );
  
  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }
  
  # instantiate anything unique to this module.
  $self->{lang} = $args{lang};
  $self->{app} = $args{app};
  
  if ($self->{app} eq "")
  {
    $self->error("app = '' is invalid!");
    return $self;
  }

  my $moduleName = $self->{lang};
  if (length $moduleName == 0) # it wasn't found, default to English.
  {
    $moduleName = $self->{lang} = "en";
  }
  if ($self->{app} eq "Portal")
  {
    if (!-e "Language/$moduleName". ".pm")
    {
      $moduleName = $self->{lang} = "en";
    }
    $moduleName = "Portal::Language::" . $moduleName;
  }
  else
  {
    $moduleName = "Portal::" . $self->{app} . "::Language::" . $moduleName;
    # Make sure it exists, else we default to en.
    eval "use $moduleName;";
    if ($@)
    {
      $moduleName = "Portal::" . $self->{app} . "::Language::en";
    }
  }

  # bring it into existance.
  eval "use $moduleName;";
  if ($@)
  {
    $self->error("Eval of $moduleName failed!<br>\nError = '$@'.<br>\n");

    return $self;
  }
  eval { $self->{phraseObj} = $moduleName->new; };  # Instantiate the Language Phrase object.  It will contain the charEncoding to use.
  if ($@)
  {
    $self->error("Instantiation of $moduleName failed!<br>\nError = '$@'.<br>\n");

    return $self;
  }
  if ($self->{phraseObj}->error)
  {
    $self->error($self->{phraseObj}->errorMessage);

    return $self;
  }

  # do validation
  if (!$self->Portal::Language::isValid())
  {
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 1 or 0 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;
  
  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (!defined $self->{phraseObj})
  {
    $self->missing("phraseObj");
  }
  elsif (!$self->{phraseObj}->isValid)
  {
    $self->invalid("phraseObj", $self->{phraseObj}, $self->{phraseObj}->errorMessage);
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  return 1;
}

1;
__END__

=head1 Exported VARIABLES

  LanguageObject phraseObj - The Language that has phrases and encoding.

  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3)

=cut
