# Log.pm - The Object Class that provides a Log Object
# Created by James Pattie & Jason A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Log;
use strict;
use Portal::Base;
use Portal::Objects::LogEntry;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.11';

=head1 NAME

Log - Object used to build a Log Object Class.

=head1 SYNOPSIS

  use Portal::Log;
  my $obj = Portal::Log->new(dbHandle => $dbHandle);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Log is a Log class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(dbHandle)

 Creates a new instance of the Log module.
 See Portal::Base(3) for a listing of required arguments.

 requires: dbHandle - DBIWrapper Object

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( dbHandle => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{dbHandle} = $args{dbHandle};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{dbHandle})
  {
    $self->missing("dbHandle");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# newEntry
#    Makes a new entry in the log_tb table.
# logEntry - Portal::Objects::LogEntry object.
sub newEntry
{
  my $self = shift;
  my %args = ( logEntry => undef, @_ );
  my $logEntry = $args{logEntry};

  if (not defined $logEntry)
  {
    $self->error("The logEntry object must be defined!<br>\n");
    return;
  }
  if (not $logEntry->isValid)
  {
    $self->error("The logEntry object is not valid!<br>\n" . $logEntry->errorMessage);
    return;
  }

  my $action = $logEntry->{action};
  my $ipAddress = $logEntry->{ipAddress};
  my $extraInfo = $logEntry->{extraInfo};
  my $userId = $logEntry->{userId};
  my $serverName = $logEntry->{serverName};

  # make sure that all ' quotes are escaped in the extraInfo string.
  #$extraInfo =~ s/'/\\'/gs;  # escape all ' quotes.
  #$extraInfo =~ s/([^\\]{2})'/$1\\'/gs;  # only do it to those ' that are not already escaped.

  # make the entry in the database.
  if ($self->{dbHandle}->{dbType} eq "mysql")
  {
    $self->{dbHandle}->write(sql => "INSERT INTO log_tb (action, ip_address, extra, user_id, action_date, action_time, server_name) VALUES (?, ?, ?, ?, now(), now(), ?)", plug => [ $action, $ipAddress, $extraInfo, $userId, $serverName ]);
  }
  elsif ($self->{dbHandle}->{dbType} eq "Pg")
  {
    $self->{dbHandle}->write(sql => "INSERT INTO log_tb (action, ip_address, extra, user_id, action_date, action_time, server_name) VALUES (?, ?, ?, ?, now(), now()::timestamp without time zone::time, ?)", plug => [ $action, $ipAddress, $extraInfo, $userId, $serverName ]);
  }
  if ($self->{dbHandle}->error)
  {
    $self->error($self->{dbHandle}->errorMessage);
    return;
  }
  $self->{dbHandle}->commit;  # make sure the entry is committed to the database.
  if ($self->{dbHandle}->error)
  {
    $self->error($self->{dbHandle}->errorMessage);
    return;
  }
}

# swapDate
# takes a date in format MM/DD/YYYY and returns YYYY/MM/DD or vice versa
sub swapDate
{
  my $self = shift;
  my %args = ( date => "", @_ );
  my $date = $args{date};

  if (length $date == 0)
  {
    $date = "01/01/2000"; # default date.
  }
  if ($date =~ /^\d{2}.\d{2}.\d{4}$/)  # MM/DD/YYYY
  {
    $date =~ s/^(\d{2})(.)(\d{2})(.)(\d{4})$/$5$2$1$4$3/;
  }
  else # YYYY/MM/DD
  {
    $date =~ s/^(\d{4})(.)(\d{2})(.)(\d{2})$/$3$2$5$4$1/;
  }

  return $date;
}

# getEntries
#    Looks up and returns an array of LogEntry objects that contains the
#    requested information.
#      startDate - A date.
#      endDate - A date.
#      startTime - A time.
#      endTime - A time.
#      All other fields are as in newEntry().
#      See Portal::Objects::LogEntry(3) for info on the structure of a
#      LogEntry.
sub getEntries
{
  my $self = shift;
  my %args = ( startDate => "now()", endDate => "now()", startTime => "", endTime => "", action => -2, ipAddress => "", extraInfo => "", userId => -2, serverName => "", @_ );
  my $startDate = $args{startDate};
  my $endDate = $args{endDate};
  my $startTime = $args{startTime};
  my $endTime = $args{endTime};
  my $action = $args{action};
  my $ipAddress = $args{ipAddress};
  my $extraInfo = $args{extraInfo};
  my $userId = $args{userId};
  my $serverName = $args{serverName};

  my @logEntries = ();

  if ($self->{dbHandle}->{dbType} eq "mysql" || ($self->{dbHandle}->{dbType} eq "Pg" && !$self->{dbHandle}->{setDateStyle}))
  {
    if ($startDate ne "now()")
    {
      $startDate = $self->swapDate(date => $startDate);
    }
    if ($endDate ne "now()")
    {
      $endDate = $self->swapDate(date => $endDate);
    }
  }

  # Build the SQL statement to execute.
  my $dateString = "action_date BETWEEN ";
  if ($self->{dbHandle}->{dbType} eq "mysql")
  {
    if ($startDate eq "now()")
    {
      $dateString .= "$startDate";
    }
    else
    {
      $dateString .= "'$startDate'";
    }
    $dateString .= " AND ";
    if ($endDate eq "now()")
    {
      $dateString .= "$endDate";
    }
    else
    {
      $dateString .= "'$endDate'";
    }
  }
  else
  {
    $dateString .= "'$startDate' AND '$endDate'";
  }
  my $sql = "SELECT action_date, action_time, action, ip_address, extra, user_id, server_name FROM log_tb WHERE ($dateString)";
  if ($action != -2)
  {
    $sql .= " AND action = $action";
  }
  if ($userId != -2)
  {
    $sql .= " AND user_id = $userId";
  }
  if (length $ipAddress > 0)
  {
    $sql .= " AND ip_address = '$ipAddress'";
  }
  if (length $extraInfo > 0)
  {
    $sql .= " AND extra LIKE '$extraInfo'";
  }
  if (length $serverName > 0)
  {
    $sql .= " AND server_name = '$serverName'";
  }
  $sql .= " ORDER BY server_name, action_date, action_time, action, user_id, ip_address";

  # Execute the SQL statement.
  my $sth = $self->{dbHandle}->read(sql => $sql);
  if ($self->{dbHandle}->error)
  {
    $self->error($self->{dbHandle}->errorMessage);

    return ();
  }

  # build up the array of LogEntries.
  while (my @entry = $sth->fetchrow_array)
  {
    my $date = $entry[0];
    if ($self->{dbHandle}->{dbType} eq "mysql" || ($self->{dbHandle}->{dbType} eq "Pg" && !$self->{dbHandle}->{setDateStyle}))
    {
      $date = $self->swapDate(date => $date);
    }
    my $entryObj = Portal::Objects::LogEntry->new(actionDate => $date, actionTime => $entry[1], action => $entry[2], ipAddress => $entry[3], extraInfo => $entry[4], userId => $entry[5], serverName => $entry[6], langObj => $self->{langObj});
    if ($entryObj->error)
    {
      $self->error($entryObj->errorMessage);

      return @logEntries;
    }
    $logEntries[++$#logEntries] = $entryObj;
  }

  return @logEntries;
}

=item int purgeEntries(olderThan)

 Deletes any log entries older than X days, where X
 defaults to 30.

 Returns the number of entries deleted.

=cut
sub purgeEntries
{
  my $self = shift;
  my $olderThan = 30;
  if (@_ == 1)
  {
    $olderThan = shift;
  }
  else
  {
    my %args = ( olderThan => 30, @_ );
    $olderThan = $args{olderThan};
  }

  # only delete entries older than todays.
  if ($olderThan >= 1)
  {
    my $result = $self->{dbHandle}->write(sql => "DELETE FROM log_tb WHERE action_date < (now() - '$olderThan days'::INTERVAL)::DATE");
    if ($self->{dbHandle}->error)
    {
      $self->error($self->{dbHandle}->errorMessage);
      return $result;
    }
    $self->dbHandle->commit;  # make sure the changes are made.
    return $result;
  }
  return 0;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

 date swapDate(date)
    takes a date in format MM/DD/YYYY and returns YYYY/MM/DD or
    vice versa.

  void newEntry(logEntry)
    Makes a new entry in the log_tb table.
      logEntry - Portal::Objects::LogEntry(3) object.

  LogEntry[] getEntries(startDate, endDate, startTime, endTime, action,
                        ipAddress, userId, extraInfo)
    Looks up and returns an array of LogEntry objects that contains the
    requested information.  If the startTime and endTime values are not
    specified, then it will ignore the time stamp of the log entries.
    If no dates are given, we default to only looking at the current
    dates log entries.  If action = -2, then we gather all action
    entries that pertain.  If userId = -2, then we gather all user
    entries that pertain.  If extraInfo is empty, it is ignored.
    The defaults for action and userId are -2.
      startDate - A date.
      endDate - A date.
      startTime - A time.
      endTime - A time.
      All other fields are as in newEntry().
      See Portal::Objects::LogEntry(3) for info on the structure of a
      LogEntry.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::Objects::LogEntry(3), Portal::Base(3)

=cut
