# AppServerObject.pm - The Object Class that provides a AppServerObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::AppServerObject;
use strict;
use overload '==' => \&objsEqual,
             '!=' => \&objsNotEqual,
             '""' => \&printStr;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.07';

=head1 NAME

AppServerObject - Object used to build a AppServerObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::AppServerObject;
  my $obj = Portal::Objects::AppServerObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

AppServerObject is a AppServerObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(id, server, port, dbHost, dbType, dbPort, counter)

 Creates a new instance of the Portal::Objects::AppServerObject module.
 See Portal::Base(3) for a listing of required arguments.

 requires:
  id
  server
  port
  dbHost
  dbType
  dbPort
  counter - UNIQUE Index used by the database to identify each entry.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( id => -1, server => "", port => -1, dbHost => "127.0.0.1", dbType => "Pg", dbPort => "5432", counter => "-2", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{id} = $args{id};
  $self->{server} = $args{server};
  $self->{port} = $args{port};
  $self->{dbHost} = $args{dbHost};
  $self->{dbType} = $args{dbType};
  $self->{dbPort} = $args{dbPort};
  $self->{counter} = $args{counter};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{id} < 0)
  {
    $self->invalid("id", "$self->{id}");
  }
  if (length $self->{server} == 0)
  {
    $self->invalid("server", "$self->{server}");
  }
  if ($self->{port} < 0)
  {
    $self->invalid("port", "$self->{port}");
  }
  if (length $self->{dbHost} == 0)
  {
    $self->missing("dbHost");
  }
  if ($self->{dbType} !~ /^(Pg|mysql)$/)
  {
    $self->invalid("dbType", "$self->{dbType}");
  }
  if ($self->{dbPort} !~ /^\d+$/)
  {
    $self->invalid("dbPort", "$self->{dbPort}");
  }
  if ($self->{counter} < -1)
  {
    $self->invalid("counter", $self->{counter});
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool objsEqual(a,b)

 returns 1 if the 2 objects are the same (content wise),
 0 otherwise.
 
=cut
sub objsEqual
{
  my $a = shift;
  my $b = shift;
  
  if (ref($a) ne "Portal::Objects::AppServerObject")
  {
    die "objsEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::AppServerObject")
  {
    die "objsEqual: b = '" . ref($b) . "'";
  }

  # do number tests.
  foreach my $key (qw(id port dbPort counter))
  {
    return 0 if ($a->{$key} != $b->{$key});
  }
  
  # do string tests
  foreach my $key (qw(server dbHost dbType))
  {
    return 0 if ($a->{$key} ne $b->{$key});
  }
  
  return 1;  # assume they are equal if we get this far.  
}

=item bool objsNotEqual(a,b)

 returns 0 if the 2 objects are the same (content wise),
 1 otherwise.
 
=cut
sub objsNotEqual
{
  my $a = shift;
  my $b = shift;
  
  if (ref($a) ne "Portal::Objects::AppServerObject")
  {
    die "objsNotEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::AppServerObject")
  {
    die "objsNotEqual: b = '" . ref($b) . "'";
  }

  # take advantage of the == overload to do the brunt of our work.
  return (!($a == $b));
}

=item scalar printStr()

 returns the printable version of this object.
 
=cut
sub printStr
{
  my $self = shift;
  my $out = "Portal::Objects::AppServerObject : ";
  
  foreach my $keys (keys %{$self})
  {
    $out .= ", " if ($out =~ /=/);  # output a , if we have an = in the string.
    $out .= "$keys = '" . $self->{$keys} . "'";
  }
  
  $out .= "\n";
  
  return $out;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES
  id     - int
  server - text
  port   - numeric(6)
  dbHost - text
  dbType - text
  dbPort - numeric(6)

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
