# ApplicationObject.pm - The Object Class that provides a ApplicationObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::ApplicationObject;
use strict;
use Portal::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.07';

=head1 NAME

ApplicationObject - Object used to build a ApplicationObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::ApplicationObject;
  my $obj = Portal::Objects::ApplicationObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

ApplicationObject is a ApplicationObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(id, name, description, iconName, server, port, cost, unit,
type, adminType, dbHost, dbType, dbPort, dbName, number, height, width, autorun, wap)

 Creates a new instance of the Portal::Objects::ApplicationObject module.
 See Portal::Base(3) for a listing of required arguments.

  id int           - uniquely identifies this application.
  name text        - Uppercase name of the Perl Module
  description text - Descriptive name to show
  iconName text    - name of image file to display
  server text      - server it is located on
  port int         - The port to connect to (80 or 443
                     indicates http(s))
  dbHost text      - server the Database resides on
  dbType text      - Database type (Pg or mysql)
  dbPort int       - Database port (any valid number)
  cost float       - The cost of this application
  unit int         - per month, user, etc.
  type text        - administration, help, bugs, news, regular
  adminType text   - normal, system (if type = administration)
  dbName text      - The database this app is using (not normally
                     used).
  number int       - The number purchased (not normally used).
  height int       - The initial height of the window.
  width int        - The initial width of the window.
  autorun bool     - Should this app be auto launched when portal
                     is logged into.
  wap bool         - Is this app WAP aware?

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( id => -1, name => "", description => "", iconName => "default.jpg", server => "localhost", port => "80", cost => "0.00", unit => 0,
               type => "regular", adminType => "normal", dbName => "", number => "0", height => "640", width => "480", autorun => 0,
               dbType => "Pg", dbPort => "5432", wap => 0, dbHost => "127.0.0.1", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{id} = $args{id};
  $self->{name} = $args{name};
  $self->{description} = $args{description};
  $self->{iconName} = $args{iconName};
  $self->{server} = $args{server};
  $self->{port} = $args{port};
  $self->{cost} = $args{cost};
  $self->{unit} = $args{unit};
  $self->{type} = $args{type};
  $self->{adminType} = $args{adminType};
  $self->{dbName} = $args{dbName};  # this is only used by the getAppsForCompany and getAppsForUser methods.
  $self->{number} = $args{number};  # this is only used by the getAppsForCompany and getAppsForUser methods.
  $self->{height} = $args{height};
  $self->{width} = $args{width};
  $self->{autorun} = $args{autorun};# this is only used by the getAppsForCompany and getAppsForUser methods.
  $self->{dbHost} = $args{dbHost};
  $self->{dbType} = $args{dbType};
  $self->{dbPort} = $args{dbPort};
  $self->{wap} = $args{wap};  # are we WAP aware?
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{name} == 0)
  {
    $self->missing("name");
  }
  if (length $self->{description} == 0)
  {
    $self->missing("description");
  }
  if (length $self->{server} == 0)
  {
    $self->missing("server");
  }
  if ($self->{port} !~ /^\d+$/)
  {
    $self->invalid("port", "$self->{port}");
  }
  if ($self->{cost} !~ /^\d+\.\d{2}$/)
  {
    $self->invalid("cost", "$self->{cost}");
  }
  if (not exists $self->variables->units->{$self->{unit}})
  {
    $self->invalid("unit", "$self->{unit}");
  }
  if (not exists $self->variables->appTypes->{$self->{type}})
  {
    $self->invalid("type", "$self->{type}");
  }
  if (not exists $self->variables->adminTypes->{$self->{adminType}})
  {
    $self->invalid("adminType", "$self->{adminType}");
  }
  if ($self->{number} < 0)  # number = 0 means unlimited licenses.
  {
    $self->invalid("number", "$self->{number}");
  }
  if ($self->{height} < 0 || $self->{height} > 4000)  # I think 4000 is way too much!
  {
    $self->invalid("height", "$self->{height}");
  }
  if ($self->{width} < 0 || $self->{width} > 4000)
  {
    $self->invalid("width", "$self->{width}");
  }
  if ($self->{autorun} !~ /^(1|0)$/)
  {
    $self->invalid("autorun", "$self->{autorun}");
  }
  if (length $self->{dbHost} == 0)
  {
    $self->missing("dbHost");
  }
  if ($self->{dbType} !~ /^(Pg|mysql)$/)
  {
    $self->invalid("dbType", "$self->{dbType}");
  }
  if ($self->{dbPort} !~ /^\d+$/)
  {
    $self->invalid("dbPort", "$self->{dbPort}");
  }
  if ($self->{wap} !~ /^(1|0)$/)
  {
    $self->invalid("wap", "$self->{wap}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Application(3), Portal::Base(3)

=cut
