# CompanyApplicationObject.pm - The Object Class that provides a CompanyApplicationObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::CompanyApplicationObject;
use strict;
use Portal::Objects::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Objects::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.08';

=head1 NAME

CompanyApplicationObject - Object used to build a CompanyApplicationObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::CompanyApplicationObject;
  my $obj = Portal::Objects::CompanyApplicationObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

CompanyApplicationObject is a CompanyApplicationObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Objects::CompanyApplicationObject module.
 See Portal::Objects::Base(3) for a listing of required arguments.

 You now have to use the populate() method to assign the
 CompanyApplicationObject parameters and have them be validated.
 new() just instantiates the object so that it can be used by the
 Portal::Forms::Apps module to manage Applications with.

 The HTMLObject::Forms data structures will be defined for
 the following cases:

 assignment - allow an admin to assign/un-assign the app
  to users of a specific company.  This form has a
  javascript code block defined which needs to be output
  into the javascript section of the HTMLObject document
  you display the form in.  (It is not an include file yet).

 serverInfo - allow an admin to edit the server info for
  the specified company_app_tb entry.

 purchaseApp - used to let the Company admin purchase the
  app and do initial server selection, etc.

 sellAppLicenses - allows an admin to sell off spare
  licenses or the app itself.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{backwardsCompatible} = 1;
  $self->{objName} = "CompanyApplicationObject";
  $self->{data}->{main} =
    { appId => { -DataType => "int", -Value => -1 },
      companyId => { -DataType => "int", -Value => -1 },
      number => { -DataType => "int", -Value => 1 },
      port => { -DataType => "int", -Value => -1 },
      unit => { -DataType => "int", -Value => 0 },
      dbPort => { -DataType => "int", -Value => "5432" },
      height => { -DataType => "int", -Value => 640 },
      width => { -DataType => "int", -Value => 480 },
      autorun => { -DataType => "int", -Value => 0 },
      wap => { -DataType => "int", -Value => 0 },
      server => { -DataType => "string", -Value => "" },
      cost => { -DataType => "float", -Value => "0.00" },
      dbHost => { -DataType => "string", -Value => "127.0.0.1" },
      dbType => { -DataType => "string", -Value => "Pg" },
      dbName => { -DataType => "string", -Value => "" },
    };

  $self->{usingForms} = 1;

  # the templates
  $self->{template}->{assignment} = <<"END_OF_CODE";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=appId#
  #FIELD=formSubmitted#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="center" valign="top">#FIELD=users#</td>
    </tr>
    <tr>
      <td align="center">#FIELD=submitForm#</td>
    </tr>
    <tr>
      <td align="center"><span style="font-size: smaller;">#LANG=DisabledAdminSysadminIndicators#</span></td>
    </tr>
  </table>
</form>
END_OF_CODE

  $self->{data}->{assignment} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                               "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("UpdateAppAssignments") },
                               "app" => { -Type => "hidden", -Value => "" },
                               "state" => { -Type => "hidden", -Value => "" },
                               "command" => { -Type => "hidden", -Value => "updateAppAssignments" },
                               "appId" => { -Type => "hidden", -Value => "" },
                               "users" => { -Type => "select-picker", -assignedName => "assignedUsers",
                                 -unassignedName => "unassignedUsers", -assignedLabel => $self->{langObj}->map("Assigned"),
                                 -unassignedLabel => $self->{langObj}->map("UnAssigned"),
                                 -seperator => ",", -assignedValues => "", -unassignedValues => "",
                                 -assignedOptions => undef, -unassignedOptions => undef, class => "edit" },
                             };

  $self->{profile}->{assignment} = { required => [],
                                  optional => [],
                                  constraints => {},
                                };

  $self->{template}->{serverInfo} = <<"END_OF_CODE";
<span style="font-size: smaller;">#LANG=ServerInfo#</span><br />
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=appId#
  #FIELD=formSubmitted#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=serverInfo#: </span></td>
      <td align="left">#FIELD=serverInfo#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=dbName#: </span></td>
      <td align="left"><span style="font-size: smaller;">#FIELD=dbName#</span></td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=autorun#: #INVALID=autorun#</span></td>
      <td align="left">#FIELD=autorun#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=wap#: #INVALID=wap#</span></td>
      <td align="left">#FIELD=wap#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=width#: #INVALID=width#</span></td>
      <td align="left">#FIELD=width#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=height#: #INVALID=height#</span></td>
      <td align="left">#FIELD=height#</td>
    </tr>
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_CODE

  $self->{data}->{serverInfo} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                               "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("UpdateServerInfo") },
                               "app" => { -Type => "hidden", -Value => "" },
                               "state" => { -Type => "hidden", -Value => "" },
                               "command" => { -Type => "hidden", -Value => "updateServerInfo" },
                               "appId" => { -Type => "hidden", -Value => "" },
                               "serverInfo" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("Server"), -Options => undef },
                               "dbName" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("DBName"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                               "autorun" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("AutoRun"), -Options => $self->{variables}->formBoolean() },
                               "wap" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("WAPEnabled"), -Options => $self->{variables}->formBoolean() },
                               "width" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("Width"), size => 4, maxlength => 4 },
                               "height" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("Height"), size => 4, maxlength => 4 },
                             };

  $self->{profile}->{serverInfo} = { required => [ qw( serverInfo autorun wap width height app state command ) ],
                                  optional => [],
                                  constraints => {},
                                };

  $self->{template}->{purchaseApp} = <<"END_OF_CODE";
<span style="font-size: smaller;">#LANG=PurchaseApplication#</span><br />
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=appId#
  #FIELD=formSubmitted#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=name#: </span></td>
      <td align="left"><span style="font-size: smaller;">#FIELD=name#</span></td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=description#: </span></td>
      <td align="left"><span style="font-size: smaller;">#FIELD=description#</span></td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=cost#: </span></td>
      <td align="left"><span style="font-size: smaller;">#FIELD=cost#</span></td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=unit#: </span></td>
      <td align="left"><span style="font-size: smaller;">#FIELD=unit#</span></td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=numLicenses#: #INVALID=numLicenses#</span></td>
      <td align="left">#FIELD=numLicenses#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=server#: </span></td>
      <td align="left">#FIELD=server#</td>
    </tr>
    <tr>
      <td align="right" valign="top"><span style="font-size: smaller;">#LABEL=dbName#: </span></td>
      <td align="left">#FIELD=dbName#</td>
    </tr>
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#<br /><span style="font-size: smaller;">#LANG=ToCancelClickManageApps#</span></td>
    </tr>
  </table>
</form>
END_OF_CODE

  $self->{data}->{purchaseApp} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                               "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("Purchase") },
                               "app" => { -Type => "hidden", -Value => "" },
                               "state" => { -Type => "hidden", -Value => "" },
                               "command" => { -Type => "hidden", -Value => "purchaseApp" },
                               "appId" => { -Type => "hidden", -Value => "" },
                               "server" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("Server"), -Options => undef },
                               "dbName" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("DBName"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                               "numLicenses" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("numLicenses"), -Options => undef },
                               "name" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("Name"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                               "description" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("Description"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                               "cost" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("Cost"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                               "unit" => { -Type => "select", -Value => "", -Options => undef, -Label => $self->{langObj}->map("Unit"), -ReadOnly => 1, -ReadOnlyMode => "text" },
                             };

  $self->{profile}->{purchaseApp} = { required => [ qw( server numLicenses dbName app state command ) ],
                                  optional => [],
                                  constraints => {},
                                };

  $self->{template}->{sellAppLicenses} = <<"END_OF_CODE";
<span style="font-size: smaller;">#SELLAPPLICENSESPHRASE#</span><br />
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=appId#
  #FIELD=formSubmitted#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="left"><span style="font-size: smaller;">#FIELD=type#</span></td>
    </tr>
    <tr>
      <td align="left"><span style="font-size: smaller;">#FIELD=numLicenses#</span></td>
    </tr>
    <tr>
      <td align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_CODE

  $self->{data}->{sellAppLicenses} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                               "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("Sell") },
                               "app" => { -Type => "hidden", -Value => "" },
                               "state" => { -Type => "hidden", -Value => "" },
                               "command" => { -Type => "hidden", -Value => "sellApp" },
                               "appId" => { -Type => "hidden", -Value => "" },
                               "type" => { -Type => "radio", -Value => "licenses", -Options => { values => [ "app", "licenses" ], names => [ $self->{langObj}->map("Application"), $self->{langObj}->map("Licenses") ] } },
                               "numLicenses" => { -Type => "select", -Value => "", -Options => undef },
                             };

  $self->{profile}->{sellAppLicenses} = { required => [ qw( type appId app state command ) ],
                                  optional => [ qw( numLicenses ) ],
                                  constraints => {},
                                };

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->get("appId") < 0)
  {
    $self->invalid("appId", $self->get("appId"));
  }
  if ($self->get("companyId") < 0)
  {
    $self->invalid("companyId", $self->get("companyId"));
  }
  if ($self->get("number") < 0)  # 0 is valid as it represents unlimited licenses.
  {
    $self->invalid("number", $self->get("number"));
  }
  if (length $self->get("server") == 0)
  {
    $self->invalid("server", $self->get("server"));
  }
  if ($self->get("port") < 0)
  {
    $self->invalid("port", $self->get("port"));
  }
  if ($self->get("cost") !~ /^\d+\.\d{2}$/)
  {
    $self->invalid("cost", $self->get("cost"));
  }
  if (!exists $self->{variables}->{units}->{$self->get("unit")})
  {
    $self->invalid("unit", $self->get("unit"));
  }
  if (length $self->get("dbName") == 0)
  {
    $self->invalid("dbName", $self->get("dbName"));
  }
  if ($self->get("height") !~ /^\d+$/ || ($self->get("height") < 0 || $self->get("height") > 4000))  # I think 4000 is way too much!
  {
    $self->invalid("height", $self->get("height"));
  }
  if ($self->get("width") !~ /^\d+$/ || ($self->get("width") < 0 || $self->get("width") > 4000))
  {
    $self->invalid("width", $self->get("width"));
  }
  if ($self->get("autorun") !~ /^(1|0)$/)
  {
    $self->invalid("autorun", $self->get("autorun"));
  }
  if (length $self->get("dbHost") == 0)
  {
    $self->missing("dbHost");
  }
  if ($self->get("dbType") !~ /^(Pg|mysql)$/)
  {
    $self->invalid("dbType", $self->get("dbType"));
  }
  if ($self->get("dbPort") !~ /^\d+$/)
  {
    $self->invalid("dbPort", $self->get("dbPort"));
  }
  if ($self->get("wap") !~ /^(1|0)$/)
  {
    $self->invalid("wap", $self->get("wap"));
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  appId     - int
  companyId - int
  number    - int (number of licenses bought)
  server    - text
  port      - numeric(6)
  dbHost    - text
  dbType    - text
  dbPort    - numeric(6)
  cost      - money (without $ and ,'s)
  unit      - int
  dbName    - text
  height    - int
  width     - int
  autorun   - bool
  wap       - bool


=head1 NOTE

 Data fields can only be accessed by treating them as
 a function call.

 print $obj->variable($newValue);
 $value = $obj->variable;

 This does not work in here documents (<<)!

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Application(3), Portal::Base(3)

=cut
