# CompanyBillingObject.pm - The Object Class that provides a CompanyBillingObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::CompanyBillingObject;
use strict;
use Portal::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

=head1 NAME

CompanyBillingObject - Object used to build a CompanyBillingObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::CompanyBillingObject;
  my $obj = Portal::Objects::CompanyBillingObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

CompanyBillingObject is a CompanyBillingObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(id, ccType, ccNum, ccExpire)

 Creates a new instance of the Portal::Objects::CompanyBillingObject module.
 See Portal::Base(3) for a listing of required arguments.
 
 requires:
   id
   ccType
   ccNum
   ccExpire

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( id => -1, ccType => "Visa", ccNum => "", ccExpire => "01/01/2000", @_ );
  
  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{id} = $args{id};
  $self->{ccType} = $args{ccType};
  $self->{ccNum} = $args{ccNum};
  $self->{ccExpire} = $args{ccExpire};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{id} < 0)
  {
    $self->invalid("id", "$self->{id}");
  }
  if (not exists $self->variables->ccTypes->{$self->{ccType}})
  {
    $self->invalid("ccType", "$self->{ccType}");
  }
  if (length $self->ccNum != 16 || $self->ccNum !~ /^\d{16}$/)
  {
    $self->invalid("ccNum", "$self->{ccNum} must be 16 <b>digits</b> long!");
  }
  if ($self->ccExpire !~ /^\d{2}(\/|-)\d{2}(\/|-)\d{4}$/)
  {
    $self->invalid("ccExpire", "$self->{ccExpire}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  id       - int
  ccType   - text (Visa Mastercard 'American Express')
  ccNum    - int (16)
  ccExpire - Date

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Auth(3), Portal::Base(3)

=cut
