# DynamicContentObject.pm - The Object Class that provides a DynamicContentObject Object
# Created by James A. Pattie, 01/22/2002.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::DynamicContentObject;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.02';

=head1 NAME

DynamicContentObject - Object used to build a DynamicContent Object Class.

=head1 SYNOPSIS

  use Portal::Objects::DynamicContentObject;
  my $obj = Portal::Objects::DynamicContentObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

DynamicContentObject is a DynamicContentObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(index, companyId, userId, callingApp, tag, app, arguments)

 Creates a new instance of the Portal::Objects::DynamicContentObject module.
 See Portal::Base(3) for a listing of required arguments.
 
 requires:
  index
  companyId
  userId
  callingApp
  tag
  app
  arguments

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( index => "-1", companyId => "-1", userId => "-1", callingApp => "",
               tag => "", app => "", arguments => "", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{index} = $args{index};
  $self->{companyId} = $args{companyId};
  $self->{userId} = $args{userId};
  $self->{callingApp} = $args{callingApp};
  $self->{tag} = $args{tag};
  $self->{app} = $args{app};
  $self->{arguments} = $args{arguments};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{index} !~ /^(-1|\d+)$/)
  {
    $self->invalid("index", "$self->{index}");
  }
  if ($self->{companyId} !~ /^(-1|\d+)$/)
  {
    $self->invalid("companyId", "$self->{companyId}");
  }
  if ($self->{userId} !~ /^(-1|\d+)$/)
  {
    $self->invalid("userId", "$self->{userId}");
  }
  if (length $self->{callingApp} == 0)
  {
    $self->missing("callingApp");
  }
  if (length $self->{tag} == 0)
  {
    $self->missing("tag");
  }
  if (length $self->{app} == 0)
  {
    $self->missing("app");
  }
  if (length $self->{arguments} == 0)
  {
    $self->missing("arguments, which at a minimum must contain state and command values");
  }
  if ($self->{companyId} eq "-1" && $self->{userId} ne "-1")
  {
    $self->invalid("companyId", "-1");
    $self->invalid("and userId", "$self->{userId}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# scalar print(void)
sub print
{
  my $self = shift;

  my $string = "index='$self->{index}', companyId='$self->{companyId}', userId='$self->{userId}',";
  $string .= "callingApp='$self->{callingApp}', tag='$self->{tag}', app='$self->{app}',";
  $string .= "arguments='$self->{arguments}'";

  return $string;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  scalar new(index, companyId, userId, callingApp, tag, app, arguments)
    Creates a new instance of the Portal::Objects::DynamicContentObject
    object.

  bool isValid(void)
    Returns 0 or 1 to indicate if the object is valid.  The error will be
    available via errorMessage().

  void error(errorString)
    requires: errorString
    returns: nothing
    Sets error = 1 and errorString = errorString passed in.

  scalar error(void)
    Returns the value of error.

  scalar errorMessage(void)
    Returns the value of errorString.
    
  scalar print(void)
    Returns the printfied version of the Object.


  NOTE:  All data fields are accessible by specifying the object
         and pointing to the data member to be modified on the
         left-hand side of the assignment.
         Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3)

=cut
