# MenuItem.pm - The Object Class that provides a MenuItem Object
# Created by James A. Pattie, 10/15/2002.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::MenuItem;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.02';

=head1 NAME

MenuItem - Object used to build a Menu Item Object Class.

=head1 SYNOPSIS

  use Portal::Objects::MenuItem;
  my $obj = Portal::Objects::MenuItem->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

MenuItem is a menuItem class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(type, url, text, title, onClick, onMouseOver, options,
             target, onChange, name)

 Creates a new instance of the Portal::Objects::MenuItem object.
 See Portal::Base(3) for a listing of required arguments.

 requires: type - 'link', 'seperator', 'text', 'hilightedText',
                  'select',
 optional: url - url of link,
           text - text to display or the contents of the <a></a> link,
           title - the title and/or mouseOver contents
           onClick - the onClick event code or empty if none
           onMouseOver - the onMouseOver event or empty if none
           options - array of option entries for type='select'
           target - the window to target for this link
           onChange - onChange event code when type='select'
           name - the name of the select box when type='select'

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( type => "link", url => "", text => "", title => "", onClick => "",
               onMouseOver => "", options => undef, target => "", onChange => "",
               name => "", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{type} = $args{type};
  $self->{url} = $args{url};
  $self->{text} = $args{text};
  $self->{title} = $args{title};
  $self->{onClick} = $args{onClick};
  $self->{onMouseOver} = $args{onMouseOver};
  $self->{options} = $args{options};
  $self->{target} = $args{target};
  $self->{onChange} = $args{onChange};
  $self->{name} = $args{name};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{type} !~ /^(link|seperator|text|hilightedText|select)$/)
  {
    $self->invalid("type", "$self->{type}");
  }
  if ($self->{type} !~ /^(seperator|select)$/)
  {
    # verify we have the text
    if ($self->{text} !~ /^(.+)$/)
    {
      $self->invalid("text", "$self->{text}");
    }
  }
  if ($self->{type} =~ /^(link)$/)
  {
    # verify we have the url and title
    if ($self->{url} !~ /^(https?:\/\/[^\/]+\.[^\/.]+(\/.+)?)$/)
    {
      $self->invalid("url", "$self->{url}");
    }
    if ($self->{title} !~ /^(.+)$/)
    {
      $self->invalid("title", "$self->{title}");
    }
  }
  if ($self->{type} =~ /^(select)$/)
  {
    # verify we have a name
    if ($self->{name} !~ /^(.+)$/)
    {
      $self->invalid("name", "$self->{name}");
    }
    # verify we have at least one option.
    if (!defined $self->{options})
    {
      $self->missing("options");
    }
    elsif (scalar @{$self->{options}} == 0)
    {
      $self->missing("options");
    }
    else
    {
      # verify each of the specified options.
      my $counter=0;
      foreach my $option (@{$self->{options}})
      {
        if (!exists $option->{value})
        {
          $self->missing("option #$counter");
        }
        elsif ($option->{value} !~ /^(.+)$/)
        {
          $self->invalid("option #$counter value", "$option->{value}");
        }
        if (!exists $option->{display})
        {
          $self->missing("option #$counter display");
        }
        elsif ($option->{display} !~ /^(.+)$/)
        {
          $self->invalid("option #$counter display", "$option->{display}");
        }
        $counter++;
      }
    }
  }
  if ($self->{onClick} && $self->{type} !~ /^(link)$/)
  {
    $self->invalid("onClick", "when type = '$self->{type}'");
  }
  if ($self->{onMouseOver} && $self->{type} !~ /^(link)$/)
  {
    $self->invalid("onMouseOver", "when type = '$self->{type}'");
  }
  if ($self->{target} && $self->{type} !~ /^(link)$/)
  {
    $self->invalid("target", "when type = '$self->{type}'");
  }
  if ($self->{onChange} && $self->{type} !~ /^(select)$/)
  {
    $self->invalid("onChange", "when type = '$self->{type}'");
  }
  if ($self->{name} && $self->{type} !~ /^(select)$/)
  {
    $self->invalid("name", "when type = '$self->{type}'");
  }
  if (defined $self->{options} && $self->{type} !~ /^(select)$/)
  {
    $self->invalid("options", "when type = '$self->{type}'");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# string print(orientation, indent)
# requires: orientation, indent - # of spaces to indent.
# returns:  generated MenuItem string.
# generates the specified MenuItem without the wrapping <tr>,<td> tags.
sub print
{
  my $self = shift;
  my %args = ( orientation => "horizontal", indent => 6, @_ );
  my $orientation = $args{orientation};
  my $indent = $args{indent};
  my $result = "";

  if ($orientation !~ /^(horizontal|vertical)$/)
  {
    $self->error("orientation = '$orientation' is invalid!<br>\n");
    return $result;
  }
  if ($indent !~ /^\d+$/)
  {
    $self->error("indent = '$indent' is invalid!<br>\n");
    return $result;
  }

  my $indentString = " " x $indent;

  if ($self->{type} eq "link")
  {
    $result = "<a class=\"menu\" href=\"$self->{url}\"";
    if ($self->{onMouseOver})
    {
      $result .= " onMouseOver=\"$self->{onMouseOver}\"";
    }
    if ($self->{onClick})
    {
      $result .= " onClick=\"$self->{onClick}\"";
    }
    if ($self->{target})
    {
      $result .= " target=\"$self->{target}\"";
    }
    $result .= " title=\"$self->{title}\">$self->{text}</a>";
  }
  elsif ($self->{type} eq "seperator")
  {
    if ($orientation eq "horizontal")
    {
      $result = "&nbsp;&nbsp;|&nbsp;";
    }
    else
    {
      $result = "<hr>";
    }
  }
  elsif ($self->{type} eq "text")
  {
    $result = "$self->{text}";
  }
  elsif ($self->{type} eq "hilightedText")
  {
    $result = "<span class=\"menuHilight\">$self->{text}</span>";
  }
  elsif ($self->{type} eq "select")
  {
    $result = "<select name=\"$self->{name}\"";
    if ($self->{onChange})
    {
      $result .= " onChange=\"$self->{onChange}\"";
    }
    $result .= ">\n";
    foreach my $option (@{$self->{options}})
    {
      $result .= "$indentString  <option value=\"$option->{value}\">$option->{display}</option>\n";
    }
    $result .= "$indentString</select>\n";
  }

  return $result;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  string print(orientation, indent)
    requires: orientation, indent - # of spaces to indent.
    returns:  generated MenuItem string.
    generates the specified MenuItem without the wrapping <tr>,<td> tags.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Methods(3), Portal::Base(3)

=cut
