# SOAPResult.pm - The Object Class that provides a SOAPResult Object
# Created by James A. Pattie, 2004-07-06.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::SOAPResult;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

=head1 NAME

SOAPResult - Class used to build a SOAPResult object.

=head1 SYNOPSIS

  use Portal::Objects::SOAPResult;
  my $obj = Portal::Objects::SOAPResult->new(langObj => $langObj);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

SOAPResult is a error return class for the SOAP related code.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(result, message)

 Creates a new instance of the Portal::Objects::SOAPResult module.
 See Portal::Base(3) for a listing of required arguments.
 
 result - integer value (optional), defaults to 0.  Can be any
   value you need to denote a result code with.
 message - string (optional).  Allows you to pass back info to 
   the caller.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( result => 0, message => "", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{result} = $args{result};
  $self->{message} = $args{message};
  
  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{result} !~ /^(-?)(\d+)$/)
  {
    $self->invalid("result", $self->{invalid}, "This is signed numeric value.");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
