# User.pm - The Object Class that provides a User Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::User;
use strict;
use overload '==' => \&objsEqual,
             '!=' => \&objsNotEqual,
             '""' => \&printStr;
use Portal::Base;
use HTMLObject::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.06';

=head1 NAME

User - Object used to build a User Object Class.

=head1 SYNOPSIS

  use Portal::Objects::User;
  my $obj = Portal::Objects::User->new;
  if ($obj->error)
  {
    die $obj->errorMessage;
  }
  my $valid = $obj->populate(specify the parameters to this
    object);
  if (!$valid)
  {
    die $obj->errorMessage;
  }
  # else we are valid, so proceed to use the object.

=head1 DESCRIPTION

User is a User Object class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Objects::User object.

 You now have to use the populate() method to assign the User
 Object parameters and have them be validated.  new() just
 instantiates the object so that it can be used by the
 Portal::Forms::User module to create or edit users.

 The HTMLObject::Forms data structures will be defined for
 the following cases:

 adminCreate - create user screen (admins only)
 adminEdit - an admin (System or Company) edit screen
 userEdit  - the user edit screen (limited functionality)
 adminPassword  - password change screen (admins only)
 userPassword - user password change screen
 view      - user view screen (non-editable)

 The #CHILD_FORM_ARGS# and #CHILD_FORM_TABLE# tags in the
 templates allow derived objects to add items to the
 form template before the table definition and in the table
 before the <submit> button and after all our items are defined.

 The #CHILD_FORM_TABLE# has to be either removed or replaced
 with valid table definitions where each row contains 2 columns.

=cut
sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{variables}->error)
  {
    $self->error($self->{variables}->errorMessage);

    return $self;
  }

  # the templates
  my $createUserPhrase = $self->{langObj}->map("CreateUser");
  $self->{template}->{sysadminCreate} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">$createUserPhrase</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=uname#: #INVALID=uname#</td>
      <td align="left">#FIELD=uname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password1#: #INVALID=password1#</td>
      <td align="left">#FIELD=password1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password2#: #INVALID=password2#</td>
      <td align="left">#FIELD=password2#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=active#: #INVALID=active#</td>
      <td align="left">#FIELD=active#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=ssn#: #INVALID=ssn#</td>
      <td align="left">#FIELD=ssn#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=empId#: #INVALID=empId#</td>
      <td align="left">#FIELD=empId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=admin#: #INVALID=admin#</td>
      <td align="left">#FIELD=admin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=sysadmin#: #INVALID=sysadmin#</td>
      <td align="left">#FIELD=sysadmin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{adminCreate} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">$createUserPhrase</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=uname#: #INVALID=uname#</td>
      <td align="left">#FIELD=uname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password1#: #INVALID=password1#</td>
      <td align="left">#FIELD=password1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password2#: #INVALID=password2#</td>
      <td align="left">#FIELD=password2#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=active#: #INVALID=active#</td>
      <td align="left">#FIELD=active#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=ssn#: #INVALID=ssn#</td>
      <td align="left">#FIELD=ssn#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=empId#: #INVALID=empId#</td>
      <td align="left">#FIELD=empId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=admin#: #INVALID=admin#</td>
      <td align="left">#FIELD=admin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{sysadminEdit} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=active#: #INVALID=active#</td>
      <td align="left">#FIELD=active#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=ssn#: #INVALID=ssn#</td>
      <td align="left">#FIELD=ssn#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=empId#: #INVALID=empId#</td>
      <td align="left">#FIELD=empId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=admin#: #INVALID=admin#</td>
      <td align="left">#FIELD=admin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=sysadmin#: #INVALID=sysadmin#</td>
      <td align="left">#FIELD=sysadmin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{adminEdit} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=active#: #INVALID=active#</td>
      <td align="left">#FIELD=active#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=ssn#: #INVALID=ssn#</td>
      <td align="left">#FIELD=ssn#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=empId#: #INVALID=empId#</td>
      <td align="left">#FIELD=empId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=admin#: #INVALID=admin#</td>
      <td align="left">#FIELD=admin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{userEdit} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{adminPassword} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=password1#: #INVALID=password1#</td>
      <td align="left">#FIELD=password1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password2#: #INVALID=password2#</td>
      <td align="left">#FIELD=password2#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{userPassword} = <<"END_OF_FORM";
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=formSubmitted#
  #CHILD_FORM_ARGS#
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  <p style="text-align: center;">#FORMERRORSTRING#</p>
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=original#: #INVALID=original#</td>
      <td align="left">#FIELD=original#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password1#: #INVALID=password1#</td>
      <td align="left">#FIELD=password1#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=password2#: #INVALID=password2#</td>
      <td align="left">#FIELD=password2#</td>
    </tr>
    #CHILD_FORM_TABLE#
    <tr>
      <td colspan="2" align="center">#FIELD=submitForm#</td>
    </tr>
  </table>
</form>
END_OF_FORM
  $self->{template}->{view} = <<"END_OF_FORM";
  <p style="text-align: center;">#LABEL=uname#: #FIELD=uname#</p>
  #CHILD_FORM_ARGS#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right" valign="top">#LABEL=active#: #INVALID=active#</td>
      <td align="left">#FIELD=active#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=fname#: #INVALID=fname#</td>
      <td align="left">#FIELD=fname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=mname#: #INVALID=mname#</td>
      <td align="left">#FIELD=mname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lname#: #INVALID=lname#</td>
      <td align="left">#FIELD=lname#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=ssn#: #INVALID=ssn#</td>
      <td align="left">#FIELD=ssn#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=empId#: #INVALID=empId#</td>
      <td align="left">#FIELD=empId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=companyId#: #INVALID=companyId#</td>
      <td align="left">#FIELD=companyId#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=language#: #INVALID=language#</td>
      <td align="left">#FIELD=language#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=timeFormat#: #INVALID=timeFormat#</td>
      <td align="left">#FIELD=timeFormat#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=tz#: #INVALID=tz#</td>
      <td align="left">#FIELD=tz#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=admin#: #INVALID=admin#</td>
      <td align="left">#FIELD=admin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=sysadmin#: #INVALID=sysadmin#</td>
      <td align="left">#FIELD=sysadmin#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=email#: #INVALID=email#</td>
      <td align="left">#FIELD=email#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=comment#: #INVALID=comment#</td>
      <td align="left">#FIELD=comment#</td>
    </tr>
    <tr>
      <td align="right" valign="top">#LABEL=lastEdited#: #INVALID=lastEdited#</td>
      <td align="left">#FIELD=lastEdited#</td>
    </tr>
    #CHILD_FORM_TABLE#
  </table>
END_OF_FORM

  # the data structures
  $self->{data}->{sysadminCreate} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                   "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("CreateUser") },
                                   "app" => { -Type => "hidden", -Value => "" },
                                   "state" => { -Type => "hidden", -Value => "" },
                                   "command" => { -Type => "hidden", -Value => "createUser" },
                                   "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("uname"), size => 26, maxlength => 25 },
                                   "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                                   "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                                   "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                                   "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                                   "empId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("empId") },
                                   "password1" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("newPassword") },
                                   "password2" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("ReEnterPassword") },
                                   "ssn" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ssn"), size => 10, maxlength => 9 },
                                   "language" => { -Type => "select", -Value => "en", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                                   "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                                   "active" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("active"), -Options => $self->{variables}->formBoolean() },
                                   "admin" => { -Type => "select", -Value => "0", -Label => $self->{langObj}->map("admin"), -Options => $self->{variables}->formBoolean() },
                                   "sysadmin" => { -Type => "select", -Value => "0", -Label => $self->{langObj}->map("sysadmin"), -Options => $self->{variables}->formBoolean() },
                                   "tz" => { -Type => "select", -Value => "CST", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                                   "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                                 };
  $self->{data}->{adminCreate} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                   "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("CreateUser") },
                                   "app" => { -Type => "hidden", -Value => "" },
                                   "state" => { -Type => "hidden", -Value => "" },
                                   "command" => { -Type => "hidden", -Value => "createUser" },
                                   "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("uname"), size => 26, maxlength => 25 },
                                   "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                                   "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                                   "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                                   "password1" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("newPassword") },
                                   "password2" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("ReEnterPassword") },
                                   "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                                   "empId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("empId") },
                                   "ssn" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ssn"), size => 10, maxlength => 9 },
                                   "language" => { -Type => "select", -Value => "en", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                                   "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                                   "active" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("active"), -Options => $self->{variables}->formBoolean() },
                                   "admin" => { -Type => "select", -Value => "0", -Label => $self->{langObj}->map("admin"), -Options => $self->{variables}->formBoolean() },
                                   "tz" => { -Type => "select", -Value => "CST", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                                   "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                                 };
  $self->{data}->{sysadminEdit} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                 "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("EditUser") },
                                 "app" => { -Type => "hidden", -Value => "" },
                                 "state" => { -Type => "hidden", -Value => "" },
                                 "command" => { -Type => "hidden", -Value => "editUser" },
                                 "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("EditUser"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                                 "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                                 "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                                 "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                                 "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                                 "empId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("empId") },
                                 "ssn" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ssn"), size => 10, maxlength => 9 },
                                 "language" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                                 "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                                 "active" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("active"), -Options => $self->{variables}->formBoolean() },
                                 "admin" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("admin"), -Options => $self->{variables}->formBoolean() },
                                 "sysadmin" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("sysadmin"), -Options => $self->{variables}->formBoolean() },
                                 "tz" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                                 "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                               };
  $self->{data}->{adminEdit} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                 "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("EditUser") },
                                 "app" => { -Type => "hidden", -Value => "" },
                                 "state" => { -Type => "hidden", -Value => "" },
                                 "command" => { -Type => "hidden", -Value => "editUser" },
                                 "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("EditUser"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                                 "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                                 "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                                 "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                                 "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                                 "empId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("empId") },
                                 "ssn" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ssn"), size => 10, maxlength => 9 },
                                 "language" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                                 "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                                 "active" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("active"), -Options => $self->{variables}->formBoolean() },
                                 "admin" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("admin"), -Options => $self->{variables}->formBoolean() },
                                 "tz" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                                 "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                               };
  $self->{data}->{userEdit} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("EditUser") },
                                "app" => { -Type => "hidden", -Value => "" },
                                "state" => { -Type => "hidden", -Value => "" },
                                "command" => { -Type => "hidden", -Value => "editUser" },
                                "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("EditUser"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                                "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                                "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                                "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                                "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                                "language" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                                "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                                "tz" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                                "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                              };
  $self->{data}->{adminPassword} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                     "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("ChangePassword") },
                                     "app" => { -Type => "hidden", -Value => "" },
                                     "state" => { -Type => "hidden", -Value => "" },
                                     "command" => { -Type => "hidden", -Value => "changePassword" },
                                     "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ChangePasswordFor"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                                     "password1" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("newPassword") },
                                     "password2" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("ReEnterPassword") },
                                   };
  $self->{data}->{userPassword} = { "formSubmitted" => { -Type => "hidden", -Value => "1" },
                                    "submitForm" => { -Type => "submit", -Value => $self->{langObj}->map("ChangePassword") },
                                    "app" => { -Type => "hidden", -Value => "" },
                                    "state" => { -Type => "hidden", -Value => "" },
                                    "command" => { -Type => "hidden", -Value => "changePassword" },
                                    "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ChangePasswordFor"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                                    "original" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("originalPassword") },
                                    "password1" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("newPassword") },
                                    "password2" => { -Type => "password", -Value => "", -Label => $self->{langObj}->map("ReEnterPassword") },
                                  };
  $self->{data}->{view} = { "uname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ViewUserInfo"), size => 26, maxlength => 25, -ReadOnly => 1, -ReadOnlyMode => "text" },
                            "fname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("fname") },
                            "mname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("mname") },
                            "lname" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lname") },
                            "email" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("email") },
                            "empId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("empId") },
                            "ssn" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("ssn"), size => 10, maxlength => 9 },
                            "language" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("language"), -Options => $self->{variables}->formLanguage() },
                            "comment" => { -Type => "textarea", -Value => "", -Label => $self->{langObj}->map("comment") },
                            "active" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("active"), -Options => $self->{variables}->formBoolean() },
                            "admin" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("admin"), -Options => $self->{variables}->formBoolean() },
                            "sysadmin" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("sysadmin"), -Options => $self->{variables}->formBoolean() },
                            "companyId" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("companyId") },
                            "tz" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("tz"), -Options => $self->{variables}->formTZ() },
                            "timeFormat" => { -Type => "select", -Value => "", -Label => $self->{langObj}->map("timeFormat"), -Options => $self->{variables}->formTimeFormat() },
                            "lastEdited" => { -Type => "text", -Value => "", -Label => $self->{langObj}->map("lastEdited") },
                          };

  # the profiles
  $self->{profile}->{sysadminCreate} = { required => [ qw( formSubmitted app state command uname active fname lname ssn empId language
                                                        timeFormat tz admin sysadmin email password1 password2 ) ],
                                      optional => [ qw( mname comment ) ],
                                      constraints => { email => 'email' },
                                    };
  $self->{profile}->{adminCreate} = { required => [ qw( formSubmitted app state command uname active fname lname ssn empId language
                                                        timeFormat tz admin email password1 password2 ) ],
                                      optional => [ qw( mname comment ) ],
                                      constraints => { email => 'email' },
                                    };
  $self->{profile}->{sysadminEdit} = { required => [ qw( formSubmitted app state command uname active fname lname ssn empId language
                                                      timeFormat tz admin sysadmin email ) ],
                                    optional => [ qw( mname comment ) ],
                                    constraints => { email => 'email' },
                                  };
  $self->{profile}->{adminEdit} = { required => [ qw( formSubmitted app state command uname active fname lname ssn empId language
                                                      timeFormat tz admin email ) ],
                                    optional => [ qw( mname comment ) ],
                                    constraints => { email => 'email' },
                                  };
  $self->{profile}->{userEdit} = { required => [ qw( formSubmitted app state command uname fname lname language timeFormat tz
                                                     email ) ],
                                   optional => [ qw( mname comment ) ],
                                   constraints => { email => 'email' },
                                 };
  $self->{profile}->{adminPassword} = { required => [ qw( formSubmitted app state command uname password1 password2 ) ],
                                        optional => [],
                                        constraints => {},
                                      };
  $self->{profile}->{userPassword} = { required => [ qw( formSubmitted app state command uname original password1 password2 ) ],
                                       optional => [],
                                       constraints => {},
                                     };
  $self->{profile}->{view} = { required => [],
                               optional => [],
                               constraints => {},
                             };

  return $self;
}

=item bool populate(id, uname, fname, mname, lname, email, password, empId,
ssn, language, comment, active, admin, sysadmin, companyId, tz,
timeFormat, passCrypted, lastEdited)

 Populates the instance of the Portal::Objects::User object with the
 specified properties and validates them via the isValid() method.

 Returns the true/false result from isValid().

 requires:
   id
   uname
   fname
   mname
   lname
   email
   password
   empId
   ssn
   language
   comment
   active
   admin
   sysadmin
   companyId
   tz
   timeFormat
   passCrypted
   lastEdited

 passCrypted indicates whether or not the password is
 currently crypted or needs to be.

=cut

sub populate
{
  my $self = shift;
  my %args = ( id => -1, uname => "", fname => "", mname => "", lname => "", email => "", password => "", empId => "", ssn => "",
               language => "en", comment => "", active => 1, admin => 0, sysadmin => 0, passCrypted => 1, companyId => -1,
               tz => "CST", timeFormat => 12, lastEdited => "now()", @_ );

  # assign to $self and update the data profiles.
  foreach my $name (qw( id companyId uname fname lname mname password passCrypted empId ssn language comment active admin sysadmin tz timeFormat lastEdited))
  {
    $self->{$name} = $args{$name};
    foreach my $profile (keys %{$self->{data}})
    {
      $self->{data}->{$profile}->{$name}->{-Value} = $args{$name} if (exists $self->{data}->{$profile}->{$name});
    }
  }

  # handle the email field special.
  ($self->{email} = $args{email}) =~ s/\\@/@/;  # replace the escaped @ with a normal @.
  foreach my $profile (keys %{$self->{data}})
  {
    $self->{data}->{$profile}->{email}->{-Value} = $self->{email} if (exists $self->{data}->{$profile}->{email});
  }


  # do validation.  Allow ourselves to be derived from.
  return $self->Portal::Objects::User::isValid;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{companyId} < 0)
  {
    $self->invalid("companyId", "$self->{companyId}");
  }
  if (length $self->{uname} == 0)
  {
    $self->missing("uname");
  }
  if (length $self->{fname} == 0)
  {
    $self->missing("fname");
  }
  if (length $self->{lname} == 0)
  {
    $self->missing("lname");
  }
  if (length $self->{password} < 6)
  {
    $self->invalid("password", "$self->{password}", "passwords must be at least 6 characters long!");
  }
  if (length $self->{email} == 0)
  {
    $self->missing("email");
  }
  if ($self->{email} !~ /.+\@.+\..+/)
  {
    $self->invalid("email", "$self->{email}");
  }
  if ($self->{ssn} !~ /^(\d{9})$/)
  {
    $self->invalid("ssn", $self->{ssn}, "Your SSN must be 9 digits long!");
  }
  my $doc = HTMLObject::Base->new;
  if (length $self->{language} == 0)
  {
    $self->missing("language");
  }
  if (length $doc->lookupLanguageName(code => $self->{language}) == 0)
  {
    $self->invalid("language", "$self->{language}");
  }
  if ($self->{active} !~ /^(1|0)$/)
  {
    $self->invalid("active", "$self->{active}");
  }
  if ($self->{admin} !~ /^(1|0)$/)
  {
    $self->invalid("admin", "$self->{admin}");
  }
  if ($self->{sysadmin} !~ /^(1|0)$/)
  {
    $self->invalid("sysadmin", "$self->{sysadmin}");
  }
  if ($self->{sysadmin} && $self->{admin} == 0)
  {
    $self->invalid("sysadmin", $self->{sysadmin}, "It is invalid to be a sysadmin while not an admin!");
  }
  if (not exists $self->variables->timeZones->{$self->{tz}})
  {
    $self->invalid("tz", "$self->{tz}");
  }
  if (not exists $self->{variables}->{timeFormats}->{$self->{timeFormat}})
  {
    $self->invalid("timeFormat", "$self->{timeFormat}");
  }
  if (length $self->{lastEdited} == 0)
  {
    $self->invalid("lastEdited", "$self->{lastEdited}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool objsEqual(a,b)

 returns 1 if the 2 objects are the same (content wise),
 0 otherwise.

=cut
sub objsEqual
{
  my $a = shift;
  my $b = shift;

  if (ref($a) ne "Portal::Objects::User")
  {
    die "objsEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::User")
  {
    die "objsEqual: b = '" . ref($b) . "'";
  }

  # do number tests.
  foreach my $key (qw(id empId ssn active admin sysadmin companyId tz timeFormat))
  {
    return 0 if ($a->{$key} != $b->{$key});
  }

  # do string tests
  foreach my $key (qw(uname fname mname lname email password language comment lastEdited))
  {
    return 0 if ($a->{$key} ne $b->{$key});
  }

  return 1;  # assume they are equal if we get this far.
}

=item bool objsNotEqual(a,b)

 returns 0 if the 2 objects are the same (content wise),
 1 otherwise.

=cut
sub objsNotEqual
{
  my $a = shift;
  my $b = shift;

  if (ref($a) ne "Portal::Objects::User")
  {
    die "objsNotEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::User")
  {
    die "objsNotEqual: b = '" . ref($b) . "'";
  }

  # take advantage of the == overload to do the brunt of our work.
  return (!($a == $b));
}

=item scalar printStr()

 returns the printable version of this object.

=cut
sub printStr
{
  my $self = shift;
  my $out = "Portal::Objects::User : ";

  foreach my $keys (keys %{$self})
  {
    $out .= ", " if ($out =~ /=/);  # output a , if we have an = in the string.
    $out .= "$keys = '" . $self->{$keys} . "'";
  }

  $out .= "\n";

  return $out;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  id        - int
  companyId - int
  uname     - text
  fname     - text
  mname     - text
  lname     - text
  email     - text
  password  - text
  empId     - text
  ssn       - text (9 characters long)
  language  - text
  comment   - text
  active    - bool
  admin     - bool
  sysadmin  - bool
  tz        - text
  timeFormat- int (12 = AM/PM, 24 = Military Time)
  lastEdited- time stamp

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Auth(3), Portal::Base(3)

=cut
