# UserApplicationObject.pm - The Object Class that provides a UserApplicationObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::UserApplicationObject;
use strict;
use overload '==' => \&objsEqual,
             '!=' => \&objsNotEqual,
             '""' => \&printStr;
use Portal::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.07';

=head1 NAME

UserApplicationObject - Object used to build a UserApplicationObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::UserApplicationObject;
  my $obj = Portal::Objects::UserApplicationObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }
  my $valid = $obj->populate(specify the parameters to this
    object);
  if (!$valid)
  {
    die $obj->errorMessage;
  }
  # else we are valid, so proceed to use the object.

=head1 DESCRIPTION

UserApplicationObject is a UserApplicationObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Objects::UserApplicationObject object.
 See Portal::Base(3) for a listing of required arguments.
 
 You now have to use the populate() method to assign the User
 Object parameters and have them be validated.  new() just
 instantiates the object so that it can be used by the
 Portal::Forms::User module to create or edit users.
 
 The HTMLObject::Forms data structures will be defined for
 the following cases:
 
 edit - used to edit the UserApplicationObject settings.
 
 view - displays the UserApplicationObject object to let the 
  user edit the UserApp settings.  It is upto the calling
  application to replace the #EDITLINK# tag with the 
  corresponding user or app info.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( companyId => -1, appId => -1, userId => -1, height => "640", width => "480", autorun => 0, wap => 0, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{variables}->error)
  {
    $self->error($self->{variables}->errorMessage);

    return $self;
  }
  
  # the HTMLObject::Forms data structures
  
  $self->{template}->{edit} = <<"END_OF_CODE";
END_OF_CODE

  $self->{data}->{edit} = {};
  
  $self->{profile}->{edit} = { required => [ qw( userId appId app state command height width autorun wap ) ],
                               optional => [ ],
                               constraints => {},
                             };
                             
  $self->{template}->{view} = <<"END_OF_CODE";
  <tr class="#TRCLASS#">
    <td><span style="font-size: smaller;">#EDITLINK#</span></td>
    <td align="right"><span style="font-size: smaller;">#FIELD=autorun#</span></td>
    <td align="right"><span style="font-size: smaller;">#FIELD=wap#</span></td>
    <td align="right"><span style="font-size: smaller;">#FIELD=width#</span></td>
    <td align="right"><span style="font-size: smaller;">#FIELD=height#</span></td>
  </tr>
END_OF_CODE
  
  $self->{data}->{view} = { "autorun" => { -Type => "text", -Value => "", -Label => "" },
                            "wap" => { -Type => "text", -Value => "", -Label => "" },
                            "width" => { -Type => "text", -Value => "", -Label => "" },
                            "height" => { -Type => "text", -Value => "", -Label => "" }
                          };
  
  $self->{profile}->{view} = { required => [ ],
                               optional => [ ],
                               constraints => {},
                             };
  return $self;
}

=item scalar populate(companyId, appId, userId, height, width, autorun, wap)

 Populates the instance of the Portal::Objects::UserApplicationObject
 object with the specified properties and validates them via the 
 isValid() method.
 
 Returns the true/false result from isValid().

 requires:
   companyId
   appId
   userId
   height
   width
   autorun
   wap

=cut
sub populate
{
  my $self = shift;
  my %args = ( companyId => -1, appId => -1, userId => -1, height => "640", width => "480", autorun => 0, wap => 0, @_ );
  
  # assign to $self and update the data profiles.
  foreach my $name (qw(companyId appId userId height width autorun wap))
  {
    $self->{$name} = $args{$name};
    foreach my $profile (keys %{$self->{data}})
    {
      $self->{data}->{$profile}->{$name}->{-Value} = $args{$name} if (exists $self->{data}->{$profile}->{$name});
    }
  }

  # do validation
  return $self->isValid;
}
    
=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{companyId} < 0)
  {
    $self->invalid("companyId", "$self->{companyId}");
  }
  if ($self->{appId} < 0)
  {
    $self->invalid("appId", "$self->{appId}");
  }
  if ($self->{userId} < 0)
  {
    $self->invalid("userId", "$self->{userId}");
  }
  if ($self->{height} < 0 || $self->{height} > 4000)
  {
    $self->invalid("height", "$self->{height}");
  }
  if ($self->{width} < 0 || $self->{width} > 4000)
  {
    $self->invalid("width", "$self->{width}");
  }
  if ($self->{autorun} !~ /^(1|0)$/)
  {
    $self->invalid("autorun", "$self->{autorun}");
  }
  if ($self->{wap} !~ /^(1|0)$/)
  {
    $self->invalid("wap", "$self->{wap}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool objsEqual(a,b)

 returns 1 if the 2 objects are the same (content wise),
 0 otherwise.
 
=cut
sub objsEqual
{
  my $a = shift;
  my $b = shift;
  
  if (ref($a) ne "Portal::Objects::UserApplicationObject")
  {
    die "objsEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::UserApplicationObject")
  {
    die "objsEqual: b = '" . ref($b) . "'";
  }

  # do number tests.
  foreach my $key (qw(userId appId companyId width height autorun wap))
  {
    return 0 if ($a->{$key} != $b->{$key});
  }
  
  # do string tests
  #foreach my $key (qw())
  #{
  #  return 0 if ($a->{$key} ne $b->{$key});
  #}
  
  return 1;  # assume they are equal if we get this far.  
}

=item bool objsNotEqual(a,b)

 returns 0 if the 2 objects are the same (content wise),
 1 otherwise.
 
=cut
sub objsNotEqual
{
  my $a = shift;
  my $b = shift;
  
  if (ref($a) ne "Portal::Objects::UserApplicationObject")
  {
    die "objsNotEqual: a = '" . ref($a) . "'";
  }
  if (ref($b) ne "Portal::Objects::UserApplicationObject")
  {
    die "objsNotEqual: b = '" . ref($b) . "'";
  }

  # take advantage of the == overload to do the brunt of our work.
  return (!($a == $b));
}

=item scalar printStr()

 returns the printable version of this object.
 
=cut
sub printStr
{
  my $self = shift;
  my $out = "Portal::Objects::UserApplicationObject : ";
  
  foreach my $keys (keys %{$self})
  {
    $out .= ", " if ($out =~ /=/);  # output a , if we have an = in the string.
    $out .= "$keys = '" . $self->{$keys} . "'";
  }
  
  $out .= "\n";
  
  return $out;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  companyId int
  appId     int
  userId    int
  height    int
  width     int
  autorun   bool
  wap       bool

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Application(3), Portal::Base(3)

=cut
