# UserRightObject.pm - The Object Class that provides a UserRightObject Object
# Created by James A. Pattie, 11/07/2000.
# Last edited 07/26/2001.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::UserRightObject;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

=head1 NAME

UserRightObject - Object used to build a UserRight Object Class.

=head1 SYNOPSIS

  use Portal::Objects::UserRightObject;
  my $obj = Portal::Objects::UserRightObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

UserRightObject is a UserRight class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(counter, permission, description, admin, app, section,
dependsOn)

 Creates a new instance of the Portal::Objects::UserRightObject object.
 See Portal::Base(3) for a listing of required arguments.

 requires:
  counter is generated by the database.
  admin is a 1/0 flag that is there to help determine if certain
        rights should be granted to everyone or just admins.  It is up
        to each application in how they use this.
        
  permission is the string representing the right (read, write, etc.)
  description fully describes what this right is (Read access to Data)
  app is the name of the app that we are dealing with (Portal, Example,
  Accounting, ..)

  section describes the part of the application this permission is
         valid in.  This allows you to have the same permission string
         (read, write, etc.) in multiple sections of your application.
         Ex. Login, Desktop, Data Input, etc.
 
 optional:
  dependsOn is either -1 to indicate no dependancies or it points to
        the counter value of the right that it needs granted before it
        is valid for the user to have.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( counter => '-1', permission => "", description => "", app => "", section => "", admin => 0, dependsOn => "-1", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{counter} = $args{counter};
  $self->{permission} = $args{permission};
  $self->{description} = $args{description};
  $self->{app} = $args{app};
  $self->{section} = $args{section};
  $self->{admin} = $args{admin};
  $self->{dependsOn} = $args{dependsOn};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{counter} !~ /^(-1|\d+)$/)
  {
    $self->invalid("counter", "$self->{counter}");
  }
  if (length $self->{permission} == 0)
  {
    $self->invalid("permission", "$self->{permission}");
  }
  if (length $self->{description} == 0)
  {
    $self->invalid("description", "$self->{description}");
  }
  if ($self->{admin} !~ /^(1|0)$/)
  {
    $self->invalid("admin", "$self->{admin}");
  }
  if (length $self->{app} == 0)
  {
    $self->invalid("app", "$self->{app}");
  }
  if (length $self->{section} == 0)
  {
    $self->invalid("section", "$self->{section}");
  }
  if ($self->{dependsOn} !~ /^(-1|\d+)$/)
  {
    $self->invalid("dependsOn", "$self->{dependsOn}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# print
# requires:
# optional:
# returns: string that represents the contents of this object.
sub print
{
  my $self = shift;
  my $string = "counter = '$self->{counter}', permission = '$self->{permission}', description = '$self->{description}', app = '$self->{app}',";
  $string .= " section = '$self->{section}', admin = '$self->{admin}', dependsOn = '$self->{dependsOn}'";

  return $string;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  string print(void)
    Returns the string that represents the contents of this object.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Auth(3), Portal::Base(3)

=cut
