# Replication.pm - The Object Class that provides a Replication Object
# Created by James A. Pattie, 07/26/2001.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Replication;
use strict;
use Portal::Base;
use Portal::Objects::ReplicationObject;
use Portal::Log;
use Portal::Objects::LogEntry;
use DBIWrapper;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

=head1 NAME

Replication - Object used to build a Database Replication Object Class.

=head1 SYNOPSIS

  use Portal::Replication;
  my $obj = Portal::Replication->new(portalDB => $portalDB);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Replication is a Database Replication class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(portalDB, sessionObj)

 Creates a new instance of the Replication module.
 See Portal::Base(3) for a listing of required arguments.

 requires: portalDB - DBIWrapper pointing to the Portal Database
 optional: sessionObj - Portal Session

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( portalDB => undef, sessionObj => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{portalDB} = $args{portalDB};
  $self->{sessionObj} = $args{sessionObj};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{portalDB})
  {
    $self->missing("portalDB");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# createEntry
# requires: replicationObj
# optional: log - 1 means log event
# returns: 1 = OK, 0 = error
sub createEntry
{
  my $self = shift;
  my %args = ( replicationObj => undef, log => 1, @_ );
  my $replicationObj = $args{replicationObj};
  my $log = $args{log};

  if (not defined $replicationObj)
  {
    $self->error("replicationObj is not defined!<br>\n");
    return 0;
  }
  if (! $replicationObj->isValid)
  {
    $self->error($replicationObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure we don't already have a ReplicationObject entry that matches
  # this one.
  my @replicationObjs = $self->getEntries(tbName => $replicationObj->tbName, dbName => $replicationObj->dbName,
                                             dbHost => $replicationObj->dbHost);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  my $found = 0;
  foreach my $tmpObj (@replicationObjs)
  {
    if ($tmpObj->dbType eq $replicationObj->dbType && $tmpObj->dbPort eq $replicationObj->dbPort)
    {
      $found = 1;
      last;
    }
  }

  if ($found)
  {
    $self->error("Replication Entry already exists for " . $replicationObj->print . "<br>\n");
    return 0;
  }
  else
  {
    $self->{portalDB}->write(sql => "INSERT INTO replication_tb (tb_name, db_name, db_host, db_type, db_port) VALUES (?, ?, ?, ?, ?)",
                             plug => [ $replicationObj->tbName, $replicationObj->dbName, $replicationObj->dbHost, $replicationObj->dbType, $replicationObj->dbPort ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    $self->{portalDB}->commit;
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    if ($log)
    { # now log the Replication Create event.
      $self->doLog(db => $self->{portalDB}, action => 21, extraInfo => "Created Replication Entry:" . $replicationObj->print . "'");
      if ($self->error)
      {
        $self->prefixError();
        return 0;
      }
    }
  }

  return 1;
}

# deleteEntry
# requires: replicationObj
# optional: log - 1 means log event
# returns: 1 = OK, 0 = error
sub deleteEntry
{
  my $self = shift;
  my %args = ( replicationObj => undef, log => 1, @_ );
  my $replicationObj = $args{replicationObj};
  my $log = $args{log};

  if (not defined $replicationObj)
  {
    $self->error("replicationObj is not defined!<br>\n");
    return 0;
  }
  if (! $replicationObj->isValid)
  {
    $self->error($replicationObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure we have a ReplicationObject entry that matches this one.
  my @replicationObjs = $self->getEntries(tbName => $replicationObj->tbName, dbName => $replicationObj->dbName,
                                             dbHost => $replicationObj->dbHost);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  my $found = 0;
  foreach my $tmpObj (@replicationObjs)
  {
    if ($tmpObj->dbType eq $replicationObj->dbType && $tmpObj->dbPort eq $replicationObj->dbPort)
    {
      $found = 1;
      last;
    }
  }

  if (!$found)
  {
    $self->error("Replication Entry does not exist for " . $replicationObj->print . "<br>\n");
    return 0;
  }
  else
  {
    $self->{portalDB}->write(sql => "DELETE FROM replication_tb WHERE tb_name = ? AND db_name = ? AND db_host = ? AND db_type = ? AND db_port = ?",
                             plug => [ $replicationObj->tbName, $replicationObj->dbName, $replicationObj->dbHost, $replicationObj->dbType, $replicationObj->dbPort ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    $self->{portalDB}->commit;
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
    if ($log)
    { # now log the Replication Delete event.
      $self->doLog(db => $self->{portalDB}, action => 22, extraInfo => "Deleted Replication Entry:" . $replicationObj->print . "'");
      if ($self->error)
      {
        $self->prefixError();
        return 0;
      }
    }
  }

  return 1;
}

# migrateEntries
# requires: replicationObj
# optional: log
# returns: 1 = OK, 0 = Error
# This routine is used to initially populate a database with the contents of the
# table that we are being asked to replicate.
sub migrateEntries
{
  my $self = shift;
  my %args = ( replicationObj => undef, log => 1, @_ );
  my $replicationObj = $args{replicationObj};
  my $log = $args{log};

  if (not defined $replicationObj)
  {
    $self->error("replicationObj is not defined!<br>\n");
    return 0;
  }
  if (! $replicationObj->isValid)
  {
    $self->error($replicationObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # establish a connection to the destination database and make sure the table exists.
  my $destDB = DBIWrapper->new(dbType => $replicationObj->dbType, dbHost => $replicationObj->dbHost,
                               dbName => $replicationObj->dbName, dbUser => $replicationObj->{configObj}->dbUser,
                               dbPasswd => $replicationObj->{configObj}->dbPasswd, dbPort => $replicationObj->dbPort);
  if ($destDB->error)
  {
    $self->error($destDB->errorMessage);
    return 0;
  }

  # define the SELECT and INSERT strings that are unique to each table.
  my $selectSQL = "SELECT ";
  my $insertSQL = "INSERT INTO $replicationObj->{tbName} (";
  if ($replicationObj->tbName eq "company_tb")
  {
    $selectSQL .= "id, parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, time_format, tz, country, last_edited, url, logo_url, billing_method FROM company_tb ORDER BY id";
    $insertSQL .= "id, parent_id, code, name, address1, address2, city, state, zip, phone, fax, email, comment, language, active, time_format, tz, country, last_edited, url, logo_url, billing_method) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "user_tb")
  {
    $selectSQL .= "id, company_id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, time_format, tz, last_edited FROM user_tb ORDER BY id";
    $insertSQL .= "id, company_id, uname, fname, mname, lname, email, password, emp_id, ssn, language, comment, active, admin, sysadmin, time_format, tz, last_edited) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "app_tb")
  {
    $selectSQL .= "id, name, description, icon_name, server, port, db_host, db_type, db_port, cost, unit, type, admin_type, height, width, wap FROM app_tb ORDER BY id";
    $insertSQL .= "id, name, description, icon_name, server, port, db_host, db_type, db_port, cost, unit, type, admin_type, height, width, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "app_servers_tb")
  {
    $selectSQL .= "id, server, port, db_host, db_type, db_port, counter FROM app_servers_tb ORDER BY counter";
    $insertSQL .= "id, server, port, db_host, db_type, db_port, counter) VALUES (?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "company_app_tb")
  {
    $selectSQL .= "app_id, company_id, number, server, port, db_host, db_type, db_port, cost, unit, db_name, height, width, autorun, wap FROM company_app_tb ORDER BY app_id, company_id";
    $insertSQL .= "app_id, company_id, number, server, port, db_host, db_type, db_port, cost, unit, db_name, height, width, autorun, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "user_app_tb")
  {
    $selectSQL .= "company_id, app_id, user_id, height, width, autorun, wap FROM user_app_tb ORDER BY user_id, app_id";
    $insertSQL .= "company_id, app_id, user_id, height, width, autorun, wap) VALUES (?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "rights_tb")
  {
    $selectSQL .= "counter, permission, description, admin, app, section, depends_on FROM rights_tb ORDER BY counter";
    $insertSQL .= "counter, permission, description, admin, app, section, depends_on) VALUES (?, ?, ?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "user_rights_tb")
  {
    $selectSQL .= "id, counter FROM user_rights_tb ORDER BY id, counter";
    $insertSQL .= "id, counter) VALUES (?, ?)";
  }
  elsif ($replicationObj->tbName eq "user_preferences_tb")
  {
    $selectSQL .= "id, name, app, module, value FROM user_preferences_tb ORDER BY id, app, module, name";
    $insertSQL .= "id, name, app, module, value) VALUES (?, ?, ?, ?, ?)";
  }
  elsif ($replicationObj->tbName eq "dynamic_content_tb")
  {
    $selectSQL .= "index, calling_app, company_id, user_id, tag, app, arguments FROM dynamic_content_tb ORDER BY calling_app, tag, app, company_id, user_id, arguments";
    $insertSQL .= "index, calling_app, company_id, user_id, tag, app, arguments) VALUES (?, ?, ?, ?, ?, ?, ?)";
  }

  # now gather the data from the specified table.
  my $sth = $self->{portalDB}->read(sql => $selectSQL);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }

  # now insert the data into the destination database/table pair.
  while (my @info = $sth->fetchrow_array)
  {
    $destDB->write(sql => $insertSQL, plug => \@info);
    if ($destDB->error)
    {
      $self->error($destDB->errorMessage);
      return 0;
    }
  }

  if ($sth->rows > 0)
  {
    $destDB->commit;  # make sure the changes take effect on the destination database.
    if ($destDB->error)
    {
      $self->error($destDB->errorMessage);
      return 0;
    }
  }

  if ($log && $sth->rows > 0)
  { # log the Replication event
    $self->doLog(db => $self->{portalDB}, action => 23, extraInfo => "Migration of :" . $replicationObj->print . " occured.  Migrated ". $sth->rows . " rows.");
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
  }

  return 1;
}

# getEntries
# requires:
# optional: tbName, dbName, dbHost, createDBIObjects
# returns: array of ReplicationObject entries representing all items that matched.
# If createDBIObjects = 1 then the ReplicationObject will be instructed to
# create a DBIWrapper instance for it's contents (after validating them) so that
# the calling program will have the database object to work with.  createDBIObjects
# defaults to 0.
sub getEntries
{
  my $self = shift;
  my %args = ( tbName => "", dbName => "", dbHost => "", createDBIObjects => 0, @_ );
  my $tbName = $args{tbName};
  my $dbName = $args{dbName};
  my $dbHost = $args{dbHost};
  my $createDBIObjects = $args{createDBIObjects};
  my @result = ();

  if (length $tbName == 0 && length $dbName == 0 && length $dbHost == 0)
  {
    $self->error("You must specify at least one of tbName, dbName or dbHost!<br>\n");
    return @result;
  }

  if (length $tbName > 0 && $tbName !~ /^(company_tb|user_tb|app_tb|app_servers_tb|company_app_tb|user_app_tb|rights_tb|user_rights_tb|user_preferences_tb|dynamic_content_tb)$/)
  {
    $self->error("tbName = '$tbName' is invalid!<br>\n");
    return @result;
  }

  if ($createDBIObjects !~ /^(1|0)$/)
  {
    $self->error("createDBIObjects = '$createDBIObjects' is invalid!<br>\n");
    return @result;
  }

  # build the SQL statement to execute./
  my $sql = "SELECT tb_name, db_name, db_host, db_type, db_port FROM replication_tb WHERE ";
  my $whereClause = "";
  if (length $tbName > 0)
  {
    $whereClause .= "tb_name = '$tbName'";
  }
  if (length $dbName > 0)
  {
    $whereClause .= " AND " if (length $whereClause > 0);
    $whereClause .= "db_name = '$dbName'";
  }
  if (length $dbHost > 0)
  {
    $whereClause .= " AND " if (length $whereClause > 0);
    $whereClause .= "db_host = '$dbHost'";
  }

  $sql .= $whereClause . " ORDER BY tb_name, db_host, db_name";

  my $sth = $self->{portalDB}->read(sql => $sql);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return @result;
  }
  my $info = $sth->fetchall_arrayref;
  my @info = @{$info};
  foreach my $elementRef (@info)
  {
    my @element = @{$elementRef};
    my $replicationObj = Portal::Objects::ReplicationObject->new(tbName => $element[0], dbName => $element[1], dbHost => $element[2],
                         dbType => $element[3], dbPort => $element[4], createDBIObject => $createDBIObjects, langObj => $self->{langObj});
    if ($replicationObj->error)
    {
      $self->error($replicationObj->errorMessage);
      return @result;
    }
    $result[++$#result] = $replicationObj;
  }

  return @result;
}

# doLog
# requires: db, action
# optional: extraInfo
# returns: 1 = OK, 0 = error
# summary: This routine generates the LogEntry, LogObj and then inserts it.
sub doLog
{
  my $self = shift;
  my %args = ( db => undef, action => -1, extraInfo => "", @_ );
  my $db = $args{db};
  my $action = $args{action};
  my $extraInfo = $args{extraInfo};

  if ($action !~ /^(\d+)$/)
  {
    $self->error("action = '$action' is invalid!<br>\n");
    return 0;
  }
  if (not defined $db)
  {
    $self->error("db is not defined!<br>\n");
    return 0;
  }
  if (!$db->isValid)
  {
    $self->error($db->errorMessage);
    return 0;
  }

  my $logEntry = Portal::Objects::LogEntry->new(action => $action, ipAddress => (exists $ENV{REMOTE_ADDR} ? $ENV{REMOTE_ADDR} : '127.0.0.1'), userId => (defined $self->{sessionObj} ? $self->{sessionObj}->{store}->{userObj}->{id} : -1), extraInfo => $extraInfo, langObj => $self->{langObj});
  if ($logEntry->error)
  {
    $self->error($logEntry->errorMessage);
    return 0;
  }

  # now instantiate the Log object.
  my $logObj = Portal::Log->new(dbHandle => $db, langObj => $self->{langObj});
  if ($logObj->error)
  {
    $self->error($logObj->errorMessage);
    return 0;
  }

  # now log it
  $logObj->newEntry(logEntry => $logEntry);
  if ($logObj->error)
  {
    $self->error($logObj->errorMessage);
    return 0;
  }

  return 1;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  int createEntry(replicationObj)
    requires: replicationObj
    optional: log - 1 means log event
    returns: 1 = OK, 0 = error

  int deleteEntry(replicationObj)
    requires: replicationObj
    optional: log - 1 means log event
    returns: 1 = OK, 0 = error

  int migrateEntries(replicationObj, log)
    requires: replicationObj
    optional: log
    returns: 1 = OK, 0 = Error
    This routine is used to initially populate a database with the
    contents of the table that we are being asked to replicate.

  @ getEntries(tbName, dbName, dbHost, createDBIObjects)
    requires:
    optional: tbName, dbName, dbHost, createDBIObjects
    returns: array of ReplicationObject entries representing all items
             that matched.
    At least one of the optional entries must be specified.
    If createDBIObjects = 1 then the ReplicationObject will be
    instructed to create a DBIWrapper instance for it's contents (after
    validating them) so that the calling program will have the database
    object to work with.  createDBIObjects defaults to 0.

  int doLog(db, action, extraInfo)
    requires: db, action
    optional: extraInfo
    returns: 1 = OK, 0 = error
    summary: This routine generates the LogEntry, LogObj and then inserts
             it.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Objects::ReplicationObject(3), Portal::Base(3)

=cut
