# Session.pm - The Object Class that provides a Session Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2004 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Session;
use strict;
use Portal::Base;
use Portal::Methods;
use Portal::Data::Variables;
use DBIWrapper;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.12';

=head1 NAME

Session - Object used to build a Session Object Class.

=head1 SYNOPSIS

  use Portal::Session;
  my $obj = Portal::Session->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Session is a Session class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(dbHandle, dbType, cookieObj, appName, cookieDomain,
      cookieLife, sessionType, fileDir, lockDir, lockTimeout,
      sessionId, mode)

 Creates a new instance of the Session object.
 See Portal::Base(3) for a listing of required arguments.

 requires: sessionType - Database or File

 if sessionType = Database
  dbHandle - DBIWrapper object pointing to Portal Database
  dbType - database type (Postgres, MySQL)
 elsif sessionType = File
  fileDir - name of Directory to store sessions in
  lockDir - Directory to use for lock files
  lockTimeout - the number of seconds to allow lock files to stay around

 optional: cookieObj - HTMLObject::ReadCookie which has the
                       Compressed Cookie already decompressed in it.
           appName - name of Application cookie to check for Session ID in.
           cookieDomain - domain the normal cookie must exist in.
           cookieLife - specifies how long the cookie is to exist for in minutes.
           sessionId - The id generated for the session, if restoring a previous session.
           mode - 'read' or 'write'.  Specifies if the Apache::Session
             is to be read-only or writeable.  Defaults to 'read'.

  If sessionId specified then cookieObj and appName must not be
  specified.  This means we are using another Session to store this
  sessions Name -> ID in for use by the SessionHandler class.

  If the sessionId passed in via the cookie existed in the cookie but
  not in the data store, then the errorCode = -1, else for all other
  errors, errorCode = 0.  This will allow the calling application to decide
  what to do, either blow an error, manually clear the cookie, etc.
  If using sessionId instead of cookie, the sessionId is used without
  doing any look ups other than for existance, etc.

  The value changed in the store is reserved by the module.

  sessionId is now available if you don't like referencing session_id.

  When you want to indicate that an entry has changed in the store, that
  is deep, update the changed variable.  Your best bet is to change it
  to store the current time value.
  Ex: $sessionObj->{store}->{changed} += 1;

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( dbHandle => undef, dbType => "Postgres", cookieObj => undef, appName => "", cookieDomain => ".pcxperience.com", cookieLife => 30,
               sessionType => "File", fileDir => "", lockDir => "", lockTimeout => "3600", sessionId => "", mode => "read", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{errorCode} = 0;  # make an initial define so AUTOLOADING can work!
  $self->{sessionType} = $args{sessionType};
  $self->{fileDir} = $args{fileDir};
  $self->{lockDir} = $args{lockDir};
  $self->{dbHandle} = $args{dbHandle};
  $self->{dbType} = $args{dbType};
  $self->{cookieObj} = $args{cookieObj};
  $self->{appName} = $args{appName};
  $self->{cookieDomain} = $args{cookieDomain};
  $self->{cookieLife} = $args{cookieLife};
  $self->{lockTimeout} = $args{lockTimeout};
  $self->{sessionId} = $args{sessionId};
  $self->{useCookie} = (defined $self->{cookieObj} && length $self->{appName} > 0 && length $self->{sessionId} == 0 ? 1 : 0);
  $self->{mode} = $args{mode};
  $self->{methods} = Portal::Methods->new(langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $self;
  }
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});

  # do validation
  if (!$self->isValid(skip_store => 1))
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  # no error's yet, check for session cookies and instantiate the Apache::Session object.
  my $errorMessage = "Debug: useCookie = '$self->{useCookie}', sessionId = '$self->{sessionId}', appName = '$self->{appName}'";
  my $session_id = "";
  my $dbh = undef;
  if ($self->{sessionType} eq "Database")
  { # protect ourselves from when we call this without passing in the DBIWrapper handle (files only)
    $dbh = $self->{dbHandle}->dbh;
  }

  if ($self->{useCookie})
  {
    # look for the cookie
    if (exists $self->{cookieObj}->{cookies}->{$self->{appName}})
    {
      $session_id = $self->{cookieObj}->{cookies}->{$self->{appName}};
    }
  }
  else
  {
    $session_id = $self->{sessionId};
  }

  $errorMessage .= ", sessionId now = '$session_id', sessionType = '$self->{sessionType}'";

  if (length $session_id > 0)
  {
    if ($self->{sessionType} eq "Database")
    {
      # we have to validate the Session Id ourselves as Apache::Session doesn't do a good enough job!
      my $sth = $self->{dbHandle}->read(sql => "SELECT id FROM sessions WHERE id = '$session_id'");
      if ($self->{dbHandle}->error)
      {
        $self->error("Checking for Session ID = '$session_id' failed!<br>\n" . $self->{dbHandle}->errorMessage . $errorMessage);
        return $self;
      }
      # check the result from the database.
      my @result = $sth->fetchrow_array;
      if (scalar @result == 0)
      {
        # The session_id doesn't exist in the Sessions table!  Set a special error.
        $self->{errorCode} = -1;
        $self->error("sessionID = '$session_id' doesn't exist in the store. $errorMessage");
        return $self;
      }
    }
    elsif ($self->{sessionType} eq "File")
    {
      if (! -e "$self->{fileDir}/$session_id")
      {
        # The session_id doesn't exist in the directory!  Set a special error.
        $self->{errorCode} = -1;
        $self->error("sessionID = '$session_id' doesn't exist in the store.");
        return $self;
      }
    }
  }

  # this is needed so that we are working with the same session.
  $self->{sessionId} = $session_id;

  # now create the session.
  if (!$self->createSession())
  {
    $self->prefixError();
    return $self;
  }

  #$self->{store}->{changed} = 0;  # flag to indicate if the session has changed so that we can trigger the update in the backend.
  $self->{session_id} = $self->{store}->{_session_id};
  $self->{sessionId} = $self->{session_id};  # store the new, better named variable.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;
  my %args = ( @_ );
  my $skip_store = ( exists $args{skip_store} ? 1 : 0 );

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{mode} !~ /^(read|write)$/)
  {
    $self->invalid("mode", $self->{mode}, "valid values are: read, write");
  }
  if ($self->{sessionType} !~ /^(Database|File)$/)
  {
    $self->invalid("sessionType", $self->{sessionType});
  }
  if ($self->{sessionType} eq "Database")
  {
    if (not defined $self->{dbHandle})
    {
      $self->missing("dbHandle");
    }
    if ($self->{dbType} !~ /^(Postgres|MySQL)$/)
    {
      $self->invalid("dbType", $self->{dbType});
    }
  }
  elsif ($self->{sessionType} eq "File")
  {
    if (length $self->{fileDir} == 0)
    {
      $self->missing("fileDir");
    }
    elsif (! -d $self->{fileDir})
    {
      $self->invalid("fileDir", $self->{fileDir});
    }

    if (length $self->{lockDir} == 0)
    {
      $self->missing("lockDir");
    }
    elsif (! -d $self->{lockDir})
    {
      $self->invalid("lockDir", $self->{lockDir});
    }

    if ($self->{lockTimeout} < 300)
    {
      $self->{lockTimeout} = 300;  # allow lock files to stay around for 5 minutes (minimum).
    }
  }
  if ($self->{useCookie})
  {
    if (not defined $self->{cookieObj})
    {
      $self->missing("cookieObj");
    }
    if (length $self->{appName} == 0)
    {
      $self->missing("appName");
    }
  }
  else
  {
    # currently no validation checks for not working with cookies as it is valid to have an empty sessionId to start out with.
  }
  if (length $self->{cookieDomain} == 0)
  {
    $self->invalid("cookieDomain", $self->{cookieDomain});
  }
  if ($self->{cookieLife} < 30)
  {
    $self->{cookieLife} = 30;  # mandate a minimum of 30 minutes.
  }
  if (!$skip_store)
  {
    if (not defined $self->{store})
    {
      $self->invalid("store", "Session Store is no longer valid!");  ## LANG
    }
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool createSession()

Does the work of tieing store to the session.
Honors the mode to determine if we are read only
or writeable.

returns 1 on success, 0 on error.

=cut
sub createSession
{
  my $self = shift;
  my $errorMessage = "Debug: useCookie = '$self->{useCookie}', sessionId = '$self->{sessionId}', appName = '$self->{appName}', sessionType = '$self->{sessionType}'";
  my $dbh = undef;
  if ($self->{sessionType} eq "Database")
  { # protect ourselves from when we call this without passing in the DBIWrapper handle (files only)
    $dbh = $self->{dbHandle}->dbh;
  }
  my $session_id = $self->{sessionId};

  if (tied %{$self->{store}})
  {
    tied (%{$self->{store}})->save();  # force the data to be saved, so that it is synced to disk before the DESTROY handler happens.
    tied (%{$self->{store}})->release_all_locks();
    untie $self->{store};
    $self->{store} = undef;
  }

  if ($self->{sessionType} eq "Database")
  {
    # tie to the Apache::Session::dbType database.
    eval "use Apache::Session::$self->{dbType};";
    if ($@)
    {
      $self->error("Eval of 'use Apache::Session::$self->{dbType}' failed!<br>\nError = '$@'.<br>\n$errorMessage");
      return 0;
    }
    if ($self->{dbType} eq "Postgres")
    {
      eval { tie %{$self->{store}}, "Apache::Session::$self->{dbType}", $session_id, { Handle => $dbh, Commit => 1, Mode => $self->{mode}, DEBUG => 0 }; };
    }
    elsif ($self->{dbType} eq "MySQL")
    {
      eval { tie %{$self->{store}}, "Apache::Session::$self->{dbType}", $session_id, { Handle => $dbh, LockHandle => $dbh, Mode => $self->{mode} }; };
    }
    if ($@)
    {
      $self->error("Eval of tie failed!<br>\nError = '$@'.<br>\n$errorMessage");
      return 0;
    }
  }
  else
  {
    # tie to the File
    eval "use Apache::Session::File;";
    if ($@)
    {
      $self->error("Eval of 'use Apache::Session::File' failed!<br>\nError = '$@'.<br>\n");
      return 0;
    }

    eval { tie %{$self->{store}}, 'Apache::Session::File', $session_id, { Directory => $self->{fileDir}, LockDirectory => $self->{lockDir}, Mode => $self->{mode} }; };
    if ($@)
    {
      $self->error("Eval of tie failed!<br>\nError = '$@'.<br>\n");
      return 0;
    }

    # do any cleanup needed.
    eval "use Apache::Session::Lock::File;";
    if ($@)
    {
      $self->error("Eval of 'use Apache::Session::Lock::File' failed!<br>\nError = '$@'.<br>\n");
      return 0;
    }
    my $l = Apache::Session::Lock::File->new;
    $l->clean($self->{lockDir}, $self->{lockTimeout});
  }

  if ($self->{mode} eq "write")
  {
    $self->bump();  # make sure the session will be written to disk.
  }

  return 1;
}

=item scalar getMode()

returns the current value of mode.

=cut
sub getMode
{
  my $self = shift;
  return $self->{mode};
}

=item bool setMode(mode)

sets mode = to the specified input and then calls
createSession() to make the mode changes.

returns 0 on error, 1 on success.

Ex:  $sessionObj->setMode("write");

=cut
sub setMode
{
  my $self = shift;
  my $mode = shift;

  if ($mode !~ /^(read|write)$/)
  {
    $self->error("mode = '$mode' is invalid!");
    return 0;
  }

  if ($mode eq $self->{mode})
  {
    return 1;
  }

  $self->{mode} = $mode;
  return $self->createSession();
}

# delete - Removes the session from the session store
sub delete
{
  my $self = shift;

  if (defined $self->{store})
  {
    $self->setMode("write");
    tied (%{$self->{store}})->delete if (tied %{$self->{store}});

    $self->{store} = undef;
  }
}

=item scalar read(key)

returns the value from store->{key}.

Ex: my $name = $sessionObj->read("name");

=cut
sub read
{
  my $self = shift;
  my $key = shift;

  return $self->{store}->{$key};
}

=item scalar write(key, value)

If mode = 'write', will set store->{key} = value.
Returns the value specified or undef if an
error occured.  Check error() for details.

If mode = 'read', returns undef.

Ex: $sessionObj->write("name", "James");

=cut
sub write
{
  my $self = shift;
  my $key = shift;
  my $value = shift;

  if ($key eq "")
  {
    $self->error("You must specify the key to write to!");
    return undef;
  }

  return undef if ($self->{mode} eq "read");
  $self->{store}->{$key} = $value;

  return $value;
}

=item void bump()

bumps the value of changed up by 1 to make sure that any deep
entries are saved in the session.

=cut
sub bump
{
  my $self = shift;

  $self->{store}->{changed} += 1;
}

sub DESTROY
{
  my $self = shift;

  if (defined $self->{store})
  {
    untie $self->{store} if (tied %{$self->{store}});  # cleanly close down the connection to the Session.
  }
}

# doc writeCookie - generates the cookie based upon the application name.
# takes:  doc - a HTMLObject document (Base, Normal or FrameSet).
sub writeCookie
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};

  if (not ref($doc))
  {
    $self->error("doc not defined!<br>\n");
    return;
  }

  if (!defined $self->{store})
  {
    $self->error("store not defined!<br>\n");
    return;
  }

  if ($self->{useCookie})
  {
    # write the cookie into the specified document.
    # make the cookie expire X minutes from now in the future (GMT time).
    my $minutes = $self->{cookieLife};
    $doc->setCookie(name => $self->{appName}, value => $self->{session_id}, domain => $self->{cookieDomain}, expires => "$minutes minutes", path => "/");
    if ($doc->error)
    {
      $self->error($doc->getErrorMessage);
      return;
    }
  }
  $self->{store}->{changed} = 1;  # The application should set this is they need the session to be saved.

  return $doc;
}

=item bool registerWindow(name)

 (pass by value, not by name)

 requires:  name - the name of the window to register in the Portal.
 returns:  1 - window registerd, 0 - window already existed or an
   error occured.  Check ->error() to make sure.

 This method will register the specified window in the Portal via the
 portalSessionObj or in the Applications sessionObj.

 All sessions should have a windows hash defined in their store which
 allows the Portal to track the initial application window and any
 Help windows.  All other windows an application opens are registered
 in that applications session so the Portal knows which applications
 have what windows open.

 If the session is in read only mode and the window name does not
 exist, then we toggle to write mode and then back to read only mode.

 Ex:  $self->{sessionObj}->registerWindow("main_menu");

=cut
sub registerWindow
{
  my $self = shift;
  my $name = shift;

  if (length $name == 0)
  {
    $self->missing("name");
    $self->error($self->genErrorString("all"));
    return 0;
  }

  if (!exists $self->{store}->{windows})
  {
    $self->error("windows hash does not exist in your Session!");
    return 0;
  }
  if (!$self->windowExists($name))
  {
    $self->setMode("write");
    $self->{store}->{windows}->{$name} = 1;
    $self->{store}->{changed} += 1;
    $self->setMode("read");
    return 1;
  }
  return 0;  # it already has been defined!
}

=item bool unregisterWindow(name)

 (pass by value, not by name)

 requires:  name - the name of the window to unregister from the Portal.
 returns:  1 - window unregisterd, 0 - window didn't exist or an
   error occured.  Check ->error() to make sure.

 This method will unregister the specified window in the Portal via the
 portalSessionObj or in the Applications sessionObj.

 All sessions should have a windows hash defined in their store which
 allows the Portal to track the initial application window and any
 Help windows.  All other windows an application opens are registered
 in that applications session so the Portal knows which applications
 have what windows open.

 If the session is in read only mode and the window name does
 exist, then we toggle to write mode and then back to read only mode.

 Ex:  $self->{sessionObj}->unregisterWindow("main_menu");

=cut
sub unregisterWindow
{
  my $self = shift;
  my $name = shift;

  if (length $name == 0)
  {
    $self->missing("name");
    $self->error($self->genErrorString("all"));
    return 0;
  }

  if (!exists $self->{store}->{windows})
  {
    $self->error("windows hash does not exist in your Session!");
    return 0;
  }
  if ($self->windowExists($name))
  {
    $self->setMode("write");
    delete $self->{store}->{windows}->{$name};
    $self->{store}->{changed} += 1;
    $self->setMode("read");
    return 1;
  }
  return 0;  # it has not been defined!
}

=item bool windowExists(name)

 (pass by value, not by name)

 requires:  name - the name of the window to check for in the Portal.
 returns:  1 - window registerd, 0 - window didn't exist or an
   error occured.  Check ->error() to make sure.

 This method will check for the specified window in the Portal via the
 portalSessionObj or in the Applications sessionObj.

 All sessions should have a windows hash defined in their store which
 allows the Portal to track the initial application window and any
 Help windows.  All other windows an application opens are registered
 in that applications session so the Portal knows which applications
 have what windows open.

 Ex:
   if ($self->{sessionObj}->windowExists("main_menu"))
   {
     # do something regarding the fact the window has been registered.
   }

=cut
sub windowExists
{
  my $self = shift;
  my $name = shift;

  if (length $name == 0)
  {
    $self->missing("name");
    $self->error($self->genErrorString("all"));
    return 0;
  }

  if (!exists $self->{store}->{windows})
  {
    $self->error("windows hash does not exist in your Session!");
    return 0;
  }
  if (exists $self->{store}->{windows}->{$name})
  {
    return 1;
  }
  return 0;  # it has not been defined!
}

=item scalar getWindows()

  If the caller wants an array, we return the window names
  else, we return the number of windows registered in this session.

=cut
sub getWindows
{
  my $self = shift;

  if (!exists $self->{store}->{windows})
  {
    $self->error("windows hash does not exist in your Session!");
    return 0;
  }

  my @windows = keys %{$self->{store}->{windows}};

  if (wantarray)
  {
    return @windows;
  }
  else
  {
    return scalar @windows;
  }
}

=item scalar generateAppWindowName(app)

 (pass by value, not by name)

 requires:  app - the name of the application to create their main
   window name for.
 returns:  The window name or "" if an error occured.

 This method will build up the unique window name that should be
 used when opening an instance of the application in the Portal.
 The windowTitleTimestamp value is used to provide the uniqueness.

=cut
sub generateAppWindowName
{
  my $self = shift;
  my $app = shift;

  if (length $app == 0)
  {
    $self->missing("app");
    $self->error($self->genErrorString("all"));
    return "";
  }

  if (!exists $self->{store}->{windowTitleTimestamp})
  {
    $self->error("windowTitleTimestamp does not exist in your Session!");
    return "";
  }

  return $app . "_" . $self->{store}->{windowTitleTimestamp};
}

=item scalar generateHelpWindowName(app)

 (pass by value, not by name)

 requires:  app - the name of the application to create their Help
   window name for.
 returns:  The window name or "" if an error occured.

 This method will build up the Help window name that should be
 used when opening an instance of the application in the Portal.
 The windowTitleTimestamp value is used to provide the uniqueness.

=cut
sub generateHelpWindowName
{
  my $self = shift;
  my $app = shift;

  if (length $app == 0)
  {
    $self->missing("app");
    $self->error($self->genErrorString("all"));
    return "";
  }

  if (!exists $self->{store}->{windowTitleTimestamp})
  {
    $self->error("windowTitleTimestamp does not exist in your Session!");
    return "";
  }

  return "help_" . $app . "_" . $self->{store}->{windowTitleTimestamp};
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  void delete(void)
    Deletes the session from the session store.  Technically the Session
    Object is no longer usable.  The store should be undefined.

  void writeCookie(doc)
    writes the cookie into the specified HTMLObject document.

=head1 Exported VARIABLES

  store - the hash that is actually the session.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
