# SessionHandler.pm - The Object Class that provides a SessionHandler Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::SessionHandler;
use strict;
use Portal::Base;
use Portal::Session;
use Portal::Data::Variables;
use HTMLObject::ReadCookie;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.10';

=head1 NAME

SessionHandler - Object used to build a SessionHandler Object Class.

=head1 SYNOPSIS

  use Portal::SessionHandler;
  my $obj = Portal::SessionHandler->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

SessionHandler is a SessionHandler class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(dbHandle, cookieName, dbType, cookieDomain,
             invalidCookieBehaviour, sessionType, fileDir, lockDir,
             lockTimeout, sessionObj, methods)

 Creates a new instance of the SessionHandler object.
 See Portal::Base(3) for a listing of required arguments.

 requires: sessionType - Database or File
           invalidCookieBehaviour - 0 = Accept, 1 = Bomb out.
           methods - Portal::Methods object

 if sessionType = Database
  dbHandle - DBIWrapper object pointing to Portal Database
  dbType - database type (Postgres, MySQL)
 elsif sessionType = File
  fileDir - name of Directory to store sessions in
  lockDir - Directory to use for lock files
  lockTimeout - the number of seconds to allow lock files to stay around

 optional: cookieObj - HTMLObject::ReadCookie which has the
                       Compressed Cookie already decompressed in it.
           cookieDomain - domain the normal cookie must exist in.
           sessionObj - The Portal::Session object that is the parent
                        session that this group of sessions is managed by.
           cookieName - the compressed cookie to manage all sessions for.

  Either specify cookieName or sessionObj.  All the apps that the cookie or
  sessionObj knows about will now be available for manipulation.

  The value changed in the store is reserved by the module.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( dbHandle => undef, dbType => "Postgres", cookieName => "pcx_session_id", cookieDomain => ".pcxperience.com", cookieLife => 30,
               sessionType => "File", fileDir => "", lockDir => "", lockTimeout => "3600", sessionObj => undef, invalidCookieBehaviour => 1,
               configObj => undef, methods => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{sessionType} = $args{sessionType};
  $self->{dbHandle} = $args{dbHandle};
  $self->{fileDir} = $args{fileDir};
  $self->{lockDir} = $args{lockDir};
  $self->{lockTimeout} = $args{lockTimeout};
  $self->{cookieName} = $args{cookieName};
  $self->{cookieObj} = HTMLObject::ReadCookie->new;
  $self->{invalidCookieBehaviour} = $args{invalidCookieBehaviour};
  $self->{dbType} = $args{dbType};
  $self->{cookieDomain} = $args{cookieDomain};
  $self->{cookieLife} = $args{cookieLife};
  $self->{sessionObj} = $args{sessionObj};
  $self->{useCookie} = (defined $self->{sessionObj} ? 0 : 1);
  $self->{apps} = {};  # The hash of applications we are keeping sessions going for.
  $self->{configObj} = $args{configObj};
  $self->{methods} = $args{methods};
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  my $apps;
  if ($self->{useCookie})
  {
    $apps = $self->{cookieObj}->getCompressedCookies(cname => $self->{cookieName});
  }
  else
  {
    if (!exists $self->{sessionObj}->{store}->{apps})
    {
      $self->error("apps hash does not exist in parent Session!<br>\n");
      return $self;
    }
    $apps = $self->{sessionObj}->{store}->{apps};  # get list of apps to work with from the parent session.
  }
  foreach my $app (keys %{$apps})
  {
    $self->makeSession(name => $app);
    if ($self->error)
    {
      $self->prefixError();
      return $self;
    }
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;
  my %args = ( @_ );
  my $skip_store = ( exists $args{skip_store} ? 1 : 0 );

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{cookieName} == 0 && !defined $self->{sessionObj})
  {
    $self->missing("cookieName");
  }
  if (defined $self->{sessionObj})
  {
    if (!$self->{sessionObj}->isValid)
    {
      $self->invalid("sessionObj", $self->{sessionObj}->errorMessage);
    }
  }
  if ($self->{sessionType} !~ /^(Database|File)$/)
  {
    $self->invalid("sessionType", $self->{sessionType});
  }
  if ($self->{sessionType} eq "Database")
  {
    if (not defined $self->{dbHandle})
    {
      $self->missing("dbHandle");
    }
    if ($self->{dbType} !~ /^(Postgres|MySQL)$/)
    {
      $self->invalid("dbType", $self->{dbType});
    }
  }
  elsif ($self->{sessionType} eq "File")
  {
    if (length $self->{fileDir} == 0)
    {
      $self->missing("fileDir");
    }
    elsif (! -d $self->{fileDir})
    {
      $self->invalid("fileDir", $self->{fileDir});
    }

    if (length $self->{lockDir} == 0)
    {
      $self->missing("lockDir");
    }
    elsif (! -d $self->{lockDir})
    {
      $self->invalid("lockDir", $self->{lockDir});
    }

    if ($self->{lockTimeout} < 300)
    {
      $self->{lockTimeout} = 300;  # allow lock files to stay around for 5 minutes (minimum).
    }
  }
  if (not defined $self->{cookieObj})
  {
    $self->missing("cookieObj");
  }
  if ($self->{invalidCookieBehaviour} !~ /^(0|1)$/)
  {
    $self->invalid("invalidCookieBehaviour", $self->{invalidCookieBehaviour});
  }
  if ($self->{cookieLife} < 30)
  {
    $self->{cookieLife} = 30;  # make sure we have some positive value at least 30 minutes in length.
  }
  if (length $self->{cookieDomain} == 0)
  {
    $self->invalid("cookieDomain", $self->{cookieDomain});
  }
  if (not defined $self->{configObj})
  {
    $self->missing("configObj");
  }
  if (not defined $self->{methods})
  {
    $self->missing("methods");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# makeSession - creates a Portal::Session object based upon the given app name and cookie object or sessionObj
# and sets it in the apps hash.  This should be used by new() and whenever an application needs to start a new
# session and wants to keep track of it transparently.
# takes:  name - App name to work with.
sub makeSession
{
  my $self = shift;
  my %args = (name => "Portal", mode => "read", @_ );
  my $name = $args{name};
  my $mode = $args{mode};

  # make sure we don't already have this App in our care.
  if (exists $self->{apps}->{$name})
  {
    $self->error("App = '$name' already exists!<br>\n");
    return;
  }

RETRY:
  my $appObj;
  if ($self->{useCookie})
  {
    $appObj = $self->{methods}->portalSession(portalDB => $self->{dbHandle}, cookieObj => $self->{cookieObj},
                                              appName => $name, configObj => $self->{configObj}, mode => $mode);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return;
    }
  }
  else
  {
    my $sessionId = $self->{sessionObj}->{store}->{apps}->{$name};
    $appObj = $self->{methods}->portalSession(portalDB => $self->{dbHandle}, sessionId => $sessionId,
                                              configObj => $self->{configObj}, mode => $mode);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return;
    }
  }
  if ($appObj->error)
  {
    if ($appObj->errorCode == -1)
    {
      # There was a cookie for the App but the session wasn't valid!  What do we do?
      if ($self->{invalidCookieBehaviour})
      {
        # bomb out!
        $self->error("App = '$name', Error = '" . $appObj->errorMessage . "'.<br>\n");

        return;
      }
      else
      {
        # delete the cookie for the app and try again.
        if ($self->{useCookie})
        {
          delete($self->{cookieObj}->{cookies}->{$name});
        }
        else
        {  # remove the session info from the parent session.
          $self->{sessionObj}->setMode("write");
          delete($self->{sessionObj}->{store}->{apps}->{$name});
          $self->{sessionObj}->setMode("read");
        }
        goto RETRY;
      }
    }
    else
    {
      $self->error("App = '$name', Error = '" . $appObj->errorMessage . "'.<br>\n");

      return;
    }
  }
  else
  {
    $self->{apps}->{$name} = $appObj;  # assign this Session Object to the App entry.
  }
  if (!$self->{useCookie})
  {
    # we need to keep track of this session in the parent session object
    if (!exists $self->{sessionObj}->{store}->{apps}->{$name})
    {
      if (!$self->{sessionObj}->setMode("write"))
      {
        $self->error($self->{sessionObj}->errorMessage);
        return;
      }
      $self->{sessionObj}->{store}->{apps}->{$name} = $appObj->sessionId;
      if (!$self->{sessionObj}->setMode("read"))
      {
        $self->error($self->{sessionObj}->errorMessage);
        return;
      }
      if (!exists $self->{sessionObj}->{store}->{apps}->{$name})
      {
        $self->error("App = '$name' was not stored in the Portal Session!");
        return;
      }
    }
  }
}

# doc writeCookie - generates the compressed cookie based upon the applications that are being kept track of.
# takes:  doc - a HTMLObject document (Base, Normal or FrameSet).
sub writeCookie
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};

  if (not ref($doc))
  {
    $self->error("doc not defined!<br>\n");
    return;
  }

  if ($self->{useCookie})
  {
    my @cookies = ();

    foreach my $app (keys %{$self->{apps}})
    {
      if (defined $self->{apps}->{$app}->{store})  # only add them to the cookie if their session is still valid.
      {
        my @temp_cookie = ();
        $temp_cookie[0] = $app;
        $temp_cookie[1] = $self->{apps}->{$app}->session_id;

        $cookies[++$#cookies] = [ @temp_cookie ];
      }
    }

    # write the compressed cookie into the specified document.
    # make the cookie expire X minutes from now in the future (GMT time).
    my $minutes = $self->{cookieLife};
    $doc->setCompressedCookie(name => $self->{cookieName}, cookies => \@cookies, domain => $self->{cookieDomain}, expires => "$minutes minutes", path => "/");
  }
  else
  {
    # have the parent session write the cookie for itself as that is all we need to have done.
    $doc = $self->{sessionObj}->writeCookie(doc => $doc);
    if ($self->{sessionObj}->error)
    {
      $self->error($self->{sessionObj}->errorMessage);
      return;
    }

    # cleanup any sessions that are no longer valid.
    foreach my $app (keys %{$self->{apps}})
    {
      if (!defined $self->{apps}->{$app}->{store})
      {
        $self->{sessionObj}->setMode("write");
        delete($self->{sessionObj}->{store}->{apps}->{$app});  # remove it from the parent Session Object.
      }
    }
    $self->{sessionObj}->setMode("read");
  }

  return $doc;
}

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  void makeSession(name)
    Creates an instance of Portal::Session and stores it in the apps
    hash if nothing goes wrong.  This is affected by the value of
    invalidCookieBehaviour.
        name - The app to start a Session for.

  void writeCookie(doc)
    Generates the compressed cookie for all managed apps in the
    specified HTMLObject document (Base, Normal or FrameSet).
        doc - HTMLObject document.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Session(3), Portal::Base(3)

=cut
