# Who.pm - The Object Class that provides a Who Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Who;
use strict;
use Portal::Base;
use Portal::Objects::WhoEntryObject;
use Portal::Data::Variables;
use Portal::Data::Config;
use Portal::Session;
use Portal::Methods;
use Digest::MD5;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.12';

=head1 NAME

Who - Object used to build a Who Object Class.

=head1 SYNOPSIS

  use Portal::Who;
  my $obj = Portal::Who->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Who is a Who class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(portalDB, configObj, methods)

 Creates a new instance of the Who module.
 See Portal::Base(3) for a listing of required arguments.
 
 requires: portalDB - DBIWrapper pointing to the Portal Database.
 
=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( portalDB => undef, configObj => undef, methods => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{portalDB} = $args{portalDB};
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{configObj} = $args{configObj};
  $self->{methods} = $args{methods};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{portalDB})
  {
    $self->missing("portalDB");
  }
  if (not defined $self->{configObj})
  {
    $self->missing("configObj");
  }
  if (not defined $self->{methods})
  {
    $self->missing("methods");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# swapDate
# takes: date
# returns: date
# takes a date in format MM/DD/YYYY and returns YYYY/MM/DD or vice versa
sub swapDate
{
  my $self = shift;
  my %args = ( date => "", @_ );
  my $date = $args{date};

  if (length $date == 0)
  {
    $date = "01/01/2000"; # default date.
  }
  if ($date =~ /^\d{2}.\d{2}.\d{4}$/)  # MM/DD/YYYY
  {
    $date =~ s/^(\d{2})(.)(\d{2})(.)(\d{4})$/$5$2$1$4$3/;
  }
  else # YYYY/MM/DD
  {
    $date =~ s/^(\d{4})(.)(\d{2})(.)(\d{2})$/$3$2$5$4$1/;
  }

  return $date;
}

# makeWhoEntry
# takes: whoEntryObj - WhoEntryObject
# returns: 1=OK, 0=Error, -1=Already exists
# This routine makes an entry in the who_tb table in the portal_db database that lets us know when the user logged into the portal,
# what company they are with, where they are coming from and what Portal Session they are.
sub makeWhoEntry
{
  my $self = shift;
  my %args = ( whoEntryObj => undef, @_ );
  my $whoEntryObj = $args{whoEntryObj};

  if (!defined $whoEntryObj)
  {
    $self->error("whoEntryObj must be defined!<br>\n");
    return 0;
  }
  if (!$whoEntryObj->isValid)
  {
    $self->error($whoEntryObj->errorMessage);
    return 0;
  }

  # make sure this entry doesn't already exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT uname, company_id FROM who_tb WHERE session = ? ORDER BY company_id, uname",
                                    plug => [ $whoEntryObj->{session} ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    return -1;  # entry already exists.
  }

  # make the entry
  my $rc = $self->{portalDB}->write(sql => "INSERT INTO who_tb (uname, company_id, session, ip_address, start_date, refresh_date) VALUES (?, ?, ?, ?, now(), now())",
                                    plug => [ $whoEntryObj->{uname}, $whoEntryObj->{companyId}, $whoEntryObj->{session}, $whoEntryObj->{ipAddress} ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  $self->{portalDB}->commit;  # make it permanent.

  return 1;
}

# deleteWhoEntry
# takes: session
# optional: removeSession = 0
# returns: 1=OK, 0=Error, -1=Doesn't exist
# This routine removes the entry from the who_tb that details the user using the specified Portal Session.
# It also removes the session from the sessions table and any application sessions that belong to that
# session.
sub deleteWhoEntry
{
  my $self = shift;
  my %args = ( session => "", removeSession => 0, @_ );
  my $session = $args{session};
  my $removeSession = $args{removeSession};

  if (length $session == 0)
  {
    $self->error("session must be specified!<br>\n");
    return 0;
  }
  if ($removeSession !~ /^(1|0)$/)
  {
    $self->error("removeSession = '$removeSession' is invalid!<br>\n");
    return 0;
  }

  # make sure this entry does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT uname, company_id FROM who_tb WHERE session = ? ORDER BY company_id, uname",
                                    plug => [ $session ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0])
  {
    return -1;  # entry doesn't exist.
  }

  # delete the entry
  my $rc = $self->{portalDB}->write(sql => "DELETE FROM who_tb WHERE session = ?",
                                    plug => [ $session ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }

  if ($removeSession)
  {
    # instantiate the Session
    my $sessionObj = $self->{methods}->portalSession(portalDB => $self->{portalDB}, configObj => $self->{configObj}, sessionId => $session);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return 0;
    }
    if ($sessionObj->error)
    {
      $self->error($sessionObj->errorMessage);
      return 0;
    }

    # Now see if we are handling any other sessions, and if so, delete them
    foreach my $app (keys %{$sessionObj->{store}->{apps}})
    {
      $self->{portalDB}->write(sql => "DELETE FROM sessions WHERE id = ?",
                              plug => [ $sessionObj->{store}->{apps}->{$app} ]);
      if ($self->{portalDB}->error)
      {
        $self->error($self->{portalDB}->errorMessage);
        return 0;
      }
    }

    # now delete the initial session from the sessions table.
    $self->{portalDB}->write(sql => "DELETE FROM sessions WHERE id = ?",
                                      plug => [ $session ]);
    if ($self->{portalDB}->error)
    {
      $self->error($self->{portalDB}->errorMessage);
      return 0;
    }
  }

  # Finally make all deletions permanent.
  $self->{portalDB}->commit;

  return 1;
}

# updateWhoEntry
# takes: session
# returns: 1=OK, 0=Error, -1=doesn't exist
# This routine updates the refresh_date field in the who_tb for the specified session to the current timestamp.
sub updateWhoEntry
{
  my $self = shift;
  my %args = ( session => "", @_ );
  my $session = $args{session};

  if (length $session == 0)
  {
    $self->error("session must be specified!<br>\n");
    return 0;
  }

  # make sure this entry does exist.
  my $sth = $self->{portalDB}->read(sql => "SELECT uname, company_id FROM who_tb WHERE session = ? ORDER BY company_id, uname",
                                    plug => [ $session ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0])
  {
    return -1;  # entry doesn't exist.
  }

  # update the entry
  my $rc = $self->{portalDB}->write(sql => "UPDATE who_tb SET refresh_date = now() WHERE session = ?",
                                    plug => [ $session ]);
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  $self->{portalDB}->commit;  # make it permanent.

  return 1;
}

# getWhoEntries
# takes: uname, companyId, ipAddress, startDate, useInterval
# returns: Array of Portal::Objects::WhoEntryObject that details all entries that matched.
# This routine optionally takes uname, companyId, startDate, and/or IPAddress values to search for in the who_tb.
# If no values are specified, then all entries are returned.  If useInterval = 1 then generate code to
# find all entries that are cookieLife+15 minutes or more stale.  startDate must not be defined for useInterval to work.
sub getWhoEntries
{
  my $self = shift;
  my %args = ( uname => "", companyId => "", ipAddress => "", startDate => "", useInterval => 0, @_ );
  my $uname = $args{uname};
  my $companyId = $args{companyId};
  my $ipAddress = $args{ipAddress};
  my $startDate = $args{startDate};
  my $useInterval = $args{useInterval};
  my @Entries = ();

  if (length $companyId > 0 && $companyId < 0)
  {
    $self->error("companyId = '$companyId' is invalid!<br>\n");
    return @Entries;
  }
  if (length $ipAddress > 0 && $ipAddress !~ /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
  {
    $self->error("ipAddress = '$ipAddress' is invalid!<br>\n");
    return @Entries;
  }
  if (length $startDate > 0 && $startDate !~ /^\d{2}(-|\/)\d{2}(-|\/)\d{4} \d{2}:\d{2}:\d{2}$/)
  {
    $self->error("startDate = '$startDate' is invalid!<br>\n");
    return @Entries;
  }
  if ($useInterval !~ /^(1|0)$/)
  {
    $self->error("useInterval = '$useInterval' is invalid!<br>\n");
    return @Entries;
  }

  my $whereStr = "";
  if (length $uname > 0)
  {
    $whereStr .= "WHERE uname = '$uname'";
  }
  if (length $companyId > 0)
  {
    $whereStr .= " AND " if (length $whereStr > 0);
    $whereStr .= "WHERE " if (length $whereStr == 0);
    $whereStr .= "company_id = '$companyId'";
  }
  if (length $ipAddress > 0)
  {
    $whereStr .= " AND " if (length $whereStr > 0);
    $whereStr .= "WHERE " if (length $whereStr == 0);
    $whereStr .= "ip_address = '$ipAddress'";
  }
  if (length $startDate > 0)
  {
    $whereStr .= " AND " if (length $whereStr > 0);
    $whereStr .= "WHERE " if (length $whereStr == 0);
    $whereStr .= "start_date = '$startDate'";
  }
  elsif ($useInterval)
  {
    $whereStr .= " AND " if (length $whereStr > 0);
    $whereStr .= "WHERE " if (length $whereStr == 0);
    my $minutes = $self->{configObj}->{cookieLife} + 15;
    $whereStr .= "refresh_date < " . ($self->{configObj}->{dbType} eq "Postgres" ? "CAST(now() - CAST('$minutes minutes' AS INTERVAL) AS TIMESTAMP)" : "now() - INTERVAL $minutes MINUTE");
  }

  # find all entries that match.
  my $sth = $self->{portalDB}->read(sql => "SELECT uname, company_id, session, ip_address, start_date, refresh_date FROM who_tb $whereStr ORDER BY company_id, uname, start_date, refresh_date, ip_address");
  if ($self->{portalDB}->error)
  {
    $self->error($self->{portalDB}->errorMessage);
    return 0;
  }
  while (my @info = $sth->fetchrow_array)
  {
    (my $ipAddress = $info[3]) =~ s/^(.*)\/(\d+)$/$1/;
    (my $dbStartDate = $info[4]) =~ s/(:\d+)(\.\d+(-\d+)?)/$1/;
    my $startDate = $self->{methods}->formatDateString(date => "$dbStartDate", format => "%F %X");
    (my $dbRefreshDate = $info[5]) =~ s/(:\d+)(\.\d+(-\d+)?)/$1/;
    my $refreshDate = $self->{methods}->formatDateString(date => "$dbRefreshDate", format => "%F %X");
    my $whoEntry = Portal::Objects::WhoEntryObject->new(uname => $info[0], companyId => $info[1], session => $info[2], ipAddress => $ipAddress, startDate => $startDate, refreshDate => $refreshDate, langObj => $self->{langObj});
    if ($whoEntry->error)
    {
      $self->error($whoEntry->errorMessage);
      return @Entries;
    }
    $Entries[++$#Entries] = $whoEntry;
  }

  return @Entries;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  scalar swapDate(date)
    takes a date in format MM/DD/YYYY and returns YYYY/MM/DD or vice
    versa

  int makeWhoEntry(whoEntryObj)
    returns: 1=OK, 0=Error, -1=Already exists
    This routine makes an entry in the who_tb table in the portal_db
    database that lets us know when the user logged into the portal,
    what company they are with, where they are coming from and what
    Portal Session they are.

  int deleteWhoEntry(session, removeSession)
    returns: 1=OK, 0=Error, -1=Doesn't exist
    This routine removes the entry from the who_tb that details the
    user using the specified Portal Session.  It also removes the session
    from the sessions table and any application sessions that belong to
    that session.  If you specify removeSession => 1 then the
    session and all it's children sessions will be removed.

  int updateWhoEntry(session)
    returns: 1=OK, 0=Error, -1=doesn't exist
    This routine updates the refresh_date field in the who_tb for the
    specified session to the current timestamp.

  WhoEntryObject[] getWhoEntries(uname, companyId, ipAddress,
                                 startDate, useInterval)
    returns: Array of Portal::Objects::WhoEntryObject that details all
             entries that matched.
    This routine optionally takes uname, companyId, startDate, and/or
    IPAddress values to search for in the who_tb.  If no values are
    specified, then all entries are returned.  If useInterval = 1
    then generate code to find all entries that are cookieLife+15
    minutes or more stale.  startDate must not be defined for
    useInterval to work.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
