# ConfigParser.pm - Will parse an XML file and return the DBI result set.
# Created by James A. Pattie.  Copyright (c) 2002-2003, Xperience, Inc.

package Portal::XML::ConfigParser;

use strict;
use XML::LibXML;
use Portal::Base;
use Portal::XML::ConfigObject;
use vars qw ($AUTOLOAD @ISA @EXPORT $VERSION);

require Exporter;
@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = "1.1";

=head1 NAME

ConfigParser - The XML Configuration Parser Module.

=head1 SYNOPSIS

  use Portal::XML::ConfigParser;
  my $obj = Portal::XML::ConfigParser->new;
  my $configObj = $obj->parse(file => "config.xml");
  # this is a ConfigObject object.

=head1 DESCRIPTION

ConfigParser will parse XML files that have been generated to the config
specification.  See the Portal::XML::ConfigObject man page for the
structure of the returned data.

=head1 FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(void)

 Creates a new instance of the Portal::XML::ConfigParser module.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );
  my $errStr = "Portal::XML::ConfigParser->new$self->{errorPhrase}";

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{configObj} = undef;
  $self->{module} = "";
  $self->{version} = "";

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  eval { $self->{xmlObj} = XML::LibXML->new(); };
  if ($@)
  {
    die "$errStr $@\n";
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# configObj parse(file, string, module, version)
# requires:  file - xml file to work with, or string - xml data
#           in a string to work with.
#           module - The Portal App the config file is for
#           version - The version of the config file required
# returns: Portal::XML::ConfigObject instance with parsed info.
sub parse
{
  my $self = shift;
  my %args = ( "file" => "", string => "", module => "", version => "", @_ );
  my $nodes = undef;
  my $errStr = "Portal::XML::ConfigParser->parse$self->{errorPhrase}";

  if (!defined $self->{xmlObj})
  {
    die "$errStr xmlObj is undefined!\n";
  }

  if (length $args{file} > 0)
  {
    if ($args{file} !~ /^(.*\.xml)$/)
    {
      die "$errStr file = '$args{file}' is not a valid file!\n";
    }
    if (! -e $args{file})
    {
      die "$errStr Can not find config file = '$args{file}'!  $!\n";
    }
    if (! -r $args{file})
    {
      die "$errStr config file = '$args{file}' is not readable!  $!\n";
    }
  }
  elsif (length $args{string} == 0)
  {
    die "$errStr You must specify either 'file' or 'string'!\n";
  }
  if (length $args{module} == 0)
  {
    die "$errStr You must specify the module this config file is for!\n";
  }
  if ($args{version} !~ /^(\d+\.\d+)$/)
  {
    die "$errStr version = '$args{version}' is invalid!\n";
  }

  $self->{module} = $args{module};
  $self->{version} = $args{version};

  $self->{dataFile} = (length $args{string} > 0 ? $args{string} : $args{file});

  if (length $args{file} > 0 && length $args{string} == 0)
  {
    eval { $self->{xmlDoc} = $self->{xmlObj}->parse_file($self->{dataFile}); };
  }
  else
  {
    eval { $self->{xmlDoc} = $self->{xmlObj}->parse_string($self->{dataFile}); };
  }
  if ($@)
  {
    die "$errStr $@\n";
  }

  # get the module
  my $module = $self->getModule;

  if ($module =~ /^($self->{module})$/)
  {
    eval { $self->{configObj} = Portal::XML::ConfigObject->new(langObj => $self->{langObj}); };
    if ($@)
    {
      die "$errStr $@\n";
    }
    if ($self->{configObj}->error)
    {
      die "$errStr" . $self->{configObj}->errorMessage;
    }

    # start by validating the version of the XML file.
    $self->validateVersion;

    # initiate the data structure.  Fill in any default values possible.
    $self->{configObj}->{version} = $self->{version};
    $self->{configObj}->{module} = $module;

    # gather the <setting> values
    $self->getSettings;
  }
  else
  {
    die "$errStr  Unknown config module = '$module'!\n";
  }

  return $self->{configObj};
}

# hash getAttributes(node)
# requires: node - XPath Node
# returns:  hash of attributes for the specified node.
sub getAttributes
{
  my $self = shift;
  my %args = ( node => undef, @_ );
  my $node = $args{node};
  my %attributes = ();
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if (!defined $node)
  {
    die "$errStr  You must specify the XPath Node to work with!\n";
  }
  if ($node->getType() != XML_ELEMENT_NODE)
  {
    die "$errStr  You did not specify an XPath Node: " . $node->getType() . "\n";
  }
  foreach my $attribute ($node->getAttributes)
  {
    my $name = $attribute->getName;
    $attributes{$name} = $attribute->getValue;
  }

  return %attributes;
}

# array getNodes(path, context)
# required: path - XPath to search for
# optional: context - the XPath object to base the search from.  Make sure your path is relative to it!
# returns:  array - array of nodes returned.  These are the XPath objects representing each node.
sub getNodes
{
  my $self = shift;
  my %args = ( path => "*", context => undef, @_ );
  my $path = $args{path};
  my $context = $args{context};
  my @nodes = ( );
  my $nodes = undef;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if (length $path == 0)
  {
    die "$errStr  You must specify a path!\n";
  }

  if (! defined $context)
  {
    $nodes = $self->{xmlDoc}->findnodes($path);
  }
  else
  {
    $nodes = $context->findnodes($path);
  }
  if (!$nodes->isa('XML::LibXML::NodeList'))
  {
    die "$errStr  Query '$path' didn't return a nodelist: " . $nodes->getType() . "\n";
  }
  if ($nodes->size)
  {
    #print "Found " . $nodes->size . " nodes...\n";
    foreach my $node ($nodes->get_nodelist)
    {
      push @nodes, $node;
    }
  }

  return @nodes;
}

# string getModule(void)
# returns the module value from the parent <config> tag.
sub getModule
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my @nodes = $self->getNodes(path => "/config");
  if (scalar @nodes == 0)
  {
    die "$errStr  Your XML file doesn't contain a <config> tag!\n";
  }
  if (scalar @nodes > 1)
  {
    die "$errStr  You have too many <config> tags!  You should only have one!\n";
  }
  my %attributes = $self->getAttributes(node => $nodes[0]);
  if (!exists $attributes{module})
  {
    die "$errStr  You do not have the module defined!\n";
  }

  return $attributes{module};
}

# string getVersion(void)
# returns the version value from the parent <config> tag.
sub getVersion
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my @nodes = $self->getNodes(path => "/config");
  if (scalar @nodes == 0)
  {
    die "$errStr  Your XML file doesn't contain a <config> tag!\n";
  }
  if (scalar @nodes > 1)
  {
    die "$errStr  You have too many <config> tags!  You should only have one!\n";
  }
  my %attributes = $self->getAttributes(node => $nodes[0]);
  if (!exists $attributes{version})
  {
    die "$errStr  You do not have the version defined!\n";
  }

  return $attributes{version};
}

# This routine looks up the <config version=""> tag and validates that the
# version specified is the same as what we know how to work with.
sub validateVersion
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my $version = $self->getVersion;
  if ($version !~ /^($self->{version})$/)
  {
    die "$errStr  '$version' is not equal to Version '$self->{version}'!\n";
  }
}

# void getSettings(void)
# requires: nothing
# returns: nothing
sub getSettings
{
  my $self = shift;
  my %args = ( @_ );
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";
  my $tag = "setting";

  my @nodes = $self->getNodes(path => "/config/$tag");
  if (scalar @nodes == 0)
  {
    die "$errStr  You do not have a <$tag> tag for <config module='$self->{module}'>!\n";
  }
  foreach my $node (@nodes)
  {
    # gather all attributes of the <setting> tag.
    my %attributes = $self->getAttributes(node => $node);
    my %encountered = ();
    foreach my $attribute (keys %attributes)
    {
      if (exists $encountered{$attribute})
      {
        die "$errStr  You have already defined '$attribute' in the <$tag> tag!'\n";
      }
      if ($attribute !~ /^(name|value)$/)
      {
        die "$errStr  '$attribute' is invalid in the <$tag> tag!'\n";
      }
      $encountered{$attribute} = 1;
      if ($attribute =~ /^(value)$/ && $attributes{$attribute} !~ /^(.*)$/)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid!  <$tag>, name='$attributes{name}'\n";
      }
      if ($attribute =~ /^(name)$/ && $attributes{name} !~ /^(.+)$/)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid!  <$tag>\n";
      }
    }
    foreach my $required ("name", "value")
    {
      if (!exists $encountered{$required})
      {
        die "$errStr  '$required' is required in the <$tag> tag!\n";
      }
    }

    # make sure it doesn't already exist.
    if (exists $self->{configObj}->{settings}->{$attributes{name}})
    {
      die "$errStr  name = '$attributes{name}' already exists with value = '$self->{configObj}->{settings}->{$attributes{name}}'!\n";
    }

    # create the item object and store it.
    $self->{configObj}->{settings}->{$attributes{name}} = $attributes{value};
  }
}

=back

=cut

1;
__END__

=head1 FUNCTIONS (non-Inline POD)

  Portal::XML::ConfigObject parse(file, string)
    Does the actual parsing of the XML file and generates the
    resulting data object and returns it.

    file points to the XML Config file to use.

    If you don't specify a file to work with then you must specify the
    xml via the string argument.  If you specify both, then the string
    will take precedence.  The file must still point to a valid file.

  string getVersion(void)
    returns the version value from the parent <config> tag.

  string getModule(void)
    returns the module value from the parent <config> tag.

=head1 VARIABLES

  configObj - ConfigObject object that represents the xml file.

  module - The Portal App name the user specified to parse().

  version - The version of the config file the user specified to parse().

  xmlObj - The XML::LibXML object being used to parse the XML File.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin at pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::XML::ConfigObject(3), Portal::Base(3)

=cut
