# DBSettings.pm - Will pull DB settings from an XML config file into a format usable by the system.
# Created by James A. Pattie.  Copyright (c) 2002-2005, Xperience, Inc.

package Portal::XML::DBSettings;

use strict;
use XML::LibXML;
use Portal::Base;
use vars qw ($AUTOLOAD @ISA @EXPORT $VERSION);

require Exporter;
@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = "1.0";

=head1 NAME

DBSettings - The XML Portal DBSettings Module.

=head1 SYNOPSIS

  use Portal::XML::DBSettings;
  my $obj = Portal::XML::DBSettings->new(langObj => $langObj);

=head1 DESCRIPTION

Portal::XML::DBSettings will contain the parsed XML file.
It provides a method to validate that it is complete and also a method
to generate a valid XML file from the data stored in the data hash.

=head1 FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(void)
 Creates a new instance of the Portal::XML::DBSettings object.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{version} = "1.0";
  $self->{module} = "";
  $self->{sections} = {};
  # each defined section is a hash with it's own groupings (add, update, delete) which contain the
  # array of hashes that make up the actual data for that section to work with.

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;
  my %sectionArgs = (
    add => {
      config => qr/^(name|value)$/,
      event => qr/^(id|label)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app|arguments)$/,
      rights => qr/^(permission|description|admin|app|section|dependsOn)$/
    },
    delete => {
      config => qr/^(name)$/,
      event => qr/^(id)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app)$/,
      rights => qr/^(permission|app|section)$/
    },
    update => {
      config => qr/^(name|value|newValue)$/,
      event => qr/^(id|label|newValue)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app|arguments|newValue)$/,
      rights => qr/^(permission|description|admin|app|section|newValue|dependsOn)$/
    }
  );
  my %requiredArgs = (
    add => {
      config => [qw(name value)],
      event => [qw(id label)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments)],
      rights => [qw(permission description admin app section)]
    },
    delete => {
      config => [qw(name)],
      event => [qw(id)],
      'dynamic-content' => [qw(callingApp companyId userId tag app)],
      rights => [qw(permission app section)]
    },
    update => {
      config => [qw(name value newValue)],
      event => [qw(id label newValue)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments newValue)],
      rights => [qw(permission description admin app section newValue)]
    }
  );

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{version} !~ /^(\d+\.\d+)$/)
  {
    $self->invalid("version", "$self->{version}");
  }
  if (length $self->{module} == 0)
  {
    $self->missing("module");
  }
  if (ref ($self->{sections}) ne "HASH")
  {
    $self->invalid("sections", ref ($self->{sections}), "Is not a HASH ref!");
  }
  elsif (scalar keys %{$self->{sections}} == 0)
  {
    $self->missing("sections");
  }
  else
  {
    # validate sections
    foreach my $section (keys %{$self->{sections}})
    {
      if (ref ($self->{sections}->{$section}) ne "HASH")
      {
        $self->invalid("section", $section, "Is not a HASH ref!");
      }
      else
      {
        # make sure the section is one we support.
        if ($section !~ /^(config|event|dynamic-content|rights)$/)
        {
          $self->invalid("section", $section, "Valid sections are: config, event, dynamic-content, rights!");
        }
        else
        {
          # make sure we have the add, update, delete groupings.
          foreach my $group (keys %{$self->{sections}->{$section}})
          {
            if ($group !~ /^(add|update|delete)$/)
            {
              $self->invalid("group", $group, "section = '$section'.  Valid groups are: add, update, delete!");
            }
            elsif (ref ($self->{sections}->{$section}->{$group}) ne "ARRAY")
            {
              $self->invalid("group", $group, "section = '$section'.  Is not an ARRAY ref!");
            }
            # make sure the group has some entries defined.
            elsif (scalar @{$self->{sections}->{$section}->{$group}} == 0)
            {
              $self->invalid("group", $group, "section = '$section'.  You must define entries for this group!");
            }
            else
            {
              # make sure the entries appear to be valid.
              for (my $entryIndex = 0; $entryIndex < scalar @{$self->{sections}->{$section}->{$group}}; $entryIndex++)
              {
                my $entry = $self->{sections}->{$section}->{$group}->[$entryIndex];
                if (ref ($entry) ne "HASH")
                {
                  $self->invalid("entry", $entryIndex, "section = '$section', group = '$group'.  Is not a HASH ref!");
                }
                else
                {
                  my %found = ();
                  # now do the checks for the different section types.
                  foreach my $name (keys %{$entry})
                  {
                    if ($name !~ $sectionArgs{$group}{$section})
                    {
                      $self->invalid("key", $name, "section = '$section', group = '$group', entry = '$entryIndex'.  Valid keys are: " . join(", ", @{$requiredArgs{$group}{$section}}) . "!");
                    }
                    else
                    {
                      if ($section eq "dynamic-content" && $name =~ /^(companyId|userId)$/ && $entry->{$name} != -1)
                      {
                        $self->invalid($name, $entry->{$name}, "section = '$section', group = '$group', entry = '$entryIndex'.  -1 is the only valid value for now!");
                      }
                      if ($section eq "rights")
                      {
                        if ($name =~ /^(admin)$/ && $entry->{$name} !~ /^(0|1)$/)
                        {
                          $self->invalid($name, $entry->{$name}, "section = '$section', group = '$group', entry = '$entryIndex'.  This is a boolean value.");
                        }
                        if ($name eq "dependsOn")
                        {
                          if (ref ($entry->{$name}) ne "HASH")
                          {
                            $self->invalid($name, $entry->{$name}, "section = '$section', group = '$group', entry = '$entryIndex'.  Is not a HASH ref!");
                          }
                          else
                          {
                            # make sure that the app, section, and permission attributes are defined.
                            my %subFound = ();
                            foreach my $subName (keys %{$entry->{$name}})
                            {
                              if ($subName !~ /^(app|section|permission)$/)
                              {
                                $self->invalid("sub-key", $subName, "section = '$section', group = '$group', entry = '$entryIndex', key = '$name'.  Valid keys are: permission, app, section!");
                              }
                              else
                              {
                                $subFound{$subName} = 1;
                              }
                            }
                            foreach my $subName (qw(app section permission))
                            {
                              if (!exists $subFound{$subName})
                              {
                                $self->missing($subName, "section = '$section', group = '$group', entry = '$entryIndex', key = '$name'.");
                              }
                            }
                          }
                        }
                      }
                      $found{$name} = 1;
                    }
                  }
                  foreach my $name (@{$requiredArgs{$group}{$section}})
                  {
                    if (!exists $found{$name})
                    {
                      $self->missing($name, "section = '$section', group = '$group', entry = '$entryIndex'.");
                    }
                  }
                }
              }
            }
          }
          if (scalar keys %{$self->{sections}->{$section}} == 0)
          {
            $self->invalid("section", $section, "You must define at least one group!");
          }
        }
      }
    }
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

sub generateXML
{
  my $self = shift;
  my $result = "";
  my %requiredArgs = (
    add => {
      config => [qw(name value)],
      event => [qw(id label)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments)],
      rights => [qw(permission description admin app section)]
    },
    delete => {
      config => [qw(name value)],
      event => [qw(id label)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments)],
      rights => [qw(permission description admin app section)]
    },
    update => {
      config => [qw(name value newValue)],
      event => [qw(id label newValue)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments newValue)],
      rights => [qw(permission description admin app section newValue)]
    }
  );

  if ($self->isValid())
  {
    $result .= <<"END_OF_XML";
<?xml version="1.0" encoding="ISO-8859-1"?>
<dbSettings module="$self->{module}" version="$self->{version}">
END_OF_XML
    foreach my $section (sort keys %{$self->{sections}})
    {
      $result .= qq{  <section name="$section">\n};
      foreach my $group (sort keys %{$self->{sections}->{$section}})
      {
        $result .= qq{    <group name="$group">\n};
        foreach my $entry (@{$self->{sections}->{$section}->{$group}})
        {
          $result .= qq{      <entry};
          foreach my $name (@{$requiredArgs{$group}{$section}})
          {
            my $value = $self->encodeEntities(string => $entry->{$name});
            $value =~ s/(\\\@)/\@/g;
            $result .= qq{ $name="$value"};
          }
          if ($section eq "rights")
          {
            if (exists $entry->{dependsOn})
            {
              $result .= qq{>\n};
              $result .= qq{        <dependsOn};
              foreach my $name (qw(app section permission))
              {
                my $value = $self->encodeEntities(string => $entry->{dependsOn}->{$name});
                $value =~ s/(\\\@)/\@/g;
                $result .= qq{ $name="$value"};
              }
              $result .= qq{ />\n};
              $result .= qq{      </entry>\n};
            }
            else
            {
              $result .= qq{ />\n};
            }
          }
          else
          {
            $result .= qq{ />\n};
          }
        }
        $result .= qq{    </group>\n};
      }
      $result .= qq{  </section>\n};
    }
    $result .= <<"END_OF_XML";
</dbSettings>
END_OF_XML
  }
  else
  {
    $result .= "Data not valid!\n\n";
    $result .= $self->errorMessage;
    die $result;
  }

  return $result;
}

# string encodeEntities(string)
# requires: string - string to encode
# optional:
# returns: string that has been encoded.
# summary: replaces all special characters with their XML entity equivalent. " => &quot;
sub encodeEntities
{
  my $self = shift;
  my %args = ( string => "", @_ );
  my $string = $args{string};

  my @entities = ('&', '"', '<', '>', '\n');
  my %entities = ('&' => '&amp;', '"' => '&quot;', '<' => '&lt;', '>' => '&gt;', '\n' => '\\n');

  return $string if (length $string == 0);

  foreach my $entity (@entities)
  {
    $string =~ s/$entity/$entities{$entity}/g;
  }

  return $string;
}

=back

=cut

1;
__END__

=head1 Exported Functions (non-Inline POD)

  string generateXML(void)
    Creates an XML file based upon the info stored in the
    Portal::XML::DBSettings object.  It first calls isValid() to make sure
    this is possible.  If not then we die with an informative error
    message.

=head1 VARIABLES

  version - version of the XML file parsed

  module - the module this config data is for

  sections - the hash that represents the different database tables
    that the dbSettings xml file allows you to manipulate.
    Possible values are:
      config
      event
      dynamic-content
      rights

    Each entry is a hash that contains the following entries used
    for grouping the database modifications:
      add
      update
      delete

    Each group is an array of hashes that specify the actual entries
    to be added, updated or deleted.  The processing order in the
    updateDBSettings.pl script is to add, then update, then delete.
    It is upto the developer to make sure that you do not add an entry
    and then turn around and delete it.

    The layouts for the different sections data entries are as follows:

    config - name, value, newValue (only when group = update)
    event - id, label, newValue (only when group = update)
    dynamic-content - callingApp, companyId, userId, tag, app,
      arguments, newValue (only when group = update)
    rights - permission, description, admin, app, section, dependsOn,
      newValue (only when group = update).
      dependsOn is a hash that contains:
        app, section, permission

    The newValue entry, when group = update, allows the config, event,
    dynamic-content or rights entry to be updated when the current entry
    matches to have the following attribute updated to be equal to
    newValue.
      config - value = newValue
      event - label = newValue
      dynamic-content - arguments = newValue
      rights - description = newValue

    If the need arises to change more than 1 attribute for the
    dynamic-content or rights entries, the xml config and data structure
    will be updated to handle as appropriate.

  An example of the dbSettings xml schema is:

<?xml version="1.0" encoding="ISO-8859-1"?>
<dbSettings module="Accounting" version="1.0">
  <section name="config">
    <group name="add">
      <entry name="Accounting_Global_VerticalMenuLocation_Default" value="left" />
      <entry name="test" value="something" />
    </group>
  </section>
  <section name="rights">
    <group name="add">
      <entry permission="read" description="Let the user read this thingy." admin="0" app="Accounting" section="Main">
        <dependsOn app="Accounting" section="Main" permission="display" />
      </entry>
      <entry permission="write" description="Let the user write this thingy." admin="1" app="Accounting" section="Main">
        <dependsOn app="Accounting" section="Main" permission="display" />
      </entry>
    </group>
  </section>
  <section name="dynamic-content">
    <group name="add">
      <entry callingApp="Accounting" companyId="-1" userId="-1" tag="ledger-summary" app="Accounting" arguments="state=Main&amp;command=display" />
    </group>
  </section>
  <section name="event">
    <group name="update">
      <entry id="30" label="Monitor Account Job" newValue="Monitor Job Account" />
    </group>
  </section>
</dbSettings>

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin at pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::XML::ConfigParser(3), Portal::Base(3)

=cut
