# Portal.pm - The Object Class that provides the Portal Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal;
use Portal::Args;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Args Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.4.00';

# global variables.
my %states = ( "Login" => "login module", "Desktop" => "desktop module", "Help" => "help module" );

=head1 NAME

Portal - Object used to build a Portal Class.

=head1 SYNOPSIS

  use Portal;
  my $obj = Portal->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Portal is the main class for the Portal.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal object.
 See Portal::Args(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = (  @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject run(state, command)

 Pulls the specified state file into existance and calls
 run(command).

 returns: The HTMLObject document to display or undef if an error.

=cut

sub run
{
  my $self = shift;
  my %args = ( state => "", command => "", @_ );
  my $state = $args{state};
  my $command = $args{command};
  my $doc = undef;

  if (length $state == 0)
  {
    my $string = $self->{langObj}->map("stateNeeded");
    $self->error(sprintf($string));
    return $doc;
  }
  if (length $command == 0)
  {
    my $string = $self->langObj->map("commandNeeded");
    $self->error(sprintf($string));
    return $doc;
  }

  # validate the state is known.
  if (!exists $states{$state})
  {
    my $string = $self->langObj->map("stateNotKnown");
    $self->error(sprintf($string, $self->formEncodeString($state)));
    return $doc;
  }

  my $stateName = "Portal::$state";
  eval "use $stateName;";
  if ($@)
  {
    my $string = $self->langObj->map("stateEvalFailed");
    $self->error(sprintf($string, $self->formEncodeString($stateName), $@));
    return $doc;
  }

  # make a logObj
  my $logObj = Portal::Log->new(dbHandle => $self->{portalDB}, langObj => $self->{langObj});
  if ($logObj->error)
  {
    $self->error($logObj->errorMessage);
    return $doc;
  }

  my $stateObj;
  my $arguments = $self->arguments();
  eval "\$stateObj = $stateName->new($arguments, logObj => \$logObj);";
  if ($@)
  {
    my $string = $self->langObj->map("stateEvalFailed");
    $self->error(sprintf($string, $self->formEncodeString($stateName) . " Object", $@));
    return $doc;
  }
  if ($stateObj->error)
  {
    $self->error($stateObj->errorMessage . "\n<br>arguments='$arguments'");
    return $doc;
  }

  eval {
  $doc = $stateObj->run(command => $command);
  };
  if ($@)
  {
    my $string = $self->langObj->map("stateEvalFailed");
    $self->error(sprintf($string, $self->formEncodeString($stateName) . " Run", $@));
    return $doc;
  }
  if ($stateObj->error)
  {
    $self->error($stateObj->errorMessage);
    return $doc;
  }

  return $doc;  # signal all ok.
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), HTMLObject(3), DBIWrapper(3), Portal::Args(3)

=cut
