# App.pm - The Object Class that provides a App Base Object
# Created by James A. Pattie, 03/26/2003.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::App;
use Portal::Base;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

=head1 NAME

App - Object used to build a base Application Object Class.

=head1 SYNOPSIS

 package Portal::Test;
 use Portal::App;
 use strict;
 use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

 require Exporter;

 @ISA = qw(Portal::App Exporter AutoLoader);
 @EXPORT = qw();

 $VERSION = '0.01';

 # global variables
 my %states = ( Main => "Main Menu", Test => "Something about Tests", );

 sub new
 {
   my $class = shift;
   # logOk = 1 - you must have a log_tb defined in your database.
   # change useAppStateCommandSecurity => 1, when you have defined user permissions.
   # change dbHost => "Portal" to "App" if not working with the portal database.
   my $self = $class->SUPER::new(@_, app => "Test", appVersion => $VERSION, states => \%states, logOk => 1, useAppStateCommandSecurity => 0, dbHost => Portal);
   my %args = ( @_ );

   if ($self->error)
   {
     $self->prefixError();
     return $self;
   }

   # instantiate anything unique to this module

   # do validation
   # The $self->Portal::Test::isValid makes sure we access our
   # isValid method and not any classes isValid method that has
   # derived from us.
   if (!$self->Portal::Test::isValid)
   {
     # the error is set in the isValid() method.
     return $self;
   }

   # do anything else you might need to do.

   return $self;
 }

 sub isValid
 {
   my $self = shift;

   # make sure our Parent class is valid.
   $self->SUPER::isValid();

   # validate our parameters.

   if ($self->numInvalid() > 0 || $self->numMissing() > 0)
   {
     $self->error($self->genErrorString("all"));
     return 0;
   }
   return 1;
 }

=head1 DESCRIPTION

AppArgs is the base class for Portal Application modules.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(langObj, configObj, cookieObj, input, portalDB,
                 browserType, browserVersion, browserMaker,
                 methods, appObj, applicationObj, userObj,
                 companyObj, portalSession, authObj, app,
                 appVersion, states, logOk, useAppStateCommandSecurity,
                 dbHost, preStateCallback, stateEvalErrorCallback,
                 setDateStyle, extraArgs)

Creates a new instance of the Portal::App
object.

requires: langObj - Portal::Language Object
          configObj - Portal::Data::Config
          cookieObj - hash of cookies
          input - hash of passed in arguments
          portalDB - DBIWrapper pointing at the Portal
                     Database
          browserType
          browserVersion
          browserMaker
          methods - Portal::Methods Object
          appObj - Current applications info from the
                   Portal Database
          applicationObj - Portal::Application Object
          userObj - Portal::Objects::User
          companyObj - Portal::Objects::CompanyObject
          portalSession - Portal::Session Object
          authObj - Portal::Auth Object

returns:  object reference

Extra variables that are specified by the derived app:
  app     - the name of the app that derived from us.
  appVersion  - version of the app.
  states  - hash of the States this app supports.
  logOk   - 1 or 0, defaults to 0.  specifies if it is safe to do
            logging to a log_tb in your database.
  dbHost  - "Portal" or "App".  See instantiatePortalApp() for
    more info.

  preStateCallback  - ran before we use the state and eval it.
     The method defined will return an array consisting of
     (state, command) which will be used to change the state that
     is evaled and the command that is sent to the run method.  It
     takes the current state and command as values.  The state
     value passed in is not just Main, for example, but is
     Portal::AppName::State, where AppName is $self->{app}
     and State is the state value passed into the run() method.

  stateEvalErrorCallback - ran if there is an error when the state
     is evaled or when the run method is executed.  The method
     defined takes the current state and command as values.  See
     the preStateCallback for a definition of the state variables
     contents.

  useAppStateCommandSecurity - 1 or 0, defaults to 1.  If 1,
     we check for a user right assigned to the user where the
     app = appName, section = state and permission = command that
     is about to be executed.  If no permission is returned for
     the user, then they are not allowed to continue and an
     error screen is displayed.  See the UserRightObject for
     details on user rights.

     The derived app should set this to 0 until they have made
     sure that permissions have been created and assigned to
     all users who are going to access the app, otherwise nothing
     will work.

  setDateStyle - 1 or 0, defaults to 0.  If 1, we instruct
     DBIWrapper to signal we want dates as US style, not ISO.

  extraArgs - string of extra arguments you want passed into the
     states.  The current limitation is this has to be static
     values (constants) as you are defining them in new().
     Ex:  extraArgs = "toto => '$toto', dancing => 1"

The langObj requires the following phrases:

B<missingArgument> - "%s is missing"

B<invalidArgument> - "%s = '%s' is invalid"

B<errorExclam> - "Error!"

=cut

sub new
{
  my $class = shift;
  my %args = ( configObj => undef, cookieObj => undef, input => undef, portalDB => undef,
               browserType => "HTML", browserVersion => "4.0",
               browserMaker => "", methods => undef, appObj => undef,
               applicationObj => undef, userObj => undef, companyObj => undef,
               portalSession => undef, authObj => undef, app => "", appVersion => "",
               logOk => 0, useAppStateCommandSecurity => 1, states => {},
               preStateCallback => "", stateEvalErrorCallback => "",
               extraArgs => "", setDateStyle => 0, dbHost => "Portal",  @_ );
  my $self = $class->SUPER::new(@_);

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{applicationObj} = $args{applicationObj};  # The Portal::Application module.
  $self->{input} = $args{'input'};
  $self->{cookieObj} = $args{cookieObj};
  $self->{configObj} = $args{configObj};
  $self->{appObj} = $args{appObj};  # info about the application being run.
  $self->{userObj} = $args{userObj};
  $self->{companyObj} = $args{companyObj};
  $self->{portalSession} = $args{portalSession};
  $self->{browserType} = $args{browserType};
  $self->{browserVersion} = $args{browserVersion};
  $self->{browserMaker} = $args{browserMaker};
  $self->{authObj} = $args{authObj};
  $self->{methods} = $args{methods};
  $self->{portalDB} = $args{portalDB};
  $self->{app} = $args{app};
  $self->{appVersion} = $args{appVersion};
  $self->{states} = $args{states};
  $self->{logOk} = $args{logOk};   # default to not being able to log.  The derived app has to explicitly enable.
  $self->{preStateCallback} = $args{preStateCallback};
  $self->{stateEvalErrorCallback} = $args{stateEvalErrorCallback};
  $self->{useAppStateCommandSecurity} = $args{useAppStateCommandSecurity};  # default to enforcing the user permissions.  The derived app has to explicitly disable.
  $self->{setDateStyle} = $args{setDateStyle};
  $self->{extraArgs} = $args{extraArgs};
  $self->{dbHost} = $args{dbHost};

  # do validation
  if (!$self->Portal::App::isValid)
  {
    # the error is set in the isValid() method.
    $self->prefixError();
    return $self;
  }

  # do anything else you might need to do.
  # now call the instantiatePortalApp() for the caller, but only if app != "".
  if ($self->{app} ne "")
  {
    $self->instantiatePortalApp(dbHost => $self->{dbHost});
    if ($self->error)
    {
      $self->prefixError();
      return $self;
    }
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  $self->SUPER::isValid();

  # validate our parameters.
  if (not defined $self->{input})
  {
    $self->missing("input");
  }
  if (not defined $self->{portalSession})
  {
    $self->missing("portalSession");
  }
  elsif (!$self->{portalSession}->isValid)
  {
    $self->invalid("portalSession", $self->{portalSession}->errorMessage);
  }
  if (not defined $self->{appObj})
  {
    $self->missing("appObj");
  }
  elsif (!$self->{appObj}->isValid)
  {
    $self->invalid("appObj", $self->{appObj}->errorMessage);
  }
  if (not defined $self->{applicationObj})
  {
    $self->missing("applicationObj");
  }
  elsif (!$self->{applicationObj}->isValid)
  {
    $self->invalid("applicationObj", $self->{applicationObj}->errorMessage);
  }
  if (not defined $self->{userObj})
  {
    $self->missing("userObj");
  }
  elsif (!$self->{userObj}->isValid)
  {
    $self->invalid("userObj", $self->{userObj}->errorMessage);
  }
  if (not defined $self->{companyObj})
  {
    $self->missing("companyObj");
  }
  elsif (!$self->{companyObj}->isValid)
  {
    $self->invalid("companyObj", $self->{companyObj}->errorMessage);
  }
  if (not defined $self->{authObj})
  {
    $self->missing("authObj");
  }
  elsif (!$self->{authObj}->isValid)
  {
    $self->invalid("authObj", $self->{authObj}->errorMessage);
  }
  if (not defined $self->{cookieObj})
  {
    $self->missing("cookieObj");
  }
  if (not defined $self->{configObj})
  {
    $self->missing("configObj");
  }
  if ($self->{browserType} !~ /^(HTML|HDML|WML)$/)
  {
    $self->invalid("browserType", $self->{browserType});
  }
  if (length $self->{browserVersion} == 0)
  {
    $self->invalid("browserVersion", $self->{browserVersion});
  }
  if (not defined $self->{methods})
  {
    $self->missing("methods");
  }
  elsif (!$self->{methods}->isValid)
  {
    $self->invalid("methods", $self->{methods}->errorMessage);
  }
  if (not defined $self->{portalDB})
  {
    $self->missing("portalDB");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item hash getStates()

 returns the states hash.

=cut
sub getStates
{
  my $self = shift;

  return %{$self->{states}};
}

=item void instantiatePortalApp(dbHost)

  requires: dbHost
  returns: nothing

  dbHost is either "Portal" or "App" and indicates if the
    database host to connect to is the Portals dbHost value
    or what is stored in the database for this company app
    pairing.  It defaults to "App".

  This routine will setup the langObj, appConfigObj, appSessions
  and dbHandle objects needed by the app.  The firstTime
  variable is also set.

  This method is to be called from the new method of a Portal
  App that has derived from Portal::App.  If you need extra
  session setup done, do it after this method has been called and
  only if no errors occured.

=cut
sub instantiatePortalApp
{
  my $self = shift;
  my %args = ( dbHost => "App", @_ );
  my $dbHost = $args{dbHost};
  if ($dbHost !~ /^(Portal|App)$/)
  {
    $self->invalid("dbHost", $dbHost, " The dbHost parameter can only be 'Portal' or 'App'.");
    $self->error($self->genErrorString("all"));
    return;
  }
  my $appName = $self->{app};

  # get our Language Object
  my $langObj;
  eval "\$langObj = Portal::Language->new(lang => \$self->{langObj}->{lang}, app => \$appName);";
  if ($@)
  {
    $self->error($@);
    return;
  }
  if ($langObj->error)
  {
    $self->error($langObj->errorMessage);
    return;
  }

  # make sure the Portals langObj is merged into the applications so we can use both.
  $self->mergeLanguageObjects(portalLangObj => $self->{langObj}, appLangObj => $langObj);
  if ($self->error)
  {
    $self->error;
    return;
  }

  $self->{langObj} = $langObj;

  # update any modules we want to have our merged langObj.
  $self->{methods}->{langObj} = $langObj;

  # instantiate our config file for our children to use
  eval "\$self->{appConfigObj} = Portal::Data::Config->new(langObj => \$self->{langObj}, app => \$appName);";
  if ($@)
  {
    $self->error($@);
    return;
  }
  if ($self->{appConfigObj}->error)
  {
    $self->error($self->{appConfigObj}->errorMessage);
    return;
  }

  # setup the Session (database location, name, user, password, etc.)
  my $appSessions = $self->{methods}->portalSessionHandler($self->extract("portalDB, configObj"),
                                                           sessionObj => $self->{portalSession});
  if ($self->{methods}->error)
  {
    $self->prefixError($self->{methods}->errorMessage);
    return;
  }
  if ($appSessions->error)
  {
    $self->prefixError($appSessions->errorMessage);
    return;
  }
  $self->{appSessions} = $appSessions;

  # see if our app has a session started yet, if not create one.
  $self->{firstTime} = 0;  # default to not the firstTime.
  if (!exists $appSessions->{apps}->{$appName})
  {
    $self->{firstTime} = 1;

    $self->{portalSession}->setMode("write");
    if ($self->{appSessions}->{sessionObj}->{mode} ne "write")
    {
      $self->error("appSessions->mode = '$self->{appSessions}->{sessionObj}->{mode}' and I just flipped portalSession to write!  They are not the same object! Argh!!!");
      return;
    }
    $self->{appSessions}->makeSession(name => "$appName", mode => "write");
    if ($self->{appSessions}->error)
    {
      $self->prefixError($self->{appSessions}->errorMessage);
      return;
    }
    $self->{portalSession}->setMode("read");

    # use the companyObj and userObj to get the info for this application as far as
    # the database server, port, type, user, password and database name to connect to.

    my $companyAppObj = $self->{applicationObj}->getCompanyAppInfo(companyId => $self->{companyObj}->id, appId => $self->{appObj}->id);
    if ($self->{applicationObj}->error)
    {
      $self->prefixError($self->{applicationObj}->errorMessage);
      $self->cleanupSession;
      return;
    }
    if (!defined $companyAppObj)
    {
      $self->error("Application = '$appName' not assigned to Company = '$self->{companyObj}->{name}'!<br>\n");
      $self->cleanupSession;
      return;
    }

    # now we need to store the database info in the session.
    $self->{appSessions}->{apps}->{$appName}->setMode("write");
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbHost}   = ($dbHost eq "Portal" ? $self->{configObj}->{dbHost} : $companyAppObj->get("dbHost"));
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbName}   = $companyAppObj->get("dbName");
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbType}   = $companyAppObj->get("dbType");
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbPort}   = $companyAppObj->get("dbPort");
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbUser}   = $self->{configObj}->dbUser;
    $self->{appSessions}->{apps}->{$appName}->{store}->{dbPasswd} = $self->{configObj}->dbPasswd;

    # store the companyAppObj in the session for this application so later states can get access to it.
    $self->{appSessions}->{apps}->{$appName}->{store}->{companyAppObj} = $companyAppObj;

    $self->{appSessions}->{apps}->{$appName}->{store}->{windows} = {};  # The hash that keeps track of what windows are open.

    $self->{appSessions}->{apps}->{$appName}->{store}->{okToLogout} = 1;
    $self->{appSessions}->{apps}->{$appName}->setMode("read");
  }

  # use the database info from the session to connect to the database and store the handle internally.
  my $dbHost = $self->{appSessions}->{apps}->{$appName}->{store}->{dbHost};
  my $dbName = $self->{appSessions}->{apps}->{$appName}->{store}->{dbName};
  my $dbType = $self->{appSessions}->{apps}->{$appName}->{store}->{dbType};
  my $dbPort = $self->{appSessions}->{apps}->{$appName}->{store}->{dbPort};
  my $dbUser = $self->{appSessions}->{apps}->{$appName}->{store}->{dbUser};
  my $dbPasswd = $self->{appSessions}->{apps}->{$appName}->{store}->{dbPasswd};

  my $dbHandle = DBIWrapper->new(dbType => $dbType, dbPort => $dbPort, dbHost => $dbHost, dbName => $dbName, dbUser => $dbUser, dbPasswd => $dbPasswd, setDateStyle => $self->{setDateStyle});
  if ($dbHandle->error)
  {
    $self->error($dbHandle->errorMessage);
    $self->cleanupSession;
    return;
  }
  $self->{dbHandle} = $dbHandle;
}

=item void mergeLanguageObjects(portalLangObj, appLangObj)

  takes: portalLangObj, appLangObj
  returns: nothing

  portalLangObj is an instance of the Portals langObj.
  appLangObj is the langObj just instantiated for the
    App in question.

  The phrases from the portalLangObj are then added to the
  appLangObj's phrases hash (not overriding).
  This allows a single langObj to be passed around and
  to have both the Portals phrases and the apps phrases
  without the user having to duplicate Portal phrases
  in their language mappings.  Any phrases that exist in
  the Portal hash and in the apps hash, are not overwritten.

=cut
sub mergeLanguageObjects
{
  my $self = shift;
  my %args = ( portalLangObj => undef, appLangObj => undef, @_ );
  my $portalLangObj = $args{portalLangObj};
  my $appLangObj = $args{appLangObj};

  if (!defined $portalLangObj)
  {
    $self->missing("portalLangObj");
  }
  elsif (!$portalLangObj->isValid)
  {
    $self->error($portalLangObj->errorMessage);
  }
  if (!defined $appLangObj)
  {
    $self->missing("appLangObj");
  }
  elsif (!$appLangObj->isValid)
  {
    $self->error($appLangObj->errorMessage);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return;
  }
  if ($self->error)
  {
    return;
  }

  foreach my $phrase (keys %{$portalLangObj->{phraseObj}->{phrases}})
  {
    if (!exists $appLangObj->{phraseObj}->{phrases}->{$phrase})
    {
      $appLangObj->{phraseObj}->{phrases}->{$phrase} = $portalLangObj->{phraseObj}->{phrases}->{$phrase};
    }
  }
}

=item void cleanupSession(void)

 If firstTime is 1, then we delete the session.
 This method is called when an error occured in the
 run() method.

=cut
sub cleanupSession
{
  my $self = shift;

  if ($self->{firstTime})
  {
    $self->{appSessions}->{apps}->{$self->{app}}->delete;  # remove the session info.
    $self->{appSessions}->{apps}->{$self->{app}} = undef;  # force the session to go out of scope.
    my $tmpDoc = HTMLObject::Base->new;
    $tmpDoc = $self->{appSessions}->writeCookie(doc => $tmpDoc);  # This will allow the parent session to be properly updated!
  }
}

=item void cleanupStateEvalError(state, command)

 rquires: state, command

 Sees if the user specified a callback to execute when an error
 occured trying to eval/run the state.

 Also runs the cleanupSession method to make sure the
 session is cleaned up if needed.

=cut
sub cleanupStateEvalError
{
  my $self = shift;
  my %args = ( state => "", command => "", @_ );
  my $state = $args{state};
  my $command = $args{command};

  if ($self->{stateEvalErrorCallback})
  {
    eval "\$self->" . $self->{stateEvalErrorCallback} . "(state => \$state, command => \$command);";
    if ($@)
    {
      $self->error(sprintf($self->langObj->map("stateEvalErrorCallbackFailed"), $@));
      $self->cleanupSession;
      return;
    }
  }
  $self->cleanupSession;
}

=item HTMLObject run(state, command)

 The main entry point for this application.
 takes: state, command
 returns: The HTMLObject document to display or undef if an error.

 Pulls the specified state file into existance and calls run(command).
 Returns the HTMLObject document that represents the output from the
 State->run() method.

 You need to make sure you set the following variables in the derived
 Portal::App modules new() method:
   appName
   appVersion
   states
   logOk
   useAppStateCommandSecurity

   preStateCallback
   stateEvalErrorCallback
   extraArgs

=cut
sub run
{
  my $self = shift;
  my %args = ( state => "", command => "", @_ );
  my $state = $args{state};
  my $command = $args{command};
  my $dbHandle = $self->{dbHandle};
  my $doc = undef;

  if (length $state == 0)
  {
    $self->missing("state");
    $self->error($self->genErrorString("all"));
    $self->cleanupSession;
    return $doc;
  }
  if (length $command == 0)
  {
    $self->missing("command");
    $self->error($self->genErrorString("all"));
    $self->cleanupSession;
    return $doc;
  }

  # validate the state is known.
  if (!exists $self->{states}->{$state})
  {
    $self->error(sprintf($self->langObj->map("stateNotKnown"), $state));
    $self->cleanupSession;
    return $doc;
  }

  # make sure the user has this application assigned to them that is attempting to run it
  if ($self->{firstTime})
  {
    my $result = $self->{applicationObj}->isAppAssignedToUser(appName => "$self->{app}", userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      $self->cleanupSession;
      return $doc;
    }
    if ($result == 0)
    {
      $self->error($self->langObj->map("applicationNotAssigned"));
      $self->cleanupSession;
      return $doc;
    }
  }

  my $stateName = "Portal::$self->{app}::$state";

  if ($self->{preStateCallback})
  {
    my @result;
    eval "\@result = \$self->" . $self->{preStateCallback} . "(state => \$stateName, command => \$command);";
    if ($@)
    {
      $self->error(sprintf($self->langObj->map("preStateCallbackFailed"), $@));
      $self->cleanupSession;
      return $doc;
    }
    if ($self->error)
    {
      $self->prefixError;
      $self->cleanupSession;
      return $doc;
    }
    $stateName = $result[0];
    $command = $result[1];
    ($state = $stateName) =~ s/^(Portal::$self->{app}::)//;
  }

  if ($self->{useAppStateCommandSecurity})
  {
    # see if the user is an admin, if yes, they don't have to have an acl defined, else check.
    if ($self->{userObj}->{admin} == 0)
    {
      my $userRight = $self->{authObj}->getUserRightInfo(app => $self->{app}, section => $state, permission => $command);
      if ($self->{authObj}->error)
      {
        $self->error($self->{authObj}->errorMessage);
        $self->cleanupStateEvalError(state => $stateName, command => $command);
        return $doc;
      }
      if (!defined $userRight)
      {
        # blow an error informing the user they don't have access to this App/State/Command.
        $self->error(sprintf($self->langObj->map("AccessDenied"), $self->{app}, $state, $command));
        $self->cleanupStateEvalError(stateName => $stateName, command => $command);
        return $doc;
      }
    }
  }

  eval "use $stateName;";
  if ($@)
  {
    $self->error(sprintf($self->langObj->map("stateEvalFailed"), $stateName, $@));
    $self->cleanupStateEvalError(stateName => $stateName, command => $command);
    return $doc;
  }

  # make a logObj
  my $logObj = Portal::Log->new(dbHandle => $dbHandle, langObj => $self->{langObj});
  if ($logObj->error)
  {
    $self->error($logObj->errorMessage);
    $self->cleanupStateEvalError(state => $stateName, command => $command);
    return $doc;
  }

  if ($self->{logOk})
  {
    if ($self->{firstTime})  # make the log entry (3) to indicate this app has been launched.
    {
      $self->{methods}->doLog(action => 3, extraInfo => $self->{app}, caller => $self, userId => $self->{userObj}->{id}, logObj => $logObj);
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        $self->cleanupStateEvalError(state => $stateName, command => $command);
        return undef;
      }
    }
  }

  my $stateObj;
  my $arguments = $self->arguments();
  eval "\$stateObj = $stateName->new($arguments, dbHandle => \$self->{dbHandle}, sessionObj => \$self->{appSessions}->{apps}->{$self->{app}}, appConfigObj => \$self->{appConfigObj}, logObj => \$logObj, $self->{extraArgs});";
  if ($@)
  {
    $self->error(sprintf($self->langObj->map("stateEvalFailed"), $stateName, $@));
    $self->cleanupStateEvalError(state => $stateName, command => $command);
    return $doc;
  }
  if ($stateObj->error)
  {
    $self->error($stateObj->errorMessage);
    $self->cleanupStateEvalError(state => $stateName, command => $command);
    return $doc;
  }

  eval { $doc = $stateObj->run(command => $command); };
  if ($@)
  {
    $self->error(sprintf($self->langObj->map("stateEvalFailed"), $stateName, $@));
    $self->cleanupStateEvalError(state => $stateName, command => $command);
    return $doc;
  }
  if ($stateObj->error)
  {
    $self->error($stateObj->errorMessage);
    $self->cleanupStateEvalError(state => $stateName, command => $command);
    return $doc;
  }
  $self->{appSessions}->writeCookie(doc => $doc);  # make sure the application session id cookie is reset.
  if (ref($doc) =~ /Normal/)
  {
    # we are a Normal document.
    $self->{methods}->setJavaScriptErrorInfo(doc => $doc, email => $self->{configObj}->{emailAddress}, appName => "Portal::$self->{app}", appVersion => $self->{appVersion});
  }

  return $doc;  # signal all ok.
}

=item scalar arguments(void)

 Returns a string that when eval'ed will specify all the
 arguments to pass into an instance of the Portal::App->new
 method.

 Ex:

 "langObj => \$self->{langObj}, input => \$self->{input}"

=cut

sub arguments
{
  my $self = shift;
  my $args = $self->SUPER::arguments();

  foreach my $arg (qw(portalDB cookieObj input configObj methods authObj applicationObj browserType browserVersion browserMaker appObj userObj companyObj portalSession app appVersion states logOk preStateCallback stateEvalErrorCallback useAppStateCommandSecurity setDateStyle))
  {
    $args .= ", " if ($args);
    $args .= "$arg => \$self->{$arg}";
  }

  return $args;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
