 # AppStateArgs.pm - The Object Class that provides a AppStateArgs Base Object
# Created by James A. Pattie, 03/26/2003.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::AppState;
use Portal::Base;
use Portal::Data::Variables;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

=head1 NAME

AppStateArgs - Object used to build a base Application State Object Class.

=head1 SYNOPSIS

 package Portal::Test::Main;
 use Portal::AppState;
 use strict;
 use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

 require Exporter;

 @ISA = qw(Portal::AppState Exporter AutoLoader);
 @EXPORT = qw();

 $VERSION = '0.01';

 my %commands = ( display => "display the state", close => "Close the App", );

 sub new
 {
   my $class = shift;
   my $self = $class->SUPER::new(@_);
   my %args = ( @_ );

   if ($self->error)
   {
     $self->prefixError();
     return $self;
   }

   # instantiate anything unique to this module
   $self->{commands} = \%commands;
   $self->{state} = "Main";  # this states name.

   # do validation
   # The $self->Portal::Test::Main::isValid makes sure we access our
   # isValid method and not any classes isValid method that has
   # derived from us.
   if (!$self->Portal::Test::Main::isValid)
   {
     # the error is set in the isValid() method.
     return $self;
   }

   # do anything else you might need to do.
   $self->buildUrls;  # populate the url variables

   return $self;
 }

 =item % getCommands(void)

   This method returns the commands hash and is only to be
   used by the registerApp method for creating the permissions,
   if using the AppStateCommandSecurity implementation.

 =cut
 sub getCommands
 {
   return %commands;
 }

 sub isValid
 {
   my $self = shift;

   # make sure our Parent class is valid.
   if (!$self->SUPER::isValid())
   {
     $self->prefixError();
     return 0;
   }

   # validate our parameters.

   if ($self->numInvalid() > 0 || $self->numMissing() > 0)
   {
     $self->error($self->genErrorString("all"));
     return 0;
   }
   return 1;
 }

=head1 DESCRIPTION

AppStateArgs is the base class for Portal Application State modules.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(langObj, configObj, cookieObj, input, dbHandle,
                 browserType, browserVersion, browserMaker,
                 methods, appObj, applicationObj, userObj,
                 companyObj, portalSession, authObj, logObj,
                 sessionObj, appConfigObj, app, appVersion,
                 logOk, useAppStateCommandSecurity, states)

Creates a new instance of the Portal::AppState
object.

requires: langObj - Portal::Language Object
          configObj - Portal::Data::Config
          appConfigObj - Portal::Data::Config
                         config object for the App in question
          cookieObj - hash of cookies
          input - hash of passed in arguments
          dbHandle - DBIWrapper pointing at the App's
                     Database
          browserType
          browserVersion
          browserMaker
          methods - Portal::Methods Object
          appObj - Current applications info from the
                   Portal Database
          applicationObj - Portal::Application Object
          userObj - Portal::Objects::User
          companyObj - Portal::Objects::CompanyObject
          portalSession - Portal::Session Object
          sessionObj - Portal::Session Object for the
                       Application
          logObj - Portal::Log Object
          authObj - Portal::Auth Object

          The following are provided by the derived
          Portal::App instance of your application.

          app   - Name of the Portal App
          appVersion - Version of your Portal App
          logOk - Do we have access to a log_tb?
          useAppStateCommandSecurity
          states - hash of states for the App.


          Provides:
          portalVariables - an instance of the
            Portal::Data::Variables module.

          browserCap - shortcut to the browserCap hash in the
            portalSession.

returns:  object reference

The langObj requires the following phrases:

B<missingArgument> - "%s is missing"

B<invalidArgument> - "%s = '%s' is invalid"

B<errorExclam> - "Error!"

=cut

sub new
{
  my $class = shift;
  my %args = ( configObj => undef, cookieObj => undef, input => undef, dbHandle => undef,
               browserType => "HTML", browserVersion => "4.0", appConfigObj => undef,
               browserMaker => "", methods => undef, appObj => undef,
               applicationObj => undef, userObj => undef, companyObj => undef,
               portalSession => undef, authObj => undef, logObj => undef, sessionObj => undef,
               app => "", appVersion => "", logOk => 0, states => undef,
               useAppStateCommandSecurity => 0, @_ );
  my $self = $class->SUPER::new(@_);

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{applicationObj} = $args{applicationObj};  # The Portal::Application module.
  $self->{input} = $args{'input'};
  $self->{cookieObj} = $args{cookieObj};
  $self->{configObj} = $args{configObj};
  $self->{appConfigObj} = $args{appConfigObj};
  $self->{appObj} = $args{appObj};  # info about the application being run.
  $self->{userObj} = $args{userObj};
  $self->{companyObj} = $args{companyObj};
  $self->{portalSession} = $args{portalSession};
  $self->{browserType} = $args{browserType};
  $self->{browserVersion} = $args{browserVersion};
  $self->{browserMaker} = $args{browserMaker};
  $self->{authObj} = $args{authObj};
  $self->{methods} = $args{methods};
  $self->{dbHandle} = $args{dbHandle};
  $self->{sessionObj} = $args{sessionObj};
  $self->{logObj} = $args{logObj};
  $self->{app} = $args{app};
  $self->{appVersion} = $args{appVersion};
  $self->{logOk} = $args{logOk};
  $self->{states} = $args{states};
  $self->{useAppStateCommandSecurity} = $args{useAppStateCommandSecurity};

  # variables unique to the State.
  $self->{commands} = {};
  $self->{state} = "";

  # do validation
  if (!$self->Portal::AppState::isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  $self->{portalVariables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{portalVariables}->error)
  {
    $self->error($self->{portalVariables}->errorMessage);
    return $self;
  }

  # make a shortcup to the browserCap hash.
  $self->{browserCap} = $self->{portalSession}->{store}->{browserCap};

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{input})
  {
    $self->missing("input");
  }
  if (not defined $self->{portalSession})
  {
    $self->missing("portalSession");
  }
  elsif (!$self->{portalSession}->isValid)
  {
    $self->invalid("portalSession", $self->{portalSession}->errorMessage);
  }
  if (not defined $self->{sessionObj})
  {
    $self->missing("sessionObj");
  }
  elsif (!$self->{sessionObj}->isValid)
  {
    $self->invalid("sessionObj", $self->{sessionObj}->errorMessage);
  }
  if (not defined $self->{appObj})
  {
    $self->missing("appObj");
  }
  elsif (!$self->{appObj}->isValid)
  {
    $self->error($self->{appObj}->errorMessage);
  }
  if (not defined $self->{applicationObj})
  {
    $self->missing("applicationObj");
  }
  elsif (!$self->{applicationObj}->isValid)
  {
    $self->error($self->{applicationObj}->errorMessage);
  }
  if (not defined $self->{userObj})
  {
    $self->missing("userObj");
  }
  elsif (!$self->{userObj}->isValid)
  {
    $self->error($self->{userObj}->errorMessage);
  }
  if (not defined $self->{companyObj})
  {
    $self->missing("companyObj");
  }
  elsif (!$self->{companyObj}->isValid)
  {
    $self->error($self->{companyObj}->errorMessage);
  }
  if (not defined $self->{authObj})
  {
    $self->missing("authObj");
  }
  elsif (!$self->{authObj}->isValid)
  {
    $self->error($self->{authObj}->errorMessage);
  }
  if (not defined $self->{cookieObj})
  {
    $self->missing("cookieObj");
  }
  if (not defined $self->{configObj})
  {
    $self->missing("configObj");
  }
  if (not defined $self->{appConfigObj})
  {
    $self->missing("appConfigObj");
  }
  if ($self->{browserType} !~ /^(HTML|HDML|WML)$/)
  {
    $self->invalid("browserType", $self->{browserType});
  }
  if (length $self->{browserVersion} == 0)
  {
    $self->invalid("browserVersion", $self->{browserVersion});
  }
  if (not defined $self->{methods})
  {
    $self->missing("methods");
  }
  elsif (!$self->{methods}->isValid)
  {
    $self->error($self->{methods}->errorMessage);
  }
  if (not defined $self->{dbHandle})
  {
    $self->missing("dbHandle");
  }
  if (not defined $self->{logObj})
  {
    $self->missing("logObj");
  }
  elsif (!$self->{logObj}->isValid)
  {
    $self->error($self->{logObj}->errorMessage);
  }
  if ($self->{app} !~ /^(.+)$/)
  {
    $self->invalid("app", $self->{app});
  }
  if ($self->{appVersion} !~ /^(.+)$/)
  {
    $self->invalid("appVersion", $self->{appVersion});
  }
  if (! defined $self->{states})
  {
    $self->missing("states");
  }
  if ($self->{useAppStateCommandSecurity} !~ /^(0|1)$/)
  {
    $self->invalid("useAppStateCommandSecurity", $self->{useAppStateCommandSecurity});
  }
  if ($self->{logOk} !~ /^(0|1)$/)
  {
    $self->invalid("logOk", $self->{logOk});
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  if ($self->error)
  {
    return 0;
  }

  return 1;
}

=item void buildUrls()

 This method will build up the following url variables.

 Call it from the new() of your derived AppState module so
 that $self->{app} is defined.

 URLS defined:

 baseUrl - the website url of the Portal, minus any portal
   specific info.  Ex:  https://www.pcxperience.com/
 url - the url to the portal/cgi-bin/index.cgi of the server
   that the CompanyAppObj points to.
 closeUrl - the url that instructs the app to Close.
 mainMenuUrl - the url that causes the Main Menu to be displayed.
 helpUrl - the url that will display the State level help topics. If the
   command in question is not display, it is tacked onto the help lookup
   string.  This may generate an invalid help topic.  You should
   generate your own helpUrl if you need better context sensitive help.
 refreshUrl - the url that will attempt to cause the current screen to
   be re-displayed.  The command is taken from $self->input->{command}.
   Update if you need extra args specified.


 A urls hash is also created and populated with all the above defined
 urls so that they can easily be passed into modules that need the
 urls but are not derived from Portal::AppState.

=cut
sub buildUrls
{
  my $self = shift;

  # build up the core url's needed by this app.
  $self->{baseUrl} = $self->methods->createBaseURL(type => "Portal", linkType => "base");
  if ($self->methods->error)
  {
    $self->error($self->methods->errorMessage());
    return $self;
  }
  $self->{url} = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $self;
  }

  $self->{closeUrl} = $self->{methods}->urlBuilder(baseUrl => $self->{url}, arguments => { app => $self->{app}, state => "Main", command=> "close" });
  $self->{mainMenuUrl} = $self->{methods}->urlBuilder(baseUrl => $self->{url}, arguments => { app => $self->{app}, state => "Main", command => "display" });
  $self->{helpUrl} = $self->{methods}->urlBuilder(baseUrl => $self->{url}, arguments => { app => "Portal", state => "Help", command => "display", helpApp => $self->{app}, content => "Normal" . ($self->{state} ne "Main" ? ",$self->{state}" : "") . ($self->{input}->{command} ne "display" ? ",$self->{input}->{command}" : "" ) });
  $self->{refreshUrl} = $self->{methods}->urlBuilder(baseUrl => $self->{url}, arguments => { app => $self->{app}, state => $self->{state}, command => $self->{input}->{command} });

  $self->{urls} = {};
  foreach my $url (qw(baseUrl url closeUrl mainMenuUrl helpUrl refreshUrl))
  {
    $self->{urls}->{$url} = $self->{$url};
  }
}

=item HTMLObject run(command)

 takes: command
 returns: The HTMLObject document to display or undef if an error.

 summary: We validate the specified command against the
   $self->{commands} hash.  If the command exists, we then
   sanitize it (replace any non a-z, A-Z, 0-9 or _ character with
   an _) and execute the function that is named c_{command} where
   {command} is the sanitized version.  This is in the scope of
   $self, so it has to be defined as a method of the derived
   Portal::AppState module.

   Ex:  State = Portal::Example::Main, command = display
   would exectue $self->c_display();

   c_display needs to be defined as follows:

   sub c_display
   {
     my $self = shift;

     my $doc = $self->setupCSS();
     if ($self->error)
     {
       $self->prefixError();
       return $doc;
     }

     # do the display code here

     return $doc;
   }

=cut
sub run
{
  my $self = shift;
  my %args = ( command => "", @_ );
  my $command = $args{command};
  my $doc;

  if (!exists $self->{commands}->{$command})
  {
    $self->error(sprintf($self->langObj->map("commandNotKnown"), $command));
    return undef;
  }

  (my $safeCommand = $command) =~ s/([^\w])/_/g;

  # now we execute the c_ command function
  eval "\$doc = \$self->c_" . $safeCommand . "();";
  if ($@)
  {
    $self->prefixError($@);
    return undef;
  }

  return $doc;
}

=item HTMLObject setupCSS(doc)

  requires:
  optional: doc
  returns:  doc

  summary:  This method will instantiate an HTMLObject::Normal
            document, if you don't specify doc, and then proceeds
            to get the current color scheme for the user and
            create the CSS entries in the doc.  We also call
            Portal::Methods->includeCoreJSFiles() to pull in
            any JavaScript files that the Portal requires for
            the generatePickerCode() method to "just work".

=cut
sub setupCSS
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};
  my $applicationObj = $self->{applicationObj};
  my $userId = $self->{userObj}->{id};

  if (!defined $doc)
  {
    $doc = HTMLObject::Normal->new;
  }

  # get the users colorScheme to work with.
  my $colorScheme = $applicationObj->getCurrentColorScheme(userId => $userId);
  if ($applicationObj->error)
  {
    $self->error($applicationObj->errorMessage);
    return $doc;
  }
  if (!defined $colorScheme)
  {
    $self->error($self->{langObj}->map("noValidColorScheme"));
    return $doc;
  }

  # create the needed css entries
  my %cssEntries = $self->{methods}->generateColorSchemeCSS(colorScheme => $colorScheme, appName => $self->{app});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return $doc;
  }
  $doc->print(%cssEntries);

  # make sure that this doc supports javascript.
  if (exists $doc->{javascriptIncludes})
  {
    $self->{methods}->includeCoreJSFiles(doc => $doc);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return $doc;
    }
  }

  return $doc;
}

=item HTMLObject prepDoc(doc, docType, title, titlePhrase, template, templateStr)

 doc - an HTMLObject instance to work with instead of instantiating
   a new HTMLObject document.  Defaults to undef, so a new doc
   will be generated by default.

 docType - specify the type of HTMLObject document to be created.
   Valid values are: 'Base', 'Normal', 'FrameSet'
   Default is 'Normal'.

   If 'Base' is specified, then the core javascript modules will
   not be included, but CSS will always be setup.

 title - specify the actual title string to be set in the Document.

 titlePhrase - specify the phrase to be looked up in the langObj and
   used for the title.  This will be sprintf'ed with the title phrase
   from the current langObj.  The current app's VERSION will be part
   of the output.
   Ex:  $doc->setTitle(sprintf($self->langObj->map("title"), $VERSION,
   $self->langObj->map($titlePhrase)));

 If neither title or titlePhrase are specified, then the title will
   be set by looking up the current command from the input hash and
   using the commands hash to figure out the langObj entry to use.
   The resulting langObj entry will then be sprintf'ed with the title
   langObj phrase and the current app's VERSION.
   Ex: $doc->setTitle(sprintf($self->langObj->map("title"), $VERSION,
   $self->langObj->map($self->commands{$self->input->{command}})));

 template - specify the type of template html code to be generated in
   the body of the document.  Valid values are: 'none', 'title',
   'menubar', 'menubar-vertical', 'menubar+sidemenu-left',
   'menubar+sidemenu-right', 'menubar+sidemenu-detect', 'sidemenu-left',
   'sidemenu-right' or 'custom'.

   Defaults to the prepDocTemplate value from your applications
   appConfigObj object.

   If 'none', then nothing will be output into the body of the document.

   If 'menubar+sidemenu-detect', we lookup the users preference for
   sidemenuLocation (which must have a value of 'left' or 'right') and
   use that to set template to 'menubar+sidemenu-left' or
   'menubar+sidemenu-right'.  If the preference does not exist, it will
   be created as long as the application has defined the default value
   for the sidemenuLocation preference.  The 'prefModule' for the
   preference, and thus also it's default config entry, MUST BE =
   'Global'.  The default config entry must use the
   prefName = 'sidemenuLocation_Default'.

   If 'title', then

   <center><h3>$title</h3></center>

   will be output into the body of the document, where $title is the
   title or titlePhrase you specified.  If you did not specify a title
   or titlePhrase value, then <title> will be output instead allowing
   you to substitute later.

   If 'menubar', then

   <menubar>
   <br />

   will be output into the body of the document so you can later
   substitute in the generated menubar.

   If 'menubar-vertical', then

   <menubar>

   will be output into the body of the document so you can later
   substitute in the generated menubar.

   If 'menubar+sidemenu-left', then

   <menubar>
   <br />
   <table border="0" cellspacing="0" cellpadding="2" width="100%">
     <tr>
       <td width="25%" valign="top"><sidemenu></td>
       <td class="main" width="75%" align="center"><maincontent></td>
     </tr>
   </table>

   will be output into the body of the document so you can later
   substitute <menubar> for the main menu, <sidemenu> for the side
   menu and <maincontent> for the content being displayed in the main
   table.

   If 'menubar+sidemenu-right', then

   <menubar>
   <br />
   <table border="0" cellspacing="0" cellpadding="2" width="100%">
     <tr>
       <td class="main" width="75%" align="center"><maincontent></td>
       <td width="25%" valign="top"><sidemenu></td>
     </tr>
   </table>

   will be output into the body of the document so you can later
   substitute <menubar> for the main menu, <sidemenu> for the side
   menu and <maincontent> for the content being displayed in the main
   table.

   If 'sidemenu-left', then

   <table border="0" cellspacing="0" cellpadding="2" width="100%">
     <tr>
       <td width="25%" valign="top"><sidemenu></td>
       <td class="main" width="75%" align="center"><maincontent></td>
     </tr>
   </table>

   will be output into the body of the document so you can later
   substitute <sidemenu> for the side menu and <maincontent> for the
   content being displayed in the main table.

   If 'sidemenu-right', then

   <table border="0" cellspacing="0" cellpadding="2" width="100%">
     <tr>
       <td class="main" width="75%" align="center"><maincontent></td>
       <td width="25%" valign="top"><sidemenu></td>
     </tr>
   </table>

   will be output into the body of the document so you can later
   substitute <sidemenu> for the side menu and <maincontent> for the
   content being displayed in the main table.

   If 'custom', then the templateStr value will be used to allow for
   custom body html to be generated.  templateStr defaults to the
   prepDocCustomTemplate value from your appConfigObj object.

 Returns the created HTMLObject or undef if an error happened.

 If the HTMLObject can support JavaScript, then the help and closeApp
 helper functions are generated, so that the developer doesn't have
 to manually do the setup in every screen.

 If template = 'menubar' or 'menubar-vertical' or
 'menubar+sidemenu-left' or 'menubar+sidemenu-right', then the
 appropriate menus will be generated using the printMenu() method and
 substituted into the template for you.  It is possible that the
 document title will be specified by the printMenu() method, thus
 allowing the application to override the title you specified or that
 was defaulted to.

 menubar-vertical, will generate the same entries as menubar, but in a
 vertical orientation, instead of horizontally.  This is designed more
 for those applications that use framesets and want the main menubar
 to go down the side, rather than across the top.

=cut
sub prepDoc
{
  my $self = shift;
  my $langObj = $self->{langObj};
  my %args = ( doc => undef, docType => "Normal", title => "", titlePhrase => "", template => $self->{appConfigObj}->{prepDocTemplate},
               templateStr => $self->{appConfigObj}->{prepDocCustomTemplate}, @_ );
  my $doc = $args{doc};
  my $docType = $args{docType};
  my $title = (length $args{title} > 0 ? $args{title} : (length $args{titlePhrase} > 0 ? sprintf($langObj->map("title"), $self->appVersion, $langObj->map($args{titlePhrase})) : undef));
  my $template = $args{template};
  my $templateStr = $args{templateStr};

  if (!defined $doc)
  {
    if ($docType !~ /^(Base|Normal|FrameSet)$/)
    {
      $self->invalid("docType", $docType, "valid values are: Base, Normal, FrameSet");
      $self->error($self->genErrorString("all"));
      return undef;
    }
    eval "\$doc = HTMLObject::$docType" . "->new();";
    if ($@)
    {
      $self->error($@);
      return undef;
    }
  }

  # validate the template
  if ($template !~ /^(none|title|menubar(\+sidemenu\-(left|right|detect)|\-vertical)?|sidemenu\-(left|right)|custom)$/)
  {
    $self->invalid("template", $template, "valid values are: none, title, menubar, menubar-vertical, menubar+sidemenu-left, menubar+sidemenu-right, sidemenu-left, sidemenu-right, custom");
    $self->error($self->genErrorString("all"));
    return undef;
  }

  # generate the CSS and any JavaScript include statements.
  $doc = $self->setupCSS(doc => $doc);

  # now set the title, if it is defined.
  if (defined $title)
  {
    $doc->setTitle($title);
  }
  else
  {
    # lookup the current command from the $self->input hash and map it through the commands hash to figure out the langObj entry to work with.
    my $langPhrase = $self->commands->{$self->input->{command}};
    $title = sprintf($langObj->map("title"), $self->appVersion, $langObj->map($langPhrase));
    $doc->setTitle($title);
  }

  # now start the body template.
  if ($template eq "menubar+sidemenu-detect")
  {
    # we have to look up the sidemenuLocation preference and set
    # template = 'menubar+sidemenu-left' or 'menubar+sidemenu-right'
    # based upon the sidemenuLocation preference.

    my $prefObj = $self->authObj->getOrCreateUserPreference(userId => $self->userObj->get("id"),
      app => $self->app, module => "Global", preference => "sidemenuLocation");
    if ($self->authObj->error)
    {
      $self->error($self->authObj->errorMessage);
      return undef;
    }
    my $location = $prefObj->get("value");
    if (! exists $self->portalVariables->sideMenuLocations->{$location})
    {
      $self->invalid("sidemenuLocation", $location, "Valid values are: " . join(", ", keys %{$self->portalVariables->sideMenuLocations}));
      $self->error($self->genErrorString("all"));
      return undef;
    }
    $template = "menubar+sidemenu-$location";
  }

  if ($template eq "title")
  {
    $title = "<title>" if (!defined $title);
    $doc->print("<center><h3>$title</h3></center>\n");
  }
  elsif ($template eq "menubar")
  {
    $doc->print("<menubar>\n<br />\n");
  }
  elsif ($template eq "menubar-vertical")
  {
    $doc->print("<menubar>\n");
  }
  elsif ($template eq "menubar+sidemenu-left")
  {
    $doc->print(<<"END_OF_BODY");
<menubar>
<br />
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top"><sidemenu></td>
    <td class="main" width="75%" align="center"><maincontent></td>
  </tr>
</table>
END_OF_BODY
  }
  elsif ($template eq "menubar+sidemenu-right")
  {
    $doc->print(<<"END_OF_BODY");
<menubar>
<br />
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td class="main" width="75%" align="center"><maincontent></td>
    <td width="25%" valign="top"><sidemenu></td>
  </tr>
</table>
END_OF_BODY
  }
  elsif ($template eq "sidemenu-left")
  {
    $doc->print(<<"END_OF_BODY");
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td width="25%" valign="top"><sidemenu></td>
    <td class="main" width="75%" align="center"><maincontent></td>
  </tr>
</table>
END_OF_BODY
  }
  elsif ($template eq "sidemenu-right")
  {
    $doc->print(<<"END_OF_BODY");
<table border="0" cellspacing="0" cellpadding="2" width="100%">
  <tr>
    <td class="main" width="75%" align="center"><maincontent></td>
    <td width="25%" valign="top"><sidemenu></td>
  </tr>
</table>
END_OF_BODY
  }
  elsif ($template eq "custom")
  {
    $doc->print($templateStr);
  }

  if (exists $doc->{javascriptIncludes})
  {
    # create the help JS function.
    $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "help", langObj => $self->{langObj});
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }

    # create the validateClose JS function.
    $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeApp", langObj => $self->{langObj});
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }

  if ($template =~ /^(menubar(\+sidemenu\-(left|right)|\-vertical)?|sidemenu\-(left|right))$/)
  {
    # generate the <menubar> code.
    if ($template =~ /^(menubar(\+sidemenu\-(left|right)|\-vertical)?)$/)
    {
      my %menu = $self->printMenu(location => "menubar", orientation => ($template eq "menubar-vertical" ? "vertical" : "horizontal"));
      if ($self->error)
      {
        $self->error($self->errorMessage);
        return undef;
      }
      $doc->setTitle($menu{title}) if (length $menu{title} > 0);
      $doc->printTag(tag => "<menubar>", value => $menu{menu});
    }

    if ($template =~ /^(menubar\+sidemenu\-(left|right)|sidemenu\-(left|right))$/)
    {
      # generate the <sidemenu> code.
      my %menu = $self->printMenu(location => "sidemenu");
      if ($self->error)
      {
        $self->error($self->errorMessage);
        return undef;
      }
      $doc->setTitle($menu{title}) if (length $menu{title} > 0);
      $doc->printTag(tag => "<sidemenu>", value => $menu{menu});
    }
  }

  return $doc;
}

=item HTMLObject processCloseAppEvent(doc, langEntry, log)

  requires: log (bool)
  optional: doc, langEntry
  returns:  doc

  summary:  This method does all the cleanup work associated with
            closing an app.  This entails closing all registered
            windows, the apps help window, if it is open, and
            then deleting the apps session.

            If your langObj doesn't have a "close" entry, then
            use langEntry to specify the string to be displayed
            that indicates you are closing the app.

  Your langObj has to have the "title" entry which takes 2 parameters
  the version and the current location of the program.

  If you don't pass in doc, then we instantiate an instance of the
  HTMLObject::Normal to work with.

=cut
sub processCloseAppEvent
{
  my $self = shift;
  my %args = ( doc => undef, log => 0, langEntry => "close", @_ );
  my $doc = $args{doc};
  my $langObj = $self->{langObj};
  my $portalSession = $self->{portalSession};
  my $sessionObj = $self->{sessionObj};
  my $applicationObj = $self->{applicationObj};
  my $userId = $self->{userObj}->{id};
  my $log = $args{log};
  my $logObj = $self->{logObj};
  my $langEntry = $args{langEntry};

  if ($log !~ /^(0|1)$/)
  {
    $self->invalid("log", $log);
  }
  if ($log && !defined $logObj)
  {
    $self->missing("logObj");
  }
  if (defined $logObj && !$logObj->isValid)
  {
    $self->error($logObj->errorMessage);
  }
  if ($langEntry !~ /^(.+)$/)
  {
    $self->invalid("langEntry", $langEntry, $self->{langObj}->map("langEntryNeeded"));
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $doc;
  }
  if ($self->error)
  {
    return $doc;
  }

  if (!$sessionObj->{store}->{okToLogout})
  {
    $doc = $self->run(command => "display");
    if ($self->error)
    {
      $self->prefixError();
      return $doc;
    }
  }
  else
  {
    my $close = $langObj->map($langEntry);
    my $title = sprintf($langObj->map("title"), $self->{appVersion}, $close);

    $doc = $self->prepDoc(doc => $doc, title => $title, template => "none");
    if ($self->error)
    {
      $self->prefixError();
      return $doc;
    }

    # generate the document that will close all open windows and then close itself.

    # create the closeWindow JS function.
    $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeWindow", langObj => $langObj);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return $doc;
    }

    my $onLoadCode = "";
    foreach my $window ($sessionObj->getWindows())
    {
      $onLoadCode .= "closeWindow('$window');\n";
    }

    my $windowName = $portalSession->generateHelpWindowName($self->{app});
    if ($portalSession->error)
    {
      $self->error($portalSession->errorMessage);
      return $doc;
    }
    if ($portalSession->windowExists($windowName))
    {
      $onLoadCode .= "closeWindow('$windowName');\n";
      $portalSession->unregisterWindow($windowName);
      if ($portalSession->error)
      {
        $self->error($portalSession->errorMessage);
        return $doc;
      }
    }

    $onLoadCode .= "\nwindow.close();\n";  # close ourself

    if (length $onLoadCode > 0)
    {
      $doc->setOnload(code => "$onLoadCode");  # output the JavaScript code that will be run on the onLoad event of the page.
    }

    $sessionObj->delete;
    if ($sessionObj->error)
    {
      $self->error($sessionObj->errorMessage);
      return $doc;
    }
    $sessionObj = undef;  # force it to go out of scope and thus delete the session.

    if ($log)
    {
      $self->doLog(action => 19, extraInfo => $self->{app}, userId => $userId, logObj => $logObj);
      if ($self->error)
      {
        $self->prefixError();
        return $doc;
      }
    }
  }

  return $doc;
}

=item HTMLObject c_display()

 Outputs the default Main Menu screen, suitable for state = Main.
 The prepDoc() template defaults to menubar+sidemenu-left if your
 app has not yet defined the prepDocTemplate config entry.

=cut
sub c_display
{
  my $self = shift;
  my $doc;

  my %args = ();
  $args{template} = "menubar+sidemenu-left" if (!exists $self->appConfigObj->{prepDocTemplate});

  $doc = $self->prepDoc(%args);
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # lookup the sidemenuLocation preference to substitute into the mainMenuMessage with.
  my $prefObj = $self->authObj->getUserPreferenceInfo(userId => $self->userObj->get("id"),
      app => $self->app, module => "Global", preference => "sidemenuLocation");
  if ($self->authObj->error)
  {
    $self->error($self->authObj->errorMessage);
    return undef;
  }

  my $location = (defined $prefObj ? $prefObj->get("value") : "left");
  my $mainMenuMessagePhrase = sprintf($self->{langObj}->map("mainMenuMessage"), $location);

  $doc->printTag(tag => "<maincontent>", value => $mainMenuMessagePhrase);

  return $doc;
}

=item HTMLObject c_close()

 The default close application command handler.

=cut
sub c_close
{
  my $self = shift;
  my $doc;

  $doc = $self->processCloseAppEvent();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  return $doc;
}

=item % printMenu(location, orientation)

 requires: location - (menubar, sidemenu)
 optional: orientation - (vertical/horizontal - only valid with
   location = 'menubar')
 returns:  hash with members:
   title - title of document to be set
   menu - generated menu string

 generates the specified Menu.

 Calls the current apps Objects::Menu module to provide any custom
 menu entries for the menu being generated.  The method, updateEntries()
 will be called and passed in the location and currently defined
 left, center, right menu arrays.  This allows the app to update the
 Refresh link, for example, and also add new menu entries or delete
 menu entries as they see fit.

 If location = 'menubar' then the following menu items will be created
 and passed into the apps Objects::Menu->updateEntries() method.

 left[0] = Close App link
 left[1] = Seperator
 left[2] = Help App link
 left[3] = Seperator
 left[4] = Refresh App link

 right[0] = User = (
 right[1] = 'user name'
 right[2] = )

 If location = 'menubar' and orientation = 'vertical', then only the left
 array entries will be defined.

 The Help entry is only generated as a link, if the
 appConfigObj->{helpDefined} entry is true (1), otherwise it is output
 as plain text.

 If location = 'sidemenu' then there are no default menu items created
 by this method.  It is upto the apps Objects::Menu->updateEntries()
 method to populate the left menu array.

=cut
sub printMenu
{
  my $self = shift;
  my %args = ( location => "menubar", orientation => "horizontal", @_ );
  my $location = $args{location};
  my $orientation = $args{orientation};
  my $state = $self->{state};
  my $command = $self->{input}->{command};
  my %result = ( title => "", menu => "" );

  if ($location !~ /^(menubar|sidemenu)$/)
  {
    $self->invalid("location", $location);
  }
  if ($location eq "menubar" && $orientation !~ /^(horizontal|vertical)$/)
  {
    $self->invalid("orientation", $orientation, "Valid values are 'horizontal' and 'vertical'.");
  }
  if (!exists $self->states->{$self->state})
  {
    $self->invalid("state", $self->state);
  }
  if (!exists $self->commands->{$self->input->{command}})
  {
    $self->invalid("command", $self->input->{command});
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return %result;
  }

  my $url = $self->url;

  my $closeUrl = $self->closeUrl;
  my $mainMenuUrl = $self->mainMenuUrl;
  my $refreshUrl = $self->refreshUrl;
  my $helpUrl = $self->helpUrl;

  my $close = $self->langObj->map("close");
  my $refresh = $self->langObj->map("refresh");
  my $help = $self->langObj->map("help");
  my $closeMessage = $self->langObj->map("closeMessage");
  my $userPhrase = $self->langObj->map("user");

  # build up the menuItems arrays
  my @itemsLeft = ();
  my @itemsRight = ();
  my @itemsCenter = ();

  if ($location eq "menubar")
  {
    my $helpWindowName = $self->{portalSession}->generateHelpWindowName($self->{app});
    if ($self->{portalSession}->error)
    {
      $self->error($self->{portalSession}->errorMessage);
      return %result;
    }
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$closeUrl", langObj => $self->{langObj},
                    text => $close, title => $close, onMouseOver => "window.status='$close'; return true;", onClick => "return validateClose();");
    $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    if ($self->{appConfigObj}->{helpDefined})
    {
      $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "link", url => "$helpUrl", langObj => $self->{langObj},
                    text => $help, onMouseOver => "window.status='$help'; return true;",
                    onClick => "showHelp('$helpWindowName', '$helpUrl', 480, 640); return false;", title => $help);
    }
    else
    {
      $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "text", text => $help, langObj => $self->{langObj});
    }
    $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "link", url => "$refreshUrl", langObj => $self->{langObj},
                    text => $refresh, onMouseOver => "window.status='$refresh'; return true;", title => "$refresh");

    if ($orientation eq "horizontal")
    {
      $itemsRight[0] = Portal::Objects::MenuItem->new(type => "text", text => "$userPhrase = (", langObj => $self->{langObj});
      $itemsRight[1] = Portal::Objects::MenuItem->new(type => "hilightedText", text => "$self->{userObj}->{uname}", langObj => $self->{langObj});
      $itemsRight[2] = Portal::Objects::MenuItem->new(type => "text", text => ")", langObj => $self->{langObj});
    }

    # now call the apps Objects::Menu->updateEntries passing in the arrays and work with the results.
    eval "use Portal::" . $self->app . "::Objects::Menu;";
    if ($@)
    {
      $self->error($@);
      return %result;
    }
    my $menuObj = undef;
    eval "\$menuObj = Portal::" . $self->app . "::Objects::Menu->new(langObj => \$self->langObj, methods => \$self->methods, urls => \$self->{urls}, portalSession => \$self->portalSession, sessionObj => \$self->sessionObj, appConfigObj => \$self->appConfigObj, state => \$self->state, commands => \$self->commands);";
    if ($@)
    {
      $self->error($@);
      return %result;
    }
    if ($menuObj->error)
    {
      $self->error($menuObj->errorMessage);
      return %result;
    }

    # now call the updateEntries() method
    my %tmpResult = $menuObj->updateEntries(location => $location, orientation => $orientation, input => $self->input, left => \@itemsLeft, center => \@itemsCenter, right => \@itemsRight);
    if ($menuObj->error)
    {
      $self->error($menuObj->errorMessage);
      return %result;
    }

    $result{title} = $tmpResult{title} if (length $tmpResult{title} > 0);

    @itemsLeft = @{$tmpResult{left}};
    @itemsCenter = @{$tmpResult{center}};
    @itemsRight = @{$tmpResult{right}};

    # now validate they created ok.
    for (my $i=0; $i < scalar @itemsLeft; $i++)
    {
      if ($itemsLeft[$i]->error)
      {
        $self->error("itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
        return %result;
      }
    }
    for (my $i=0; $i < scalar @itemsCenter; $i++)
    {
      if ($itemsCenter[$i]->error)
      {
        $self->error("itemsCenter[$i]\n<br>" . $itemsCenter[$i]->errorMessage);
        return %result;
      }
    }
    for (my $i=0; $i < scalar @itemsRight; $i++)
    {
      if ($itemsRight[$i]->error)
      {
        $self->error("itemsRight[$i]\n<br>" . $itemsRight[$i]->errorMessage);
        return %result;
      }
    }
    my %menuArgs = ();
    $menuArgs{itemsCenter} = \@itemsCenter if (scalar @itemsCenter > 0);
    $menuArgs{itemsRight} = \@itemsRight if (scalar @itemsRight > 0);

    $result{menu} = $self->{methods}->displayMenu(orientation => $orientation,
                    itemsLeft => \@itemsLeft, indent => 6, %menuArgs);
  }
  elsif ($location eq "sidemenu")
  {
    # now call the apps Objects::Menu->updateEntries passing in the arrays and work with the results.
    eval "use Portal::" . $self->app . "::Objects::Menu;";
    if ($@)
    {
      $self->error($@);
      return %result;
    }
    my $menuObj = undef;
    eval "\$menuObj = Portal::" . $self->app . "::Objects::Menu->new(langObj => \$self->langObj, methods => \$self->methods, urls => \$self->{urls}, portalSession => \$self->portalSession, sessionObj => \$self->sessionObj, appConfigObj => \$self->appConfigObj, state => \$self->state, commands => \$self->commands);";
    if ($@)
    {
      $self->error($@);
      return %result;
    }
    if ($menuObj->error)
    {
      $self->error($menuObj->errorMessage);
      return %result;
    }

    # now call the updateEntries() method
    my %tmpResult = $menuObj->updateEntries(location => $location, input => $self->input, left => \@itemsLeft);
    if ($menuObj->error)
    {
      $self->error($menuObj->errorMessage);
      return %result;
    }

    $result{title} = $tmpResult{title} if (length $tmpResult{title} > 0);

    @itemsLeft = @{$tmpResult{left}};

    # now validate they created ok.
    for (my $i=0; $i < scalar @itemsLeft; $i++)
    {
      if ($itemsLeft[$i]->error)
      {
        $self->error("itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
        return %result;
      }
    }
    $result{menu} = $self->{methods}->displayMenu(orientation => "vertical",
                    itemsLeft => \@itemsLeft, indent => 6);
  }
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return %result;
  }

  return %result;
}

=item scalar arguments(void)

 Returns a string that when eval'ed will specify all the
 arguments to pass into an instance of the Portal::AppState->new
 method.

 Ex:

 "langObj => \$self->{langObj}, input => \$self->{input}"

=cut

sub arguments
{
  my $self = shift;
  my $args = $self->SUPER::arguments();

  foreach my $arg (qw(cookieObj input logObj configObj appConfigObj methods authObj applicationObj browserType browserVersion browserMaker dbHandle appObj userObj companyObj portalSession sessionObj app appVersion states commands logOk useAppStateCommandSecurity portalVariables))
  {
    $args .= ", " if ($args);
    $args .= "$arg => \$self->{$arg}";
  }

  return $args;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
