# Application.pm - The Object Class that provides a Application Object
# Created by James A. Pattie, 11/07/2000.
# Last Updated 07/27/2001 - POD documentation, Logging, setError use.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Application;
use strict;
use Portal::Base;
use DBIWrapper;
use Portal::Data::Variables;
use Portal::Objects::ApplicationObject;
use Portal::Objects::AppServerObject;
use Portal::Objects::CompanyObject;
use Portal::Objects::CompanyApplicationObject;
use Portal::Objects::User;
use Portal::Auth;  # This is needed for the Company and User validation routines.
use Portal::Objects::LicenseObject;
use Portal::Objects::UserApplicationObject;
use Portal::Replication;
use Portal::Objects::ReplicationObject;
use Portal::Objects::DynamicContentObject;
use Portal::Data::ColorScheme;
use Portal::Session;
use Portal::Methods;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.13';

=head1 NAME

Application - Object used to build a Application Object Class.

=head1 SYNOPSIS

  use Portal::Application;
  my $obj = Portal::Application->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Application is a Application class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(portalDB, sessionObj)

 Creates a new instance of the Application module.
 See Portal::Base(3) for a listing of required arguments.

 requires: portalDB - DBIWrapper pointing to the Portal Database
 optional: sessionObj - Portal Session

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( portalDB => undef, sessionObj => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->langObj);
  $self->{portalDB} = $args{portalDB};
  $self->{sessionObj} = $args{sessionObj};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  $self->{replicationObj} = Portal::Replication->new(portalDB => $self->portalDB, sessionObj => $self->sessionObj, langObj => $self->langObj);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return $self;
  }

  $self->{methods} = Portal::Methods->new(langObj => $self->langObj);
  if ($self->methods->error)
  {
    $self->prefixError($self->methods->errorMessage);
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->portalDB)
  {
    $self->missing("portalDB");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# Application helper functions.

=item ApplicationObject getApplicationInfo(id, name)

 takes: id or name
 returns: ApplicationObject or undef if Application not found.

 Uses either the id or name value for the application and looks them
 up and returns the ApplicationObject with the info for that
 application.  If the application is not found, we return undef.  If
 an error occurred, we also return undef and set the error condition.

=cut

sub getApplicationInfo
{
  my $self = shift;
  my %args = ( @_ );
  my $type = "";
  my $applicationObj = undef;

  if (exists $args{id})
  {
    $type = "id";
    if ($args{$type} !~ /^(\d+)$/)
    {
      $self->invalid("id", $args{$type});
    }
  }
  elsif (exists $args{name})
  {
    $type = "name";
    if (length $args{$type} == 0)
    {
      $self->missing("name");
    }
  }
  else
  {
    $self->missing("id or name");
  }

  # see if we encountered any errors.
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my $sth = $self->portalDB->read(sql => "SELECT id, name, description, icon_name, server, port, cost, unit, type, admin_type, height, width, db_host, db_type, db_port, wap FROM app_tb WHERE $type = ?",
                                plug => [ $args{$type} ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return undef;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $applicationObj = Portal::Objects::ApplicationObject->new(id => $info[0], name => $info[1], description => $info[2], iconName => $info[3], server => $info[4], port => $info[5],
                  cost => $info[6], unit => $info[7], type => $info[8], adminType => $info[9], height => $info[10], width => $info[11], dbHost => $info[12], dbType => $info[13], dbPort => $info[14],
                  wap => $info[15], langObj => $self->langObj);
  }

  return $applicationObj;
}

# createApplication
# takes: applicationObj - ApplicationObject
# optional: log
# returns: 1=OK, 0=Error, -1 if it already exists.
sub createApplication
{
  my $self = shift;
  my %args = ( applicationObj => undef, log => 1, @_ );
  my $applicationObj = $args{applicationObj};
  my $log = $args{log};

  if (not defined $applicationObj)
  {
    $self->error("applicationObj is not defined!<br>\n");
    return 0;
  }

  if (!$applicationObj->isValid)
  {
    $self->error($applicationObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # first make sure this application doesn't exist.
  my $tempApplication = $self->getApplicationInfo(name => $applicationObj->get("name"));
  if (not defined $tempApplication)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # application doesn't exist yet!
    {
      my $description = $applicationObj->get("description");
      # make sure that an ' quotes are escaped in the comment string.
      $description =~ s/([^\\][^\\])'/$1\\'/g;  # only do it to those ' that are not already escaped.
      my $wap = $applicationObj->get("wap");

      $self->portalDB->write(sql => "INSERT INTO app_tb (name, description, icon_name, server, port, cost, unit, type, admin_type, height, width, db_host, db_type, db_port, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                               plug => [ $applicationObj->get("name"), $description, $applicationObj->get("iconName"), $applicationObj->get("server"), $applicationObj->get("port"), $applicationObj->get("cost"),
                                         $applicationObj->get("unit"), $applicationObj->get("type"), $applicationObj->get("adminType"), $applicationObj->get("height"), $applicationObj->get("width"),
                                         $applicationObj->get("dbHost"), $applicationObj->get("dbType"), $applicationObj->get("dbPort"), $wap ]);
      if ($self->portalDB->error)
      {
        $self->error($self->portalDB->errorMessage);
        return 0;
      }
      $self->portalDB->commit;  # make the changes permanent.
      if ($self->portalDB->error)
      {
        $self->error($self->portalDB->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Application Create event.
        $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 24, extraInfo => "Application = '" . $applicationObj->get("name") . "' Registered", userId => (defined $self->sessionObj ? "" : -1));
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return 0;
        }
      }

      # now see if we need to replicate this change.
      my @repDBs = $self->replicationObj->getEntries(tbName => "app_tb", createDBIObjects => 1);
      if ($self->replicationObj->error)
      {
        $self->error($self->replicationObj->errorMessage);
        return 0;
      }
      my $appId = -1;
      if (scalar @repDBs > 0)
      {
        my $tempApplication = $self->getApplicationInfo(name => $applicationObj->get("name"));
        if (not defined $tempApplication)
        {
          if ($self->error)
          {
            $self->error("In Replication:<br>\n" . $self->errorMessage);
            return 0;
          }
          else  # application doesn't exist!
          {
            $self->error("In Replication:<br>\nApplication = '" . $applicationObj->get("name") . "' no longer exists and we just Registered it!<br>\n");
            return 0;
          }
        }
        else
        {
          if ($tempApplication->error)
          {
            $self->error("In Replication:<br>\n" . $tempApplication->errorMessage);
            return 0;
          }
          else
          {
            $appId = $tempApplication->get("id");
          }
        }
      }
      foreach my $repObj (@repDBs)
      {
        # do the replication.
        $repObj->dbObj->write(sql => "INSERT INTO app_tb (id, name, description, icon_name, server, port, cost, unit, type, admin_type, height, width, db_host, db_type, db_port, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                               plug => [ $appId, $applicationObj->get("name"), $description, $applicationObj->get("iconName"), $applicationObj->get("server"), $applicationObj->get("port"), $applicationObj->get("cost"),
                                         $applicationObj->get("unit"), $applicationObj->get("type"), $applicationObj->get("adminType"), $applicationObj->get("height"), $applicationObj->get("width"),
                                         $applicationObj->get("dbHost"), $applicationObj->get("dbType"), $applicationObj->get("dbPort"), $wap ]);
        if ($repObj->dbObj->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
          return 0;
        }
        else
        {
          $repObj->dbObj->commit;  # have to save the changes in the database.
          if ($repObj->dbObj->error)
          {
            $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
            return 0;
          }

          # now log the replication event
          $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Replicated App = '" . $applicationObj->get("name") . "' to " . $repObj->print, userId => (defined $self->sessionObj ? "" : -1));
          if ($self->methods->error)
          {
            $self->error($self->methods->errorMessage);
            return 0;
          }

          # log the replication event into the replication database.
          $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Replicated App = '" . $applicationObj->get("name") . "'", userId => (defined $self->sessionObj ? "" : -1));
          if ($self->methods->error)
          {
            $self->error($self->methods->errorMessage);
            return 0;
          }
        }
      } # end of replication code
    }
  }
  else
  {
    if ($tempApplication->error)
    {
      $self->error($tempApplication->errorMessage);
      return 0;
    }
    else
    {
      return -1;
    }
  }

  return 1;
}

# updateApplication
# takes: applicationObj - ApplicationObject
# optional: log
# returns: 1 if ok, 0 if error, -1 if app doesn't exist
# This method will take the ApplicationObject and update all fields in the database other than the id field.
sub updateApplication
{
  my $self = shift;
  my %args = ( applicationObj => undef, log => 1, @_ );
  my $applicationObj = $args{applicationObj};
  my $log = $args{log};

  if (not defined $applicationObj)
  {
    $self->error("applicationObj is not defined!<br>\n");
    return 0;
  }
  if (!$applicationObj->isValid)
  {
    $self->error("applicationObj is not valid!<br>\n" . $applicationObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # validate the application exists in the database.
  my $appInfo = $self->getApplicationInfo(id => $applicationObj->get("id"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (not defined $appInfo)  # The application doesn't exist!
  {
    return -1;
  }

  # update the entry.
  my $wap = $applicationObj->get("wap");

  $self->portalDB->write(sql => "UPDATE app_tb SET name = ?, description = ?, icon_name = ?, server = ?, port = ?, cost = ?, unit = ?, type = ?, admin_type = ?, height = ?, width = ?, db_host = ?, db_type = ?, db_port = ?, wap = ? WHERE id = ?",
                           plug => [ $applicationObj->get("name"), $applicationObj->get("description"), $applicationObj->get("iconName"), $applicationObj->get("server"), $applicationObj->get("port"),
                                     $applicationObj->get("cost"), $applicationObj->get("unit"), $applicationObj->get("type"), $applicationObj->get("adminType"), $applicationObj->get("height"),
                                     $applicationObj->get("width"), $applicationObj->get("dbHost"), $applicationObj->get("dbType"), $applicationObj->get("dbPort"), $wap, $applicationObj->get("id") ] );
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Update event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 10, extraInfo => "Application = '" . $applicationObj->get("name") . "' Updated", userId => (defined $self->sessionObj ? "" : -1));
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "UPDATE app_tb SET name = ?, description = ?, icon_name = ?, server = ?, port = ?, cost = ?, unit = ?, type = ?, admin_type = ?, height = ?, width = ?, db_host = ?, db_type = ?, db_port = ?, wap = ? WHERE id = ?",
                           plug => [ $applicationObj->get("name"), $applicationObj->get("description"), $applicationObj->get("iconName"), $applicationObj->get("server"), $applicationObj->get("port"),
                                     $applicationObj->get("cost"), $applicationObj->get("unit"), $applicationObj->get("type"), $applicationObj->get("adminType"), $applicationObj->get("height"),
                                     $applicationObj->get("width"), $applicationObj->get("dbHost"), $applicationObj->get("dbType"), $applicationObj->get("dbPort"), $wap, $applicationObj->get("id") ] );
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Replicated Updated App = '" . $applicationObj->get("name") . "' to " . $repObj->print, userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Replicated Updated App = '" . $applicationObj->get("name") . "'");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# backupAppServerEntry
# takes: appServerObj - AppServerObject
# returns: 1 if ok, 0 if error, -1 if already backed up, -2 if app id doesn't exist
# This method, checks to see if the specified app id exists in the app_tb.  If it doesn't, it then returns -2, else
# it checks to see if the specified combination of id, server and port exists, if it does, then return -1, else
# create the entry in app_servers_tb and return 1 if ok, else return 0 and set the error.

# This method is only going to be used by applications that are installing themselves in a Portal.
sub backupAppServerEntry
{
  my $self = shift;
  my %args = ( appServerObj => undef, @_ );
  my $appServerObj = $args{appServerObj};

  if (not defined $appServerObj)
  {
    $self->error("appServerObj is not defined!<br>\n");
    return 0;
  }
  if (!$appServerObj->isValid)
  {
    $self->error("appServerObj is not valid!<br>\n" . $appServerObj->errorMessage);
    return 0;
  }
  # validate the application exists in the database.
  my $appInfo = $self->getApplicationInfo(id => $appServerObj->get("id"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (not defined $appInfo)  # The application doesn't exist!
  {
    return -2;
  }

  # check and see if it has already been backed up.
  my $sth = $self->portalDB->read(sql => "SELECT id, server, port, db_host, db_type, db_port FROM app_servers_tb WHERE id = ? AND server = ? AND port = ? AND db_host = ? AND db_type = ? AND db_port = ?",
                                    plug => [ $appServerObj->get("id"), $appServerObj->get("server"), $appServerObj->get("port"), $appServerObj->get("dbHost"), $appServerObj->get("dbType"), $appServerObj->get("dbPort") ] );
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @entry = $sth->fetchrow_array;
  if (defined $entry[0]) # the entry exists!
  {
    return -1;
  }
  # let's go ahead and make the backup entry.
  $self->portalDB->write(sql => "INSERT INTO app_servers_tb (id, server, port, db_host, db_type, db_port) VALUES (?, ?, ?, ?, ?, ?)",
                           plug => [ $appServerObj->get("id"), $appServerObj->get("server"), $appServerObj->get("port"), $appServerObj->get("dbHost"), $appServerObj->get("dbType"), $appServerObj->get("dbPort") ] );
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # commit the transaction.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "app_servers_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "INSERT INTO app_servers_tb (id, server, port, db_host, db_type, db_port) VALUES (?, ?, ?, ?, ?, ?)",
                           plug => [ $appServerObj->get("id"), $appServerObj->get("server"), $appServerObj->get("port"), $appServerObj->get("dbHost"), $appServerObj->get("dbType"), $appServerObj->get("dbPort") ] );
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Replicated AppServer = '" . $appServerObj->get("id") . "' to " . $repObj->print, userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Replicated AppServer = '" . $appServerObj->get("id") . "'", userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;  # backup succeeded.
}

=item @AppServerObject getAppServerEntries(id)

  takes: id
  returns: array of AppServerObject entries
  This method, returns an array of the entries in app_server_tb for the
  specified application ordered by the counter field with the entry
  from the app_tb having counter = -1.

  This allows you to see all the possible Portal Servers that
  Companies can run their app from.

=cut
sub getAppServerEntries
{
  my $self = shift;
  my %args = ( id => -1, @_ );
  my $id = $args{id};
  my @Entries = ();

  if ($id < 0)
  {
    $self->error("id = '$id' is not valid!<br>\n");
    return @Entries;
  }

  # get all entries for this application id.
  my $sth = $self->portalDB->read(sql => "SELECT id, server, port, db_host, db_type, db_port, (SELECT -1) AS counter FROM app_tb WHERE id = ? UNION SELECT id, server, port, db_host, db_type, db_port, counter FROM app_servers_tb WHERE id = ? ORDER BY counter",
                                    plug => [ $id, $id ] );
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @Entries;
  }
  while (my @entry = $sth->fetchrow_array)
  {
    my $obj = Portal::Objects::AppServerObject->new(id => $entry[0], server => $entry[1], port => $entry[2], dbHost => $entry[3], dbType => $entry[4], dbPort => $entry[5], counter => $entry[6], langObj => $self->langObj);
    if ($obj->error)
    {
      $self->error($obj->errorMessage);
      return @Entries;
    }
    $Entries[++$#Entries] = $obj;
  }

  return @Entries;
}

# assignAppToCompany
# takes: companyAppObj - CompanyApplicationObject
# optional: log
# returns: 1 if the assignment was possible and ok, 0 if an error occurred, -1 if app doesn't exist, -2 if company doesn't exist, -3 if assignment already exists
# This method will create the entry in company_app_tb for this companyAppObj.
sub assignAppToCompany
{
  my $self = shift;
  my %args = ( companyAppObj => undef, log => 1, @_ );
  my $companyAppObj = $args{companyAppObj};
  my $log = $args{log};

  if (not defined $companyAppObj)
  {
    $self->error("companyAppObj is not defined!<br>\n");
    return 0;
  }
  if (!$companyAppObj->isValid)
  {
    $self->error("companyAppObj is not valid!<br>\n" . $companyAppObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # validate the application exists in the database.
  my $appInfo = $self->getApplicationInfo(id => $companyAppObj->get("appId"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (not defined $appInfo)  # The application doesn't exist!
  {
    return -1;
  }

  # validate the company exists in the database.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  my $companyInfo = $authObj->getCompanyInfo(id => $companyAppObj->get("companyId"));
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (not defined $companyInfo)  # The company doesn't exist!
  {
    return -2;
  }

  # check and see if this assignment has already happened.
  my $sth = $self->portalDB->read(sql => "SELECT app_id, company_id FROM company_app_tb WHERE app_id = ? AND company_id = ?",
                                    plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0]) # entry exists!
  {
    return -3;
  }
  my $autorun = $companyAppObj->get("autorun");
  my $wap = $companyAppObj->get("wap");

  # create the entry in company_app_tb.
  $self->portalDB->write(sql => "INSERT INTO company_app_tb (app_id, company_id, number, server, port, cost, unit, db_name, height, width, autorun, db_host, db_type, db_port, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId"), $companyAppObj->get("number"), $companyAppObj->get("server"), $companyAppObj->get("port"), $companyAppObj->get("cost"),
                                     $companyAppObj->get("unit"), $companyAppObj->get("dbName"), $companyAppObj->get("height"), $companyAppObj->get("width"), $autorun,
                                     $companyAppObj->get("dbHost"), $companyAppObj->get("dbType"), $companyAppObj->get("dbPort"), $wap ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make it permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Assignment to Company event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 11, extraInfo => "Application = '" . $appInfo->get("name") . "' Purchased by Company = '" . $companyInfo->get("code") . "'", userId => (defined $self->sessionObj ? "" : -1));
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "company_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "INSERT INTO company_app_tb (app_id, company_id, number, server, port, cost, unit, db_name, height, width, autorun, db_host, db_type, db_port, wap) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId"), $companyAppObj->get("number"), $companyAppObj->get("server"), $companyAppObj->get("port"), $companyAppObj->get("cost"),
                                     $companyAppObj->get("unit"), $companyAppObj->get("dbName"), $companyAppObj->get("height"), $companyAppObj->get("width"), $autorun,
                                     $companyAppObj->get("dbHost"), $companyAppObj->get("dbType"), $companyAppObj->get("dbPort"), $wap ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Replicated App with ID = '" . $companyAppObj->get("appId") . "' for Company with ID = '" . $companyAppObj->get("companyId") . "' to " . $repObj->print, userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Replicated App with ID = '" . $companyAppObj->get("appId") . "' for Company with ID = '" . $companyAppObj->get("companyId") . "'", userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# getAppsForCompany
# takes: companyObj - CompanyObject
# optional: wap
# returns: array of ApplicationObjects where the server, port, cost, unit, height, width and autorun values are filled in from the company_app_tb but everything else is from the app_tb.
# This method gathers all applications assigned to the specified company and returns the array of ApplicationObjects that define them.  If specified, wap is used
# to select only those apps that match the value of wap.
sub getAppsForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $wapSearch = (exists $args{wap} ? $args{wap} : "");
  my @Entries = ();

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return @Entries;
  }
  if (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return @Entries;
  }
  if (length $wapSearch > 0 && $wapSearch !~ /^(1|0)$/)
  {
    $self->error("wap = '$wapSearch' is invalid!<br>\n");
    return @Entries;
  }

  my $sth = $self->portalDB->read(sql => "SELECT app_id, company_id, number, server, port, cost, unit, db_name, height, width, autorun, db_host, db_type, db_port, wap FROM company_app_tb WHERE company_id = ?" .
                                    (length $wapSearch > 0 ? " AND wap = '$wapSearch'" : "") . " ORDER BY app_id",
                                    plug => [ $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @Entries;
  }
  while (my @info = $sth->fetchrow_array)
  {
    # get the ApplicationObject for this app and update it.
    my $appObj = $self->getApplicationInfo(id => $info[0]);
    if ($self->error)
    {
      $self->prefixError();
      return @Entries;
    }
    if ($appObj->error)
    {
      $self->error($appObj->errorMessage);
      return @Entries;
    }
    $appObj->set(number => $info[2]);
    $appObj->set(server => $info[3]);
    $appObj->set(port => $info[4]);
    $appObj->set(cost => $info[5]);
    $appObj->set(unit => $info[6]);
    $appObj->set(dbName => $info[7]);
    $appObj->set(height => $info[8]);
    $appObj->set(width => $info[9]);
    $appObj->set(autorun => $info[10]);
    $appObj->set(dbHost => $info[11]);
    $appObj->set(dbType => $info[12]);
    $appObj->set(dbPort => $info[13]);
    $appObj->set(wap => $info[14]);

    $Entries[++$#Entries] = $appObj;
  }

  return @Entries;
}

=item @ApplicationObject getUnAssignedAppsForCompany(companyObj, wap)

 takes: companyObj - CompanyObject
 optional: wap
 returns: array of ApplicationObjects where the app is not assigned
 to the specified company.  If specified, wap is used
 to select only those apps that match the value of wap.

=cut
sub getUnAssignedAppsForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $wapSearch = (exists $args{wap} ? $args{wap} : "");
  my @Entries = ();

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return @Entries;
  }
  if (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return @Entries;
  }
  if (length $wapSearch > 0 && $wapSearch !~ /^(1|0)$/)
  {
    $self->error("wap = '$wapSearch' is invalid!<br>\n");
    return @Entries;
  }

  my $sth = $self->portalDB->read(sql => "SELECT id FROM app_tb WHERE id NOT IN (SELECT app_id FROM company_app_tb WHERE company_id = ?)" .
                                    (length $wapSearch > 0 ? " AND wap = '$wapSearch'" : "") . " ORDER BY id",
                                    plug => [ $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @Entries;
  }
  while (my @info = $sth->fetchrow_array)
  {
    # get the ApplicationObject for this app and update it.
    my $appObj = $self->getApplicationInfo(id => $info[0]);
    if ($self->error)
    {
      $self->prefixError();
      return @Entries;
    }
    if ($appObj->error)
    {
      $self->error($appObj->errorMessage);
      return @Entries;
    }

    $Entries[++$#Entries] = $appObj;
  }

  return @Entries;
}

=item @User getUsersAssignedToAppForCompany(companyObj, appObj)

 takes: companyObj - CompanyObject
        appObj - ApplicationObj
 returns: array of User objects for those users that are assigned
 this app in the specified company.

=cut
sub getUsersAssignedToAppForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, appObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $appObj = $args{appObj};
  my @Users = ();

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return @Users;
  }
  elsif (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return @Users;
  }
  if (not defined $appObj)
  {
    $self->error("appObj is not defined!<br>\n");
    return @Users;
  }
  elsif (!$appObj->isValid)
  {
    $self->error("appObj is not valid!<br>\n" . $appObj->errorMessage);
    return @Users;
  }

  my $sth = $self->portalDB->read(sql => "SELECT user_id FROM user_app_tb WHERE app_id = ? AND company_id = ? ORDER BY user_id",
                                    plug => [ $appObj->get("id"), $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @Users;
  }

  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return @Users;
  }

  while (my @info = $sth->fetchrow_array)
  {
    # get the User Object for this entry.
    my $userObj = $authObj->getUserInfo(id => $info[0]);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return @Users;
    }
    if (not defined $userObj)  # The user doesn't exist!  This shouldn't happen!
    {
      $self->error("user with id = '$info[0]' does not exist, but has app = '" . $appObj->get("name") . "' assigned to them!");
      return @Users;
    }
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return @Users;
    }

    $Users[++$#Users] = $userObj;
  }

  return @Users;
}

=item UserApplicationObject getUserAppInfo(userObj, appObj)

 takes: userObj - User Object
        appObj - ApplicationObj
 returns: UserApplicationObject representing the users settings for
 this app or undef on error.

=cut
sub getUserAppInfo
{
  my $self = shift;
  my %args = ( userObj => undef, appObj => undef, @_ );
  my $userObj = $args{userObj};
  my $appObj = $args{appObj};
  my $userAppObj = undef;

  if (not defined $userObj)
  {
    $self->error("userObj is not defined!<br>\n");
    return $userAppObj;
  }
  elsif (!$userObj->isValid)
  {
    $self->error("userObj is not valid!<br>\n" . $userObj->errorMessage);
    return $userAppObj;
  }
  if (not defined $appObj)
  {
    $self->error("appObj is not defined!<br>\n");
    return $userAppObj;
  }
  elsif (!$appObj->isValid)
  {
    $self->error("appObj is not valid!<br>\n" . $appObj->errorMessage);
    return $userAppObj;
  }

  my $sth = $self->portalDB->read(sql => "SELECT height, width, autorun, wap FROM user_app_tb WHERE app_id = ? AND company_id = ? AND user_id = ?",
                                    plug => [ $appObj->get("id"), $userObj->get("companyId"), $userObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return $userAppObj;
  }

  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $userAppObj = Portal::Objects::UserApplicationObject->new(langObj => $self->langObj);
    $userAppObj->populate(companyId => $userObj->get("companyId"),
      appId => $appObj->get("id"), userId => $userObj->get("id"), height => $info[0],
      width => $info[1], autorun => $info[2], wap => $info[3]);
  }

  return $userAppObj;
}

# getCompanyAppInfo
# takes: companyId, appId
# returns: companyAppObj CompanyApplicationObject or undef if error or not found.
sub getCompanyAppInfo
{
  my $self = shift;
  my %args = ( companyId => -1, appId => -1, @_ );
  my $companyId = $args{companyId};
  my $appId = $args{appId};
  my $companyAppObj = undef;

  if ($companyId < 0)
  {
    $self->error("companyId = '$companyId' is invalid!<br>\n");
    return undef;
  }
  if ($appId < 0)
  {
    $self->error("appId = '$appId' is invalid!<br>\n");
    return undef;
  }
  my $sth = $self->portalDB->read(sql => "SELECT app_id, company_id, number, server, port, cost, unit, db_name, height, width, autorun, db_host, db_type, db_port, wap FROM company_app_tb WHERE app_id = ? AND company_id = ?",
                                    plug => [ $appId, $companyId ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return undef;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0]) # entry exists!
  {
    $companyAppObj = Portal::Objects::CompanyApplicationObject->new(langObj => $self->langObj);
    if ($companyAppObj->error)
    {
      $self->error($companyAppObj->errorMessage);
      return undef;
    }
    $companyAppObj->populate(appId => $info[0], companyId => $info[1], number => $info[2], server => $info[3],
      port => $info[4], cost => $info[5], unit => $info[6], dbName => $info[7], height => $info[8], width => $info[9],
      autorun => $info[10], dbHost => $info[11], dbType => $info[12], dbPort => $info[13], wap => $info[14]);
    if ($companyAppObj->error)
    {
      $self->error($companyAppObj->errorMessage);
      return undef;
    }
  }
  else
  {
    return undef;  # not found!
  }

  return $companyAppObj;
}

=item int updateCompanyAppInfo(companyAppObj, log)

 takes: companyAppObj - CompanyApplicationObject
 optional: log
 returns:
   1 if the update was ok,
   0 if an error occurred,
   -1 if app doesn't exist,
   -2 if company doesn't exist,
   -3 if app not assigned to company

 This method will update the entry in company_app_tb for this
 companyAppObj.

=cut
sub updateCompanyAppInfo
{
  my $self = shift;
  my %args = ( companyAppObj => undef, log => 1, @_ );
  my $companyAppObj = $args{companyAppObj};
  my $log = $args{log};

  if (not defined $companyAppObj)
  {
    $self->error("companyAppObj is not defined!<br>\n");
    return 0;
  }
  if (!$companyAppObj->isValid)
  {
    $self->error("companyAppObj is not valid!<br>\n" . $companyAppObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # validate the application exists in the database.
  my $appInfo = $self->getApplicationInfo(id => $companyAppObj->get("appId"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (not defined $appInfo)  # The application doesn't exist!
  {
    return -1;
  }

  # validate the company exists in the database.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  my $companyInfo = $authObj->getCompanyInfo(id => $companyAppObj->get("companyId"));
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (not defined $companyInfo)  # The company doesn't exist!
  {
    return -2;
  }

  # check and see if this assignment has already happened.
  my $sth = $self->portalDB->read(sql => "SELECT app_id, company_id FROM company_app_tb WHERE app_id = ? AND company_id = ?",
                                    plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0]) # entry doesn't exist!
  {
    return -3;
  }
  my $autorun = $companyAppObj->get("autorun");
  my $wap = $companyAppObj->get("wap");

  # create the entry in company_app_tb.
  $self->portalDB->write(sql => "UPDATE company_app_tb SET server = ?, port = ?, db_host = ?, db_port = ?, db_type = ?, height = ?, width = ?, autorun = ?, wap = ?, number = ? WHERE app_id = ? AND company_id = ?",
                           plug => [ $companyAppObj->get("server"), $companyAppObj->get("port"), $companyAppObj->get("dbHost"), $companyAppObj->get("dbPort"), $companyAppObj->get("dbType"), $companyAppObj->get("height"), $companyAppObj->get("width"), $autorun, $wap, $companyAppObj->get("number"), $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make it permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Assignment to Company event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 12, extraInfo => "CompanyApplicationObject: " . $companyAppObj);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "company_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "UPDATE company_app_tb SET server = ?, port = ?, db_host = ?, db_port = ?, db_type = ?, height = ?, width = ?, autorun = ?, wap = ? WHERE app_id = ? AND company_id = ?",
                           plug => [ $companyAppObj->get("server"), $companyAppObj->get("port"), $companyAppObj->get("dbHost"), $companyAppObj->get("dbPort"), $companyAppObj->get("dbType"), $companyAppObj->get("height"), $companyAppObj->get("width"), $autorun, $wap, $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Replicated CompanyApplicationObject: " . $companyAppObj . " to " . $repObj->print);
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Replicated CompanyApplicationObject: " . $companyAppObj);
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# unAssignAppFromCompany
# takes: companyAppObj - CompanyApplicationObject (The number of licenses can be 1)
# optional: log
# returns: 1 if the unassignment was possible and ok, 0 if an error occurred, -1 if assignment didn't exist
# This method will delete the entry from the company_app_tb and also delete all entries from the user_app_tb for this company and app combination.
sub unAssignAppFromCompany
{
  my $self = shift;
  my %args = ( companyAppObj => undef, log => 1, @_ );
  my $companyAppObj = $args{companyAppObj};
  my $log = $args{log};

  if (not defined $companyAppObj)
  {
    $self->error("companyAppObj is not defined!<br>\n");
    return 0;
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->error("companyAppObj is not valid!<br>\n" . $companyAppObj->errorMessage);
    return 0;
  }

  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # verify that the company, app and assignment exists.
  my $sth = $self->portalDB->read(sql => "SELECT db_name FROM company_app_tb WHERE app_id = ? AND company_id = ?", plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (not defined $info[0])
  {
    return -1;  # The assignment doesn't exist!
  }
  else
  {
    # delete the entry and then delete all entries from the user_app_tb also for this appId, companyId pair.
    $self->portalDB->write(sql => "DELETE FROM company_app_tb WHERE app_id = ? AND company_id = ?", plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
    if ($self->portalDB->error)
    {
      $self->error($self->portalDB->errorMessage);
      return 0;
    }
    $self->portalDB->write(sql => "DELETE FROM user_app_tb WHERE app_id = ? AND company_id = ?", plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
    if ($self->portalDB->error)
    {
      $self->error($self->portalDB->errorMessage);
      return 0;
    }
    $self->portalDB->commit;
    if ($self->portalDB->error)
    {
      $self->error($self->portalDB->errorMessage);
      return 0;
    }

    if ($log)
    { # log the Application UnPurchase event.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 15, extraInfo => "Application with ID = '" . $companyAppObj->get("appId") . "' UnPurchased by Company with ID = '" . $companyAppObj->get("companyId") . "'");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }

    # now see if we need to replicate this change.
    my @repDBs = $self->replicationObj->getEntries(tbName => "company_app_tb", createDBIObjects => 1);
    if ($self->replicationObj->error)
    {
      $self->error($self->replicationObj->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->dbObj->write(sql => "DELETE FROM company_app_tb WHERE app_id = ? AND company_id = ?", plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }
      else
      {
        $repObj->dbObj->commit;  # have to save the changes in the database.
        if ($repObj->dbObj->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Unassigned App with ID = '" . $companyAppObj->get("appId") . "' from Company with ID = '" . $companyAppObj->get("companyId") . "' for " . $repObj->print);
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Unassigned App with ID = '" . $companyAppObj->get("appId") . "' for Company with ID = '" . $companyAppObj->get("companyId") . "'");
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return 0;
        }
      }
    } # end of replication code

    my @repDBs = $self->replicationObj->getEntries(tbName => "user_app_tb", createDBIObjects => 1);
    if ($self->replicationObj->error)
    {
      $self->error($self->replicationObj->errorMessage);
      return 0;
    }
    foreach my $repObj (@repDBs)
    {
      # do the replication.
      $repObj->dbObj->write(sql => "DELETE FROM user_app_tb WHERE app_id = ? AND company_id = ?", plug => [ $companyAppObj->get("appId"), $companyAppObj->get("companyId") ]);
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }
      else
      {
        $repObj->dbObj->commit;  # have to save the changes in the database.
        if ($repObj->dbObj->error)
        {
          $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
          return 0;
        }

        # now log the replication event
        $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Unassigned App with ID = '" . $companyAppObj->get("appId") . "' from Company with ID = '" . $companyAppObj->get("companyId") . "' from all users of the company for " . $repObj->print);
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return 0;
        }

        # log the replication event into the replication database.
        $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Unassigned App with ID = '" . $companyAppObj->get("appId") . "' for Company with ID = '" . $companyAppObj->get("companyId") . "' from all users of the company.");
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return 0;
        }
      }
    } # end of replication code
  }

  return 1;
}

# removeAllAppsForCompany
# takes: companyObj
# optional: log
# returns: 1=OK, 0=Error
sub removeAllAppsForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, log => 1, @_ );
  my $companyObj = $args{companyObj};
  my $log = $args{log};

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return 0;
  }
  if (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return 0;
  }

  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # delete all app entries and then delete all entries from the user_app_tb also for this companyId.
  $self->portalDB->write(sql => "DELETE FROM company_app_tb WHERE company_id = ?", plug => [ $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->write(sql => "DELETE FROM user_app_tb WHERE company_id = ?", plug => [ $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Create event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 15, extraInfo => "All applications removed from Company = '" . $companyObj->get("code") . "'");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "company_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "DELETE FROM company_app_tb WHERE company_id = ?", plug => [ $companyObj->get("id") ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Unassigned All Apps from Company with ID = '" . $companyObj->get("id") . "' for " . $repObj->print);
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Unassigned All Apps for Company with ID = '" . $companyObj->get("id") . "'");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  my @repDBs = $self->replicationObj->getEntries(tbName => "user_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "DELETE FROM user_app_tb WHERE company_id = ?", plug => [ $companyObj->get("id") ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Unassigned All Apps from Company with ID = '" . $companyObj->get("id") . "' from all users of the company for " . $repObj->print);
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Unassigned All Apps for Company with ID = '" . $companyObj->get("id") . "' from all users of the company.");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# getCompanyLicenseInfo
# takes: companyId, appId
# returns: LicenseObject or undef if error occurred.
# This method checks to make sure the app is assigned to the company.  If not then errorCode = -1, else errorCode = 0 on normal error.
# Then the value in the assignment is used for the numTotal value.  A count of all users assigned that app in that company is then
# done and placed in numFree as numTotal - count.  The cost and unit values from the assignment record are then placed in the object.
sub getCompanyLicenseInfo
{
  my $self = shift;
  my %args = ( companyId => -1, appId => -1, @_ );
  my $companyId = $args{companyId};
  my $appId = $args{appId};
  my $licenseObj = undef;

  if ($companyId < 0)
  {
    $self->error("companyId = '$companyId' is invalid!<br>\n");
    $self->errorCode(0);
    return $licenseObj;
  }
  if ($appId < 0)
  {
    $self->error("appId = '$appId' is invalid!<br>\n");
    $self->errorCode(0);
    return $licenseObj;
  }

  # lookup the entry in company_app_tb.
  my $sth = $self->portalDB->read(sql => "SELECT number, cost, unit FROM company_app_tb WHERE app_id = ? AND company_id = ?",
                                    plug => [ $appId, $companyId ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return $licenseObj;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0])
  {
    # not found!
    $self->errorCode(-1);
    return $licenseObj;
  }
  my $numTotal = $info[0];
  my $cost = $info[1];
  my $unit = $info[2];

  # calculate the number of licenses being used.
  $sth = $self->portalDB->read(sql => "SELECT COUNT(user_id) FROM user_app_tb WHERE app_id = ? AND company_id = ?",
                                 plug => [ $appId, $companyId ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return $licenseObj;
  }
  @info = $sth->fetchrow_array;
  my $inUse = $info[0];

  $licenseObj = Portal::Objects::LicenseObject->new(companyId => $companyId, appId => $appId, numTotal => $numTotal, numFree => ($numTotal == 0 ? $inUse : ($numTotal-$inUse)), cost => $cost, unit => $unit, langObj => $self->langObj);
  if ($licenseObj->error)
  {
    $self->error($licenseObj->errorMessage);
    return undef;
  }

  return $licenseObj;
}

# assignAppToUser
# takes: userAppObj - UserApplicationObject
# optional: log
# returns: 1 if ok, 0 if error, -1 if app doesn't exist, -2 if company doesn't exist, -3 if user doesn't exist, -4 if assignment already done, -5 if app not assigned to company, -6 if license
# allotment will be violated, -7 if app is administration and user is not an admin
# This method will assign the specified company application to the specified user in the user_app_tb.
sub assignAppToUser
{
  my $self = shift;
  my %args = ( userAppObj => undef, log => 1, @_ );
  my $userAppObj = $args{userAppObj};
  my $log = $args{log};

  if (not defined $userAppObj)
  {
    $self->error("userAppObj is not defined!<br>\n");
    return 0;
  }
  if (!$userAppObj->isValid)
  {
    $self->error("userAppObj is not valid!<br>\n" . $userAppObj->errorMessage);
    return 0;
  }

  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # validate the app exists
  my $appInfo = $self->getApplicationInfo(id => $userAppObj->get("appId"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (not defined $appInfo)  # The application doesn't exist!
  {
    return -1;
  }

  # validate the company exists
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  my $companyInfo = $authObj->getCompanyInfo(id => $userAppObj->get("companyId"));
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (not defined $companyInfo)  # The company doesn't exist!
  {
    return -2;
  }

  # validate the user exists
  my $userObj = $authObj->getUserInfo(id => $userAppObj->get("userId"));
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (not defined $userObj)  # The user doesn't exist!
  {
    return -3;
  }

  # make sure we are not trying to assign an administration app to a non-admin user.
  if ($appInfo->get("type") eq "administration" && $userObj->get("admin") == 0)
  {
    return -7;  # error!
  }

  # make sure the app is assigned to the company and that we have enough free licenses to make the assignment.
  my $licenseObj = $self->getCompanyLicenseInfo(companyId => $userAppObj->get("companyId"), appId => $userAppObj->get("appId"));
  if (!defined $licenseObj)
  {
    if ($self->errorCode == -1)
    {
      return -5;  # This app is not assigned to the Company!
    }
    else
    {
      $self->prefixError();
      return 0;
    }
  }

  if ($licenseObj->get("numTotal") > 0 && $licenseObj->get("numFree") == 0)
  {
    return -6;  # there are no available licenses to use for this application.
  }

  # now see if this assignment has already been done.
  my $sth = $self->portalDB->read(sql => "SELECT height FROM user_app_tb WHERE company_id = ? AND app_id = ? AND user_id = ?",
                                    plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    return -4;  # the assignment has already been done.
  }

  my $autorun = $userAppObj->get("autorun");
  my $wap = $userAppObj->get("wap");

  # make the assignment.
  $self->portalDB->write(sql => "INSERT INTO user_app_tb (company_id, app_id, user_id, height, width, autorun, wap) VALUES (?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId"), $userAppObj->get("height"), $userAppObj->get("width"), $autorun, $wap ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Create event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 14, extraInfo => "Application = '" . $appInfo->get("name") . "' Assigned to User = '" . $userObj->get("uname") . "'", userId => (defined $self->sessionObj ? "" : -1));
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "user_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "INSERT INTO user_app_tb (company_id, app_id, user_id, height, width, autorun, wap) VALUES (?, ?, ?, ?, ?, ?, ?)",
                           plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId"), $userAppObj->get("height"), $userAppObj->get("width"), $autorun, $wap ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "Assigned App = '" . $userAppObj->get("appId") . "' for User with ID = '" . $userAppObj->get("userId") . "' for " . $repObj->print, userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "Assigned App = '" . $userAppObj->get("appId") . "' for User with ID = '" . $userAppObj->get("userId") . "'", userId => (defined $self->sessionObj ? "" : -1));
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# unAssignAppFromUser
# takes: userAppObj - UserApplicationObject
# optional: log
# returns: 1 if ok, 0 if error, -1 if assignment didn't exist
# This method will delete the entry from the user_app_tb for this app, company, user pair.
sub unAssignAppFromUser
{
  my $self = shift;
  my %args = ( userAppObj => undef, log => 1, @_ );
  my $userAppObj = $args{userAppObj};
  my $log = $args{log};

  if (not defined $userAppObj)
  {
    $self->error("userAppObj is not defined!<br>\n");
    return 0;
  }
  if (!$userAppObj->isValid)
  {
    $self->error("userAppObj is not valid!<br>\n" . $userAppObj->errorMessage);
    return 0;
  }

  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }

  # see if the assignment exists.
  my $sth = $self->portalDB->read(sql => "SELECT height FROM user_app_tb WHERE company_id = ? AND app_id = ? AND user_id = ?",
                                    plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  my @info = $sth->fetchrow_array;
  if (!defined $info[0])
  {
    return -1;  # assignment doesn't exist.
  }

  # delete the assignment.
  $self->portalDB->write(sql => "DELETE FROM user_app_tb WHERE company_id = ? AND app_id = ? AND user_id = ?",
                           plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make it permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Application Create event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 15, extraInfo => "Application with ID = '" . $userAppObj->get("appId") . "' Removed from User with ID = '" . $userAppObj->get("userId") . "'");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  # now see if we need to replicate this change.
  my @repDBs = $self->replicationObj->getEntries(tbName => "user_app_tb", createDBIObjects => 1);
  if ($self->replicationObj->error)
  {
    $self->error($self->replicationObj->errorMessage);
    return 0;
  }
  foreach my $repObj (@repDBs)
  {
    # do the replication.
    $repObj->dbObj->write(sql => "DELETE FROM user_app_tb WHERE company_id = ? AND app_id = ? AND user_id = ?",
                           plug => [ $userAppObj->get("companyId"), $userAppObj->get("appId"), $userAppObj->get("userId") ]);
    if ($repObj->dbObj->error)
    {
      $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
      return 0;
    }
    else
    {
      $repObj->dbObj->commit;  # have to save the changes in the database.
      if ($repObj->dbObj->error)
      {
        $self->error("In Replication:<br>\n" . $repObj->dbObj->errorMessage);
        return 0;
      }

      # now log the replication event
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 23, extraInfo => "UnAssigned App = '" . $userAppObj->get("appId") . "' for User with ID = '" . $userAppObj->get("userId") . "' for " . $repObj->print);
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }

      # log the replication event into the replication database.
      $self->methods->doLog(sessionObj => $self->sessionObj, db => $repObj->dbObj, action => 23, extraInfo => "UnAssigned App = '" . $userAppObj->get("appId") . "' for User with ID = '" . $userAppObj->get("userId") . "'");
      if ($self->methods->error)
      {
        $self->error($self->methods->errorMessage);
        return 0;
      }
    }
  } # end of replication code

  return 1;
}

# getAppsForUser
# takes: userId
# optional: wap
# returns: hash of hashes where the top level is the application type and each application type has a hash of ApplicationObjects where the server, port, cost and unit values are filled
# in from the company_app_tb but everything else is from the app_tb.  This method gathers all applications assigned to the specified user and returns the array of ApplicationObjects that
# define them.  If wap is defined, then only those apps that meet the state of wap are returned.
# If the user doesn't exist, then errorCode = -1, if company doesn't exist, errorCode = -2 (shouldn't ever be returned)
sub getAppsForUser
{
  my $self = shift;
  my %args = ( userId => -1, @_ );
  my $userId = $args{userId};
  my $wapSearch = (exists $args{wap} ? $args{wap} : "");
  my %apps = ();

  if ($userId < 0)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return %apps;
  }
  if (length $wapSearch > 0 && $wapSearch !~ /^(1|0)$/)
  {
    $self->error("wap = '$wapSearch' is invalid!<br>\n");
    return %apps;
  }

  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }

  # validate the user exists
  my $userObj = $authObj->getUserInfo(id => $userId);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return %apps;
  }
  if (not defined $userObj)  # The user doesn't exist!
  {
    $self->errorCode(-1);
    return %apps;
  }

  my $companyObj = $authObj->getCompanyInfo(id => $userObj->get("companyId"));
  if ($companyObj->error)
  {
    $self->error($companyObj->errorMessage);
    return %apps;
  }
  if (not defined $companyObj) # The company doesn't exist!  This shouldn't happen, should have been caught in the getUserInfo call.
  {
    $self->errorCode(-2);
    return %apps;
  }

  # now gather all apps that are assigned to the company this user is with.
  my @companyApps = $self->getAppsForCompany(companyObj => $companyObj);
  if ($self->error)
  {
    $self->prefixError();
    return %apps;
  }

  if (scalar @companyApps == 0)  # no apps assigned to company!  Thus no apps assigned to this user!
  {
    return %apps;
  }
  # now group these apps for easier access
  my %companyApps = $self->groupAppArray(apps => \@companyApps);

  # now gather all apps that are assigned to this user.
  my $sth = $self->portalDB->read(sql => "SELECT app_id, height, width, autorun, wap FROM user_app_tb WHERE company_id = ? AND user_id = ?" .
                                    (length $wapSearch > 0 ? " AND wap = '$wapSearch'" : "") . " ORDER BY app_id",
                                    plug => [ $userObj->get("companyId"), $userId ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return %apps;
  }

  # traverse the returned apps, getting the app info stuff from the companyApps array and insert them into the apps hash by type
  while (my @info = $sth->fetchrow_array)
  {
    my $appId   = $info[0];
    my $height  = $info[1];
    my $width   = $info[2];
    my $autorun = $info[3];
    my $wap     = $info[4];

    my $appObj = undef;

    # find this app in the companyApps hash.
    foreach my $appType (keys %companyApps)
    {
      foreach my $app (keys %{$companyApps{$appType}})
      {
        if ($companyApps{$appType}->{$app}->id == $appId)
        {
          $appObj = $companyApps{$appType}->{$app};
          last;  # jump out.
        }
      }
    }
    if (not defined $appObj)
    {
      # error!!
      $self->error("appId = '$appId' not found in company assigned apps!<br>\n");
      return %apps;
    }
    # update the user specific items.
    $appObj->set(height => $height);
    $appObj->set(width => $width);
    $appObj->set(autorun => $autorun);
    $appObj->set(wap => $wap);

    # now insert into the hash.
    if (!exists $apps{$appObj->get("type")})
    {
      $apps{$appObj->get("type")} = ();
    }
    $apps{$appObj->get("type")}->{$appObj->get("name")} = $appObj;  # make the app entry under type using the apps name.
  }

  return %apps;
}

# isAppAssignedToCompany
# requires: appName, companyObj
# optional:
# returns: 1 = Assigned, 0 = Not Assigned
# summary: Determines if the specified app is assigned to the specified company.
sub isAppAssignedToCompany
{
  my $self = shift;
  my %args = ( appName => "", companyObj => undef, @_ );
  my $appName = $args{appName};
  my $companyObj = $args{companyObj};

  if (length $appName == 0)
  {
    $self->error("appName = '$appName' is invalid!<br>\n");
    return 0;
  }
  if (!defined $companyObj)
  {
    $self->error("companyObj must be defined!<br>\n");
    return 0;
  }
  if (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return 0;
  }

  my @apps = $self->getAppsForCompany(companyObj => $companyObj);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }

  my $found = 0;
  foreach my $appObj (@apps)
  {
    if ($appObj->get("name") eq $appName)
    {
      $found = 1;
      last;  # jump out
    }
  }

  return $found;
}

# isAppAssignedToUser
# requires: appName, userId
# optional:
# returns: 1 = Assigned, 0 = Not Assigned
# summary: Determines if the specified app is assigned to the specified user.
sub isAppAssignedToUser
{
  my $self = shift;
  my %args = ( appName => "", userId => -1, @_ );
  my $appName = $args{appName};
  my $userId = $args{userId};

  if (length $appName == 0)
  {
    $self->error("appName = '$appName' is invalid!<br>\n");
    return 0;
  }
  if ($userId !~ /^(\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }

  my %apps = $self->getAppsForUser(userId => $userId);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }

  # now traverse all app types and see if we can find the app in question.
  foreach my $appType (keys %apps)
  {
    if (exists $apps{$appType}->{$appName})
    {
      return 1;
    }
  }

  return 0;
}

# getAllUserAppsForCompany
# takes: companyObj
# returns: Array of ApplicationObjects for all User App Assignments for this company.
sub getAllUserAppsForCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my @apps = ();

  if (not defined $companyObj)
  {
    $self->error("companyObj is not defined!<br>\n");
    return @apps;
  }
  if (!$companyObj->isValid)
  {
    $self->error("companyObj is not valid!<br>\n" . $companyObj->errorMessage);
    return @apps;
  }

  # now gather all apps that are assigned to the company for all users of the company.
  my @companyApps = $self->getAppsForCompany(companyObj => $companyObj);
  if ($self->error)
  {
    $self->prefixError();
    return @apps;
  }

  if (scalar @companyApps == 0)  # no apps assigned to company!  Thus no apps assigned to any user!
  {
    return @apps;
  }
  # now group these apps for easier access
  my %companyApps = $self->groupAppArray(apps => \@companyApps);

  # now gather all apps that are assigned to this company for any user.
  my $sth = $self->portalDB->read(sql => "SELECT app_id, height, width, autorun, user_id, wap FROM user_app_tb WHERE company_id = ? ORDER BY user_id, app_id",
                                    plug => [ $companyObj->get("id") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @apps;
  }

  # traverse the returned apps, getting the app info stuff from the companyApps array and insert them into the apps array
  while (my @info = $sth->fetchrow_array)
  {
    my $appId   = $info[0];
    my $height  = $info[1];
    my $width   = $info[2];
    my $autorun = $info[3];
    my $userId  = $info[4];
    my $wap     = $info[5];

    my $appObj = undef;

    # find this app in the companyApps hash.
    foreach my $appType (keys %companyApps)
    {
      foreach my $app (keys %{$companyApps{$appType}})
      {
        if ($companyApps{$appType}->{$app}->id == $appId)
        {
          $appObj = $companyApps{$appType}->{$app};
          last;  # jump out.
        }
      }
    }
    if (not defined $appObj)
    {
      # error!!
      $self->error("appId = '$appId' not found in company assigned apps!<br>\n");
      return @apps;
    }
    # update the user specific items.
    $appObj->set(height => $height);
    $appObj->set(width => $width);
    $appObj->set(autorun => $autorun);
    $appObj->set(userId => $userId);
    $appObj->set(wap => $wap);

    # now insert into the array.
    $apps[++$#apps] = $appObj;
  }

  return @apps;
}

# flattenAppHash
# takes: apps - hash ref of ApplicationObject entries.
# returns: apps - array of ApplicationObject entries sorted by app type.
sub flattenAppHash
{
  my $self = shift;
  my %args = ( apps => undef, @_ );
  my $apps = $args{apps};  # reference to the hash
  my @appArray = ();

  foreach my $type (sort keys %{$apps})
  {
    my %typeApps = %{$apps->{$type}};
    foreach my $app (sort keys %typeApps)
    {
      $appArray[++$#appArray] = $typeApps{$app};
    }
  }

  return @appArray;
}

# groupAppArray
# takes: apps - array of ApplicationObject entries.
# returns: apps - hash of ApplicationObject entries grouped by app type.
sub groupAppArray
{
  my $self = shift;
  my %args = ( apps => undef, @_ );
  my $apps = $args{apps};  # reference to the array
  my %appHash = ();

  for (my $i=0; $i < scalar @{$apps}; $i++)
  {
    if (!exists $appHash{$apps->[$i]->get("type")})
    {
      $appHash{$apps->[$i]->get("type")} = ();
    }
    $appHash{$apps->[$i]->get("type")}->{$apps->[$i]->get("name")} = $apps->[$i];
  }

  return %appHash;
}

# getCompanyAppsForServer
# requires: serverName (text string), appObj (ApplicationObject)
# optional:
# returns: array of CompanyApplicationObjects
# summary: Returns a list of CompanyApplicationObjects that defines all companies that have an instance of the specified Application on the specified server.
sub getCompanyAppsForServer
{
  my $self = shift;
  my %args = (serverName => "", appObj => undef, @_);
  my $serverName = $args{serverName};
  my $appObj = $args{appObj};
  my @result = ();

  if (length $serverName == 0)
  {
    $self->error("serverName must be specified!<br>\n");
    return @result;
  }
  if (!defined $appObj || !$appObj->isValid)
  {
    $self->error("appObj must be specified or is invalid!<br>\n");
    return @result;
  }

  # gather a list of all entries from the company_app_tb table that fits the Application ID and serverName values.
  my $sth = $self->portalDB->read(sql => "SELECT app_id, company_id, number, server, port, db_host, db_type, db_port, cost, unit, db_name, height, width, autorun, wap FROM company_app_tb WHERE app_id = ? AND server = ? ORDER BY company_id",
                                    plug => [$appObj->get("id"), $serverName]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @result;
  }

  while (my @info = $sth->fetchrow_array)
  {
    # build the CompanyApplicationObject and store it in the results array.
    my $companyAppObj = Portal::Objects::CompanyApplicationObject->new(langObj => $self->langObj);
    if ($companyAppObj->error)
    {
      $self->error($companyAppObj->errorMessage);
      return @result;
    }
    $companyAppObj->populate(appId => $info[0], companyId => $info[1], number => $info[2], server => $info[3],
      port => $info[4], dbHost => $info[5], dbType => $info[6], dbPort => $info[7], cost => $info[8], unit => $info[9],
      dbName => $info[10], height => $info[11], width => $info[12], autorun => $info[13], wap => $info[14]);
    if ($companyAppObj->error)
    {
      $self->error($companyAppObj->errorMessage);
      return @result;
    }
    $result[++$#result] = $companyAppObj;
  }

  return @result;
}

# void lockApp(app, type, users, company, log)
# requires: app - id of app, type - 'a' all users, 'u' specified users, 'c' company
# optional: users - comma seperated list of user id's if type = 'u', company - id of company if type = 'c',
#           log - 1/0, specifies if we should log the event.
# returns: nothing
# summary: locks the specified app for use only by the specified admin user based upon the type.
sub lockApp
{
  my $self = shift;
  my %args = (app => 0, type => "a", users => "", company => 0, log => 1, @_);
  my $app = $args{app};
  my $type = $args{type};
  my $users = $args{users};
  my $log = $args{log};
  my $company = $args{company};

  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return;
  }
  my $admin = $self->sessionObj->store->{userObj}->id;
  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return;
  }
  if ($type !~ /^(a|u|c)$/)
  {
    $self->error("type = '$type' is invalid!<br>\n");
    return;
  }
  if ($type eq "u" && $users !~ /^(\d+(,\d+)*)$/)
  {
    $self->error("users = '$users' is invalid!<br>\n");
    return;
  }
  if ($type eq "c" && $company !~ /^(\d+)$/)
  {
    $self->error("company = '$company' is invalid!<br>\n");
    return;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return;
  }
  if ($type eq "a")
  {
    $users = "";
    $company = 0;
  }
  elsif ($type eq "u")
  {
    $company = 0;
  }
  elsif ($type eq "c")
  {
    $users = "";
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }

  # now lookup and make sure the specified user still exists and is an administrator.
  my $userInfo = $authObj->getUserInfo(id => $admin);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }
  if (!defined $userInfo)
  {
    $self->error("User with id = '$admin' does not exist!<br>\n");
    return;
  }
  if ($userInfo->get("admin") == 0 && $userInfo->get("sysadmin") == 0)
  {
    $self->error("User '" . $userInfo->get("uname") . "' is not an administrator!<br>\n");
    return;
  }

  # now verify the specified app exists
  my $appInfo = $self->getApplicationInfo(id => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  if (!defined $appInfo)
  {
    $self->error("App with id = '$app' does not exist!<br>\n");
    return;
  }

  # if the user is not a systems administrator, make sure the app is assigned to them.
  if ($userInfo->get("sysadmin") == 0)
  {
    my $result = $self->isAppAssignedToUser(appName => $appInfo->get("name"), userId => $admin);
    if ($self->error)
    {
      $self->prefixError();
      return;
    }
    if (!$result)
    {
      $self->error("User '" . $userInfo->get("uname") . "' is not a systems administrator and does not have app '" . $appInfo->get("name") . "' assigned to them!<br>\n");
      return;
    }
  }

  # make sure the app is not already locked.
  my $result = $self->isAppLocked(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  if ($result)
  {
    my @lockedApps = $self->getLockedAppInfo(app => $app);
    if ($self->error)
    {
      $self->prefixError();
      return;
    }
    foreach my $lockedApp (@lockedApps)
    {
      if ($lockedApp->{type} eq $type && $lockedApp->{users} eq $users && $lockedApp->{company} eq $company && $lockedApp->{admin} eq $userInfo->get("id"))
      {
        $self->error("You asked to lock app = '$app' but we are already locked!");
        return;
      }
    }
  }

  my $sql = "INSERT INTO app_lock_tb (admin, app, type, users, company) VALUES (?, ?, ?, ?, ?)";
  my @plug = ();
  $plug[0] = $admin;
  $plug[1] = $app;
  $plug[2] = $type;
  if ($type eq "a") # lock all users out of this app.
  {
    if ($userInfo->get("sysadmin") == 0)
    {
      $self->error("User '" . $userInfo->get("uname") . "' is not a systems administrator and thus can not lock out all users!<br>\n");
      return;
    }
    $plug[3] = $users;
    $plug[4] = $company;
  }
  elsif ($type eq "u") # lock out the specified users.
  {
    # make sure that each user exists.
    my @users = split /,/, $users;
    foreach my $user (@users)
    {
      my $tmpUserInfo = $authObj->getUserInfo(id => $user);
      if ($authObj->error)
      {
        $self->error($authObj->errorMessage);
        return;
      }
      if (!defined $tmpUserInfo)
      {
        $self->error("User with id = '$user' does not exist!<br>\n");
        return;
      }
      if ($userInfo->get("sysadmin") == 0 && $tmpUserInfo->get("companyId") != $userInfo->get("companyId"))
      {
        $self->error("User '" . $userInfo->get("uname") . "' is not a systems administrator and user '" . $tmpUserInfo->get("uname") . "' is not a member of your company!<br>\n");
        return;
      }
    }

    $plug[3] = $users;
    $plug[4] = $company;
  }
  elsif ($type eq "c") # lock out the users of the specified company.
  {
    # make sure the specified company exists.
    my $companyInfo = $authObj->getCompanyInfo(id => $company);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return;
    }
    if (not defined $companyInfo)  # The company doesn't exist!
    {
      $self->error("Company with id = '$company' does not exist!<br>\n");
      return;
    }

    # if the user is not a systems administrator, make sure they are a member of this company.
    if ($userInfo->get("sysadmin") == 0 && $userInfo->get("companyId") != $company)
    {
      $self->error("User = '" . $userInfo->get("uname") . "' is not a systems administrator and is not a member of Company = '" . $companyInfo->get("name") . "'!<br>\n");
      return;
    }

    $plug[3] = $users;
    $plug[4] = $company;
  }

  # now insert into the database and lock this app.
  $self->portalDB->write(sql => $sql, plug => \@plug);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return;
  }
  $self->portalDB->commit;

  # now log the App lock event.
  if ($log)
  {
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 25, extraInfo => "Application = '" . $appInfo->get("name") . "' Locked by user = '" . $userInfo->get("uname") . "'");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return;
    }
  }
}

# void unlockApp(app, type, users, company, log)
# requires: app, type
# optional: users, company, log - 1/0
# returns: nothing
# summary: unlocks the specified app, but only if the admin user specified was the one who locked it.
sub unlockApp
{
  my $self = shift;
  my %args = (app => 0, type => "", users => "", company => 0, log => 1, @_);
  my $app = $args{app};
  my $type = $args{type};
  my $users = $args{users};
  my $company = $args{company};
  my $log = $args{log};

  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return;
  }
  my $admin = $self->sessionObj->store->{userObj}->id;
  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return;
  }
  if ($type !~ /^(a|c|u)$/)
  {
    $self->error("type = '$type' is invalid!<br>\n");
    return;
  }
  if ($type eq "u" && $users !~ /^(\d+(,\d+)*)$/)
  {
    $self->error("users = '$users' is invalid!<br>\n");
    return;
  }
  if ($type eq "c" && $company !~ /^(\d+)$/)
  {
    $self->error("company = '$company' is invalid!<br>\n");
    return;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return;
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }

  # now lookup and make sure the specified user still exists and is an administrator.
  my $userInfo = $authObj->getUserInfo(id => $admin);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }
  if (!defined $userInfo)
  {
    $self->error("User with id = '$admin' does not exist!<br>\n");
    return;
  }
  if ($userInfo->get("admin") == 0 && $userInfo->get("sysadmin") == 0)
  {
    $self->error("User '" . $userInfo->get("uname") . "' is not an administrator!<br>\n");
    return;
  }

  # verify the app exists.
  my $appInfo = $self->getApplicationInfo(id => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  if (!defined $appInfo)
  {
    $self->error("App with id = '$app' does not exist!<br>\n");
    return;
  }

  # see if the app is locked by us.
  my $result = $self->isAppLockedByMe(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  if (!$result)
  {
    # do we blow an error, or just return?
    $self->error("You asked to unlock app = '$app' but we have not locked it!<br>\n");
    return;
  }
  # if locked, then make sure the admin user specified is who locked it.
  my @lockedApps = $self->getLockedAppInfo(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  # make sure that the exact app you specified is locked.
  my $found = 0;
  foreach my $lockedApp (@lockedApps)
  {
    if ($lockedApp->{type} eq $type && $lockedApp->{company} eq $company && $lockedApp->{users} eq $users && $lockedApp->{admin} == $admin)
    {
      $found = 1;
    }
  }
  if (!$found)
  {
    $self->error("App = '" . $appInfo->get("name") . "' was not locked by you!<br>\n");
    return;
  }

  my $sql = "DELETE FROM app_lock_tb WHERE admin = ? AND app = ? AND type = ? AND users = '$users' AND company = ?";
  my @plug = ();
  $plug[0] = $admin;
  $plug[1] = $app;
  $plug[2] = $type;
  $plug[3] = $company;

  # now delete the lock from the database and unlock this app.
  $self->portalDB->write(sql => $sql, plug => \@plug);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return;
  }
  $self->portalDB->commit;
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return;
  }

  # now log the App unlock event.
  if ($log)
  {
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 26, extraInfo => "Application = '" . $appInfo->get("name") . "', type = '$type', users = '$users', company = '$company' Un-Locked by user = '" . $userInfo->get("uname") . "'");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return;
    }
  }
}

# void breakAppLock(admin, app, type, users, company, log)
# requires: admin, app, type
# optional: users, company, log - 1/0
# returns: nothing
# summary: unlocks the specified app, but you have to be an administrator
#          in the same company as the admin who locked it or be a
#          system administrator.
sub breakAppLock
{
  my $self = shift;
  my %args = (admin => 0, app => 0, type => "", users => "", company => 0, log => 1, @_);
  my $admin = $args{admin};
  my $app = $args{app};
  my $type = $args{type};
  my $users = $args{users};
  my $company = $args{company};
  my $log = $args{log};

  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return;
  }
  my $breakingAdmin = $self->sessionObj->store->{userObj}->id;
  if ($admin !~ /^(\d+)$/)
  {
    $self->error("admin = '$admin' is invalid!<br>\n");
    return;
  }
  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return;
  }
  if ($type !~ /^(a|c|u)$/)
  {
    $self->error("type = '$type' is invalid!<br>\n");
    return;
  }
  if ($type eq "u" && $users !~ /^(\d+(,\d+)*)$/)
  {
    $self->error("users = '$users' is invalid!<br>\n");
    return;
  }
  if ($type eq "c" && $company !~ /^(\d+)$/)
  {
    $self->error("company = '$company' is invalid!<br>\n");
    return;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return;
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }

  # now lookup and make sure the specified user still exists and is an administrator.
  my $breakingUserInfo = $authObj->getUserInfo(id => $breakingAdmin);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }
  if (!defined $breakingUserInfo)
  {
    $self->error("User with id = '$breakingAdmin' does not exist!<br>\n");
    return;
  }
  if ($breakingUserInfo->get("admin") == 0 && $breakingUserInfo->get("sysadmin") == 0)
  {
    $self->error("User '" . $breakingUserInfo->get("uname") . "' is not an administrator!<br>\n");
    return;
  }

  my $userInfo = $authObj->getUserInfo(id => $admin);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return;
  }
  if (!defined $userInfo)
  {
    $self->error("User with id = '$admin' does not exist!<br>\n");
    return;
  }
  if ($userInfo->get("admin") == 0 && $userInfo->get("sysadmin") == 0)
  {
    $self->error("User '" . $userInfo->get("uname") . "' is not an administrator!<br>\n");
    return;
  }

  # verify the app exists.
  my $appInfo = $self->getApplicationInfo(id => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  if (!defined $appInfo)
  {
    $self->error("App with id = '$app' does not exist!<br>\n");
    return;
  }

  # See if the app is locked and we can find an exact match.
  my @lockedApps = $self->getLockedAppInfo(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return;
  }
  # make sure that the exact app you specified is locked.
  my $found = 0;
  foreach my $lockedApp (@lockedApps)
  {
    if ($lockedApp->{type} eq $type && $lockedApp->{company} eq $company && $lockedApp->{users} eq $users && $lockedApp->{admin} == $admin)
    {
      $found = 1;
    }
  }
  if (!$found)
  {
    $self->error("App = '" . $appInfo->get("name") . "' was not locked by admin = '$admin'!<br>\n");
    return;
  }

  # now verify that the breakingAdmin has the right to break this lock.
  if ($breakingUserInfo->get("sysadmin") == 0)
  {
    # make sure they are in the same company as the admin who locked the app.
    if ($breakingUserInfo->get("companyId") != $userInfo->get("companyId"))
    {
      $self->error("User = '" . $breakingUserInfo->get("uname") . "' is not a systems administrator and is not in the same company as locking admin = '" . $userInfo->get("uname") . "'!<br>\n");
      return;
    }
  }

  my $sql = "DELETE FROM app_lock_tb WHERE admin = ? AND app = ? AND type = ? AND users = '$users' AND company = ?";
  my @plug = ();
  $plug[0] = $admin;
  $plug[1] = $app;
  $plug[2] = $type;
  $plug[3] = $company;

  # now delete the lock from the database and unlock this app.
  $self->portalDB->write(sql => $sql, plug => \@plug);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return;
  }
  $self->portalDB->commit;
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return;
  }

  # now log the App unlock event.
  if ($log)
  {
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 26, extraInfo => "Application = '" . $appInfo->get("name") . "', admin = '$admin', type = '$type', users = '$users', company = '$company' Lock Broken by user = '" . $breakingUserInfo->get("uname") . "'");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return;
    }
  }
}

# bool isAppLocked(app)
# requires: app
# optional:
# returns: 0 - not locked, 1 - locked
# summary: checks to see if the specified app is locked.
sub isAppLocked
{
  my $self = shift;
  my %args = (app => 0, @_);
  my $app = $args{app};

  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return 0;
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }

  my $userId = $self->sessionObj->store->{userObj}->id;
  # now lookup and make sure the specified user still exists.
  my $userInfo = $authObj->getUserInfo(id => $userId);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (!defined $userInfo)
  {
    $self->error("User with id = '$userId' does not exist!<br>\n");
    return 0;
  }

  # verify the app exists.
  my $appInfo = $self->getApplicationInfo(id => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (!defined $appInfo)
  {
    $self->error("App with id = '$app' does not exist!<br>\n");
    return 0;
  }

  # see if the app is locked for the current user/company.
  my @lockedApps = $self->getLockedAppInfo(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }

  foreach my $lockedApp (@lockedApps)
  {
    if ($lockedApp->{type} eq 'a')
    {
      return 1;
    }
    elsif ($lockedApp->{type} eq 'c')
    {
      if ($userInfo->get("companyId") == $lockedApp->{company})
      {
        return 1;
      }
    }
    elsif ($lockedApp->{type} eq 'u')
    {
      # see if the user is one of the locked out users.
      my @users = split /,/, $lockedApp->{users};
      foreach my $lockedUser (@users)
      {
        if ($lockedUser == $userId)
        {
          return 1;
        }
      }
    }
  }

  return 0;  # by default it isn't locked.
}

# bool isAppLockedByMe(app)
# requires: app
# optional:
# returns: 0 - no, 1 - yes
# summary: checks to see if the specified app is locked by us.
sub isAppLockedByMe
{
  my $self = shift;
  my %args = (app => 0, @_);
  my $app = $args{app};

  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return 0;
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }

  my $userId = $self->sessionObj->store->{userObj}->id;
  # now lookup and make sure the specified user still exists.
  my $userInfo = $authObj->getUserInfo(id => $userId);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return 0;
  }
  if (!defined $userInfo)
  {
    $self->error("User with id = '$userId' does not exist!<br>\n");
    return 0;
  }

  # verify the app exists.
  my $appInfo = $self->getApplicationInfo(id => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (!defined $appInfo)
  {
    $self->error("App with id = '$app' does not exist!<br>\n");
    return 0;
  }

  # see if the app is locked by us.
  my @lockedApps = $self->getLockedAppInfo(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }

  foreach my $lockedApp (@lockedApps)
  {
    if ($lockedApp->{admin} == $userId)
    {
      return 1;
    }
  }

  return 0;  # by default it isn't locked.
}

# bool isUserLockedOut(app)
# requires: app - app id
# optional:
# returns: 0 - not locked out, 1 - locked out
# summary: checks to see if the specified user is locked out of the specified app.
sub isUserLockedOut
{
  my $self = shift;
  my %args = (app => 0, @_);
  my $app = $args{app};

  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return 0;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  # see if the app is locked and if the user is locked out.
  my @lockedApps = $self->getLockedAppInfo(app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  foreach my $lockedApp (@lockedApps)
  {
    # see if they are the admin who locked the app.
    if ($userObj->get("id") == $lockedApp->{admin})
    {
      # they are, so we don't lock them out!
      next;  # continue checking w/ the next lock
    }
    if ($lockedApp->{type} eq "u")
    {
      # see if the user is one of the locked out users.
      my @users = split /,/, $lockedApp->{users};
      foreach my $lockedUser (@users)
      {
        if ($lockedUser == $userObj->get("id"))
        {
          return 1;
        }
      }
    }
    elsif ($lockedApp->{type} eq "c")
    {
      # see if the users company is locked out.
      if ($userObj->get("companyId") == $lockedApp->{company})
      {
        return 1;
      }
    }
    elsif ($lockedApp->{type} eq "a")
    {
      # since they aren't the admin who locked the app, they
      # are automatically locked out!
      return 1;
    }
  }

  return 0;  # by default it isn't locked.
}

# int getLockedAppInfo(app)
# requires: app - app id
# optional:
# returns: array of hashes of id of user who locked the app, type of app, company, user values
#          or empty array if not locked.
# summary: returns the id of the user who locked the specified app.
sub getLockedAppInfo
{
  my $self = shift;
  my %args = (app => 0, @_);
  my $app = $args{app};
  my @result = ();

  if ($app !~ /^(\d+)$/)
  {
    $self->error("app = '$app' is invalid!<br>\n");
    return @result;
  }

  # see if the app is locked.
  my $sql = "SELECT admin, type, users, company FROM app_lock_tb WHERE app = ? ORDER BY type DESC";
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $app ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @result;
  }
  while (my @info = $sth->fetchrow_array)
  {
    my %values = (admin => $info[0], type => $info[1], users => $info[2], company => $info[3]);
    $result[++$#result] = \%values;
  }

  return @result;
}

# int createDynamicContentEntry(dynamicContentObj, log)
#  requires: dynamicContentObj
#  optional: log = 1
#  returns: 1 - entry created, 0 - error, -1 - entry already created,
#           -2 - permission denied
sub createDynamicContentEntry
{
  my $self = shift;
  my %args = (dynamicContentObj => undef, log => 1, @_);
  my $dynamicContentObj = $args{dynamicContentObj};
  my $log = $args{log};

  if (not defined $dynamicContentObj)
  {
    $self->error("You must define dynamicContentObj!<br>\n");
    return 0;
  }
  if (!$dynamicContentObj->isValid())
  {
    $self->error($dynamicContentObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }
  if ($dynamicContentObj->get("index") ne "-1")
  {
    $self->error("index = '" . $dynamicContentObj->get("index") . "' is invalid!  It must be -1.<br>\n");
    return 0;
  }

  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  # make sure we are a sysadmin
  if ($userObj->get("syadmin") == 0)
  {
    if ($dynamicContentObj->get("companyId") == -1)
    {
      # you can not work with a global entry (you are at most a company admin)!
      return -2;
    }
    if ($userObj->get("admin") == 0)
    {
      if ($userObj->get("id") != $dynamicContentObj->get("userId"))
      {
        # as a normal user, you can only delete your own entries!
        return -2;
      }
    }
    if ($userObj->get("companyId") != $dynamicContentObj->get("companyId"))
    {
      # you can not work with an entry for some other company!
      return -2;
    }
    # verify the user is a member of your company
    if ($dynamicContentObj->get("userId") != -1)
    {
      # instantiate an Auth instance.
      my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
      if ($authObj->error)
      {
        $self->error($authObj->errorMessage);
        return 0;
      }

      # now lookup and make sure the specified user still exists.
      my $userInfo = $authObj->getUserInfo(id => $dynamicContentObj->get("userId"));
      if ($authObj->error)
      {
        $self->error($authObj->errorMessage);
        return 0;
      }
      if (!defined $userInfo)
      {
        $self->error("User with id = '" . $dynamicContentObj->get("userId") . "' does not exist!<br>\n");
        return 0;
      }
      if ($userInfo->get("companyId") != $userObj->get("companyId"))
      {
        $self->error("User with id = '" . $dynamicContentObj->get("userId") . "' does not belong to your company!<br>\n");
        return 0;
      }
    }
  }

  # first make sure this entry doesn't exist.
  my @entries = $self->getDynamicContentEntries(companyId => $dynamicContentObj->get("companyId"),
                userId => $dynamicContentObj->get("userId"), callingApp => $dynamicContentObj->get("callingApp"),
                tag => $dynamicContentObj->get("tag"), app => $dynamicContentObj->get("app"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (scalar @entries > 0)
  { # see if our exact entry is included.
    my $found = 0;
    foreach my $entry (@entries)
    {
      if ($entry->get("arguments") eq $dynamicContentObj->get("arguments"))
      {
        return -1;
      }
    }
  }

  # now create the entry
  $self->portalDB->write(sql => "INSERT INTO dynamic_content_tb (company_id, user_id, calling_app, tag, app, arguments) VALUES (?, ?, ?, ?, ?, ?)",
                          plug => [ $dynamicContentObj->get("companyId"), $dynamicContentObj->get("userId"),
                                    $dynamicContentObj->get("callingApp"), $dynamicContentObj->get("tag"),
                                    $dynamicContentObj->get("app"), $dynamicContentObj->get("arguments") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Dynamic Content Entry Create event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 27, extraInfo => "Dynamic Content Entry: " . $dynamicContentObj->print);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

# dynamicContentObj[] getDynamicContentEntries(index, companyId, userId,
#                      callingApp, tag, app, global)
#  requires:
#  optional: index or any of companyId, userId, callingApp, tag, app,
#            global - 1/0 (defaults to 0)
#  returns: array of dynamicContentObj entries that represent all
#          dynamic_content_tb table entries that matched the specified
#          parameters.  The returned values are sorted by
#          companyId, userId, callingApp, tag, app and arguments
#          If global = 1, then we include the companyId = -1 or
#          companyId = companyId and userId = -1 checks, but this is only
#          valid when the user specifies a companyId and a userId.
sub getDynamicContentEntries
{
  my $self = shift;
  my %args = (index => "", companyId => "", userId => "", callingApp => "",
              tag => "", app => "", global => 0, @_);
  my $index = $args{index};
  my $companyId = $args{companyId};
  my $userId = $args{userId};
  my $callingApp = $args{callingApp};
  my $tag = $args{tag};
  my $app = $args{app};
  my $global = $args{global};
  my @entries = ();

  if ($index !~ /^(|\d+)$/)
  {
    $self->error("index = '$index' is invalid!<br>\n");
    return @entries;
  }
  if ($companyId !~ /^(|-1|\d+)$/)
  {
    $self->error("companyId = '$companyId' is invalid!<br>\n");
    return @entries;
  }
  if ($userId !~ /^(|-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return @entries;
  }
  if ($global !~ /^(1|0)$/)
  {
    $self->error("global = '$global' is invalid!<br>\n");
    return @entries;
  }
  if ($global && ($companyId !~ /^(\d+)$/ || $userId !~ /^(\d+)$/))
  {
    $self->error("You must specify both companyId and userId when global = 1!<br>\n");
    return @entries;
  }

  my $sql = "SELECT nindex, company_id, user_id, calling_app, tag, app, arguments FROM dynamic_content_tb WHERE";
  if ($index)
  {
    $sql .= " nindex = '$index'";
  }
  else
  {
    my $seperator = 0;
    if ($global)
    {
      $sql .= " ((company_id = '$companyId' AND user_id = '$userId') OR (company_id = -1) OR (company_id = '$companyId' AND user_id = -1))";
      $seperator = 1;
    }
    else
    {
      if ($companyId)
      {
        $sql .= " company_id = '$companyId'";
        $seperator = 1;
      }
      if ($userId)
      {
        $sql .= " AND" if ($seperator);
        $sql .= " user_id = '$userId'";
        $seperator = 1;
      }
    }
    if ($callingApp)
    {
      $sql .= " AND" if ($seperator);
      $sql .= " calling_app = '$callingApp'";
      $seperator = 1;
    }
    if ($tag)
    {
      $sql .= " AND" if ($seperator);
      $sql .= " tag = '$tag'";
      $seperator = 1;
    }
    if ($app)
    {
      $sql .= " AND" if ($seperator);
      $sql .= " app = '$app'";
    }

    # order by
    $sql .= " ORDER BY company_id, user_id, calling_app, tag, app, arguments";
  }

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @entries;
  }
  while (my @info = $sth->fetchrow_array)
  {
    my $entryObj = Portal::Objects::DynamicContentObject->new(index => $info[0],
                   companyId => $info[1], userId => $info[2], callingApp => $info[3],
                   tag => $info[4], app => $info[5], arguments => $info[6], langObj => $self->langObj);
    if ($entryObj->error)
    {
      $self->error($entryObj->errorMessage);
      return @entries;
    }
    # we must enforce security.
    my $isAssigned = 1;
    if ($entryObj->get("app") ne "Portal")
    {
      $isAssigned = $self->applicationObj->isAppAssignedToUser(appName => $entryObj->get("app"), userId => $entryObj->get("userId"));
      if ($self->applicationObj->error)
      {
        $self->error($self->applicationObj->errorMessage);
        return @entries;
      }
    }
    $entries[++$#entries] = $entryObj if ($isAssigned);
  }

  return @entries;
}

# int updateDynamicContentEntry(dynamicContentObj, log)
#  requires: dynamicContentObj
#  optional: log = 1
#  returns: 1 - entry updated, 0 - error, -1 - entry does not exist,
#           -2 - permission denied
sub updateDynamicContentEntry
{
  my $self = shift;
  my %args = (dynamicContentObj => undef, log => 1, @_);
  my $dynamicContentObj = $args{dynamicContentObj};
  my $log = $args{log};

  if (not defined $dynamicContentObj)
  {
    $self->error("You must define dynamicContentObj!<br>\n");
    return 0;
  }
  if (!$dynamicContentObj->isValid())
  {
    $self->error($dynamicContentObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }
  if ($dynamicContentObj->get("index") eq "-1")
  {
    $self->error("index = '" . $dynamicContentObj->get("index") . "' is invalid!  It must not be -1.<br>\n");
    return 0;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  # make sure we are a sysadmin
  if ($userObj->get("syadmin") == 0)
  {
    if ($dynamicContentObj->get("companyId") == -1)
    {
      # you can not work with a global entry (you are at most a company admin)!
      return -2;
    }
    if ($userObj->get("admin") == 0)
    {
      if ($userObj->get("id") != $dynamicContentObj->get("userId"))
      {
        # as a normal user, you can only update your own entries!
        return -2;
      }
    }
    if ($userObj->get("companyId") != $dynamicContentObj->get("companyId"))
    {
      # you can not work with an entry for some other company!
      return -2;
    }
    # verify the user is a member of your company
    if ($dynamicContentObj->get("userId") != -1)
    {
      # instantiate an Auth instance.
      my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
      if ($authObj->error)
      {
        $self->error($authObj->errorMessage);
        return 0;
      }

      # now lookup and make sure the specified user still exists.
      my $userInfo = $authObj->getUserInfo(id => $dynamicContentObj->get("userId"));
      if ($authObj->error)
      {
        $self->error($authObj->errorMessage);
        return 0;
      }
      if (!defined $userInfo)
      {
        $self->error("User with id = '" . $dynamicContentObj->get("userId") . "' does not exist!<br>\n");
        return 0;
      }
      if ($userInfo->get("companyId") != $userObj->get("companyId"))
      {
        $self->error("User with id = '" . $dynamicContentObj->get("userId") . "' does not belong to your company!<br>\n");
        return 0;
      }
    }
  }

  # first make sure this entry does exist.
  my @entries = $self->getDynamicContentEntries(index => $dynamicContentObj->get("index"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (scalar @entries == 0)
  {
    return -1;
  }
  if (scalar @entries > 1)
  {
    $self->error("There was '" . (scalar @entries) . "' entries returned for index = '" . $dynamicContentObj->get("index") . "'!<br>\n");
    return 0;
  }

  # now we update the entry
  $self->portalDB->write(sql => "UPDATE dynamic_content_tb SET company_id = ?, user_id = ?, calling_app = ?, tag = ?, app = ?, arguments = ? WHERE nindex = ?",
                          plug => [ $dynamicContentObj->get("companyId"), $dynamicContentObj->get("userId"),
                                    $dynamicContentObj->get("callingApp"), $dynamicContentObj->get("tag"),
                                    $dynamicContentObj->get("app"), $dynamicContentObj->get("arguments"),
                                    $dynamicContentObj->get("index") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Dynamic Content Entry Update event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 28, extraInfo => "Dynamic Content Entry: " . $dynamicContentObj->print);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

# int deleteDynamicContentEntry(dynamicContentObj, log)
#  requires: dynamicContentObj
#  optional: log = 1
#  returns: 1 - entry deleted, 0 - error, -1 - entry does not exist,
#           -2 - permission denied
sub deleteDynamicContentEntry
{
  my $self = shift;
  my %args = (dynamicContentObj => undef, log => 1, @_);
  my $dynamicContentObj = $args{dynamicContentObj};
  my $log = $args{log};

  if (not defined $dynamicContentObj)
  {
    $self->error("You must define dynamicContentObj!<br>\n");
    return 0;
  }
  if (!$dynamicContentObj->isValid())
  {
    $self->error($dynamicContentObj->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return 0;
  }
  if ($dynamicContentObj->get("index") eq "-1")
  {
    $self->error("index = '" . $dynamicContentObj->get("index") . "' is invalid!  It must not be -1.<br>\n");
    return 0;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  # make sure we are a sysadmin
  if ($userObj->get("syadmin") == 0)
  {
    if ($dynamicContentObj->get("companyId") == -1)
    {
      # you can not work with a global entry (you are at most a company admin)!
      return -2;
    }
    if ($userObj->get("admin") == 0)
    {
      if ($userObj->get("id") != $dynamicContentObj->get("userId"))
      {
        # as a normal user, you can only delete your own entries!
        return -2;
      }
    }
    if ($userObj->get("companyId") != $dynamicContentObj->get("companyId"))
    {
      # you can not work with an entry for some other company!
      return -2;
    }
  }

  # first make sure this entry does exist.
  my @entries = $self->getDynamicContentEntries(index => $dynamicContentObj->get("index"));
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (scalar @entries == 0)
  {
    return -1;
  }
  if (scalar @entries > 1)
  {
    $self->error("There was '" . (scalar @entries) . "' entries returned for index = '" . $dynamicContentObj->get("index") . "'!<br>\n");
    return 0;
  }

  # now we update the entry
  $self->portalDB->write(sql => "DELETE FROM dynamic_content_tb WHERE nindex = ?",
                          plug => [ $dynamicContentObj->get("index") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the Dynamic Content Entry Delete event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 29, extraInfo => "Dynamic Content Entry: " . $dynamicContentObj->print);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

# int deleteDynamicContentEntries(companyId, userId, callingApp, tag,
#                                app, log)
#  requires:
#  optional: companyId, userId, callingApp, tag, app, log = 1
#  returns: >= 0 - number entries deleted, -1 - error,
#           -2 - permission denied
sub deleteDynamicContentEntries
{
  my $self = shift;
  my %args = (companyId => "", userId => "", callingApp => "",
              tag => "", app => "", log => 1, @_);
  my $companyId = $args{companyId};
  my $userId = $args{userId};
  my $callingApp = $args{callingApp};
  my $tag = $args{tag};
  my $app = $args{app};
  my $log = $args{log};
  my $numDeleted = 0;

  if ($companyId !~ /^(|-1|\d+)$/)
  {
    $self->error("companyId = '$companyId' is invalid!<br>\n");
    return -1;
  }
  if ($userId !~ /^(|-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return -1;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return -1;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  my @entries = $self->getDynamicContentEntries(companyId => $companyId,
                userId => $userId, callingApp => $callingApp, tag => $tag, app => $app);
  if ($self->error)
  {
    $self->prefixError();
    return -1;
  }
  foreach my $entry (@entries)
  {
    my $result = $self->deleteDynamicContentEntry(dynamicContentObj => $entry, log => $log);
    if ($self->error)
    {
      $self->prefixError();
      return -1;
    }
    if ($result == -1)
    {
      $self->error("Entry: " . entry->print . " does not exist!<br>\n");
      return -1;
    }
    elsif ($result == -2)
    {
      $self->error("You do not have permission to Delete Entry: " . $entry->print . "<br>\n");
      return -1;
    }
    $numDeleted++;  # keep track of number deleted.
  }

  return $numDeleted;
}

# colorScheme getColorScheme(name, userId)
#    requires: name - name of colorScheme,
#              userId - -1 or id of user
#    optional:
#    returns:  ColorScheme object (Portal::Data::ColorScheme) or undef
#              on error.
sub getColorScheme
{
  my $self = shift;
  my %args = (name => "", userId => "", @_);
  my $name = $args{name};
  my $userId = $args{userId};
  my $colorScheme = undef;

  if ($userId !~ /^(-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return $colorScheme;
  }
  if (length $name == 0)
  {
    $self->error("name = '$name' is invalid!<br>\n");
    return $colorScheme;
  }
  if ($userId != -1 && $name !~ /^(.+)( \(user\))$/)
  {
    $self->error("name = '$name' is invalid for userId = '$userId'!<br>\n");
    return $colorScheme;
  }
  if ($userId != -1)
  {
    # instantiate an Auth instance.
    my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return $colorScheme;
    }

    # now make sure the specified user exists
    my $userInfo = $authObj->getUserInfo(id => $userId);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return $colorScheme;
    }
    if (!defined $userInfo)
    {
      $self->error("User with id = '$userId' does not exist!<br>\n");
      return $colorScheme;
    }
  }

  my $sql = "SELECT value, derived_from FROM colorscheme_tb WHERE userid = ? AND name = ?";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $userId, $name ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return $colorScheme;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    eval { $colorScheme = Portal::Data::ColorScheme->new(name => $name, string => $info[0],
                          userId => $userId, derivedFrom => $info[1], langObj => $self->langObj); };
    if ($@)
    {
      $self->error($@);
      return $colorScheme;
    }
  }

  return $colorScheme;
}

# colorScheme getCurrentColorScheme(userId)
#    requires: userId - id of real user or -1 for system default colors
#    optional:
#    returns:  ColorScheme object or undef on error.  The colorScheme
#              returned is based upon the users set preferences.
sub getCurrentColorScheme
{
  my $self = shift;
  my %args = (userId => "", @_);
  my $userId = $args{userId};
  my $colorScheme = undef;

  if ($userId !~ /^(-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return $colorScheme;
  }

  # instantiate an Auth instance.
  my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
  if ($authObj->error)
  {
    $self->error($authObj->errorMessage);
    return $colorScheme;
  }

  my $userInfo = undef;
  if ($userId != -1)
  {
    # now make sure the specified user exists
    $userInfo = $authObj->getUserInfo(id => $userId);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return $colorScheme;
    }
    if (!defined $userInfo)
    {
      $self->error("User with id = '$userId' does not exist!<br>\n");
      return $colorScheme;
    }
  }

  my $colorSchemeName = "";

  if ($userId != -1)
  {
    # now look up this users preference for their current colorScheme.
    my $userPrefObj = $authObj->getUserPreferenceInfo(userId => $userId, app => "Portal",
                                module => "Global", preference => "colorScheme");
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return $colorScheme;
    }
    if (!defined $userPrefObj)
    {
      $self->error("User = '" . $userInfo->get("uname") . "' does not have preference colorScheme set for app='Portal', module='Global'!<br>\n");
      return $colorScheme;
    }
    $colorSchemeName = $userPrefObj->get("value");
    if ($colorSchemeName !~ /^(.+)( \(user\))$/)
    {
      # we have to set userId = -1 since the user is using a default color scheme
      $userId = -1;
    }
  }
  else
  {
    # lookup the default colorScheme name.
    my $prefName = "Portal_Global_colorScheme_Default";
    $colorSchemeName = $self->methods->getConfigValue(name => $prefName, portalDB => $self->portalDB);
    if ($self->error)
    {
      $self->prefixError();
      return $colorScheme;
    }
    if (!defined $colorSchemeName)
    {
      $self->error("config entry name = '$prefName' does not exist!<br>\nYou need to upgrade the database!<br>\n");
      return $colorScheme;
    }
  }

  $colorScheme = $self->getColorScheme(name => $colorSchemeName, userId => $userId);
  if ($self->error)
  {
    $self->prefixError();
    return $colorScheme;
  }

  return $colorScheme;
}

# [] getAvailableColorSchemes(userId)
#    requires: userId - -1 or id of user
#    optional:
#    returns:  array of available colorScheme entries for the specified
#              user.
sub getAvailableColorSchemes
{
  my $self = shift;
  my %args = (userId => "", @_);
  my $userId = $args{userId};
  my @entries = ();

  if ($userId !~ /^(-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return @entries;
  }

  my $sql = "SELECT name FROM colorscheme_tb WHERE userid = ? ORDER BY name";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $userId ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return @entries;
  }
  while (my @info = $sth->fetchrow_array)
  {
    $entries[++$#entries] = $info[0];
  }

  return @entries;
}

# {} getDerivedColorSchemes(name)
#    requires: name - name of system colorScheme that users have derived from.
#    optional:
#    returns:  hash of colorScheme name => userId values that have been derived from name.
sub getDerivedColorSchemes
{
  my $self = shift;
  my %args = (name => "", @_);
  my $name = $args{name};
  my %entries = ();

  if ($name !~ /^(.+)$/ || $name =~ /^(.+)( \(user\))$/)
  {
    $self->error("name = '$name' is invalid!<br>\n");
    return %entries;
  }

  my $sql = "SELECT name, userid FROM colorscheme_tb WHERE derived_from = ? ORDER BY name, userid";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $name ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return %entries;
  }
  while (my @info = $sth->fetchrow_array)
  {
    $entries{$info[0]} = $info[1];
  }

  return %entries;
}

# int createColorScheme(colorScheme, log)
#    requires: colorScheme - ColorScheme object
#    optional: log = 1
#    returns:  1 - OK, 0 - error, -1 - already exists,
#              -2 - permission denied
sub createColorScheme
{
  my $self = shift;
  my %args = (colorScheme => undef, log => 1, @_);
  my $colorScheme = $args{colorScheme};
  my $log = $args{log};

  if (!defined $colorScheme)
  {
    $self->error("colorScheme must be defined!<br>\n");
    return 0;
  }
  if (!$colorScheme->isValid())
  {
    $self->error($colorScheme->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return -1;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  my $name = $colorScheme->get("name");
  my $userId = $colorScheme->get("userId");
  if ($userId != -1)
  {
    # instantiate an Auth instance.
    my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }

    # now make sure the specified user exists
    my $userInfo = $authObj->getUserInfo(id => $userId);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }
    if (!defined $userInfo)
    {
      $self->error("User with id = '$userId' does not exist!<br>\n");
      return 0;
    }

    # now make sure we are the same user creating this entry or at least we are a sysadmin
    if ($userId != $userObj->get("id"))
    {
      if ($userObj->get("sysadmin") == 0)
      {
        # see if we are an admin in the same company as this user.
        if ($userObj->get("admin") == 0)
        {
          # we are not an admin!
          return -2;
        }
        elsif ($userObj->get("companyId") != $userInfo->get("companyId"))
        {
          # we are an admin, but not in the same company as the user we are working for.
          return -2;
        }
      }
    }
  }
  elsif ($userObj->get("sysadmin") == 0)
  {
    # we are not a sysadmin, so we can't make -1 entries!
    return -2;
  }

  # now make sure this colorScheme is not already created.
  my $tmpColorScheme = $self->getColorScheme(name => $name, userId => $userId);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (defined $tmpColorScheme)
  {
    return -1;
  }

  # generate the xml string to work with.
  my $value = $colorScheme->export();
  if ($colorScheme->error)
  {
    $self->error($colorScheme->errorMessage);
    return 0;
  }

  my $sql = "INSERT INTO colorscheme_tb (name, userId, value, derived_from) VALUES (?, ?, ?, ?)";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $name, $userId, $value, $colorScheme->get("derivedFrom") ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the ColorScheme Entry Create event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 27, extraInfo => "ColorScheme Entry: " . $colorScheme->print);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

# int updateColorScheme(colorScheme, log)
#    requires: colorScheme - ColorScheme object
#    optional: log = 1
#    returns:  1 - OK, 0 - error, -1 - does not exist,
#              -2 - permission denied
#    summary: only the value part can be updated in the database.
sub updateColorScheme
{
  my $self = shift;
  my %args = (colorScheme => undef, log => 1, @_);
  my $colorScheme = $args{colorScheme};
  my $log = $args{log};

  if (!defined $colorScheme)
  {
    $self->error("colorScheme must be defined!<br>\n");
    return 0;
  }
  if (!$colorScheme->isValid())
  {
    $self->error($colorScheme->errorMessage);
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return -1;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  my $name = $colorScheme->get("name");
  my $userId = $colorScheme->get("userId");
  if ($userId != -1)
  {
    # instantiate an Auth instance.
    my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }

    # now make sure the specified user exists
    my $userInfo = $authObj->getUserInfo(id => $userId);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }
    if (!defined $userInfo)
    {
      $self->error("User with id = '$userId' does not exist!<br>\n");
      return 0;
    }

    # now make sure we are the same user updating this entry or at least we are a sysadmin
    if ($userId != $userObj->get("id"))
    {
      if ($userObj->get("sysadmin") == 0)
      {
        # see if we are an admin in the same company as this user.
        if ($userObj->get("admin") == 0)
        {
          # we are not an admin!
          return -2;
        }
        elsif ($userObj->get("companyId") != $userInfo->get("companyId"))
        {
          # we are an admin, but not in the same company as the user we are working for.
          return -2;
        }
      }
    }
  }
  elsif ($userObj->get("sysadmin") == 0)
  {
    # we are not a sysadmin, so we can't update -1 entries!
    return -2;
  }

  # now make sure this colorScheme does exist.
  my $tmpColorScheme = $self->getColorScheme(name => $name, userId => $userId);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (!defined $tmpColorScheme)
  {
    return -1;
  }

  # generate the xml string to work with.
  my $value = $colorScheme->export();
  if ($colorScheme->error)
  {
    $self->error($colorScheme->errorMessage);
    return 0;
  }

  my $sql = "UPDATE colorscheme_tb SET value = ? WHERE userId = ? AND name = ?";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $value, $userId, $name ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the ColorScheme Entry Update event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 28, extraInfo => "ColorScheme Entry: " . $colorScheme->print);
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

# int deleteColorScheme(name, userId, log)
#    requires: name - name of colorScheme,
#              userId - -1 or id of user
#    optional: log = 1
#    returns:  1 - OK, 0 - error, -1 - does not exist,
#              -2 - permission denied
sub deleteColorScheme
{
  my $self = shift;
  my %args = (name => "", userId => "", log => 1, @_);
  my $name = $args{name};
  my $userId = $args{userId};
  my $log = $args{log};

  if ($userId !~ /^(-1|\d+)$/)
  {
    $self->error("userId = '$userId' is invalid!<br>\n");
    return 0;
  }
  if (length $name == 0)
  {
    $self->error("name = '$name' is invalid!<br>\n");
    return 0;
  }
  if ($userId != -1 && $name !~ /^(.+)( \(user\))$/)
  {
    $self->error("name = '$name' is invalid!<br>\n");
    return 0;
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->error("log = '$log' is invalid!<br>\n");
    return -1;
  }
  if (!defined $self->sessionObj || !$self->sessionObj->isValid)
  {
    $self->error("the sessionObj is not valid!<br>\n");
    return 0;
  }
  my $userObj = $self->sessionObj->store->{userObj};

  if ($userId != -1)
  {
    # instantiate an Auth instance.
    my $authObj = Portal::Auth->new(portalDB => $self->portalDB, langObj => $self->langObj);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }

    # now make sure the specified user exists
    my $userInfo = $authObj->getUserInfo(id => $userId);
    if ($authObj->error)
    {
      $self->error($authObj->errorMessage);
      return 0;
    }
    if (!defined $userInfo)
    {
      $self->error("User with id = '$userId' does not exist!<br>\n");
      return 0;
    }

    # now make sure we are the same user deleting this entry or at least we are a sysadmin
    if ($userId != $userObj->get("id"))
    {
      if ($userObj->get("sysadmin") == 0)
      {
        # see if we are an admin in the same company as this user.
        if ($userObj->get("admin") == 0)
        {
          # we are not an admin!
          return -2;
        }
        elsif ($userObj->get("companyId") != $userInfo->get("companyId"))
        {
          # we are an admin, but not in the same company as the user we are working for.
          return -2;
        }
      }
    }
  }
  elsif ($userObj->get("sysadmin") == 0)
  {
    # we are not a sysadmin, so we can't delete -1 entries!
    return -2;
  }

  # now make sure this colorScheme does exist.
  my $tmpColorScheme = $self->getColorScheme(name => $name, userId => $userId);
  if ($self->error)
  {
    $self->prefixError();
    return 0;
  }
  if (!defined $tmpColorScheme)
  {
    return -1;
  }

  #if ($userId == -1)
  #{
    # do we have to delete any derivedFrom entries?  This is a placeholder in case I decide to in the future.
    # I'm going to say no, since if we delete an entry for a user and it was their default, we now have to
    # fix that and that is way too much work for now.  There is nothing that is tying us to the original,
    # except for when we update the original, we want some of those changes to possibly trickle down.
  #}

  my $sql = "DELETE FROM colorscheme_tb WHERE userId = ? AND name = ?";

  # execute the sql statement.
  my $sth = $self->portalDB->read(sql => $sql, plug => [ $userId, $name ]);
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }
  $self->portalDB->commit;  # make the changes permanent.
  if ($self->portalDB->error)
  {
    $self->error($self->portalDB->errorMessage);
    return 0;
  }

  if ($log)
  { # log the ColorScheme Entry Delete event.
    $self->methods->doLog(sessionObj => $self->sessionObj, db => $self->portalDB, action => 29, extraInfo => "ColorScheme Entry: userId='$userId', name='$name'");
    if ($self->methods->error)
    {
      $self->error($self->methods->errorMessage);
      return 0;
    }
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (Non-Inline POD)

  int createApplication(applicationObj, log)
    Takes the applicationObj ApplicationObject and creates the entry in
    the portal database after validating the application doesn't exist
    yet.  Returns 1 if successfull, 0 if error, -1 if already exists.
    optional: log - 1 means log event

  int updateApplication(applicationObj, log)
    takes: applicationObj - ApplicationObject
    optional: log
    returns: 1 if ok, 0 if error, -1 if app doesn't exist
    This method will take the ApplicationObject and update all fields
    in the database other than the id field.

  int backupAppServerEntry(appServerObj)
    takes: appServerObj - AppServerObject
    returns: 1 if ok, 0 if error, -1 if already backed up, -2 if app id
             doesn't exist
    This method, checks to see if the specified app id exists in the
    app_tb.  If it doesn't, it then returns -2, else it checks to see if
    the specified combination of id, server and port exists, if it does,
    then return -1, else create the entry in app_servers_tb and return 1
    if ok, else return 0 and set the error.

    This method is only going to be used by applications that are
    installing themselves in a Portal.

  int assignAppToCompany(companyAppObj, log)
    takes: companyAppObj - CompanyApplicationObject
    optional: log
    returns: 1 if the assignment was possible and ok, 0 if an error
             occurred, -1 if app doesn't exist, -2 if company doesn't
             exist, -3 if assignment already exists
    This method will create the entry in company_app_tb for this
    companyAppObj.

  @ getAppsForCompany(companyObj, wap)
    takes: companyObj - CompanyObject
    optional: wap
    returns: array of ApplicationObjects where the server, port, cost,
             unit, height, width and autorun values are filled in from
             the company_app_tb but everything else is from the app_tb.
    This method gathers all applications assigned to the specified
    company and returns the array of ApplicationObjects that define them.
    If specified, wap is used to select only those apps that match the
    value of wap.

  companyAppObj getCompanyAppInfo(companyId, appId)
    takes: companyId, appId
    returns: companyAppObj CompanyApplicationObject or undef if error or
             not found.

  int unAssignAppFromCompany(companyAppObj, log)
    takes: companyAppObj - CompanyApplicationObject (The number of
           licenses can be 1)
    optional: log
    returns: 1 if the unassignment was possible and ok, 0 if an error
             occurred, -1 if assignment didn't exist
    This method will delete the entry from the company_app_tb and also
    delete all entries from the user_app_tb for this company and app
    combination.

  int removeAllAppsForCompany(companyObj, log)
    takes: companyObj
    optional: log
    returns: 1=OK, 0=Error

  LicenseObject getCompanyLicenseInfo(companyId, appId)
    takes: companyId, appId
    returns: LicenseObject or undef if error occurred.
    This method checks to make sure the app is assigned to the company.
    If not then errorCode = -1, else errorCode = 0 on normal error.
    Then the value in the assignment is used for the numTotal value.  A
    count of all users assigned that app in that company is then
    done and placed in numFree as numTotal - count.  The cost and unit
    values from the assignment record are then placed in the object.

  int assignAppToUser(userAppObj, log)
    takes: userAppObj - UserApplicationObject
    optional: log
    returns: 1 if ok, 0 if error, -1 if app doesn't exist, -2 if company
             doesn't exist, -3 if user doesn't exist, -4 if assignment
             already done, -5 if app not assigned to company, -6 if license
             allotment will be violated, -7 if app is administration and
             user is not an admin
    This method will assign the specified company application to the
    specified user in the user_app_tb.

  int unAssignAppFromUser(userAppObj, log)
    takes: userAppObj - UserApplicationObject
    optional: log
    returns: 1 if ok, 0 if error, -1 if assignment didn't exist
    This method will delete the entry from the user_app_tb for this app,
    company, user pair.

  % getAppsForUser(userId, wap)
    takes: userId
    optional: wap
    returns: hash of hashes where the top level is the application type
             and each application type has a hash of ApplicationObjects
             where the server, port, cost and unit values are filled
             in from the company_app_tb but everything else is from the
             app_tb.  This method gathers all applications assigned to
             the specified user and returns the array of
             ApplicationObjects that define them.  If wap is defined,
             then only those apps that meet the state of wap are
             returned.  If the user doesn't exist, then errorCode = -1,
             if company doesn't exist, errorCode = -2 (shouldn't ever be
             returned)

  bool isAppAssignedToCompany(appName, companyObj)
    requires: appName, companyObj
    optional:
    returns: 1 = Assigned, 0 = Not Assigned
    summary: Determines if the specified app is assigned to the specified
             company.

  bool isAppAssignedToUser(appName, userId)
    requires: appName, userId
    optional:
    returns: 1 = Assigned, 0 = Not Assigned
    summary: Determines if the specified app is assigned to the specified
             user.

  @ getAllUserAppsForCompany(companyObj)
    takes: companyObj
    returns: Array of ApplicationObjects for all User App Assignments for
             this company.

  @ flattenAppHash(apps)
    takes: apps - hash ref of ApplicationObject entries.
    returns: apps - array of ApplicationObject entries sorted by app type.

  % groupAppArray(apps)
    takes: apps - array of ApplicationObject entries.
    returns: apps - hash of ApplicationObject entries grouped by app type.

  @ getCompanyAppsForServer(serverName, appObj)
    requires: serverName (text string), appObj (ApplicationObject)
    optional:
    returns: array of CompanyApplicationObjects
    summary: Returns a list of CompanyApplicationObjects that defines all
             companies that have an instance of the specified Application
             on the specified server.

  void lockApp(app, type, users, company, log)
    requires: app - id of app, type - 'a' all
              users, 'u' specified users, 'c' company
    optional: users - comma seperated list of user id's if type = 'u',
              company - id of company if type = 'c', log - 1/0
    returns: nothing
    summary: locks the specified app for use only by the admin user in
             the session based upon the type.

  void unlockApp(app, type, users, company, log)
    requires: app, type
    optional: users, company, log - 1/0
    returns: nothing
    summary: unlocks the specified app, but only if the user
             in the session was the one who locked it.

  void breakAppLock(admin, app, type, users, company, log)
    requires: admin, app, type
    optional: users, company, log - 1/0
    returns: nothing
    summary: unlocks the specified app, but you have to be an
             administrator in the same company as the admin who locked it
             or be a system administrator.

  bool isAppLocked(app)
    requires: app
    optional:
    returns: 0 - not locked, 1 - locked
    summary: checks to see if the specified app is locked for the company,
             the specific user, or all users of the person calling us.

  bool isAppLockedByMe(app)
    requires: app
    optional:
    returns: 0 - no, 1 - yes
    summary: checks to see if the specified app is locked by us.

  bool isUserLockedOut(app)
    requires: app - app id
    optional:
    returns: 0 - not locked out, 1 - locked out
    summary: checks to see if the user in the session is locked out of
             the specified app.

  int getLockedAppInfo(app)
    requires: app - app id
    optional:
    returns: array of hashes containing admin - id of user who locked
             the app, type - type of lock, users - users being locked out,
             company - company id being locked out.
             An empty array is returned if the app is not locked.
    summary: returns the info the specified app if it is locked.

  int createDynamicContentEntry(dynamicContentObj, log)
    requires: dynamicContentObj
    optional: log = 1
    returns: 1 - entry created, 0 - error, -1 - entry already created,
             -2 - permission denied

  dynamicContentObj[] getDynamicContentEntries(index, companyId, userId,
                         callingApp, tag, app, global)
    requires:
    optional: index or any of companyId, userId, callingApp, tag, app,
              global - 1/0 (defaults to 0)
    returns: array of dynamicContentObj entries that represent all
             dynamic_content_tb table entries that matched the specified
             parameters.  The returned values are sorted by
             companyId, userId, callingApp, tag, app and arguments
             If global = 1, then we include the companyId = -1 or
             companyId = companyId and userId = -1 checks, but this is
             only valid when the user specifies a companyId and a userId.


  int updateDynamicContentEntry(dynamicContentObj, log)
    requires: dynamicContentObj
    optional: log = 1
    returns: 1 - entry updated, 0 - error, -1 - entry does not exist,
             -2 - permission denied

  int deleteDynamicContentEntry(dynamicContentObj, log)
    requires: dynamicContentObj
    optional: log = 1
    returns: 1 - entry deleted, 0 - error, -1 - entry does not exist,
             -2 - permission denied

  int deleteDynamicContentEntries(companyId, userId, callingApp, tag,
                                  app, log)
    requires:
    optional: companyId, userId, callingApp, tag, app, log = 1
    returns: >= 0 - number entries deleted, -1 - error,
             -2 - permission denied

  colorScheme getColorScheme(name, userId)
    requires: name - name of colorScheme,
              userId - -1 or id of user
    optional:
    returns:  ColorScheme object (Portal::Data::ColorScheme) or undef
              on error.

  colorScheme getCurrentColorScheme(userId)
    requires: userId - id of real user or -1 for system default colors
    optional:
    returns:  ColorScheme object or undef on error.  The colorScheme
              returned is based upon the users set preferences.

  [] getAvailableColorSchemes(userId)
    requires: userId - -1 or id of user
    optional:
    returns:  array of available colorScheme entries for the specified
              user.

  {} getDerivedColorSchemes(name)
     requires: name - name of system colorScheme that users have derived
               from.
     optional:
     returns:  hash of colorScheme name => userId values that have been
               derived from name.

  int createColorScheme(colorScheme, log)
    requires: colorScheme - ColorScheme object
    optional: log = 1
    returns:  1 - OK, 0 - error, -1 - already exists,
              -2 - permission denied

  int updateColorScheme(colorScheme, log)
    requires: colorScheme - ColorScheme object
    optional: log = 1
    returns:  1 - OK, 0 - error, -1 - does not exist,
              -2 - permission denied
    summary: only the value part can be updated in the database.

  int deleteColorScheme(name, userId, log)
    requires: name - name of colorScheme,
              userId - -1 or id of user
    optional: log = 1
    returns:  1 - OK, 0 - error, -1 - does not exist,
              -2 - permission denied

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::Objects::ApplicationObject(3), Portal::Base(3)

=cut
