# Args.pm - The Object Class that provides a Args Base Object
# Created by James A. Pattie, 03/26/2003.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Args;
use Portal::Base;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

=head1 NAME

Args - Object used to build a base Application Object Class.

=head1 SYNOPSIS

 package Portal::Test;
 use Portal::Args;
 use strict;
 use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

 require Exporter;

 @ISA = qw(Portal::Args Exporter AutoLoader);
 @EXPORT = qw();

 $VERSION = '0.01';

 sub new
 {
   my $class = shift;
   my $self = $class->SUPER::new(@_);
   my %args = ( something => 'Hello World!', @_ );

   if ($self->error)
   {
     $self->prefixError();
     return $self;
   }

   # instantiate anything unique to this module
   $self->{something} = $args{something};

   # do validation
   # The $self->Portal::Test::isValid makes sure we access our
   # isValid method and not any classes isValid method that has
   # derived from us.
   if (!$self->Portal::Test::isValid)
   {
     # the error is set in the isValid() method.
     return $self;
   }

   # do anything else you might need to do.
   return $self;
 }

 sub isValid
 {
   my $self = shift;

   # make sure our Parent class is valid.
   if (!$self->SUPER::isValid())
   {
     $self->prefixError();
     return 0;
   }

   # validate our parameters.
   if ($self->{something} !~ /^(.+)$/)
   {
     $self->invalid("something", $self->something);
   }

   if ($self->numInvalid() > 0 || $self->numMissing() > 0)
   {
     $self->error($self->genErrorString("all"));
     return 0;
   }
   return 1;
 }

=head1 DESCRIPTION

Args is the base class for Portal.pm.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(langObj, configObj, cookieObj, input, portalDB,
                 browserType, browserVersion, browserMaker,
                 methods, applicationObj, authObj)

Creates a new instance of the Portal::Args
object.

requires: langObj - Portal::Language Object
          configObj - Portal::Data::Config
          cookieObj - hash of cookies
          input - hash of passed in arguments
          portalDB - DBIWrapper pointing at the Portal
                     Database
          browserType
          browserVersion
          browserMaker
          methods - Portal::Methods Object
          applicationObj - Portal::Application Object
          authObj - Portal::Auth Object

returns:  object reference

The langObj requires the following phrases:

B<missingArgument> - "%s is missing"

B<invalidArgument> - "%s = '%s' is invalid"

B<errorExclam> - "Error!"


=cut

sub new
{
  my $class = shift;
  my %args = ( configObj => undef, cookieObj => undef, input => undef, portalDB => undef,
               browserType => "HTML", browserVersion => "4.0",
               browserMaker => "", methods => undef,
               applicationObj => undef, authObj => undef, @_ );
  my $self = $class->SUPER::new(@_);

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{applicationObj} = $args{applicationObj};  # The Portal::Application module.
  $self->{input} = $args{'input'};
  $self->{cookieObj} = $args{cookieObj};
  $self->{configObj} = $args{configObj};
  $self->{browserType} = $args{browserType};
  $self->{browserVersion} = $args{browserVersion};
  $self->{browserMaker} = $args{browserMaker};
  $self->{authObj} = $args{authObj};
  $self->{methods} = $args{methods};
  $self->{portalDB} = $args{portalDB};

  # do validation
  if (!$self->Portal::Args::isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().


=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{input})
  {
    $self->missing("input");
  }
  if (not defined $self->{applicationObj})
  {
    $self->missing("applicationObj");
  }
  elsif (!$self->{applicationObj}->isValid)
  {
    $self->invalid("applicationObj", $self->{applicationObj}->errorMessage);
  }
  if (not defined $self->{authObj})
  {
    $self->missing("authObj");
  }
  elsif (!$self->{authObj}->isValid)
  {
    $self->invalid("authObj", $self->{authObj}->errorMessage);
  }
  if (not defined $self->{cookieObj})
  {
    $self->missing("cookieObj");
  }
  if (not defined $self->{configObj})
  {
    $self->missing("configObj");
  }
  if ($self->{browserType} !~ /^(HTML|HDML|WML)$/)
  {
    $self->invalid("browserType", $self->{browserType});
  }
  if (length $self->{browserVersion} == 0)
  {
    $self->invalid("browserVersion", $self->{browserVersion});
  }
  if (not defined $self->{methods})
  {
    $self->missing("methods");
  }
  elsif (!$self->{methods}->isValid)
  {
    $self->invalid("methods", $self->{methods}->errorMessage);
  }
  if (not defined $self->{portalDB})
  {
    $self->missing("portalDB");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item scalar arguments(void)

 Returns a string that when eval'ed will specify all the
 arguments to pass into an instance of the Portal::Args->new
 method.
 
 Ex:  
 
 "langObj => \$self->{langObj}, portalDB => \$self->{portalDB}"
 
=cut

sub arguments
{
  my $self = shift;
  my $args = $self->SUPER::arguments();

  foreach my $arg (qw(portalDB cookieObj input configObj methods authObj applicationObj browserType browserVersion browserMaker))
  {
    $args .= ", " if ($args);
    $args .= "$arg => \$self->{$arg}";
  }

  return $args;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3)

=cut
