# Help.pm - The Object Class that provides a Help Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Data::Help;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.10';

# Help Data structure
my %Topics = (
             en => {
                     Normal => {
                               "_order_" => "Logout,_seperator_,Refresh,Help,_line_,System,Crash & Burn,Test",
                               "_overview_" => "This is the top level of help topics for the Portal. &nbsp;Select a topic to get more info. &nbsp;Topics with a &gt; to their left, indicate they have other topics grouped under them.",
                               "Logout" => "This Logs you out of the Portal, closes all open program windows that we are keeping track of, and brings you back to the login screen.",
                               "Refresh" => "This redisplays the Portal Desktop. &nbsp;Use this if you were just assigned a new application and it isn't showing up on your desktop.",
                               "Help" => "This is what you are currently seeing.",
                               "System" => "Go <help app=\"\" content=\"System\">here</help> for System help.",
                               "Crash & Burn" => "Go <help app=\"UserProperties\" content=\"Normal\">here</help> to see help in the UserProperties App.",
                               "Test" => {
                                         "_order_" => "Link,Yet another level down",
                                         "_overview_" => "This shows an _overview_ in action and briefly mentions that topics can now have multiple &lt;help&gt; entries in them and the engine now properly handles them all.",
                                         "_summary_" => "This is a test grouping.",
                                         "Yet another level down" => {
                                           "_order_" => "Link2,_seperator_,_line_,_seperator_,Link3",
                                           "_overview_" => "This is the third level down.  You could keep going forever if you really wanted to.",
                                           "_summary_" => "Summary test 2",
                                           "Link2" => "This is link2 sublevel",
                                           "Link3" => "This is link3 sublevel",
                                         },
                                         "Link" => "Linking to the previous menu <help app=\"\" content=\"Normal,Logout\">now what are you_ going to</help> do.&nbsp; Testing having multiple <help app=\"\" content=\"Normal,System\">links</help> in a topic.",
                                         },
                               },
                     System => {
                               "_order_" => "This should never be reached",
                               "This should never be reached" => "The Portal always runs in Normal mode!<br><br>Go <help app=\"\" content=\"Normal,Help\">here</help> to return to the Normal Help System.",
                               },
                   },
             );

=head1 NAME

Help - Object used to build a Help Object Class.

=head1 SYNOPSIS

  use Portal::Data::Help;
  my $obj = Portal::Data::Help->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Help is a Help class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(lang)

 Creates a new instance of the Portal::Data::Help module.
 See Portal::Base(3) for a listing of required arguments.
 
 requires: lang - the language code to work with.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( lang => "en", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{lang} = $args{lang};  # we will assume this language is valid as the Language module will have already validated it for us.
  $self->{Topics} = \%Topics;   # store the help data in the class.
  if (!exists $self->{Topics}->{$self->{lang}})
  {
    $self->{lang} = "en";  # This must always exist!
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  Topics - Help topics hash.  The language code is the first index,
           followed by Normal and System.  They provide the main indexes
           for the Normal and System modes.  Under them are the topics
           of interest for the 2 modes.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Help(3), Portal::Base(3);

=cut
