# Desktop.pm - The Object Class that provides a Desktop Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Desktop;
use strict;
use Portal::Data::Variables;
use Portal::PortalArgs;
use Portal::Who;
use Portal::DynamicContent;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::PortalArgs Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.16';

# global variables.
my %commands = ( display => "Desktop", logout => "Logout" );

=head1 NAME

Desktop - Object used to build a Desktop Object Class.

=head1 SYNOPSIS

  use Portal::Desktop;
  my $obj = Portal::Desktop->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Desktop is a Desktop class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(startSession)

 Creates a new instance of the Portal::Desktop object.

 startSession defines whethere we have to create a sessionObj or not.
              It is a bool value, defaults to 0.

 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( startSession => 0, @_ );

  if ($self->error)
  {
    $self->prefixError($self->errorMessage);
    return $self;
  }

  # instantiate anything unique to this module
  my $variableObj = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{variables} = $variableObj;

  $self->{startSession} = $args{startSession};

  my $whoObj = Portal::Who->new(portalDB => $self->{portalDB}, langObj => $self->{langObj},
                                configObj => $self->{configObj}, methods => $self->{methods});
  if ($whoObj->error)
  {
    $self->error($whoObj->errorMessage);
    return $self;
  }
  $self->{whoObj} = $whoObj;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{startSession} !~ /^(1|0)$/)
  {
    $self->invalid("startSession", $self->{startSession});
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject run(state, command)

 Pulls the specified state file into existance and calls
 run(command).

 returns: The HTMLObject document to display or undef if an error.

=cut

sub run
{
  my $self = shift;
  my %args = ( @_ );
  my $command = $args{command};
  my $doc;
  my $appName = $self->{configObj}->{myHostName};

  # first generate the portal session
  my $session = $self->{methods}->portalSession(portalDB => $self->{portalDB}, cookieObj => $self->{cookieObj}, appName => "pcx_session_id", configObj => $self->{configObj}, sessionId => "", mode => ($self->startSession ? "write" : "read"));
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    $self->{input}->{removeCookies} = 1;  # signal that the cookie needs to be removed.
    return undef;
  }
  if ($session->error)
  {
    $self->error($session->errorMessage);
    $self->{input}->{removeCookies} = 1;  # signal that the cookie needs to be removed.
    return undef;
  }

#   if (!$self->startSession)
#   {
#     # lets walk the session and see what is in it.
#     $doc = HTMLObject::Base->new();
#     $doc->print("session->mode = '" . $session->{mode} . "', sessionId = '" . $session->sessionId . "', session->store = '" . $session->{store} . "' : '" . (tied %{$session->{store}}) . "'<br /><br />\n");
#     foreach my $entry (keys %{$session->{store}})
#     {
#       $doc->print("$entry = '" . $session->{store}->{$entry} . "'<br />\n");
#     }
#     return $doc;
#   }

  # get the info about the user and create the entry in the session.
  my $userObj;
  if ($self->{startSession})
  {
    $userObj = $self->{authObj}->getUserInfo(uname => $self->{input}->{uname});
  }
  else
  {
    $userObj = $self->{authObj}->getUserInfo(id => $session->{store}->{userObj}->{id});
  }
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage . "startSession = '$self->{startSession}'.");
    return undef;
  }
  if (defined $userObj)
  {
    if ($userObj->error)
    {
      $self->error($userObj->errorMessage);
      return undef;
    }
  }
  else
  {
    my $userNotFound = $self->langObj->map("userNotFound");
    $doc = $self->logout(checkOK => 0, error => sprintf($userNotFound, ($self->{startSession} ? $self->{input}->{uname} : $session->{store}->{userObj}->{uname})), session => $session);
    if ($self->error)
    {
      $self->error($self->errorMessage);
      return undef;
    }
    return $doc;
  }
  my $toggleMode = 0;
  if ($self->startSession || $session->{store}->{userObj} != $userObj)
  {
    $toggleMode = 1 if ($session->getMode() eq "read");
    $session->setMode("write");
    $session->{store}->{userObj} = $userObj;  # store the userObj in the session.
    $session->{store}->{changed} += 1;
  }

  # get the company object.
  my $companyObj = $self->{authObj}->getCompanyInfo(id => $session->{store}->{userObj}->{companyId});
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return undef;
  }
  if (defined $companyObj)
  {
    if ($companyObj->error)
    {
      $self->error($companyObj->errorMessage);
      return undef;
    }
  }
  else
  {
    my $companyDoesNotExist = $self->langObj->map("companyDoesNotExist");
    $doc = $self->logout(checkOK => 0, error => sprintf($companyDoesNotExist, $session->{store}->{userObj}->{companyId}), session => $session);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
    return $doc;
  }
  if ($self->startSession) # I can't compare companyObj's yet -- || $session->{store}->{companyObj} != $companyObj)
  {
    $toggleMode = 1 if ($session->getMode() eq "read");
    $session->setMode("write");
    $session->{store}->{companyObj} = $companyObj;  # store the companyObj in the session.
    $session->{store}->{changed} += 1;
  }

  # set the User ID in the input hash for error handling in index.cgi
  $self->{input}->{uid} = $session->{store}->{userObj}->{id};

  # make sure that the company and user are still active.
  if ($companyObj->active == 0)
  {
    my $inactiveCompany = $self->langObj->map("inactiveCompany");
    $doc = $self->logout(checkOK => 0, error => sprintf($inactiveCompany, $companyObj->{name}), session => $session);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
    return $doc;
  }
  if ($userObj->active == 0)
  {
    my $inactiveUser = $self->langObj->map("inactiveUser");
    $doc = $self->logout(checkOK => 0, error => $inactiveUser, session => $session);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
    return $doc;
  }

  if ($self->startSession)
  {
    $toggleMode = 1 if ($session->getMode() eq "read");
    $session->setMode("write");
    # store the browser capabilities info in the session.
    $session->{store}->{browserCap} = {};
    if ($self->{browserType} eq "HTML")
    {
      my $browserCapCookie = $self->{cookieObj}->{cookies}->{pcx_browser_cap};
      $session->{store}->{browserCap}->{'document.getElementById'} = 0;
      $session->{store}->{browserCap}->{'document.all'} = 0;
      $session->{store}->{browserCap}->{'document.layers'} = 0;
      $session->{store}->{browserCap}->{'document.createElement'} = 0;

      foreach my $cap (split (/\|/, $browserCapCookie))
      {
        $session->{store}->{browserCap}->{$cap} = 1;
      }
    }
    elsif ($self->{browserType} eq "HDML")
    {
    }
    elsif ($self->{browserType} eq "WML")
    {
    }

    # create the hash that will store the application session info
    $session->{store}->{apps} = {};

    # create the hash which will notify us of any Portal specific windows being open.
    $session->{store}->{windows} = {};
    $session->{store}->{changed} += 1;

    # get the current time as seconds past the epoc, to use as a unique window modifier for help and initial app windows.
    $session->{store}->{windowTitleTimestamp} = $self->{methods}->getCurrentDate(format => "%s");

    # make the who_tb entry
    my $whoEntryObj = Portal::Objects::WhoEntryObject->new(uname => $userObj->{uname}, companyId => $userObj->{companyId}, ipAddress => (exists $ENV{REMOTE_ADDR} ? $ENV{REMOTE_ADDR} : '127.0.0.1'),
                                                           session => $session->{sessionId}, skipDates => 1, langObj => $self->{langObj});
    if ($whoEntryObj->error)
    {
      $self->error($whoEntryObj->errorMessage);
      return undef;
    }
    my $result = $self->{whoObj}->makeWhoEntry(whoEntryObj => $whoEntryObj);
    if ($self->{whoObj}->error)
    {
      $self->error($self->{whoObj}->errorMessage);
      return undef;
    }
    if ($result == -1)
    {  # the entry already exists, so the user logged out and is coming back in, just update their entry.
      $result = $self->{whoObj}->updateWhoEntry(session => $session->{session_id});
      if ($self->{whoObj}->error)
      {
        $self->error($self->{whoObj}->errorMessage);
        return undef;
      }
      if ($result == -1)
      {
        $self->error("session = '$session->{session_id}' no longer exists in the who_tb on attempt to update!<br>\n");
        return undef;
      }
    }
  }
  else
  {
    # update the who_tb entry
    my $result = $self->{whoObj}->updateWhoEntry(session => $session->{session_id});
    if ($self->{whoObj}->error)
    {
      $self->error($self->{whoObj}->errorMessage);
      return undef;
    }
    if ($result == -1)
    {
      $self->error("session = '$session->{session_id}' no longer exists in the who_tb on attempt to update (non-create)!<br>\n");
      return undef;
    }
  }

  $session->setMode("read");

  if (!exists $commands{$command})
  {
    $self->error(sprintf($self->langObj->map("commandNotKnown"), $command));
    return undef;
  }

  # store the browserCapabilities hash before the session may go out of scope.
  my %browserCap = %{$session->{store}->{browserCap}};

  if ($command eq "display")
  {
    # get the users colorScheme to work with.
    my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => $session->{store}->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    if (!defined $colorScheme)
    {
      $self->error("No valid colorScheme returned!<br>\n");
      return undef;
    }

    my $logout = $self->langObj->map("logout");
    my $refresh = $self->langObj->map("refresh");
    my $userEquals = $self->langObj->map("userEquals");
    my $companyEquals = $self->langObj->map("companyEquals");
    my $userName = $session->{store}->{userObj}->fname;
    my $runPhrase = $self->langObj->map("run");
    my $help = $self->langObj->map("help");
    if (length $session->{store}->{userObj}->{mname} > 0)
    {
      $userName .= " " . $session->{store}->{userObj}->{mname};
    }
    $userName .= " " . $session->{store}->{userObj}->{lname};
    my $myUrl = $self->{methods}->createBaseURL(type => "Portal", linkType => "cgi");
    my $portalVersion = sprintf($self->langObj->map("portalVersion"), "black", $Portal::VERSION);
    my $uname = $session->{store}->{userObj}->{uname};

    my $companyName = $session->{store}->{companyObj}->{name};
    my $companyCode = $session->{store}->{companyObj}->{code};

    # calculate the Date for the user, synced to their timezone.
    my $tz = $session->{store}->{userObj}->{tz};
    my $tzOffset = $self->{variables}->{timeZones}->{$tz};
    my $timeFormatString = ($session->{store}->{userObj}->{timeFormat} == 12 ? "%I" : "%H");
    my $amPMString = ($session->{store}->{userObj}->{timeFormat} == 12 ? "%p" : "");
    my %userDate = ();
    foreach my $part ("%m", "%d", "%Y", "$timeFormatString", "%M", "$amPMString")
    {
      # lookup this part of the date
      if ($part)
      {
        $userDate{$part} = $self->{methods}->getCurrentLocalizedDate(tz => $tz, format => $part);
      }
    }
    my $userDate = "<span class='menuHilight'>$userDate{'%Y'}</span>-<span class='menuHilight'>$userDate{'%m'}</span>-<span class='menuHilight'>$userDate{'%d'}</span> &nbsp;&nbsp;<span class='menuHilight'>$userDate{$timeFormatString}</span>:<span class='menuHilight'>$userDate{'%M'}" . ($amPMString ? $userDate{$amPMString} : "") . " $tz</span>";

    my %apps = $self->{applicationObj}->getAppsForUser(userId => $session->{store}->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }

    $doc = HTMLObject::Normal->new;
    $doc = $session->writeCookie(doc => $doc);            # output the session info to the document.
    if ($session->error)
    {
      $self->error($session->errorMessage);
      return undef;
    }

    # create the needed css entries
    my %cssEntries = $self->{methods}->generateColorSchemeCSS(colorScheme => $colorScheme);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $doc->print(%cssEntries);

    # generate the desktop.
    $doc->setTitle(sprintf($self->langObj->map("portal") . " - " . $self->langObj->map("desktop"), $Portal::VERSION));

    # create the launchApp JS function.
    $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "launchApp", langObj => $self->{langObj});
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }

    # create the validateClose JS function.
    $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "logOut", langObj => $self->{langObj});
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }

    my $url = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $myUrl, arguments => { app => "Portal", state => "Desktop", command => "display" });
    my $minutes = 5;
    $minutes = $minutes * 60; # get the number of seconds.
    $doc->refresh(seconds => $minutes, url => $url);

    # generate JavaScript to fire off 1 minute after the refresh should have happened.
    $minutes += 60;    # add a minute.
    $minutes *= 1000;  # convert to milliseconds.
    my $onLoadCode = "window.setTimeout(\"window.location='$url'\", $minutes);\n";

    my $helpUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $myUrl, arguments => { app => "Portal", state => "Help", command => "display", helpApp => "Portal", content => "Normal" });
    my $helpWindow = "help_Portal" . "_" . $session->{store}->{windowTitleTimestamp};

    # build up the menuItems arrays
    my @itemsLeft = ();
    my @itemsRight = ();
    my @itemsCenter = ();

    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => $self->{methods}->urlBuilder(doc => $doc, baseUrl => $myUrl, arguments => { app => "Portal", state => "Desktop", command => "logout" }),
                    text => $logout, title => $logout, onMouseOver => "window.status='$logout'; return true;", onClick => "return validateClose();", langObj => $self->{langObj});
    $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "link", url => $self->{methods}->urlBuilder(doc => $doc, baseUrl => $myUrl, arguments => { app => "Portal", state => "Desktop", command => "display" }),
                    text => $refresh, onMouseOver => "window.status='$refresh'; return true;", title => "$refresh", langObj => $self->{langObj});
    $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
    $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "link", url => $helpUrl,
                    text => $help, onMouseOver => "window.status='$help'; return true;",
                    onClick => "launchApp('$helpWindow', '$helpUrl', 480, 640); return false;", title => $help, langObj => $self->{langObj});
    $itemsCenter[0] = Portal::Objects::MenuItem->new(type => "text", text => $userDate, langObj => $self->{langObj});
    $itemsRight[0] = Portal::Objects::MenuItem->new(type => "hilightedText", text => "$portalVersion&nbsp;", langObj => $self->{langObj});

    # now validate they created ok.
    for (my $i=0; $i < scalar @itemsLeft; $i++)
    {
      if ($itemsLeft[$i]->error)
      {
        $self->error("itemsLeft[$i]\n<br>" . $itemsLeft[$i]->errorMessage);
        return undef;
      }
    }
    for (my $i=0; $i < scalar @itemsCenter; $i++)
    {
      if ($itemsCenter[$i]->error)
      {
        $self->error("itemsCenter[$i]\n<br>" . $itemsCenter[$i]->errorMessage);
        return undef;
      }
    }
    for (my $i=0; $i < scalar @itemsRight; $i++)
    {
      if ($itemsRight[$i]->error)
      {
        $self->error("itemsRight[$i]\n<br>" . $itemsRight[$i]->errorMessage);
        return undef;
      }
    }
    my $menuString = $self->{methods}->displayMenu(orientation => "horizontal",
                     itemsLeft => \@itemsLeft, itemsCenter => \@itemsCenter, itemsRight => \@itemsRight,
                     indent => 6);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }

    # generate the actual desktop
    $doc->setFocus("body");
    $doc->print(<<"END_OF_CODE");
<table width="100%" cellspacing="0" cellpadding="0" border="0">
  <tr class="menu">
    <td>
$menuString
    </td>
  </tr>
  <tr class="menu">
    <td>
      <table border="0" width="100%" cellpadding="0" cellspacing="0">
        <tr>
          <td align="center" width="50%" valign="bottom">$userEquals&nbsp;<span class="menuHilight">$userName&nbsp;(</span>$uname<span class="menuHilight">)</span></td>
          <td align="center" width="50%" valign="bottom">$companyEquals&nbsp;<span class="menuHilight">$companyName&nbsp;(</span>$companyCode<span class="menuHilight">)</span></td>
        </tr>
      </table>
    </td>
  </tr>
  <tr class="menu">
    <td>
      <table border="0" width="100%" cellpadding="0" cellspacing="0">
        <tr>
          <td align="center" width="15%"><a class="menu" href="$self->{configObj}->{hostingURL}" target="host" title="$self->{configObj}->{hostingAltTag}"><img src="$self->{configObj}->{hostingLogoUrl}" border="0" alt="$self->{configObj}->{hostingAltTag}"></a></td>
          <td align="center" width="70%">
            <!-- Dynamic Content From Applications goes here -->
            <table border="0" width="100%" cellpadding="0" cellspacing="0">
END_OF_CODE
    # gather the applications that want to do dynamic content for this user and give them a row each.
    my $arguments = $self->arguments();
    my $dynContentObj = undef;
    eval "\$dynContentObj = Portal::DynamicContent->new($arguments, portalSession => \$session);";
    if ($@)
    {
      my $string = $self->langObj->map("stateEvalFailed");
      $self->error(sprintf($string, $self->formEncodeString("Portal::DynamicContent"), $@));
      return undef;
    }
    if ($dynContentObj->error)
    {
      $self->error($dynContentObj->errorMessage);
      return undef;
    }

    my $dynContentString = $dynContentObj->display(callingApp => "Portal", tag => "desktop", indent => 14,
                                                   prefName => "dynamicContent", prefModule => "Desktop");
    if ($dynContentObj->error)
    {
      $self->error($dynContentObj->errorMessage);
      return undef;
    }
    $doc->print($dynContentString);
    $doc->print(<<"END_OF_CODE");
            </table>
          </td>
          <td align="center" width="15%"><a class="menu" href="$companyObj->{url}" target="company" title="$companyObj->{name}"><img src="$companyObj->{logoUrl}" border="0" alt="$companyObj->{name}"></a></td>
        </tr>
      </table>
    </td>
  </tr>
  <tr>
    <td>
      <br>
    </td>
  </tr>
  <tr>
    <!-- Application groups go here. -->
END_OF_CODE

    my @appGroups = ();
    my $displayAllGroups = $self->displayAppsPreference(portalSession => $session);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }

    foreach my $appGroup (keys %{$self->{variables}->{appTypes}})
    {
      if ($appGroup eq "administration")
      {
        if ($session->{store}->{userObj}->{admin})  # only process the administration group if we are an admin.
        {
          if (exists $apps{$appGroup})  # applications for this group.
          {
            push @appGroups, $appGroup;
          }
          elsif ($displayAllGroups)
          {
            push @appGroups, $appGroup;
          }
        }
      }
      else
      {
        if (exists $apps{$appGroup})  # applications for this group.
        {
          push @appGroups, $appGroup;
        }
        elsif ($displayAllGroups)
        {
          push @appGroups, $appGroup;
        }
      }
    }
    my $numAppTypes = scalar @appGroups;

    my $colwidth = 2;  # show 2 columns per row.
    $numAppTypes += ($numAppTypes % $colwidth > 0 ? ($colwidth - ($numAppTypes % $colwidth)) : 0);  # calculate the total number of apps needing to be displayed.

    # calculate the number of rows we are going to be displaying.
    my $cellwidth = (100 / $colwidth);  # calculate the percentage that each cell should take up.

    # start the master table display
    $doc->print(<<"END_OF_CODE");
    <td>
      <table border="0" cellpadding="0" cellspacing="0" width="100%">
        <tr>
END_OF_CODE

    # ************************************************************************
    # display the visible groups
    my $currType = 0;
    foreach my $type (sort @appGroups)
    {
      $currType++;  # indicate we've displayed another one.
      my $typeName = $self->{variables}->{appTypes}->{$type};
      $doc->print(<<"END_OF_CODE");
          <td align="left" width="$cellwidth\%" valign="top">
            <table border="0" cellpadding="4" cellspacing="0" width="100%">
              <tr>
                <td>
                  <table border="0" cellpadding="5" cellspacing="0" width="100%" class="cell">
                    <tr class="cellTitle">
                      <th class="cellTitle" align="center">$typeName</th>
                    </tr>
                    <tr>
                      <td align="center" valign="middle">
END_OF_CODE
      # display the programs or "none installed" if none found.
      if (exists $apps{$type})  # applications for this group.
      {
        # create another table that displays the apps (3) wide.
        my $appColWidth = 4;
        my $appCellWidth = (100 / $appColWidth);
        my %typeApps = %{$apps{$type}};
        my $numApps = int (scalar keys %typeApps);
        $numApps += ($numApps % $appColWidth > 0 ? ($appColWidth - ($numApps % $appColWidth)) : 0);
        my $currApp = 0; # keep track of # of apps displayed sofar.

        $doc->print(<<"END_OF_CODE");
                        <table border="0" cellpadding="0" cellspacing="0" width="100%">
                          <tr>
END_OF_CODE

        foreach my $app (sort keys %typeApps)
        {
          my $appObj = $typeApps{$app};
          my $appUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $appObj), arguments => { app => $appObj->name, state => "Main", command => "display" });
          my $width = $appObj->{width};
          my $height = $appObj->{height};
          my $description = $appObj->{description};
          my $icon = $self->{methods}->createBaseURL(type => "App", linkType => "image", appConfigObj => $appObj, appName => $appObj->{name}) . $appObj->{iconName};
          my $iconWidth = 40;
          my $iconHeight = 40;
          my $windowName = $app . "_" . $session->{store}->{windowTitleTimestamp};

          # display the cell with this app in it.
          $doc->print(<<"END_OF_CODE");
                            <td align="center" width="$appCellWidth\%" valign="top">
                              <a class="icon" href="$appUrl" onClick="javascript:launchApp('$windowName', '$appUrl', $height, $width);return false;" onMouseOver="window.status='$runPhrase $app'; return true;" title="$runPhrase $app"><img src="$icon" border=0 alt="$description" width="$iconWidth" height="$iconHeight"><br><br>$app</a>
                            </td>
END_OF_CODE
          $currApp++;
          if ($currApp % $appColWidth == 0)
          {
            # we need to close the row and start a new one (if necessary).
            $doc->print(<<"END_OF_CODE");
                          </tr>
END_OF_CODE
            if ($currApp < $numApps) # we still have more to display
            {  # open the next row
            $doc->print(<<"END_OF_CODE");
                          <tr>
END_OF_CODE
            }
          }
        }

        # now make sure the last row is completed off if needed.
        my $leftover = ($numApps - $currApp);
        if ($leftover > 0)
        {
          for (my $i=0; $i < $leftover; $i++)
          {
            $doc->print(<<"END_OF_CODE");
                            <td align="center" width="$appCellWidth\%" valign="top">
                              &nbsp;
                            </td>
END_OF_CODE
          }
        }

        $doc->print(<<"END_OF_CODE");  # close off the grouping table.
                          </tr>
                        </table>
END_OF_CODE
      }
      else  # no Applications for this group.
      {
        $doc->print(<<"END_OF_CODE");
                        <br>
                        <br>
                        none installed
                        <br>&nbsp;
END_OF_CODE
      }

      $doc->print(<<"END_OF_CODE");
                      </td>
                    </tr>
                  </table>
                </td>
              </tr>
            </table>
          </td>
END_OF_CODE
      if ($currType % $colwidth == 0)
      {
        #close out the row and start the next one if necessary.
        $doc->print(<<"END_OF_CODE");
        </tr>
END_OF_CODE
        if ($currType < $numAppTypes)
        {
        $doc->print(<<"END_OF_CODE");
        <tr>
          <td>
            <br>
          </td>
        </tr>
        <tr>
END_OF_CODE
        }
      }
    }

    # now make sure the last row is completed off if needed.
    my $leftover = ($numAppTypes - $currType);
    if ($leftover > 0)
    {
      for (my $i=0; $i < $leftover; $i++)
      {
        $doc->print(<<"END_OF_CODE");
          <td align="center" width="$cellwidth\%" valign="top">
            &nbsp;
          </td>
END_OF_CODE
      }
      $doc->print(<<"END_OF_CODE");
        </tr>
END_OF_CODE
    }
    # end the master table display
    $doc->print(<<"END_OF_CODE");
      </table>
    </td>
  </tr>
</table>
END_OF_CODE

    # see if we need to launch any of the apps.
    if ($self->startSession)
    {
      # flatten the hash of apps to work with (makes it easier).
      my @apps = $self->{applicationObj}->flattenAppHash(apps => \%apps);

      foreach my $appObj (@apps)
      {
        if ($appObj->{autorun})
        {
          my $name = $appObj->{name};
          my $appUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $appObj), arguments => { app => $appObj->name, state => "Main", command => "display" });
          my $height = $appObj->{height};
          my $width = $appObj->{width};
          my $appWindow = $name . "_" . $session->{store}->{windowTitleTimestamp};
          $onLoadCode .= "launchApp('$appWindow', '$appUrl', $height, $width);\n";
        }
      }
    }
    if (exists $self->{input}->{appToRun})
    {
      if (exists $self->{input}->{appToRunArgs})
      {
        # validate the appToRun exists and is assigned to this user.
        my $app = $self->{input}->{appToRun};
        if (length $app == 0)
        {
          $onLoadCode .= "window.alert('Error:  appToRun=\"$app\" is invalid!');\n";
        }
        else
        {
          # flatten the hash of apps to work with (makes it easier).
          my @apps = $self->{applicationObj}->flattenAppHash(apps => \%apps);
          my $found = 0;

          foreach my $appObj (@apps)
          {
            if ($appObj->{name} eq $app)
            {
              # validate we have state and command in the appToRunArgs variable.
              my $args = $self->{input}->{appToRunArgs};
              if ($args !~ /^(state=[^&]+&command=[^&]+(&[^&=]+=[^&]+)*)$/)
              {
                $onLoadCode .= "window.alert('Error:  appToRunArgs=\"$args\" is invalid!  appToRun=\"$app\".');\n";
              }
              else
              {
                my $appUrl = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $appObj), arguments => { app => $appObj->name }) . "&" . $args;
                my $height = $appObj->{height};
                my $width = $appObj->{width};
                my $appWindow = $app . "_" . $session->{store}->{windowTitleTimestamp};
                $onLoadCode .= "launchApp('$appWindow', '$appUrl', $height, $width);\n";
              }
              $found = 1;
              last;  # we can jump out now.
            }
          }
          if (!$found)
          {
            $onLoadCode .= "window.alert('Error:  appToRun=\"$app\" is not assigned to you!');\n";
          }
        }
      }
      else
      {
        $onLoadCode .= "window.alert('Error:  appToRunArgs must be specified!  appToRun=\"$self->{input}->{appToRun}\".');\n";
      }
    }
    if (length $onLoadCode > 0)
    {
      $doc->setOnload(code => "$onLoadCode");
    }
  }
  elsif ($command eq "logout")  # **************************** LOGOUT **********************************************************
  {
    $doc = $self->logout(session => $session);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
  }

  if (defined $doc)
  {
    $self->{methods}->setJavaScriptErrorInfo(doc => $doc, email => $self->{configObj}->{emailAddress}, appName => "Portal", appVersion => $VERSION);
  }

  $session = undef;

  return $doc;
}

# logout
# takes: error - error message to display, checkOK - 1 means check for okToLogout, 0 means skip the check., session
# returns doc
sub logout
{
  my $self = shift;
  my %args = ( checkOK => 1, error => "", session => undef, @_ );
  my $doc = HTMLObject::Normal->new;
  my $checkOK = $args{checkOK};
  my $session = $args{session};
  my $error = $args{error};

  # generate the document that will refresh back to the login screen while at the same time
  # setting the session cookies as being deleted.
  $doc->setCookie(name => "pcx_session_id", expires => "yesterday", domain => $self->{configObj}->{cookieDomain}, path => "/");

  my $refreshTime = (length $error > 0 ? 15 : 1);
  my $url = $self->{methods}->urlBuilder(doc => $doc, baseUrl => $self->{methods}->createBaseURL(type => "Portal", linkType => "cgi"), arguments => { app => "Portal", state => "Login", command => "display" });
  $doc->setMetaTag('http-equiv' => "Refresh", content => "$refreshTime; URL=$url");
  $doc->setTitle(sprintf($self->langObj->map("portal") . " - " . $self->langObj->map("logout"), $Portal::VERSION));

  # create the closeWindow JS function.
  $doc = $self->{methods}->displayJSHelper(doc => $doc, type => "closeWindow", langObj => $self->{langObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  # get the users colorScheme to work with.
  my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => $session->{store}->{userObj}->{id});
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return undef;
  }
  if (!defined $colorScheme)
  {
    $self->error("No valid colorScheme returned!<br>\n");
    return undef;
  }

  # create the needed css entries
  my %cssEntries = $self->{methods}->generateColorSchemeCSS(colorScheme => $colorScheme);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  $doc->print(%cssEntries);

  # get all the application sessions to cleanup. - All sessions must be stored in a central portalDB database.
  my $appSessions = $self->{methods}->portalSessionHandler(configObj => $self->{configObj}, portalDB => $self->{portalDB},
                                      sessionObj => $session);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  if ($appSessions->error)
  {
    $self->error($appSessions->errorMessage);
    return undef;
  }

  my $okToLogout = 1;  # assume we are ok to logout.

  # check the okToLogout flag set by each app to indicate if it is in the middle of doing something that it must finish first before it can close.
  if ($checkOK)
  {
    foreach my $app (keys %{$appSessions->{apps}})
    {
      if (!$appSessions->{apps}->{$app}->{store}->{okToLogout})
      {
        $okToLogout = 0;
      }
    }
  }

  if (!$okToLogout)
  {
    $doc = $self->run(command => "display");  # redisplay the desktop.
    if ($self->error)
    {
      $self->prefixError("command = '$self->{input}->{command}'");
      return undef;
    }
    return $doc;
  }

  # generate the body of the onLoad function call.
  my $onLoadCode = "";

  foreach my $app (keys %{$appSessions->{apps}})
  {
    my $appWindow = $session->generateAppWindowName($app);
    if ($session->error)
    {
      $self->error($session->errorMessage);
      return undef;
    }
    $onLoadCode .= "closeWindow('$appWindow');\n";

    # generate the JavaScript code to close all windows this app has registered.
    foreach my $window ($appSessions->{apps}->{$app}->getWindows())
    {
      $onLoadCode .= "closeWindow('$window');\n";
    }
    $appSessions->{apps}->{$app}->delete;  # delete the app session
    $appSessions->{apps}->{$app} = undef;  # force it out of scope.
  }

  # signal (in the case of an error) that we need the pcx_session_id cookie to be deleted.
  $self->{input}->{removeCookies} = 1;

  my $userId;
  $userId = $session->{store}->{userObj}->{id};
  my $sessionId;
  $sessionId = $session->{sessionId};

  # see if we have any Portal windows that need to be closed.
  foreach my $window ($session->getWindows())
  {
    $onLoadCode .= "closeWindow('$window');\n";
  }

  # delete the session from the database.
  $session->delete;

  # signal (in the case of an error) that we need the pcx_session_id cookie to be deleted.
  $self->{input}->{removeCookies} = 1;

  # remove the who_tb entry for this session.

  my $result;
  $result = $self->{whoObj}->deleteWhoEntry(session => $sessionId);
  if ($self->{whoObj}->error)
  {
    $self->error($self->{whoObj}->errorMessage);
    return undef;
  }
  if ($result == -1)
  {
    # do we care that the session no longer exists in the who_tb?  Not right now.
  }

  # make the logout log entry.
  my $logEntry;
  $logEntry = Portal::Objects::LogEntry->new(action => 1, ipAddress => (exists $ENV{REMOTE_ADDR} ? $ENV{REMOTE_ADDR} : '127.0.0.1'), userId => $userId, langObj => $self->{langObj});
  if ($logEntry->error)
  {
    $self->error($logEntry->errorMessage);
    return undef;
  }

  # log the logout event.
  $self->{logObj}->newEntry(logEntry => $logEntry);
  if ($self->{logObj}->error)
  {
    $self->error($self->{logObj}->errorMessage);
    return undef;
  }

  # at this point the user should be logged out of the system!

  if (length $onLoadCode > 0)
  {
    $doc->setOnload(code => "$onLoadCode");  # output the JavaScript code that will be run on the onLoad event of the page.
  }

  # set the JavaScript Error Handler Email Address
  $self->{methods}->setJavaScriptErrorInfo(doc => $doc, email => $self->{configObj}->{emailAddress}, appName => "Portal", appVersion => $VERSION);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }

  if (length $error > 0)
  {
    $doc->setFocus("body");
    $doc->print("<center><h1><span class=\"error\">" . $self->langObj->map("error") . ":&nbsp;</span></h1><span class=\"errorText\">$error</span></center>\n");
  }

  return $doc;
}

# looks up the Portal, Desktop, displayEmptyGroups preference and creates it if it doesn't exist.
# returns the preference value (1 or 0).
# takes: portalSession
sub displayAppsPreference
{
  my $self = shift;
  my %args = ( portalSession => undef, @_ );
  my $portalSession = $args{portalSession};

  if (! defined $portalSession)
  {
    $self->missing("portalSession");
    return undef;
  }

  my $callingApp = "Portal";
  my $prefModule = "Desktop";
  my $prefName = "displayEmptyAppGroups";
  my $result = undef;

  # check and see if this user has their preference set.
  my $userId = $portalSession->{store}->{userObj}->{id};
  my $preferenceObj = $self->{authObj}->getOrCreateUserPreference(userId => $userId,
                      app => $callingApp, module => $prefModule, preference => $prefName);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return $result;
  }
  if (!$preferenceObj->isValid)
  {
    $self->error($preferenceObj->errorMessage);
    return $result;
  }

  return $preferenceObj->value;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::PortalArgs(3)

=cut
